/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.model.application.validation;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.google.common.base.Joiner;
import com.yahoo.tensor.TensorType;
import com.yahoo.tensor.serialization.JsonFormat;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ConstantTensorJsonValidator {
    private static final String FIELD_CELLS = "cells";
    private static final String FIELD_ADDRESS = "address";
    private static final String FIELD_VALUE = "value";
    private static final String FIELD_VALUES = "values";
    private static final String FIELD_BLOCKS = "blocks";
    private static final String FIELD_TYPE = "type";
    private static final JsonFactory jsonFactory = new JsonFactory();
    private JsonParser parser;
    private final TensorType tensorType;
    private final Map<String, TensorType.Dimension> tensorDimensions = new HashMap<String, TensorType.Dimension>();
    private final List<String> denseDims = new ArrayList<String>();
    private final List<String> mappedDims = new ArrayList<String>();
    private int numIndexedDims = 0;
    private int numMappedDims = 0;
    private boolean seenCells = false;
    private boolean seenValues = false;
    private boolean seenBlocks = false;
    private boolean seenType = false;
    private boolean seenSimpleMapValue = false;

    private boolean isScalar() {
        return this.numIndexedDims == 0 && this.numMappedDims == 0;
    }

    private boolean isDense() {
        return this.numIndexedDims > 0 && this.numMappedDims == 0;
    }

    private boolean isSparse() {
        return this.numIndexedDims == 0 && this.numMappedDims > 0;
    }

    private boolean isSingleDense() {
        return this.numIndexedDims == 1 && this.numMappedDims == 0;
    }

    private boolean isSingleSparse() {
        return this.numIndexedDims == 0 && this.numMappedDims == 1;
    }

    private boolean isMixed() {
        return this.numIndexedDims > 0 && this.numMappedDims > 0;
    }

    public ConstantTensorJsonValidator(TensorType type) {
        this.tensorType = type;
        for (TensorType.Dimension dim : type.dimensions()) {
            this.tensorDimensions.put(dim.name(), dim);
            switch (dim.type()) {
                case mapped: {
                    ++this.numMappedDims;
                    this.mappedDims.add(dim.name());
                    break;
                }
                case indexedBound: 
                case indexedUnbound: {
                    ++this.numIndexedDims;
                    this.denseDims.add(dim.name());
                }
            }
        }
    }

    public void validate(String fileName, Reader tensorData) {
        if (fileName.endsWith(".json")) {
            this.validateTensor(tensorData);
        } else if (!fileName.endsWith(".json.lz4") && !fileName.endsWith(".tbf")) {
            throw new IllegalArgumentException("Ranking constant file names must end with either '.json' or '.json.lz4'");
        }
    }

    private void validateTensor(Reader tensorData) {
        try {
            this.parser = jsonFactory.createParser(tensorData);
            JsonToken top = this.parser.nextToken();
            if (top == JsonToken.START_ARRAY && this.isDense()) {
                this.consumeValuesArray();
                return;
            }
            if (top == JsonToken.START_OBJECT) {
                this.consumeTopObject();
                return;
            }
            if (this.isScalar()) {
                throw new InvalidConstantTensorException(this.parser, String.format("Invalid type %s: Only tensors with dimensions can be stored as file constants", this.tensorType.toString()));
            }
            throw new InvalidConstantTensorException(this.parser, String.format("Unexpected first token '%s' for constant with type %s", this.parser.getText(), this.tensorType.toString()));
        }
        catch (IOException e) {
            if (this.parser != null) {
                throw new InvalidConstantTensorException(this.parser, e);
            }
            throw new InvalidConstantTensorException(e);
        }
    }

    private void consumeValuesArray() throws IOException {
        this.consumeValuesNesting(0);
    }

    private void consumeTopObject() throws IOException {
        JsonToken cur = this.parser.nextToken();
        while (cur != JsonToken.END_OBJECT) {
            String fieldName;
            this.assertCurrentTokenIs(JsonToken.FIELD_NAME);
            switch (fieldName = this.parser.currentName()) {
                case "type": {
                    this.consumeTypeField();
                    break;
                }
                case "values": {
                    this.consumeValuesField();
                    break;
                }
                case "cells": {
                    this.consumeCellsField();
                    break;
                }
                case "blocks": {
                    this.consumeBlocksField();
                    break;
                }
                default: {
                    this.consumeAnyField(fieldName, this.parser.nextToken());
                }
            }
            cur = this.parser.nextToken();
        }
        if (this.seenSimpleMapValue) {
            if (!this.isSingleSparse()) {
                throw new InvalidConstantTensorException(this.parser, String.format("Cannot use {label: value} format for constant of type %s", this.tensorType.toString()));
            }
            if (this.seenCells || this.seenValues || this.seenBlocks || this.seenType) {
                throw new InvalidConstantTensorException(this.parser, String.format("Cannot use {label: value} format together with '%s'", this.seenCells ? FIELD_CELLS : (this.seenValues ? FIELD_VALUES : (this.seenBlocks ? FIELD_BLOCKS : FIELD_TYPE))));
            }
        }
        if (this.seenCells && (this.seenValues || this.seenBlocks)) {
            throw new InvalidConstantTensorException(this.parser, String.format("Cannot use both '%s' and '%s' at the same time", FIELD_CELLS, this.seenValues ? FIELD_VALUES : FIELD_BLOCKS));
        }
        if (this.seenValues && this.seenBlocks) {
            throw new InvalidConstantTensorException(this.parser, String.format("Cannot use both '%s' and '%s' at the same time", FIELD_VALUES, FIELD_BLOCKS));
        }
    }

    private void consumeCellsField() throws IOException {
        JsonToken cur = this.parser.nextToken();
        if (cur == JsonToken.START_ARRAY) {
            this.consumeLiteralFormArray();
            this.seenCells = true;
        } else if (cur == JsonToken.START_OBJECT) {
            this.consumeSimpleMappedObject();
            this.seenCells = true;
        } else {
            this.consumeAnyField(FIELD_BLOCKS, cur);
        }
    }

    private void consumeLiteralFormArray() throws IOException {
        while (this.parser.nextToken() != JsonToken.END_ARRAY) {
            this.validateLiteralFormCell();
        }
    }

    private void consumeSimpleMappedObject() throws IOException {
        if (!this.isSingleSparse()) {
            throw new InvalidConstantTensorException(this.parser, String.format("Cannot use {label: value} format for constant of type %s", this.tensorType.toString()));
        }
        JsonToken cur = this.parser.nextToken();
        while (cur != JsonToken.END_OBJECT) {
            this.assertCurrentTokenIs(JsonToken.FIELD_NAME);
            this.validateNumeric(this.parser.currentName(), this.parser.nextToken());
            cur = this.parser.nextToken();
        }
    }

    private void validateLiteralFormCell() throws IOException {
        this.assertCurrentTokenIs(JsonToken.START_OBJECT);
        boolean seenAddress = false;
        boolean seenValue = false;
        block8: for (int i = 0; i < 2; ++i) {
            String fieldName;
            this.assertNextTokenIs(JsonToken.FIELD_NAME);
            switch (fieldName = this.parser.currentName()) {
                case "address": {
                    this.validateTensorAddress(new HashSet<String>(this.tensorDimensions.keySet()));
                    seenAddress = true;
                    continue block8;
                }
                case "value": {
                    this.validateNumeric(FIELD_VALUE, this.parser.nextToken());
                    seenValue = true;
                    continue block8;
                }
                default: {
                    throw new InvalidConstantTensorException(this.parser, String.format("Only '%s' or '%s' fields are permitted within a cell object", FIELD_ADDRESS, FIELD_VALUE));
                }
            }
        }
        if (!seenAddress) {
            throw new InvalidConstantTensorException(this.parser, String.format("Missing '%s' field in cell object", FIELD_ADDRESS));
        }
        if (!seenValue) {
            throw new InvalidConstantTensorException(this.parser, String.format("Missing '%s' field in cell object", FIELD_VALUE));
        }
        this.assertNextTokenIs(JsonToken.END_OBJECT);
    }

    private void validateTensorAddress(Set<String> cellDimensions) throws IOException {
        this.assertNextTokenIs(JsonToken.START_OBJECT);
        while (this.parser.nextToken() != JsonToken.END_OBJECT) {
            this.assertCurrentTokenIs(JsonToken.FIELD_NAME);
            String dimensionName = this.parser.currentName();
            TensorType.Dimension dimension = this.tensorDimensions.get(dimensionName);
            if (dimension == null) {
                throw new InvalidConstantTensorException(this.parser, String.format("Tensor dimension '%s' does not exist", dimensionName));
            }
            if (!cellDimensions.contains(dimensionName)) {
                throw new InvalidConstantTensorException(this.parser, String.format("Duplicate tensor dimension '%s'", dimensionName));
            }
            cellDimensions.remove(dimensionName);
            this.validateLabel(dimension);
        }
        if (!cellDimensions.isEmpty()) {
            throw new InvalidConstantTensorException(this.parser, String.format("Tensor address missing dimension(s) %s", Joiner.on((String)", ").join(cellDimensions)));
        }
    }

    private void validateLabel(TensorType.Dimension dimension) throws IOException {
        JsonToken token = this.parser.nextToken();
        if (token != JsonToken.VALUE_STRING) {
            throw new InvalidConstantTensorException(this.parser, String.format("Tensor label is not a string (%s)", token.toString()));
        }
        if (dimension instanceof TensorType.IndexedBoundDimension) {
            this.validateBoundIndex((TensorType.IndexedBoundDimension)dimension);
        } else if (dimension instanceof TensorType.IndexedUnboundDimension) {
            this.validateUnboundIndex(dimension);
        }
    }

    private void validateBoundIndex(TensorType.IndexedBoundDimension dimension) throws IOException {
        try {
            int value = Integer.parseInt(this.parser.getValueAsString());
            if (value >= ((Long)dimension.size().get()).intValue()) {
                throw new InvalidConstantTensorException(this.parser, String.format("Index %s not within limits of bound dimension '%s'", value, dimension.name()));
            }
        }
        catch (NumberFormatException e) {
            this.throwCoordinateIsNotInteger(this.parser.getValueAsString(), dimension.name());
        }
    }

    private void validateUnboundIndex(TensorType.Dimension dimension) throws IOException {
        try {
            Integer.parseInt(this.parser.getValueAsString());
        }
        catch (NumberFormatException e) {
            this.throwCoordinateIsNotInteger(this.parser.getValueAsString(), dimension.name());
        }
    }

    private void throwCoordinateIsNotInteger(String value, String dimensionName) {
        throw new InvalidConstantTensorException(this.parser, String.format("Index '%s' for dimension '%s' is not an integer", value, dimensionName));
    }

    private void validateNumeric(String where, JsonToken token) throws IOException {
        if (token == JsonToken.VALUE_NUMBER_FLOAT || token == JsonToken.VALUE_NUMBER_INT || token == JsonToken.VALUE_NULL) {
            return;
        }
        if (token == JsonToken.VALUE_STRING) {
            String input = this.parser.getValueAsString();
            try {
                double d = JsonFormat.decodeNumberString((String)input);
                return;
            }
            catch (NumberFormatException e) {
                throw new InvalidConstantTensorException(this.parser, String.format("Inside '%s': cell value '%s' is not a number", where, input));
            }
        }
        throw new InvalidConstantTensorException(this.parser, String.format("Inside '%s': cell value is not a number (%s)", where, token.toString()));
    }

    private void assertCurrentTokenIs(JsonToken wantedToken) {
        this.assertTokenIs(this.parser.currentToken(), wantedToken);
    }

    private void assertNextTokenIs(JsonToken wantedToken) throws IOException {
        this.assertTokenIs(this.parser.nextToken(), wantedToken);
    }

    private void assertTokenIs(JsonToken token, JsonToken wantedToken) {
        if (token != wantedToken) {
            throw new InvalidConstantTensorException(this.parser, String.format("Expected JSON token %s, but got %s", wantedToken.toString(), token.toString()));
        }
    }

    private void consumeValuesNesting(int level) throws IOException {
        Long sz;
        this.assertCurrentTokenIs(JsonToken.START_ARRAY);
        if (level >= this.denseDims.size()) {
            throw new InvalidConstantTensorException(this.parser, String.format("Too deep array nesting for constant with type %s", this.tensorType.toString()));
        }
        TensorType.Dimension dim = this.tensorDimensions.get(this.denseDims.get(level));
        long count = 0L;
        JsonToken cur = this.parser.nextToken();
        while (cur != JsonToken.END_ARRAY) {
            if (level + 1 == this.denseDims.size()) {
                this.validateNumeric(FIELD_VALUES, cur);
            } else if (cur == JsonToken.START_ARRAY) {
                this.consumeValuesNesting(level + 1);
            } else {
                throw new InvalidConstantTensorException(this.parser, String.format("Unexpected token %s '%s'", cur.toString(), this.parser.getText()));
            }
            ++count;
            cur = this.parser.nextToken();
        }
        if (dim.size().isPresent() && (sz = (Long)dim.size().get()) != count) {
            throw new InvalidConstantTensorException(this.parser, String.format("Dimension '%s' has size %d but array had %d values", dim.name(), sz, count));
        }
    }

    private void consumeTypeField() throws IOException {
        JsonToken cur = this.parser.nextToken();
        if (cur == JsonToken.VALUE_STRING) {
            this.seenType = true;
        } else if (this.isSingleSparse()) {
            this.validateNumeric(FIELD_TYPE, cur);
            this.seenSimpleMapValue = true;
        } else {
            throw new InvalidConstantTensorException(this.parser, String.format("Field '%s' should contain the tensor type as a string, got %s", FIELD_TYPE, this.parser.getText()));
        }
    }

    private void consumeValuesField() throws IOException {
        JsonToken cur = this.parser.nextToken();
        if (this.isDense() && cur == JsonToken.START_ARRAY) {
            this.consumeValuesArray();
            this.seenValues = true;
        } else {
            this.consumeAnyField(FIELD_VALUES, cur);
        }
    }

    private void consumeBlocksField() throws IOException {
        JsonToken cur = this.parser.nextToken();
        if (cur == JsonToken.START_ARRAY) {
            this.consumeBlocksArray();
            this.seenBlocks = true;
        } else if (cur == JsonToken.START_OBJECT) {
            this.consumeBlocksObject();
            this.seenBlocks = true;
        } else {
            this.consumeAnyField(FIELD_BLOCKS, cur);
        }
    }

    private void consumeAnyField(String fieldName, JsonToken cur) throws IOException {
        if (!this.isSingleSparse()) {
            throw new InvalidConstantTensorException(this.parser, String.format("Unexpected content '%s' for field '%s'", this.parser.getText(), fieldName));
        }
        this.validateNumeric(FIELD_CELLS, cur);
        this.seenSimpleMapValue = true;
    }

    private void consumeBlocksArray() throws IOException {
        if (!this.isMixed()) {
            throw new InvalidConstantTensorException(this.parser, String.format("Cannot use blocks format:[] for constant of type %s", this.tensorType.toString()));
        }
        while (this.parser.nextToken() != JsonToken.END_ARRAY) {
            this.assertCurrentTokenIs(JsonToken.START_OBJECT);
            boolean seenAddress = false;
            boolean seenValues = false;
            block9: for (int i = 0; i < 2; ++i) {
                String fieldName;
                this.assertNextTokenIs(JsonToken.FIELD_NAME);
                switch (fieldName = this.parser.currentName()) {
                    case "address": {
                        this.validateTensorAddress(new HashSet<String>(this.mappedDims));
                        seenAddress = true;
                        continue block9;
                    }
                    case "values": {
                        this.assertNextTokenIs(JsonToken.START_ARRAY);
                        this.consumeValuesArray();
                        seenValues = true;
                        continue block9;
                    }
                    default: {
                        throw new InvalidConstantTensorException(this.parser, String.format("Only '%s' or '%s' fields are permitted within a block object", FIELD_ADDRESS, FIELD_VALUES));
                    }
                }
            }
            if (!seenAddress) {
                throw new InvalidConstantTensorException(this.parser, String.format("Missing '%s' field in block object", FIELD_ADDRESS));
            }
            if (!seenValues) {
                throw new InvalidConstantTensorException(this.parser, String.format("Missing '%s' field in block object", FIELD_VALUES));
            }
            this.assertNextTokenIs(JsonToken.END_OBJECT);
        }
    }

    private void consumeBlocksObject() throws IOException {
        if (this.numMappedDims > 1 || !this.isMixed()) {
            throw new InvalidConstantTensorException(this.parser, String.format("Cannot use blocks:{} format for constant of type %s", this.tensorType.toString()));
        }
        while (this.parser.nextToken() != JsonToken.END_OBJECT) {
            this.assertCurrentTokenIs(JsonToken.FIELD_NAME);
            this.assertNextTokenIs(JsonToken.START_ARRAY);
            this.consumeValuesArray();
        }
    }

    static class InvalidConstantTensorException
    extends IllegalArgumentException {
        InvalidConstantTensorException(JsonParser parser, String message) {
            super(message + " " + parser.currentLocation().toString());
        }

        InvalidConstantTensorException(JsonParser parser, Exception base) {
            super("Failed to parse JSON stream " + parser.currentLocation().toString(), base);
        }

        InvalidConstantTensorException(IOException base) {
            super("Failed to parse JSON stream: " + base.getMessage(), base);
        }
    }
}

