package com.manageengine.apminsight.agent.api;

import com.manageengine.apminsight.agent.JavaAgentAPIHandler;
import com.manageengine.apminsight.agent.trackers.Tracker;

/**
 * Allows user to monitor a block of code in their application. <br/>
 * This can be viewed in the trace details of the current transaction.
 * 
 * @since 3.5.0
 * @author adithyan-2116
 */
public final class CustomTracker
{
	private Tracker tracker;
	
	/**
	 * Marks beginning of block of code to be monitored.
	 * 
	 * @param name - To be used in trace details, for easy identification. Only A-Za-z _0-9 characters are allowed. Max length is 64.
	 * 
	 * @since 3.6.0
	 */
	public CustomTracker(String name)
	{
		this(name, null);
	}
	
	/**
	 * Marks beginning of block of code to be monitored.
	 * 
	 * @param name - To be used in trace details, for easy identification. Only A-Za-z _0-9 characters are allowed. Max length is 64.
	 * @param componentName - Creates a new component for the block of code being monitored. Only A-Za-z _0-9 characters are allowed. Max length is 32.
	 * 
	 * @since 3.6.0
	 */
	public CustomTracker(String name, String componentName)
	{
		try {
			tracker = JavaAgentAPIHandler.startTracker(name, componentName);
		}
		catch (Throwable th) {
			// do nothing - Only when JavaAgentAPIHandler class is not available(no agent or lower version), it would reach here
		}
	}
	
	/**
	 * Marks end of block to be monitored.
	 * 
	 * @since 3.6.0
	 */
	public void stop()
	{
		stop(null);
	}
	
	/**
	 * Marks end of block to be monitored and associate the exception with this block
	 * 
	 * @param throwable - Exception object to be associated with this block
	 * 
	 * @since 3.6.0
	 */
	public void stop(Throwable throwable)
	{
		try {
			JavaAgentAPIHandler.stopTracker(tracker, throwable);
			tracker = null; // Helping GC + preventing multiple stop() invocations.
		} 
		catch (Throwable th) {
			// do nothing
		}
	}
	
	/**
	 * Associates the exception to the currently running transaction. Accepts custom exception message.
	 * 
	 * @param message - Custom error message
	 * @param throwable - Exception object to be tracked
	 * 
	 * @since 3.5.0
	 */
	public static void trackException(String message, Throwable throwable)
	{
		try {
			JavaAgentAPIHandler.handleException(System.currentTimeMillis(), message, throwable);
		}
		catch(Throwable th) {
			// do nothing
		}
	}
	
	/**
	 * Associates the exception to the currently running transaction
	 * 
	 * @param throwable - Exception object to be tracked
	 * 
	 * @since 3.5.0
	 */
	public static void trackException(Throwable throwable)
	{
		if (throwable != null) {
			trackException(throwable.getMessage(), throwable);
		}
	}
	
	/**
	 * Allows you to send additional parameters for the current transaction.<br/>
	 * These parameters can be viewed under trace of the specific transaction.
	 * 
	 * @param key - Name of the parameter
	 * @param value - Value for the parameter
	 * 
	 * @since 4.7.0
	 */
	public static void addParameter(String key, Object value)
	{
		try {
			JavaAgentAPIHandler.addParameter(key, value);
		}
		catch(Throwable th) {
			// do nothing
		}
	}
	
	/**
	 * Replaces or sets the transaction name (both web & backgroud)
	 * 
	 * @param name - Custom name for the current transaction
	 * 
	 * @since 4.7.0
	 */
	public static void setTransactionName(String name)
	{
		try {
			JavaAgentAPIHandler.setTransactionName(name);
		}
		catch(Throwable th) {
			// do nothing
		}
	}
}