/*
 * Decompiled with CFR 0.152.
 */
package com.rabbitmq.client.impl;

import java.security.cert.Certificate;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.net.ssl.SSLSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TlsUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(TlsUtils.class);
    private static final List<String> KEY_USAGE = Collections.unmodifiableList(Arrays.asList("digitalSignature", "nonRepudiation", "keyEncipherment", "dataEncipherment", "keyAgreement", "keyCertSign", "cRLSign", "encipherOnly", "decipherOnly"));
    private static final Map<String, String> EXTENDED_KEY_USAGE = Collections.unmodifiableMap(new HashMap<String, String>(){
        {
            this.put("1.3.6.1.5.5.7.3.1", "TLS Web server authentication");
            this.put("1.3.6.1.5.5.7.3.2", "TLS Web client authentication");
            this.put("1.3.6.1.5.5.7.3.3", "Signing of downloadable executable code");
            this.put("1.3.6.1.5.5.7.3.4", "E-mail protection");
            this.put("1.3.6.1.5.5.7.3.8", "Binding the hash of an object to a time from an agreed-upon time");
        }
    });
    private static String PARSING_ERROR = "<parsing-error>";
    private static final Map<String, ExtensionStringConverter> EXTENSIONS = Collections.unmodifiableMap(new HashMap<String, ExtensionStringConverter>(){
        {
            this.put("2.5.29.14", new ExtensionStringConverter(){

                @Override
                public String convert(byte[] derOctetString, X509Certificate certificate) {
                    return "SubjectKeyIdentifier = " + TlsUtils.octetStringHexDump(derOctetString);
                }
            });
            this.put("2.5.29.15", new ExtensionStringConverter(){

                @Override
                public String convert(byte[] derOctetString, X509Certificate certificate) {
                    return "KeyUsage = " + TlsUtils.keyUsageBitString(certificate.getKeyUsage(), derOctetString);
                }
            });
            this.put("2.5.29.16", new HexDumpConverter("PrivateKeyUsage"));
            this.put("2.5.29.17", new ExtensionStringConverter(){

                @Override
                public String convert(byte[] derOctetString, X509Certificate certificate) {
                    try {
                        return "SubjectAlternativeName = " + TlsUtils.sans(certificate, "/");
                    }
                    catch (CertificateParsingException e) {
                        return "SubjectAlternativeName = " + PARSING_ERROR;
                    }
                }
            });
            this.put("2.5.29.18", new HexDumpConverter("IssuerAlternativeName"));
            this.put("2.5.29.19", new ExtensionStringConverter(){

                @Override
                public String convert(byte[] derOctetString, X509Certificate certificate) {
                    return "BasicConstraints = " + TlsUtils.basicConstraints(derOctetString);
                }
            });
            this.put("2.5.29.30", new HexDumpConverter("NameConstraints"));
            this.put("2.5.29.33", new HexDumpConverter("PolicyMappings"));
            this.put("2.5.29.35", new ExtensionStringConverter(){

                @Override
                public String convert(byte[] derOctetString, X509Certificate certificate) {
                    return "AuthorityKeyIdentifier = " + TlsUtils.authorityKeyIdentifier(derOctetString);
                }
            });
            this.put("2.5.29.36", new HexDumpConverter("PolicyConstraints"));
            this.put("2.5.29.37", new ExtensionStringConverter(){

                @Override
                public String convert(byte[] derOctetString, X509Certificate certificate) {
                    return "ExtendedKeyUsage = " + TlsUtils.extendedKeyUsage(derOctetString, certificate);
                }
            });
        }
    });

    public static void logPeerCertificateInfo(SSLSession session) {
        if (LOGGER.isDebugEnabled()) {
            try {
                Certificate[] peerCertificates = session.getPeerCertificates();
                if (peerCertificates != null && peerCertificates.length > 0) {
                    LOGGER.debug(TlsUtils.peerCertificateInfo(peerCertificates[0], "Peer's leaf certificate"));
                    for (int i = 1; i < peerCertificates.length; ++i) {
                        LOGGER.debug(TlsUtils.peerCertificateInfo(peerCertificates[i], "Peer's certificate chain entry"));
                    }
                }
            }
            catch (Exception e) {
                LOGGER.debug("Error while logging peer certificate info: {}", (Object)e.getMessage());
            }
        }
    }

    public static String peerCertificateInfo(Certificate certificate, String prefix) {
        X509Certificate c = (X509Certificate)certificate;
        try {
            return String.format("%s subject: %s, subject alternative names: %s, issuer: %s, not valid after: %s, X.509 usage extensions: %s", TlsUtils.stripCRLF(prefix), TlsUtils.stripCRLF(c.getSubjectDN().getName()), TlsUtils.stripCRLF(TlsUtils.sans(c, ",")), TlsUtils.stripCRLF(c.getIssuerDN().getName()), c.getNotAfter(), TlsUtils.stripCRLF(TlsUtils.extensions(c)));
        }
        catch (Exception e) {
            return "Error while retrieving " + prefix + " certificate information";
        }
    }

    private static String sans(X509Certificate c, String separator) throws CertificateParsingException {
        ArrayList sans = c.getSubjectAlternativeNames();
        sans = sans == null ? new ArrayList() : sans;
        return TlsUtils.join(separator, new ArrayList(sans));
    }

    public static String extensionPrettyPrint(String oid, byte[] derOctetString, X509Certificate certificate) {
        try {
            ExtensionStringConverter converter = EXTENSIONS.get(oid);
            if (converter == null) {
                converter = new HexDumpConverter(oid);
            }
            return converter.convert(derOctetString, certificate);
        }
        catch (Exception e) {
            return oid + " = " + PARSING_ERROR;
        }
    }

    public static String stripCRLF(String value) {
        return value.replaceAll("\r", "").replaceAll("\n", "");
    }

    private static String extensions(X509Certificate certificate) {
        ArrayList<String> extensions = new ArrayList<String>();
        for (String oid : certificate.getCriticalExtensionOIDs()) {
            extensions.add(TlsUtils.extensionPrettyPrint(oid, certificate.getExtensionValue(oid), certificate) + " (critical)");
        }
        for (String oid : certificate.getNonCriticalExtensionOIDs()) {
            extensions.add(TlsUtils.extensionPrettyPrint(oid, certificate.getExtensionValue(oid), certificate) + " (non-critical)");
        }
        return TlsUtils.join(", ", extensions);
    }

    private static String octetStringHexDump(byte[] derOctetString) {
        if (derOctetString.length > 4 && derOctetString[0] == 4 && derOctetString[2] == 4) {
            return TlsUtils.hexDump(4, derOctetString);
        }
        return TlsUtils.hexDump(0, derOctetString);
    }

    private static String hexDump(int start, byte[] derOctetString) {
        ArrayList<String> hexs = new ArrayList<String>();
        for (int i = start; i < derOctetString.length; ++i) {
            hexs.add(String.format("%02X", derOctetString[i]));
        }
        return TlsUtils.join(":", hexs);
    }

    private static String keyUsageBitString(boolean[] keyUsage, byte[] derOctetString) {
        if (keyUsage != null) {
            ArrayList<String> usage = new ArrayList<String>();
            for (int i = 0; i < keyUsage.length; ++i) {
                if (!keyUsage[i]) continue;
                usage.add(KEY_USAGE.get(i));
            }
            return TlsUtils.join("/", usage);
        }
        return TlsUtils.hexDump(0, derOctetString);
    }

    private static String basicConstraints(byte[] derOctetString) {
        if (derOctetString.length == 4 && derOctetString[3] == 0) {
            return "CA:FALSE";
        }
        if (derOctetString.length >= 7 && derOctetString[2] == 48 && derOctetString[4] == 1) {
            return "CA:" + (derOctetString[6] == 0 ? "FALSE" : "TRUE");
        }
        return TlsUtils.hexDump(0, derOctetString);
    }

    private static String authorityKeyIdentifier(byte[] derOctetString) {
        if (derOctetString.length == 26 && derOctetString[0] == 4) {
            return "keyid:" + TlsUtils.hexDump(6, derOctetString);
        }
        return TlsUtils.hexDump(0, derOctetString);
    }

    private static String extendedKeyUsage(byte[] derOctetString, X509Certificate certificate) {
        List<String> extendedKeyUsage = null;
        try {
            extendedKeyUsage = certificate.getExtendedKeyUsage();
            if (extendedKeyUsage == null) {
                return TlsUtils.hexDump(0, derOctetString);
            }
            ArrayList<String> extendedKeyUsageLiteral = new ArrayList<String>();
            for (String oid : extendedKeyUsage) {
                String literal = EXTENDED_KEY_USAGE.get(oid);
                extendedKeyUsageLiteral.add(literal == null ? oid : literal);
            }
            return TlsUtils.join("/", extendedKeyUsageLiteral);
        }
        catch (CertificateParsingException e) {
            return PARSING_ERROR;
        }
    }

    private static String join(String separator, List<?> items) {
        StringBuilder builder = new StringBuilder();
        if (items != null) {
            for (int i = 0; i < items.size(); ++i) {
                if (i > 0) {
                    builder.append(separator);
                }
                builder.append(items.get(i));
            }
        }
        return builder.toString();
    }

    private static class HexDumpConverter
    implements ExtensionStringConverter {
        private final String extension;

        private HexDumpConverter(String extension) {
            this.extension = extension;
        }

        @Override
        public String convert(byte[] derOctetString, X509Certificate certificate) {
            return this.extension + " = " + TlsUtils.hexDump(0, derOctetString);
        }
    }

    private static interface ExtensionStringConverter {
        public String convert(byte[] var1, X509Certificate var2);
    }
}

