///////////////////////////////////////////////////////////////////////////////////////////////
// checkstyle: Checks Java source code and other text files for adherence to a set of rules.
// Copyright (C) 2001-2023 the original author or authors.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///////////////////////////////////////////////////////////////////////////////////////////////

package com.puppycrawl.tools.checkstyle.checks.javadoc;

import com.puppycrawl.tools.checkstyle.StatelessCheck;
import com.puppycrawl.tools.checkstyle.api.AbstractCheck;
import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;
import com.puppycrawl.tools.checkstyle.utils.JavadocUtil;

/**
 * <p>
 * Checks that Javadocs are located at the correct position. As specified at
 * <a href="https://docs.oracle.com/en/java/javase/11/docs/specs/doc-comment-spec.html">
 * Documentation Comment Specification for the Standard Doclet</a>, Javadocs are recognized
 * only when placed immediately before module, package, class, interface,
 * constructor, method, or field declarations. Any other position, like in the
 * body of a method, will be ignored by the javadoc tool and is considered
 * invalid by this check.
 * </p>
 * <p>
 * To configure the check:
 * </p>
 * <pre>
 * &lt;module name="InvalidJavadocPosition"/&gt;
 * </pre>
 * <p>
 * The following code produces a violation because Javadocs should be before all annotations of
 * the Javadoc's target:
 * </p>
 * <pre>
 * &#064;SuppressWarnings("serial")
 * /**
 *  * This comment looks like javadoc but it at an invalid location.
 *  * Therefore, the text will not get into TestClass.html and the check will produce a violation.
 *  *&#47;
 * public class TestClass {
 * }
 * </pre>
 * <p>
 * Parent is {@code com.puppycrawl.tools.checkstyle.TreeWalker}
 * </p>
 * <p>
 * Violation Message Keys:
 * </p>
 * <ul>
 * <li>
 * {@code invalid.position}
 * </li>
 * </ul>
 *
 * @since 8.23
 */
@StatelessCheck
public class InvalidJavadocPositionCheck extends AbstractCheck {

    /**
     * A key is pointing to the warning message text in "messages.properties"
     * file.
     */
    public static final String MSG_KEY = "invalid.position";

    @Override
    public int[] getDefaultTokens() {
        return new int[] {
            TokenTypes.BLOCK_COMMENT_BEGIN,
        };
    }

    @Override
    public int[] getAcceptableTokens() {
        return getDefaultTokens();
    }

    @Override
    public int[] getRequiredTokens() {
        return getDefaultTokens();
    }

    @Override
    public boolean isCommentNodesRequired() {
        return true;
    }

    @Override
    public void visitToken(DetailAST ast) {
        final String commentContent = JavadocUtil.getBlockCommentContent(ast);

        if (JavadocUtil.isJavadocComment(commentContent)
                && !JavadocUtil.isCorrectJavadocPosition(ast)) {
            log(ast, MSG_KEY);
        }
    }

}
