/*
 * Decompiled with CFR 0.152.
 */
package com.palantir.dialogue.core;

import com.codahale.metrics.Meter;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Suppliers;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningScheduledExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.palantir.conjure.java.client.config.ClientConfiguration;
import com.palantir.dialogue.Endpoint;
import com.palantir.dialogue.EndpointChannel;
import com.palantir.dialogue.HttpMethod;
import com.palantir.dialogue.Request;
import com.palantir.dialogue.RequestBody;
import com.palantir.dialogue.Response;
import com.palantir.dialogue.core.Config;
import com.palantir.dialogue.core.DialogueClientMetrics;
import com.palantir.dialogue.core.DialogueExecutors;
import com.palantir.dialogue.core.Responses;
import com.palantir.dialogue.futures.DialogueFutures;
import com.palantir.logsafe.Arg;
import com.palantir.logsafe.SafeArg;
import com.palantir.logsafe.exceptions.SafeRuntimeException;
import com.palantir.logsafe.logger.SafeLogger;
import com.palantir.logsafe.logger.SafeLoggerFactory;
import com.palantir.tracing.DetachedSpan;
import com.palantir.tracing.TagTranslator;
import com.palantir.tracing.Tracers;
import com.palantir.tritium.metrics.MetricRegistries;
import com.palantir.tritium.metrics.registry.DefaultTaggedMetricRegistry;
import com.palantir.tritium.metrics.registry.SharedTaggedMetricRegistries;
import com.palantir.tritium.metrics.registry.TaggedMetricRegistry;
import java.io.IOException;
import java.io.OutputStream;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.time.Duration;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.DoubleSupplier;
import java.util.function.Function;
import java.util.function.Supplier;
import org.jspecify.annotations.Nullable;

final class RetryingChannel
implements EndpointChannel {
    private static final SafeLogger log = SafeLoggerFactory.get(RetryingChannel.class);
    private static final String SCHEDULER_NAME = "dialogue-RetryingChannel-scheduler";
    static final Supplier<ScheduledExecutorService> sharedScheduler = Suppliers.memoize(() -> DialogueExecutors.newSharedSingleThreadScheduler(MetricRegistries.instrument((TaggedMetricRegistry)SharedTaggedMetricRegistries.getSingleton(), (ThreadFactory)new ThreadFactoryBuilder().setNameFormat("dialogue-RetryingChannel-scheduler-%d").setDaemon(true).build(), (String)SCHEDULER_NAME)));
    private static final BiFunction<Endpoint, Response, Throwable> qosThrowable = (_endpoint, response) -> new SafeRuntimeException("Received retryable response", new Arg[]{SafeArg.of((String)"status", (Object)response.code())});
    private static final BiFunction<Endpoint, Response, Throwable> serverErrorThrowable = (endpoint, response) -> new SafeRuntimeException("Received server error, but http method is safe to retry", new Arg[]{SafeArg.of((String)"status", (Object)response.code()), SafeArg.of((String)"method", (Object)endpoint.httpMethod())});
    private final ListeningScheduledExecutorService scheduler;
    private final EndpointChannel delegate;
    private final Endpoint endpoint;
    private final String channelName;
    private final int maxRetries;
    private final ClientConfiguration.ServerQoS serverQoS;
    private final ClientConfiguration.RetryOnTimeout retryOnTimeout;
    private final Duration backoffSlotSize;
    private final DoubleSupplier jitter;
    private final Supplier<Meter> retryDueToServerError;
    private final Supplier<Meter> retryDueToQosResponse;
    private final Function<Throwable, Meter> retryDueToThrowable;

    static EndpointChannel create(Config cf, EndpointChannel channel, Endpoint endpoint) {
        ClientConfiguration clientConf = cf.clientConf();
        if (clientConf.maxNumRetries() == 0) {
            return channel;
        }
        return new RetryingChannel(channel, endpoint, cf.channelName(), clientConf.taggedMetricRegistry(), clientConf.maxNumRetries(), clientConf.backoffSlotSize(), clientConf.serverQoS(), clientConf.retryOnTimeout(), cf.scheduler(), cf.random()::nextDouble);
    }

    @VisibleForTesting
    RetryingChannel(EndpointChannel delegate, Endpoint endpoint, String channelName, int maxRetries, Duration backoffSlotSize, ClientConfiguration.ServerQoS serverQoS, ClientConfiguration.RetryOnTimeout retryOnTimeout) {
        this(delegate, endpoint, channelName, (TaggedMetricRegistry)new DefaultTaggedMetricRegistry(), maxRetries, backoffSlotSize, serverQoS, retryOnTimeout, sharedScheduler.get(), () -> ThreadLocalRandom.current().nextDouble());
    }

    private RetryingChannel(EndpointChannel delegate, Endpoint endpoint, String channelName, TaggedMetricRegistry metrics, int maxRetries, Duration backoffSlotSize, ClientConfiguration.ServerQoS serverQoS, ClientConfiguration.RetryOnTimeout retryOnTimeout, ScheduledExecutorService scheduler, DoubleSupplier jitter) {
        this.delegate = delegate;
        this.endpoint = endpoint;
        this.channelName = channelName;
        this.maxRetries = maxRetries;
        this.backoffSlotSize = backoffSlotSize;
        this.serverQoS = serverQoS;
        this.retryOnTimeout = retryOnTimeout;
        this.scheduler = RetryingChannel.instrument(scheduler, metrics);
        this.jitter = jitter;
        DialogueClientMetrics dialogueClientMetrics = DialogueClientMetrics.of(metrics);
        this.retryDueToServerError = Suppliers.memoize(() -> dialogueClientMetrics.requestRetry().channelName(channelName).reason("serverError").build());
        this.retryDueToQosResponse = Suppliers.memoize(() -> dialogueClientMetrics.requestRetry().channelName(channelName).reason("qosResponse").build());
        this.retryDueToThrowable = throwable -> dialogueClientMetrics.requestRetry().channelName(channelName).reason(throwable.getClass().getSimpleName()).build();
    }

    public ListenableFuture<Response> execute(Request request) {
        Optional<SafeRuntimeException> debugStacktrace = log.isDebugEnabled() ? Optional.of(new SafeRuntimeException("Exception for stacktrace", new Arg[0])) : Optional.empty();
        return new RetryingCallback(this.endpoint, request, debugStacktrace).execute();
    }

    public String toString() {
        return "RetryingChannel{maxRetries=" + this.maxRetries + ", serverQoS=" + String.valueOf(this.serverQoS) + " delegate=" + String.valueOf(this.delegate) + "}";
    }

    private static boolean isEtimedoutException(Throwable throwable) {
        return throwable != null && SocketException.class.equals(throwable.getClass()) && "Connection timed out".equals(throwable.getMessage());
    }

    private static Request trackNonRepeatableBodyConsumption(Request request) {
        if (request.body().isEmpty() || ((RequestBody)request.body().get()).repeatable()) {
            return request;
        }
        return Request.builder().from(request).body((RequestBody)new ConsumptionTrackingRequestBody((RequestBody)request.body().get())).build();
    }

    private static boolean safeToRetry(HttpMethod httpMethod) {
        return switch (httpMethod) {
            default -> throw new IncompatibleClassChangeError();
            case HttpMethod.GET, HttpMethod.HEAD, HttpMethod.OPTIONS, HttpMethod.PUT, HttpMethod.DELETE -> true;
            case HttpMethod.POST, HttpMethod.PATCH -> false;
        };
    }

    private static ListeningScheduledExecutorService instrument(ScheduledExecutorService delegate, TaggedMetricRegistry metrics) {
        return MoreExecutors.listeningDecorator((ScheduledExecutorService)Tracers.wrap((String)SCHEDULER_NAME, (ScheduledExecutorService)MetricRegistries.instrument((TaggedMetricRegistry)metrics, (ScheduledExecutorService)delegate, (String)SCHEDULER_NAME)));
    }

    private final class RetryingCallback {
        private final Endpoint endpoint;
        private final Request request;
        private final Optional<SafeRuntimeException> callsiteStacktrace;
        private final DetachedSpan span = DetachedSpan.start((String)"Dialogue-RetryingChannel");
        private int failures = 0;

        private RetryingCallback(Endpoint endpoint, Request request, Optional<SafeRuntimeException> callsiteStacktrace) {
            this.endpoint = endpoint;
            this.request = RetryingChannel.trackNonRepeatableBodyConsumption(request);
            this.callsiteStacktrace = callsiteStacktrace;
        }

        ListenableFuture<Response> execute() {
            ListenableFuture<Response> result = this.wrap((ListenableFuture<Response>)RetryingChannel.this.delegate.execute(this.request));
            result.addListener(() -> {
                if (this.failures > 0) {
                    this.span.complete((TagTranslator)RetryingCallbackTranslator.INSTANCE, (Object)this);
                }
            }, DialogueFutures.safeDirectExecutor());
            return result;
        }

        private boolean requestCanBeRetried() {
            if (this.request.body().isEmpty()) {
                return true;
            }
            RequestBody body = (RequestBody)this.request.body().get();
            if (body instanceof ConsumptionTrackingRequestBody) {
                ConsumptionTrackingRequestBody consumptionTrackingRequestBody = (ConsumptionTrackingRequestBody)body;
                return consumptionTrackingRequestBody.requestBodyCanBeRetried();
            }
            return body.repeatable();
        }

        private ListenableFuture<Response> wrap(ListenableFuture<Response> input) {
            ListenableFuture result = input;
            result = DialogueFutures.transformAsync(result, this::handleHttpResponse);
            result = DialogueFutures.catchingAllAsync((ListenableFuture)result, this::handleThrowable);
            return result;
        }

        private ListenableFuture<Response> handleHttpResponse(Response response) {
            boolean canRetryRequest = this.requestCanBeRetried();
            if (canRetryRequest && this.isRetryableQosStatus(response)) {
                return this.incrementFailuresAndMaybeRetry(response, qosThrowable, RetryingChannel.this.retryDueToQosResponse.get());
            }
            if (canRetryRequest && Responses.isInternalServerError(response) && RetryingChannel.safeToRetry(this.endpoint.httpMethod())) {
                return this.incrementFailuresAndMaybeRetry(response, serverErrorThrowable, RetryingChannel.this.retryDueToServerError.get());
            }
            return Futures.immediateFuture((Object)response);
        }

        private ListenableFuture<Response> handleThrowable(Throwable clientSideThrowable) {
            if (++this.failures <= RetryingChannel.this.maxRetries) {
                if (this.requestCanBeRetried() && this.shouldAttemptToRetry(clientSideThrowable)) {
                    this.callsiteStacktrace.ifPresent(clientSideThrowable::addSuppressed);
                    Meter retryReason = RetryingChannel.this.retryDueToThrowable.apply(clientSideThrowable);
                    long backoffNanoseconds = this.failures <= 1 ? 0L : this.getBackoffNanoseconds();
                    this.infoLogRetry(backoffNanoseconds, OptionalInt.empty(), clientSideThrowable);
                    return this.scheduleRetry(retryReason, backoffNanoseconds);
                }
                if (log.isDebugEnabled()) {
                    this.callsiteStacktrace.ifPresent(clientSideThrowable::addSuppressed);
                    if (log.isDebugEnabled()) {
                        log.debug("Not attempting to retry failure. channel: {}, service: {}, endpoint: {}", (Arg)SafeArg.of((String)"channelName", (Object)RetryingChannel.this.channelName), (Arg)SafeArg.of((String)"serviceName", (Object)this.endpoint.serviceName()), (Arg)SafeArg.of((String)"endpoint", (Object)this.endpoint.endpointName()), clientSideThrowable);
                    }
                }
            }
            return Futures.immediateFailedFuture((Throwable)clientSideThrowable);
        }

        private ListenableFuture<Response> incrementFailuresAndMaybeRetry(Response response, BiFunction<Endpoint, Response, Throwable> failureSupplier, Meter meter) {
            int proxyUpstreamRequestAttempts = Responses.getProxyUpstreamRequestAttempts(response);
            if (proxyUpstreamRequestAttempts > 1) {
                log.info("Received a Proxy-Upstream-Request-Attempts response header indicating retries have already occurred", (Arg)SafeArg.of((String)"proxyUpstreamRequestAttempts", (Object)proxyUpstreamRequestAttempts));
            }
            this.failures += Math.max(1, proxyUpstreamRequestAttempts);
            if (this.failures <= RetryingChannel.this.maxRetries) {
                response.close();
                Throwable throwableToLog = log.isTraceEnabled() ? failureSupplier.apply(this.endpoint, response) : null;
                long backoffNanos = Responses.isRetryOther(response) ? 0L : this.getBackoffNanoseconds();
                this.infoLogRetry(backoffNanos, OptionalInt.of(response.code()), throwableToLog);
                return this.scheduleRetry(meter, backoffNanos);
            }
            this.infoLogRetriesExhausted(response);
            return Futures.immediateFuture((Object)response);
        }

        private ListenableFuture<Response> scheduleRetry(Meter meter, long backoffNanoseconds) {
            meter.mark();
            if (backoffNanoseconds <= 0L) {
                return this.wrap((ListenableFuture<Response>)RetryingChannel.this.delegate.execute(this.request));
            }
            DetachedSpan backoffSpan = this.span.childDetachedSpan("retry-backoff");
            final SettableFuture responseFuture = SettableFuture.create();
            RetryingChannel.this.scheduler.schedule(() -> {
                backoffSpan.complete((TagTranslator)RetryingCallbackTranslator.INSTANCE, (Object)this);
                if (responseFuture.isDone()) {
                    return;
                }
                final ListenableFuture delegateResult = RetryingChannel.this.delegate.execute(this.request);
                DialogueFutures.addDirectCallback((ListenableFuture)delegateResult, (FutureCallback)new FutureCallback<Response>(){

                    public void onSuccess(@Nullable Response result) {
                        if (result != null && !responseFuture.set((Object)result)) {
                            result.close();
                        }
                    }

                    public void onFailure(Throwable throwable) {
                        if (delegateResult.isCancelled()) {
                            responseFuture.cancel(false);
                        } else if (!responseFuture.setException(throwable)) {
                            log.info("Response future completed before delegate threw", throwable);
                        }
                    }
                });
                DialogueFutures.addDirectListener((ListenableFuture)responseFuture, () -> {
                    if (responseFuture.isCancelled()) {
                        delegateResult.cancel(false);
                    }
                });
            }, backoffNanoseconds, TimeUnit.NANOSECONDS);
            return this.wrap((ListenableFuture<Response>)responseFuture);
        }

        private long getBackoffNanoseconds() {
            if (this.failures == 0) {
                return 0L;
            }
            int upperBound = (int)Math.pow(2.0, this.failures - 1);
            return Math.round((double)RetryingChannel.this.backoffSlotSize.toNanos() * RetryingChannel.this.jitter.getAsDouble() * (double)upperBound);
        }

        private boolean isRetryableQosStatus(Response response) {
            return switch (RetryingChannel.this.serverQoS) {
                default -> throw new IncompatibleClassChangeError();
                case ClientConfiguration.ServerQoS.AUTOMATIC_RETRY -> Responses.isRetryableQos(response);
                case ClientConfiguration.ServerQoS.PROPAGATE_429_and_503_TO_CALLER -> Responses.isQosStatus(response) && !Responses.isTooManyRequests(response) && !Responses.isUnavailable(response) && Responses.isRetryableQos(response);
            };
        }

        private boolean shouldAttemptToRetry(Throwable throwable) {
            if (RetryingChannel.this.retryOnTimeout == ClientConfiguration.RetryOnTimeout.DISABLED) {
                if (throwable instanceof SocketTimeoutException) {
                    SocketTimeoutException socketTimeout = (SocketTimeoutException)throwable;
                    return socketTimeout.getMessage() != null && socketTimeout.getMessage().contains("connect timed out");
                }
                if (RetryingChannel.isEtimedoutException(throwable)) {
                    return false;
                }
            }
            return throwable instanceof IOException;
        }

        private void infoLogRetriesExhausted(Response response) {
            if (log.isInfoEnabled()) {
                SafeRuntimeException stacktrace = this.callsiteStacktrace.orElse(null);
                log.info("Exhausted {} retries, returning last received response with status {}, channel: {}, service: {}, endpoint: {}", (Arg)SafeArg.of((String)"retries", (Object)RetryingChannel.this.maxRetries), (Arg)SafeArg.of((String)"status", (Object)response.code()), (Arg)SafeArg.of((String)"channelName", (Object)RetryingChannel.this.channelName), (Arg)SafeArg.of((String)"serviceName", (Object)this.endpoint.serviceName()), (Arg)SafeArg.of((String)"endpoint", (Object)this.endpoint.endpointName()), (Throwable)stacktrace);
            }
        }

        private void infoLogRetry(long backoffNanoseconds, OptionalInt responseStatus, @Nullable Throwable throwable) {
            if (log.isInfoEnabled()) {
                log.info("Retrying call after failure {}/{} backoff: {}, channel: {}, service: {}, endpoint: {}, status: {}", (Arg)SafeArg.of((String)"failures", (Object)this.failures), (Arg)SafeArg.of((String)"maxRetries", (Object)RetryingChannel.this.maxRetries), (Arg)SafeArg.of((String)"backoffMillis", (Object)TimeUnit.NANOSECONDS.toMillis(backoffNanoseconds)), (Arg)SafeArg.of((String)"channelName", (Object)RetryingChannel.this.channelName), (Arg)SafeArg.of((String)"serviceName", (Object)this.endpoint.serviceName()), (Arg)SafeArg.of((String)"endpoint", (Object)this.endpoint.endpointName()), (Arg)SafeArg.of((String)"status", responseStatus.isPresent() ? Integer.valueOf(responseStatus.getAsInt()) : null), throwable);
            }
        }

        private String channelName() {
            return RetryingChannel.this.channelName;
        }
    }

    private static final class ConsumptionTrackingRequestBody
    implements RequestBody {
        private final RequestBody delegate;
        private volatile boolean consumed;

        ConsumptionTrackingRequestBody(RequestBody delegate) {
            this.delegate = delegate;
        }

        public void writeTo(OutputStream output) throws IOException {
            this.consumed = true;
            this.delegate.writeTo(output);
        }

        public String contentType() {
            return this.delegate.contentType();
        }

        public OptionalLong contentLength() {
            return this.delegate.contentLength();
        }

        public boolean repeatable() {
            return this.delegate.repeatable();
        }

        public void close() {
            this.consumed = true;
            this.delegate.close();
        }

        boolean requestBodyCanBeRetried() {
            return this.repeatable() || !this.consumed;
        }

        public String toString() {
            return "ConsumptionTrackingRequestBody{" + String.valueOf(this.delegate) + "}";
        }
    }

    private static enum RetryingCallbackTranslator implements TagTranslator<RetryingCallback>
    {
        INSTANCE;


        public <T> void translate(TagTranslator.TagAdapter<T> sink, T target, RetryingCallback data) {
            sink.tag(target, "serviceName", data.endpoint.serviceName());
            sink.tag(target, "endpointName", data.endpoint.endpointName());
            sink.tag(target, "failures", Integer.toString(data.failures));
            sink.tag(target, "channel", data.channelName());
        }
    }
}

