/*
 * Decompiled with CFR 0.152.
 */
package com.palantir.dialogue.core;

import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import com.google.common.util.concurrent.ListenableFuture;
import com.palantir.dialogue.Endpoint;
import com.palantir.dialogue.EndpointChannel;
import com.palantir.dialogue.Request;
import com.palantir.dialogue.Response;
import com.palantir.dialogue.ResponseAttachments;
import com.palantir.dialogue.core.Config;
import com.palantir.dialogue.futures.DialogueFutures;
import com.palantir.logsafe.Arg;
import com.palantir.logsafe.Preconditions;
import com.palantir.logsafe.exceptions.SafeUncheckedIoException;
import com.palantir.logsafe.logger.SafeLogger;
import com.palantir.logsafe.logger.SafeLoggerFactory;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import java.util.zip.GZIPInputStream;
import org.jspecify.annotations.Nullable;

final class ContentDecodingChannel
implements EndpointChannel {
    private static final SafeLogger log = SafeLoggerFactory.get(ContentDecodingChannel.class);
    private static final String ACCEPT_ENCODING = "accept-encoding";
    private static final String CONTENT_ENCODING = "content-encoding";
    private static final String CONTENT_LENGTH = "content-length";
    private static final String GZIP = "gzip";
    private static final String PREFER_COMPRESSED_RESPONSE_TAG = "prefer-compressed-response";
    private final EndpointChannel delegate;
    private final BooleanSupplier sendAcceptGzip;

    private ContentDecodingChannel(EndpointChannel delegate, BooleanSupplier sendAcceptGzip) {
        this.delegate = (EndpointChannel)Preconditions.checkNotNull((Object)delegate, (String)"Channel is required");
        this.sendAcceptGzip = sendAcceptGzip;
    }

    static EndpointChannel create(Config cf, EndpointChannel delegate, Endpoint endpoint) {
        BooleanSupplier sendAcceptGzip = ContentDecodingChannel.shouldSendAcceptGzip(cf, endpoint);
        return new ContentDecodingChannel(delegate, sendAcceptGzip);
    }

    private static BooleanSupplier shouldSendAcceptGzip(Config cf, Endpoint endpoint) {
        if (endpoint.tags().contains(PREFER_COMPRESSED_RESPONSE_TAG)) {
            return () -> true;
        }
        return () -> ((List)cf.uris().get()).size() == 1;
    }

    public ListenableFuture<Response> execute(Request request) {
        Request augmentedRequest = ContentDecodingChannel.acceptEncoding(request, this.sendAcceptGzip.getAsBoolean());
        return DialogueFutures.transform((ListenableFuture)this.delegate.execute(augmentedRequest), ContentDecodingChannel::decompress);
    }

    private static Request acceptEncoding(Request request, boolean sendAcceptGzip) {
        if (!sendAcceptGzip || request.headerParams().containsKey((Object)ACCEPT_ENCODING)) {
            return request;
        }
        return Request.builder().from(request).putHeaderParams(ACCEPT_ENCODING, GZIP).build();
    }

    private static Response decompress(Response input) {
        Optional contentEncoding = input.getFirstHeader(CONTENT_ENCODING);
        if (contentEncoding.isPresent() && GZIP.equals(contentEncoding.get())) {
            return new ContentDecodingResponse(input);
        }
        return input;
    }

    public String toString() {
        return "ContentDecodingChannel{delegate=" + String.valueOf(this.delegate) + ", sendAcceptGzip=" + String.valueOf(this.sendAcceptGzip) + "}";
    }

    private static final class ContentDecodingResponse
    implements Response {
        private final Response delegate;
        private final ListMultimap<String, String> headers;
        private final InputStream body;

        ContentDecodingResponse(Response delegate) {
            this.delegate = delegate;
            this.headers = Multimaps.filterKeys((ListMultimap)delegate.headers(), ContentDecodingResponse::allowHeader);
            this.body = new DeferredGzipInputStream(delegate.body());
        }

        public InputStream body() {
            return this.body;
        }

        public int code() {
            return this.delegate.code();
        }

        public ListMultimap<String, String> headers() {
            return this.headers;
        }

        public Optional<String> getFirstHeader(String header) {
            if (!ContentDecodingResponse.allowHeader(header)) {
                return Optional.empty();
            }
            return this.delegate.getFirstHeader(header);
        }

        private static boolean allowHeader(String headerName) {
            return !ContentDecodingChannel.CONTENT_ENCODING.equalsIgnoreCase(headerName) && !ContentDecodingChannel.CONTENT_LENGTH.equalsIgnoreCase(headerName);
        }

        public String toString() {
            return "ContentDecodingResponse{delegate=" + String.valueOf(this.delegate) + "}";
        }

        public ResponseAttachments attachments() {
            return this.delegate.attachments();
        }

        public void close() {
            try {
                this.body.close();
            }
            catch (IOException e) {
                log.warn("Failed to close encoded body", (Throwable)e);
            }
            finally {
                this.delegate.close();
            }
        }
    }

    private static class DeferredGzipInputStream
    extends InputStream {
        private static final int BUFFER_SIZE = 8192;
        private final InputStream original;
        private @Nullable InputStream delegate;

        DeferredGzipInputStream(InputStream original) {
            this.original = original;
        }

        private InputStream getDelegate() throws IOException {
            if (this.delegate == null) {
                this.delegate = new BufferedInputStream(new GZIPInputStream(this.original, 8192), 8192);
            }
            return this.delegate;
        }

        private InputStream getDelegateSafely() {
            try {
                return this.getDelegate();
            }
            catch (IOException e) {
                throw new SafeUncheckedIoException("Failed to create a GZIPInputStream", e, new Arg[0]);
            }
        }

        @Override
        public int read() throws IOException {
            return this.getDelegate().read();
        }

        @Override
        public int read(byte[] buffer) throws IOException {
            return this.getDelegate().read(buffer);
        }

        @Override
        public int read(byte[] buffer, int off, int len) throws IOException {
            return this.getDelegate().read(buffer, off, len);
        }

        @Override
        public long skip(long requested) throws IOException {
            return this.getDelegate().skip(requested);
        }

        @Override
        public int available() throws IOException {
            return this.getDelegate().available();
        }

        @Override
        public void close() throws IOException {
            if (this.delegate == null) {
                this.original.close();
            } else {
                this.delegate.close();
            }
        }

        @Override
        public void mark(int readlimit) {
            this.getDelegateSafely().mark(readlimit);
        }

        @Override
        public void reset() throws IOException {
            this.getDelegate().reset();
        }

        @Override
        public boolean markSupported() {
            return this.getDelegateSafely().markSupported();
        }

        public String toString() {
            return "DeferredGzipInputStream{original=" + String.valueOf(this.original) + "}";
        }
    }
}

