/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.observable;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.StandardId;
import com.opengamma.strata.basics.index.Index;
import com.opengamma.strata.data.FieldName;
import com.opengamma.strata.data.ObservableId;
import com.opengamma.strata.data.ObservableSource;

/**
 * An identifier used to access the current value of an index.
 * <p>
 * This identifier can also be used to access the historic time-series of values.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class IndexQuoteId
    implements ObservableId, ImmutableBean, Serializable {

  /**
   * The index.
   */
  @PropertyDefinition(validate = "notNull")
  private final Index index;
  /**
   * The field name in the market data record that contains the market data item.
   * The most common field name is {@linkplain FieldName#MARKET_VALUE market value}.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final FieldName fieldName;
  /**
   * The source of observable market data.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final ObservableSource observableSource;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance used to obtain an observable value of the index.
   * <p>
   * The field name containing the data is {@link FieldName#MARKET_VALUE} and the market
   * data source is {@link ObservableSource#NONE}.
   *
   * @param index  the index
   * @return the identifier
   */
  public static IndexQuoteId of(Index index) {
    return new IndexQuoteId(index, FieldName.MARKET_VALUE, ObservableSource.NONE);
  }

  /**
   * Obtains an instance used to obtain an observable value of the index.
   * <p>
   * The market data source is {@link ObservableSource#NONE}.
   *
   * @param index  the index
   * @param fieldName  the name of the field in the market data record holding the data
   * @return the identifier
   */
  public static IndexQuoteId of(Index index, FieldName fieldName) {
    return new IndexQuoteId(index, fieldName, ObservableSource.NONE);
  }

  /**
   * Obtains an instance used to obtain an observable value of the index,
   * specifying the source of observable market data.
   *
   * @param index  the index
   * @param fieldName  the name of the field in the market data record holding the data
   * @param obsSource  the source of observable market data
   * @return the identifier
   */
  public static IndexQuoteId of(Index index, FieldName fieldName, ObservableSource obsSource) {
    return new IndexQuoteId(index, fieldName, obsSource);
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the identifier of the data.
   * <p>
   * This returns an artificial identifier with a scheme of 'OG-Index' and
   * a value of the index name.
   * 
   * @return the standard identifier
   */
  @Override
  public StandardId getStandardId() {
    return StandardId.of("OG-Index", index.getName());
  }

  @Override
  public IndexQuoteId withObservableSource(ObservableSource obsSource) {
    return new IndexQuoteId(index, fieldName, obsSource);
  }

  @Override
  public String toString() {
    return new StringBuilder(32)
        .append("IndexQuoteId:")
        .append(index)
        .append('/')
        .append(fieldName)
        .append(observableSource.equals(ObservableSource.NONE) ? "" : "/" + observableSource)
        .toString();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code IndexQuoteId}.
   */
  private static final TypedMetaBean<IndexQuoteId> META_BEAN =
      LightMetaBean.of(
          IndexQuoteId.class,
          MethodHandles.lookup(),
          new String[] {
              "index",
              "fieldName",
              "observableSource"},
          new Object[0]);

  /**
   * The meta-bean for {@code IndexQuoteId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<IndexQuoteId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private IndexQuoteId(
      Index index,
      FieldName fieldName,
      ObservableSource observableSource) {
    JodaBeanUtils.notNull(index, "index");
    JodaBeanUtils.notNull(fieldName, "fieldName");
    JodaBeanUtils.notNull(observableSource, "observableSource");
    this.index = index;
    this.fieldName = fieldName;
    this.observableSource = observableSource;
  }

  @Override
  public TypedMetaBean<IndexQuoteId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the index.
   * @return the value of the property, not null
   */
  public Index getIndex() {
    return index;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the field name in the market data record that contains the market data item.
   * The most common field name is {@linkplain FieldName#MARKET_VALUE market value}.
   * @return the value of the property, not null
   */
  @Override
  public FieldName getFieldName() {
    return fieldName;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the source of observable market data.
   * @return the value of the property, not null
   */
  @Override
  public ObservableSource getObservableSource() {
    return observableSource;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      IndexQuoteId other = (IndexQuoteId) obj;
      return JodaBeanUtils.equal(index, other.index) &&
          JodaBeanUtils.equal(fieldName, other.fieldName) &&
          JodaBeanUtils.equal(observableSource, other.observableSource);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(index);
      hash = hash * 31 + JodaBeanUtils.hashCode(fieldName);
      hash = hash * 31 + JodaBeanUtils.hashCode(observableSource);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
