/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.market.curve;

import java.io.Serializable;
import java.time.LocalDate;
import java.time.YearMonth;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableConstructor;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ImmutableList;
import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.collect.array.DoubleArray;
import com.opengamma.strata.market.ShiftType;
import com.opengamma.strata.market.ValueType;

/**
 * Provides the definition of how to calibrate a curve for inflation, optionally including seasonality.
 * <p>
 * This allows a "normal" curve definition to be combined with the last fixing,
 * optionally adding seasonality.
 */
@BeanDefinition(builderScope = "private", metaScope = "private")
final class InflationNodalCurveDefinition
    implements NodalCurveDefinition, ImmutableBean, Serializable {

  /**
   * The no seasonality definition.
   */
  private static final SeasonalityDefinition NO_SEASONALITY_DEFINITION =
      SeasonalityDefinition.of(DoubleArray.filled(12, 1d), ShiftType.SCALED);

  /**
   * The curve name.
   */
  @PropertyDefinition(validate = "notNull")
  private final NodalCurveDefinition curveWithoutFixingDefinition;
  /**
   * Last fixing date.
   */
  @PropertyDefinition(validate = "notNull")
  private final YearMonth lastFixingMonth;
  /**
   * Last fixing value.
   */
  @PropertyDefinition
  private final double lastFixingValue;
  /**
   * The seasonality definition associated to the curve.
   */
  @PropertyDefinition(validate = "notNull")
  private final SeasonalityDefinition seasonalityDefinition;

  //-------------------------------------------------------------------------
  // restricted constructor
  @ImmutableConstructor
  InflationNodalCurveDefinition(
      NodalCurveDefinition curveWithoutFixing,
      YearMonth lastFixingMonth,
      double lastFixingValue,
      SeasonalityDefinition seasonalityDefinition) {

    this.curveWithoutFixingDefinition = curveWithoutFixing;
    this.lastFixingMonth = lastFixingMonth;
    this.lastFixingValue = lastFixingValue;
    if (seasonalityDefinition == null) {
      this.seasonalityDefinition = NO_SEASONALITY_DEFINITION;
    } else {
      this.seasonalityDefinition = seasonalityDefinition;
    }
  }

  @Override
  public CurveName getName() {
    return curveWithoutFixingDefinition.getName();
  }

  @Override
  public ValueType getYValueType() {
    return curveWithoutFixingDefinition.getYValueType();
  }

  @Override
  public ImmutableList<CurveNode> getNodes() {
    return curveWithoutFixingDefinition.getNodes();
  }

  @Override
  public NodalCurveDefinition filtered(LocalDate valuationDate, ReferenceData refData) {
    return curveWithoutFixingDefinition.filtered(valuationDate, refData);
  }

  @Override
  public CurveMetadata metadata(LocalDate valuationDate, ReferenceData refData) {
    return curveWithoutFixingDefinition.metadata(valuationDate, refData);
  }

  @Override
  public NodalCurve curve(LocalDate valuationDate, CurveMetadata metadata, DoubleArray parameters) {
    NodalCurve curveWithoutFixing = curveWithoutFixingDefinition.curve(valuationDate, metadata, parameters);
    return InflationNodalCurve.of(
        curveWithoutFixing, valuationDate, lastFixingMonth, lastFixingValue, seasonalityDefinition);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code InflationNodalCurveDefinition}.
   * @return the meta-bean, not null
   */
  public static MetaBean meta() {
    return InflationNodalCurveDefinition.Meta.INSTANCE;
  }

  static {
    MetaBean.register(InflationNodalCurveDefinition.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  @Override
  public MetaBean metaBean() {
    return InflationNodalCurveDefinition.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the curve name.
   * @return the value of the property, not null
   */
  public NodalCurveDefinition getCurveWithoutFixingDefinition() {
    return curveWithoutFixingDefinition;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets last fixing date.
   * @return the value of the property, not null
   */
  public YearMonth getLastFixingMonth() {
    return lastFixingMonth;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets last fixing value.
   * @return the value of the property
   */
  public double getLastFixingValue() {
    return lastFixingValue;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the seasonality definition associated to the curve.
   * @return the value of the property, not null
   */
  public SeasonalityDefinition getSeasonalityDefinition() {
    return seasonalityDefinition;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      InflationNodalCurveDefinition other = (InflationNodalCurveDefinition) obj;
      return JodaBeanUtils.equal(curveWithoutFixingDefinition, other.curveWithoutFixingDefinition) &&
          JodaBeanUtils.equal(lastFixingMonth, other.lastFixingMonth) &&
          JodaBeanUtils.equal(lastFixingValue, other.lastFixingValue) &&
          JodaBeanUtils.equal(seasonalityDefinition, other.seasonalityDefinition);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(curveWithoutFixingDefinition);
    hash = hash * 31 + JodaBeanUtils.hashCode(lastFixingMonth);
    hash = hash * 31 + JodaBeanUtils.hashCode(lastFixingValue);
    hash = hash * 31 + JodaBeanUtils.hashCode(seasonalityDefinition);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("InflationNodalCurveDefinition{");
    buf.append("curveWithoutFixingDefinition").append('=').append(JodaBeanUtils.toString(curveWithoutFixingDefinition)).append(',').append(' ');
    buf.append("lastFixingMonth").append('=').append(JodaBeanUtils.toString(lastFixingMonth)).append(',').append(' ');
    buf.append("lastFixingValue").append('=').append(JodaBeanUtils.toString(lastFixingValue)).append(',').append(' ');
    buf.append("seasonalityDefinition").append('=').append(JodaBeanUtils.toString(seasonalityDefinition));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code InflationNodalCurveDefinition}.
   */
  private static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code curveWithoutFixingDefinition} property.
     */
    private final MetaProperty<NodalCurveDefinition> curveWithoutFixingDefinition = DirectMetaProperty.ofImmutable(
        this, "curveWithoutFixingDefinition", InflationNodalCurveDefinition.class, NodalCurveDefinition.class);
    /**
     * The meta-property for the {@code lastFixingMonth} property.
     */
    private final MetaProperty<YearMonth> lastFixingMonth = DirectMetaProperty.ofImmutable(
        this, "lastFixingMonth", InflationNodalCurveDefinition.class, YearMonth.class);
    /**
     * The meta-property for the {@code lastFixingValue} property.
     */
    private final MetaProperty<Double> lastFixingValue = DirectMetaProperty.ofImmutable(
        this, "lastFixingValue", InflationNodalCurveDefinition.class, Double.TYPE);
    /**
     * The meta-property for the {@code seasonalityDefinition} property.
     */
    private final MetaProperty<SeasonalityDefinition> seasonalityDefinition = DirectMetaProperty.ofImmutable(
        this, "seasonalityDefinition", InflationNodalCurveDefinition.class, SeasonalityDefinition.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "curveWithoutFixingDefinition",
        "lastFixingMonth",
        "lastFixingValue",
        "seasonalityDefinition");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -249814055:  // curveWithoutFixingDefinition
          return curveWithoutFixingDefinition;
        case -1842439587:  // lastFixingMonth
          return lastFixingMonth;
        case -1834546866:  // lastFixingValue
          return lastFixingValue;
        case 1835044115:  // seasonalityDefinition
          return seasonalityDefinition;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends InflationNodalCurveDefinition> builder() {
      return new InflationNodalCurveDefinition.Builder();
    }

    @Override
    public Class<? extends InflationNodalCurveDefinition> beanType() {
      return InflationNodalCurveDefinition.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -249814055:  // curveWithoutFixingDefinition
          return ((InflationNodalCurveDefinition) bean).getCurveWithoutFixingDefinition();
        case -1842439587:  // lastFixingMonth
          return ((InflationNodalCurveDefinition) bean).getLastFixingMonth();
        case -1834546866:  // lastFixingValue
          return ((InflationNodalCurveDefinition) bean).getLastFixingValue();
        case 1835044115:  // seasonalityDefinition
          return ((InflationNodalCurveDefinition) bean).getSeasonalityDefinition();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code InflationNodalCurveDefinition}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<InflationNodalCurveDefinition> {

    private NodalCurveDefinition curveWithoutFixingDefinition;
    private YearMonth lastFixingMonth;
    private double lastFixingValue;
    private SeasonalityDefinition seasonalityDefinition;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -249814055:  // curveWithoutFixingDefinition
          return curveWithoutFixingDefinition;
        case -1842439587:  // lastFixingMonth
          return lastFixingMonth;
        case -1834546866:  // lastFixingValue
          return lastFixingValue;
        case 1835044115:  // seasonalityDefinition
          return seasonalityDefinition;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -249814055:  // curveWithoutFixingDefinition
          this.curveWithoutFixingDefinition = (NodalCurveDefinition) newValue;
          break;
        case -1842439587:  // lastFixingMonth
          this.lastFixingMonth = (YearMonth) newValue;
          break;
        case -1834546866:  // lastFixingValue
          this.lastFixingValue = (Double) newValue;
          break;
        case 1835044115:  // seasonalityDefinition
          this.seasonalityDefinition = (SeasonalityDefinition) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public InflationNodalCurveDefinition build() {
      return new InflationNodalCurveDefinition(
          curveWithoutFixingDefinition,
          lastFixingMonth,
          lastFixingValue,
          seasonalityDefinition);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("InflationNodalCurveDefinition.Builder{");
      buf.append("curveWithoutFixingDefinition").append('=').append(JodaBeanUtils.toString(curveWithoutFixingDefinition)).append(',').append(' ');
      buf.append("lastFixingMonth").append('=').append(JodaBeanUtils.toString(lastFixingMonth)).append(',').append(' ');
      buf.append("lastFixingValue").append('=').append(JodaBeanUtils.toString(lastFixingValue)).append(',').append(' ');
      buf.append("seasonalityDefinition").append('=').append(JodaBeanUtils.toString(seasonalityDefinition));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
