package cn.org.atool.generator.javafile.template;

import cn.org.atool.fluent.mybatis.metadata.DbType;
import cn.org.atool.generator.database.config.impl.RelationConfig;
import cn.org.atool.generator.database.model.ConfigKey;
import cn.org.atool.generator.database.model.TableField;
import cn.org.atool.generator.database.model.TableSetter;
import com.squareup.javapoet.*;

import javax.lang.model.element.Modifier;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import static cn.org.atool.generator.database.model.Naming.capitalFirst;
import static cn.org.atool.generator.util.ClassNames.*;
import static cn.org.atool.generator.util.GeneratorHelper.isBlank;

/**
 * Entity代码生成构造
 *
 * @author wudarui
 */
@SuppressWarnings("rawtypes")
public class EntityFile extends AbstractTemplateFile {

    public EntityFile(TableSetter table) {
        super(table);
        this.packageName = entityPackage(table);
        this.klassName = entityClass(table);
    }

    public static TypeName entityName(TableSetter table) {
        return ClassName.get(entityPackage(table), entityClass(table));
    }

    public static String entityPackage(TableSetter table) {
        return table.gc().getBasePackage() + ".entity";
    }

    /**
     * 返回Entity类名
     *
     * @param table TableSetter
     * @return ignore
     */
    public static String entityClass(TableSetter table) {
        String entityPrefix = table.getEntityPrefix();
        if (entityPrefix.endsWith(table.getEntitySuffix())) {
            return entityPrefix;
        } else {
            return entityPrefix + table.getEntitySuffix();
        }
    }

    @Override
    protected void build(TypeSpec.Builder spec) {
        spec.addAnnotation(AnnotationSpec.builder(SuppressWarnings.class)
            .addMember("value", "{$S, $S}", "rawtypes", "unchecked")
            .build());
        if (table.gc().isLombok()) {
            spec.addAnnotation(Lombok_Data);
            if (table.gc().isSetterChain()) {
                spec.addAnnotation(AnnotationSpec.builder(Lombok_Accessors).addMember("chain", "true").build());
            }
            spec.addAnnotation(AnnotationSpec.builder(Lombok_EqualsAndHashCode).addMember("callSuper", "false").build());
        }
        spec.addAnnotation(this.fluentMybatisAnnotation())
            .addJavadoc("$T: 数据映射实体定义\n\n", super.className())
            .addJavadoc("$L", "@author Powered By Fluent Mybatis");

        if (table.gc().isRichEntity()) {
            spec.superclass(FM_RichEntity);
        } else {
            spec.superclass(FM_BaseEntity);
        }
        this.addSuperInterface(spec, table.getEntityInterfaces());

        spec.addField(FieldSpec.builder(long.class, "serialVersionUID",
                Modifier.STATIC, Modifier.FINAL, Modifier.PRIVATE)
            .initializer("1L")
            .build());
        for (TableField field : table.getFields()) {
            FieldSpec.Builder fb = FieldSpec.builder(field.getJavaType(), field.getName(), Modifier.PRIVATE);
            fb.addJavadoc("$L", field.getComment());
            if (field.isPrimary() && !"user".equals(table.getSeqName())) {
                fb.addAnnotation(this.getTableIdAnnotation(field));
            } else {
                fb.addAnnotation(this.getTableFieldAnnotation(field));
            }
            // 判断是否逻辑删除字段
            if (Objects.equals(field.getColumnName(), table.getLogicDeleted())) {
                fb.addAnnotation(FM_LogicDelete);
            }
            // 判断是否乐观锁字段
            if (Objects.equals(field.getColumnName(), table.getVersionField())) {
                fb.addAnnotation(FM_Version);
            }
            spec.addField(fb.build());
            if (!table.gc().isLombok()) {
                String capital = capitalFirst(field.getName());
                spec.addMethod(this.m_getter(capital, field));
                spec.addMethod(this.m_setter(capital, field, table.gc().isSetterChain()));
            }
        }

        spec.addMethod(this.m_entityClass());
        for (RelationConfig relation : table.getRelations()) {
            spec.addMethod(this.m_relation(relation));
        }
    }

    private MethodSpec m_getter(String capital, TableField field) {
        return MethodSpec.methodBuilder("get" + capital)
            .addModifiers(Modifier.PUBLIC)
            .returns(field.getJavaType())
            .addStatement("return this.$L", field.getName())
            .build();
    }

    private MethodSpec m_setter(String capital, TableField field, boolean setterChain) {
        MethodSpec.Builder spec = MethodSpec.methodBuilder("set" + capital)
            .addModifiers(Modifier.PUBLIC);
        if (setterChain) {
            spec.returns(this.className());
        }
        spec.addParameter(field.getJavaType(), field.getName())
            .addStatement("this.$L = $L", field.getName(), field.getName());
        if (setterChain) {
            spec.addStatement("return this");
        }
        return spec.build();
    }

    private MethodSpec m_entityClass() {
        return MethodSpec.methodBuilder("entityClass")
            .addModifiers(Modifier.PUBLIC, Modifier.FINAL)
            .addAnnotation(Override.class)
            .returns(Class.class)
            .addStatement("return $L.class", entityClass(table))
            .build();
    }

    /**
     * 设置关联字段和关联get方法
     *
     * @param relation RelationConfig
     * @return MethodSpec
     */
    private MethodSpec m_relation(RelationConfig relation) {
        String methodName = relation.methodName();
        String entityName = entityClass(table);
        return MethodSpec.methodBuilder(methodName)
            .addModifiers(Modifier.PUBLIC)
            .addAnnotation(relation.refMethodAnnotation())
            .returns(relation.returnType())
            .addJavadoc("实现 {@link $L.IEntityRelation#$LOf$L($L)}",
                table.gc().getBasePackage(), methodName, entityName, entityName)
            .addStatement("return super.invoke($S, $L)", methodName, relation.isCached())
            .build();
    }

    /**
     * 构造 @TableField(...)
     *
     * @param field TableField
     * @return ignore
     */
    private AnnotationSpec getTableFieldAnnotation(TableField field) {
        AnnotationSpec.Builder builder = AnnotationSpec.builder(FM_TableField)
            .addMember("value", "$S", field.getColumnName());
        if (!isBlank(field.getInsert())) {
            builder.addMember("insert", "$S", field.getInsert());
        }
        if (!isBlank(field.getUpdate())) {
            builder.addMember("update", "$S", field.getUpdate());
        }
        if (field.getIsLarge() != null && !field.getIsLarge()) {
            builder.addMember("notLarge", "$L", Boolean.FALSE.toString());
        }
        if (field.getTypeHandler() != null) {
            builder.addMember("typeHandler", "$T.class", field.getTypeHandler());
        }
        return builder.build();
    }

    /**
     * 构造 @TableId(...)
     *
     * @param field TableField
     * @return ignore
     */
    private AnnotationSpec getTableIdAnnotation(TableField field) {
        AnnotationSpec.Builder builder = AnnotationSpec.builder(FM_TableId)
            .addMember("value", "$S", field.getColumnName());
        String seqName = table.getSeqName();
        if (!this.isAuto(seqName, field.isPrimaryId())) {
            builder.addMember("auto", "$L", Boolean.FALSE.toString());
        }
        if (!isBlank(seqName) && !"auto".equals(seqName)) {
            builder.addMember("seqName", "$S", seqName);
        }
        boolean before = table.selectKeyBefore();
        if (before) {
            builder.addMember("before", "$L", true);
        }
        if (field.getTypeHandler() != null) {
            builder.addMember("typeHandler", "$T.class", field.getTypeHandler());
        }
        return builder.build();
    }

    boolean isAuto(String seqName, boolean isPrimaryId) {
        return "auto".equals(seqName) || ("".equals(seqName) && isPrimaryId);
    }

    private void addSuperInterface(TypeSpec.Builder builder, List<Class> interfaces) {
        if (interfaces == null || interfaces.size() == 0) {
            return;
        }
        for (Class _interface : interfaces) {
            if (hasEntityType(_interface.getSimpleName(), _interface.getTypeParameters())) {
                builder.addSuperinterface(parameterizedType(ClassName.get(_interface), this.className()));
            } else {
                builder.addSuperinterface(_interface);
            }
        }
    }

    /**
     * 泛型参数只有一个，且可以设置为Entity对象
     *
     * @param interfaceName 接口名称
     * @param typeVariables 接口泛型参数列表
     * @return ignore
     */
    private boolean hasEntityType(String interfaceName, TypeVariable[] typeVariables) {
        if (typeVariables.length != 1) {
            return false;
        }
        for (Type bound : typeVariables[0].getBounds()) {
            String tn = bound.getTypeName();
            if (Objects.equals(tn, interfaceName) || Allow_Entity_Bounds.contains(tn)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Entity自定义接口的泛型如果是下列类型子类型，可以使用Entity作为泛型参数
     */
    private static final Set<String> Allow_Entity_Bounds = new HashSet<>();

    static {
        Allow_Entity_Bounds.add("cn.org.atool.fluent.mybatis.base.IEntity");
        Allow_Entity_Bounds.add(Object.class.getName());
        Allow_Entity_Bounds.add(Serializable.class.getName());
    }

    /**
     * 生成 @FluentMybatis注解
     *
     * @return ignore
     */
    private AnnotationSpec fluentMybatisAnnotation() {
        AnnotationSpec.Builder builder = AnnotationSpec
            .builder(FM_FluentMybatis);

        builder.addMember("table", "$S", table.getTableName());
        if (!table.gc().getSchema().trim().isEmpty()) {
            builder.addMember("schema", "$S", table.gc().getSchema().trim());
        }
        if (table.gc().isUseCached()) {
            builder.addMember("useCached", "true");
        }
        if (!isBlank(table.getMapperBeanPrefix())) {
            builder.addMember("mapperBeanPrefix", "$S", table.getMapperBeanPrefix());
        }
        if (table.getDefaults() != null) {
            builder.addMember("defaults", "$T.class", table.getDefaults());
        }
        if (table.getSuperMapper() != null) {
            builder.addMember("superMapper", "$T.class", table.getSuperMapper());
        }
        if (!Objects.equals(table.getEntitySuffix(), ConfigKey.Entity_Default_Suffix)) {
            builder.addMember("suffix", "$S", table.getEntitySuffix());
        }
        if (DbType.MYSQL != table.gc().getDbType()) {
            builder.addMember("dbType", "$T.$L",
                FM_FluentDbType, table.gc().getDbType().name());
        }
        return builder.build();
    }

    @Override
    protected boolean isInterface() {
        return false;
    }
}