package cn.org.atool.generator.database.config.impl;

import cn.org.atool.fluent.mybatis.metadata.DbType;
import cn.org.atool.generator.database.model.DataSourceSetter;
import cn.org.atool.generator.database.model.Naming;
import cn.org.atool.generator.util.GeneratorHelper;
import com.squareup.javapoet.TypeName;
import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.mybatis.generator.api.JavaTypeResolver;

import javax.sql.DataSource;
import java.sql.Connection;
import java.util.Objects;

/**
 * 策略配置项
 *
 * @author darui.wu
 */
@Data
@Accessors(chain = true)
public class GlobalConfig {
    /**
     * 数据库表映射到实体的命名策略
     */
    private Naming tableNaming = Naming.underline_to_camel;
    /**
     * 数据库表字段映射到实体的命名策略
     * <p>未指定按照 naming 执行</p>
     */
    @Getter(AccessLevel.NONE)
    private Naming columnNaming;
    /**
     * Boolean类型字段是否移除is前缀（默认 false）<br>
     * 比如 : 数据库字段名称 : 'is_xxx',类型为 : tinyint. 在映射实体的时候则会去掉is,在实体类中映射最终结果为 xxx
     */
    private boolean removeIsPrefix = false;
    /**
     * schemaName
     */
    @Getter
    private String schema;
    /**
     * 代码package前缀
     */
    @Getter(AccessLevel.NONE)
    private String basePackage;

    private String daoPackage;

    @Setter(AccessLevel.NONE)
    private String packageDir;

    @Setter(AccessLevel.NONE)
    private String daoDir;
    /**
     * 代码生成路径
     */
    @Setter(AccessLevel.NONE)
    private String outputDir = System.getProperty("user.dir") + "/target/generate/base";
    /**
     * 测试代码生成路径
     */
    @Setter(AccessLevel.NONE)
    private String testOutputDir = System.getProperty("user.dir") + "/target/generate/test";
    /**
     * dao代码生成路径
     */
    @Setter(AccessLevel.NONE)
    private String daoOutputDir = System.getProperty("user.dir") + "/target/generate/dao";
    /**
     * 开发人员
     */
    private String author = "generate code";
    /**
     * 字段按字母序排列
     */
    @Setter
    private boolean alphabetOrder = true;
    /**
     * Entity类使用 Lombok 注解
     */
    @Setter
    private boolean lombok = true;

    @Setter
    private boolean setterChain = true;

    @Setter
    private boolean richEntity = true;

    @Setter
    private int selectKeyBefore = 0;

    @Setter
    private JavaTypeResolver javaTypeResolver;

    @Setter
    @Getter
    private boolean useCached;

    public Naming getColumnNaming() {
        return columnNaming == null ? tableNaming : columnNaming;
    }

    public GlobalConfig setBasePackage(String basePackage) {
        this.basePackage = basePackage;
        this.packageDir = '/' + basePackage.replace('.', '/') + '/';
        if (GeneratorHelper.isBlank(this.daoPackage)) {
            this.setDaoPackage(basePackage);
        }
        return this;
    }

    public GlobalConfig setDaoPackage(String daoPackage) {
        this.daoPackage = daoPackage;
        this.daoDir = '/' + daoPackage.replace('.', '/') + '/';
        return this;
    }

    public String getBasePackage() {
        if (GeneratorHelper.isBlank(basePackage)) {
            throw new RuntimeException("The base package should be set.");
        }
        return basePackage;
    }

    public GlobalConfig setOutputDir(String outputDir) {
        return this.setOutputDir(outputDir, outputDir, outputDir);
    }

    public GlobalConfig setOutputDir(String outputDir, String testOutputDir, String daoOutputDir) {
        this.outputDir = outputDir;
        this.testOutputDir = testOutputDir;
        this.daoOutputDir = daoOutputDir;
        return this;
    }

    /**
     * 数据源配置
     */
    private DataSourceSetter dataSourceSetter;

    public GlobalConfig setDataSource(DbType dbType, DataSource dataSource) {
        this.dataSourceSetter = new DataSourceSetter(dbType, dataSource);
        return this;
    }

    public GlobalConfig setDataSource(String url, String username, String password) {
        return this.setDataSource(DbType.MYSQL, "com.mysql.jdbc.Driver", url, username, password);
    }

    public GlobalConfig setDataSource(DbType dbType, String driver, String url, String username, String password) {
        this.dataSourceSetter = new DataSourceSetter(dbType, driver, url, username, password);
        return this;
    }

    public Connection getConnection() {
        return this.dataSourceSetter.getConn();
    }

    public DbType getDbType() {
        return this.dataSourceSetter.getDbType();
    }

    /**
     * 是否需要去掉is前缀
     *
     * @param fieldName 字段名称
     * @param fieldType 字段类型
     * @return ignore
     */
    public boolean needRemoveIsPrefix(String fieldName, TypeName fieldType) {
        if (!this.isRemoveIsPrefix()) {
            return false;
        } else if (!Objects.equals(boolean.class.getName(), fieldType.toString())) {
            return false;
        } else {
            return fieldName.startsWith("is");
        }
    }
}