/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.client.core.cnc.tracing;

import com.couchbase.client.core.cnc.EventBus;
import com.couchbase.client.core.cnc.InternalSpan;
import com.couchbase.client.core.cnc.RequestSpan;
import com.couchbase.client.core.cnc.RequestTracer;
import com.couchbase.client.core.cnc.events.tracing.OverThresholdRequestsRecordedEvent;
import com.couchbase.client.core.cnc.tracing.ThresholdInternalSpan;
import com.couchbase.client.core.cnc.tracing.ThresholdRequestSpan;
import com.couchbase.client.core.deps.org.jctools.queues.MpscUnboundedArrayQueue;
import com.couchbase.client.core.error.InvalidArgumentException;
import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.client.core.msg.Request;
import com.couchbase.client.core.msg.Response;
import com.couchbase.client.core.service.ServiceType;
import com.couchbase.client.core.util.HostAndPort;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import reactor.core.publisher.Mono;

public class ThresholdRequestTracer
implements RequestTracer {
    private static final AtomicInteger REQUEST_TRACER_ID = new AtomicInteger();
    private static final String KEY_TOTAL_MICROS = "total_us";
    private static final String KEY_DISPATCH_MICROS = "last_dispatch_us";
    private static final String KEY_ENCODE_MICROS = "encode_us";
    private static final String KEY_SERVER_MICROS = "server_us";
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final Queue<Request<?>> overThresholdQueue;
    private final EventBus eventBus;
    private final Thread worker;
    private final long kvThreshold;
    private final long queryThreshold;
    private final long viewThreshold;
    private final long searchThreshold;
    private final long analyticsThreshold;
    private final long emitIntervalNanos;
    private final int sampleSize;

    public static Builder builder(EventBus eventBus) {
        return new Builder(eventBus);
    }

    public static ThresholdRequestTracer create(EventBus eventBus) {
        return ThresholdRequestTracer.builder(eventBus).build();
    }

    private ThresholdRequestTracer(Builder builder) {
        this.eventBus = builder.eventBus;
        this.overThresholdQueue = new MpscUnboundedArrayQueue(builder.queueLength);
        this.kvThreshold = builder.kvThreshold.toNanos();
        this.analyticsThreshold = builder.analyticsThreshold.toNanos();
        this.searchThreshold = builder.searchThreshold.toNanos();
        this.viewThreshold = builder.viewThreshold.toNanos();
        this.queryThreshold = builder.queryThreshold.toNanos();
        this.sampleSize = builder.sampleSize;
        this.emitIntervalNanos = builder.emitInterval.toNanos();
        this.worker = new Thread(new Worker());
        this.worker.setDaemon(true);
    }

    @Override
    public InternalSpan internalSpan(String operationName, RequestSpan parent) {
        return new ThresholdInternalSpan(this, operationName, parent);
    }

    @Override
    public RequestSpan requestSpan(String operationName, RequestSpan parent) {
        return ThresholdRequestSpan.INSTANCE;
    }

    void finish(ThresholdInternalSpan span) {
        Request<? extends Response> request = span.requestContext().request();
        if (!this.isOverThreshold(request) || !this.overThresholdQueue.offer(request)) {
            // empty if block
        }
    }

    private boolean isOverThreshold(Request<?> request) {
        long tookNanos = request.context().logicalRequestLatency();
        ServiceType serviceType = request.serviceType();
        if (serviceType == ServiceType.KV && tookNanos >= this.kvThreshold) {
            return true;
        }
        if (serviceType == ServiceType.QUERY && tookNanos >= this.queryThreshold) {
            return true;
        }
        if (serviceType == ServiceType.ANALYTICS && tookNanos >= this.analyticsThreshold) {
            return true;
        }
        if (serviceType == ServiceType.SEARCH && tookNanos >= this.searchThreshold) {
            return true;
        }
        return serviceType == ServiceType.VIEWS && tookNanos >= this.viewThreshold;
    }

    @Override
    public Mono<Void> start() {
        return Mono.defer(() -> {
            if (this.running.compareAndSet(false, true)) {
                this.worker.start();
            }
            return Mono.empty();
        });
    }

    @Override
    public Mono<Void> stop(Duration timeout) {
        return Mono.defer(() -> {
            if (this.running.compareAndSet(true, false)) {
                this.worker.interrupt();
            }
            return Mono.empty();
        });
    }

    public static class Builder {
        private final EventBus eventBus;
        private static final Duration DEFAULT_EMIT_INTERVAL = Duration.ofSeconds(10L);
        private static final int DEFAULT_QUEUE_LENGTH = 1024;
        private static final Duration DEFAULT_KV_THRESHOLD = Duration.ofMillis(500L);
        private static final Duration DEFAULT_QUERY_THRESHOLD = Duration.ofSeconds(1L);
        private static final Duration DEFAULT_VIEW_THRESHOLD = Duration.ofSeconds(1L);
        private static final Duration DEFAULT_SEARCH_THRESHOLD = Duration.ofSeconds(1L);
        private static final Duration DEFAULT_ANALYTICS_THRESHOLD = Duration.ofSeconds(1L);
        private static final int DEFAULT_SAMPLE_SIZE = 10;
        private Duration emitInterval = DEFAULT_EMIT_INTERVAL;
        private int queueLength = 1024;
        private int sampleSize = 10;
        private Duration kvThreshold = DEFAULT_KV_THRESHOLD;
        private Duration queryThreshold = DEFAULT_QUERY_THRESHOLD;
        private Duration viewThreshold = DEFAULT_VIEW_THRESHOLD;
        private Duration searchThreshold = DEFAULT_SEARCH_THRESHOLD;
        private Duration analyticsThreshold = DEFAULT_ANALYTICS_THRESHOLD;

        Builder(EventBus eventBus) {
            this.eventBus = eventBus;
        }

        public ThresholdRequestTracer build() {
            return new ThresholdRequestTracer(this);
        }

        public Builder emitInterval(Duration emitInterval) {
            if (emitInterval.isZero()) {
                throw InvalidArgumentException.fromMessage("Emit interval needs to be greater than 0");
            }
            this.emitInterval = emitInterval;
            return this;
        }

        public Builder queueLength(int queueLength) {
            this.queueLength = queueLength;
            return this;
        }

        public Builder kvThreshold(Duration kvThreshold) {
            this.kvThreshold = kvThreshold;
            return this;
        }

        public Builder queryThreshold(Duration queryThreshold) {
            this.queryThreshold = queryThreshold;
            return this;
        }

        public Builder viewThreshold(Duration viewThreshold) {
            this.viewThreshold = viewThreshold;
            return this;
        }

        public Builder searchThreshold(Duration searchThreshold) {
            this.searchThreshold = searchThreshold;
            return this;
        }

        public Builder analyticsThreshold(Duration analyticsThreshold) {
            this.analyticsThreshold = analyticsThreshold;
            return this;
        }

        public Builder sampleSize(int sampleSize) {
            this.sampleSize = sampleSize;
            return this;
        }
    }

    private class Worker
    implements Runnable {
        private final long workerSleepMs = Long.parseLong(System.getProperty("com.couchbase.thresholdRequestTracerSleep", "100"));
        private final Comparator<Request<?>> THRESHOLD_COMPARATOR = Comparator.comparingLong(o -> o.context().logicalRequestLatency());
        private final Queue<Request<?>> kvThresholds = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> n1qlThresholds = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> viewThresholds = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> ftsThresholds = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> analyticsThresholds = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private long kvThresholdCount = 0L;
        private long n1qlThresholdCount = 0L;
        private long viewThresoldCount = 0L;
        private long ftsThresholdCount = 0L;
        private long analyticsThresholdCount = 0L;
        private long lastThresholdLog;
        private boolean hasThresholdWritten;

        private Worker() {
        }

        @Override
        public void run() {
            Thread.currentThread().setName("cb-tracing-" + REQUEST_TRACER_ID.incrementAndGet());
            while (ThresholdRequestTracer.this.running.get()) {
                try {
                    this.handleOverThresholdQueue();
                    Thread.sleep(this.workerSleepMs);
                }
                catch (InterruptedException ex) {
                    if (!ThresholdRequestTracer.this.running.get()) {
                        return;
                    }
                    Thread.currentThread().interrupt();
                }
                catch (Exception exception) {}
            }
        }

        private void handleOverThresholdQueue() {
            long now = System.nanoTime();
            if (now > this.lastThresholdLog + ThresholdRequestTracer.this.emitIntervalNanos) {
                this.prepareAndlogOverThreshold();
                this.lastThresholdLog = now;
            }
            Request request;
            while ((request = (Request)ThresholdRequestTracer.this.overThresholdQueue.poll()) != null) {
                ServiceType serviceType = request.serviceType();
                if (serviceType == ServiceType.KV) {
                    this.updateThreshold(this.kvThresholds, request);
                    ++this.kvThresholdCount;
                    continue;
                }
                if (serviceType == ServiceType.QUERY) {
                    this.updateThreshold(this.n1qlThresholds, request);
                    ++this.n1qlThresholdCount;
                    continue;
                }
                if (serviceType == ServiceType.VIEWS) {
                    this.updateThreshold(this.viewThresholds, request);
                    ++this.viewThresoldCount;
                    continue;
                }
                if (serviceType == ServiceType.SEARCH) {
                    this.updateThreshold(this.ftsThresholds, request);
                    ++this.ftsThresholdCount;
                    continue;
                }
                if (serviceType != ServiceType.ANALYTICS) continue;
                this.updateThreshold(this.analyticsThresholds, request);
                ++this.analyticsThresholdCount;
            }
            return;
        }

        private void prepareAndlogOverThreshold() {
            if (!this.hasThresholdWritten) {
                return;
            }
            this.hasThresholdWritten = false;
            ArrayList<Map<String, Object>> output = new ArrayList<Map<String, Object>>();
            if (!this.kvThresholds.isEmpty()) {
                output.add(this.convertThresholdMetadata(this.kvThresholds, this.kvThresholdCount, "kv"));
                this.kvThresholds.clear();
                this.kvThresholdCount = 0L;
            }
            if (!this.n1qlThresholds.isEmpty()) {
                output.add(this.convertThresholdMetadata(this.n1qlThresholds, this.n1qlThresholdCount, "query"));
                this.n1qlThresholds.clear();
                this.n1qlThresholdCount = 0L;
            }
            if (!this.viewThresholds.isEmpty()) {
                output.add(this.convertThresholdMetadata(this.viewThresholds, this.viewThresoldCount, "view"));
                this.viewThresholds.clear();
                this.viewThresoldCount = 0L;
            }
            if (!this.ftsThresholds.isEmpty()) {
                output.add(this.convertThresholdMetadata(this.ftsThresholds, this.ftsThresholdCount, "search"));
                this.ftsThresholds.clear();
                this.ftsThresholdCount = 0L;
            }
            if (!this.analyticsThresholds.isEmpty()) {
                output.add(this.convertThresholdMetadata(this.analyticsThresholds, this.analyticsThresholdCount, "analytics"));
                this.analyticsThresholds.clear();
                this.analyticsThresholdCount = 0L;
            }
            this.logOverThreshold(output);
        }

        private Map<String, Object> convertThresholdMetadata(Queue<Request<?>> requests, long count, String ident) {
            HashMap<String, Object> output = new HashMap<String, Object>();
            ArrayList top = new ArrayList();
            for (Request request : requests) {
                long serverDuration;
                long dispatchDuration;
                long encodeDuration;
                String localId;
                HashMap<String, Object> entry = new HashMap<String, Object>();
                entry.put(ThresholdRequestTracer.KEY_TOTAL_MICROS, TimeUnit.NANOSECONDS.toMicros(request.context().logicalRequestLatency()));
                String operationId = request.operationId();
                if (operationId != null) {
                    entry.put("last_operation_id", operationId);
                }
                entry.put("operation_name", request.getClass().getSimpleName());
                HostAndPort local = request.context().lastDispatchedFrom();
                HostAndPort peer = request.context().lastDispatchedTo();
                if (local != null) {
                    entry.put("last_local_address", RedactableArgument.redactSystem(local).toString());
                }
                if (peer != null) {
                    entry.put("last_remote_address", RedactableArgument.redactSystem(peer).toString());
                }
                if ((localId = request.context().lastChannelId()) != null) {
                    entry.put("last_local_id", RedactableArgument.redactSystem(localId).toString());
                }
                if ((encodeDuration = request.context().encodeLatency()) > 0L) {
                    entry.put(ThresholdRequestTracer.KEY_ENCODE_MICROS, encodeDuration);
                }
                if ((dispatchDuration = request.context().dispatchLatency()) > 0L) {
                    entry.put(ThresholdRequestTracer.KEY_DISPATCH_MICROS, TimeUnit.NANOSECONDS.toMicros(dispatchDuration));
                }
                if ((serverDuration = request.context().serverLatency()) > 0L) {
                    entry.put(ThresholdRequestTracer.KEY_SERVER_MICROS, TimeUnit.NANOSECONDS.toMicros(serverDuration));
                }
                top.add(entry);
            }
            top.sort((o1, o2) -> ((Long)o2.get(ThresholdRequestTracer.KEY_TOTAL_MICROS)).compareTo((Long)o1.get(ThresholdRequestTracer.KEY_TOTAL_MICROS)));
            output.put("service", ident);
            output.put("count", count);
            output.put("top", top);
            return output;
        }

        void logOverThreshold(List<Map<String, Object>> toLog) {
            ThresholdRequestTracer.this.eventBus.publish(new OverThresholdRequestsRecordedEvent(Duration.ofNanos(ThresholdRequestTracer.this.emitIntervalNanos), toLog));
        }

        private void updateThreshold(Queue<Request<?>> thresholds, Request<?> request) {
            thresholds.add(request);
            while (thresholds.size() > ThresholdRequestTracer.this.sampleSize) {
                thresholds.remove();
            }
            this.hasThresholdWritten = true;
        }
    }
}

