/*
 * Decompiled with CFR 0.152.
 */
package com.browserup.bup.proxy;

import com.browserup.bup.BrowserUpProxyServer;
import com.browserup.bup.exception.ProxyExistsException;
import com.browserup.bup.exception.ProxyPortsExhaustedException;
import com.browserup.bup.proxy.auth.AuthType;
import com.browserup.bup.util.BrowserUpProxyUtil;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import java.lang.ref.WeakReference;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class ProxyManager {
    private static final Logger LOG = LoggerFactory.getLogger(ProxyManager.class);
    private int lastPort;
    private final int minPort;
    private final int maxPort;
    private final Cache<Integer, BrowserUpProxyServer> proxyCache;
    private final ConcurrentMap<Integer, BrowserUpProxyServer> proxies;
    private static final int EXPIRED_PROXY_CLEANUP_INTERVAL_SECONDS = 60;

    @Inject
    public ProxyManager(@Named(value="minPort") Integer minPort, @Named(value="maxPort") Integer maxPort, final @Named(value="ttl") Integer ttl) {
        this.minPort = minPort;
        this.maxPort = maxPort;
        this.lastPort = maxPort;
        if (ttl > 0) {
            RemovalListener<Integer, BrowserUpProxyServer> removalListener = new RemovalListener<Integer, BrowserUpProxyServer>(){

                public void onRemoval(RemovalNotification<Integer, BrowserUpProxyServer> removal) {
                    try {
                        BrowserUpProxyServer proxy = (BrowserUpProxyServer)removal.getValue();
                        if (proxy != null) {
                            LOG.info("Expiring ProxyServer on port {} after {} seconds without activity", (Object)proxy.getPort(), (Object)ttl);
                            proxy.stop();
                        }
                    }
                    catch (Exception ex) {
                        LOG.warn("Error while stopping an expired proxy on port " + removal.getKey(), (Throwable)ex);
                    }
                }
            };
            this.proxyCache = CacheBuilder.newBuilder().expireAfterAccess((long)ttl.intValue(), TimeUnit.SECONDS).removalListener((RemovalListener)removalListener).build();
            this.proxies = this.proxyCache.asMap();
            ScheduledExecutorHolder.expiredProxyCleanupExecutor.scheduleWithFixedDelay(new ProxyCleanupTask(this.proxyCache), 60L, 60L, TimeUnit.SECONDS);
        } else {
            this.proxies = new ConcurrentHashMap<Integer, BrowserUpProxyServer>();
            this.proxyCache = null;
        }
    }

    public BrowserUpProxyServer create(String upstreamHttpProxy, String proxyUsername, String proxyPassword, Integer port, String bindAddr, String serverBindAddr, boolean useEcc, boolean trustAllServers) {
        return this.create(upstreamHttpProxy, false, null, proxyUsername, proxyPassword, port, bindAddr, serverBindAddr, useEcc, trustAllServers);
    }

    public BrowserUpProxyServer create(String upstreamHttpProxy, boolean upstreamProxyHttps, String proxyUsername, String proxyPassword, Integer port, String bindAddr, String serverBindAddr, boolean useEcc, boolean trustAllServers) {
        return this.create(upstreamHttpProxy, upstreamProxyHttps, null, proxyUsername, proxyPassword, port, bindAddr, serverBindAddr, useEcc, trustAllServers);
    }

    public BrowserUpProxyServer create(String upstreamHttpProxy, boolean upstreamProxyHttps, List<String> upstreamNonProxyHosts, String proxyUsername, String proxyPassword, Integer port, String bindAddr, String serverBindAddr, boolean useEcc, boolean trustAllServers) {
        LOG.debug("Instantiate ProxyServer...");
        BrowserUpProxyServer proxy = new BrowserUpProxyServer();
        if (useEcc) {
            LOG.info("Using Elliptic Curve Cryptography for certificate impersonation");
            proxy.setUseEcc(true);
        }
        if (trustAllServers) {
            proxy.setTrustAllServers(true);
        }
        if (proxyUsername != null && proxyPassword != null) {
            proxy.chainedProxyAuthorization(proxyUsername, proxyPassword, AuthType.BASIC);
        }
        if (upstreamHttpProxy != null) {
            try {
                InetSocketAddress chainedProxyAddress = BrowserUpProxyUtil.inetSocketAddressFromString((String)upstreamHttpProxy);
                proxy.setChainedProxy(chainedProxyAddress);
            }
            catch (URISyntaxException e) {
                LOG.error("Invalid upstream http proxy specified: " + upstreamHttpProxy + ". Must use host:port format.");
                throw new RuntimeException("Invalid upstream http proxy");
            }
            proxy.setChainedProxyHTTPS(upstreamProxyHttps);
            if (upstreamNonProxyHosts != null) {
                proxy.setChainedProxyNonProxyHosts(upstreamNonProxyHosts);
            }
        }
        InetAddress clientBindAddress = null;
        if (bindAddr != null) {
            LOG.debug("Bind ProxyServer to `{}`...", (Object)bindAddr);
            try {
                clientBindAddress = InetAddress.getByName(bindAddr);
            }
            catch (UnknownHostException e) {
                LOG.error("Unable to bind proxy to address: " + bindAddr + "; proxy will not be created.", (Throwable)e);
                throw new RuntimeException("Unable to bind proxy to address: ", e);
            }
        }
        InetAddress serverInetAddress = null;
        if (serverBindAddr != null) {
            LOG.debug("Bind ProxyServer serverAddress to `{}`...", (Object)serverBindAddr);
            try {
                serverInetAddress = InetAddress.getByName(serverBindAddr);
            }
            catch (UnknownHostException e) {
                LOG.error("Unable to bind proxy to server address: " + serverBindAddr + "; proxy will not be created.", (Throwable)e);
                throw new RuntimeException("Unable to bind proxy to server address: ", e);
            }
        }
        if (port != null) {
            return this.startProxy(proxy, port, clientBindAddress, serverInetAddress);
        }
        while (this.proxies.size() <= this.maxPort - this.minPort) {
            LOG.debug("Use next available port for new ProxyServer...");
            port = this.nextPort();
            try {
                return this.startProxy(proxy, port, clientBindAddress, serverInetAddress);
            }
            catch (ProxyExistsException ex) {
                LOG.debug("Proxy already exists at port {}", (Object)port);
            }
        }
        throw new ProxyPortsExhaustedException();
    }

    public BrowserUpProxyServer create(String upstreamHttpProxy, String proxyUsername, String proxyPassword, Integer port, String bindAddr, boolean useEcc, boolean trustAllServers) {
        return this.create(upstreamHttpProxy, false, null, proxyUsername, proxyPassword, port, null, null, false, false);
    }

    public BrowserUpProxyServer create(String upstreamHttpProxy, String proxyUsername, String proxyPassword, Integer port) {
        return this.create(upstreamHttpProxy, false, null, proxyUsername, proxyPassword, port, null, null, false, false);
    }

    public BrowserUpProxyServer create(String upstreamHttpProxy, String proxyUsername, String proxyPassword) {
        return this.create(upstreamHttpProxy, false, null, proxyUsername, proxyPassword, null, null, null, false, false);
    }

    public BrowserUpProxyServer create() {
        return this.create(null, false, null, null, null, null, null, null, false, false);
    }

    public BrowserUpProxyServer create(int port) {
        return this.create(null, false, null, null, null, port, null, null, false, false);
    }

    public BrowserUpProxyServer get(int port) {
        return (BrowserUpProxyServer)this.proxies.get(port);
    }

    private BrowserUpProxyServer startProxy(BrowserUpProxyServer proxy, int port, InetAddress clientBindAddr, InetAddress serverBindAddr) {
        BrowserUpProxyServer old;
        if (port != 0 && (old = this.proxies.putIfAbsent(port, proxy)) != null) {
            LOG.info("Proxy already exists at port {}", (Object)port);
            throw new ProxyExistsException(port);
        }
        try {
            proxy.start(port, clientBindAddr, serverBindAddr);
            if (port == 0) {
                int realPort = proxy.getPort();
                this.proxies.put(realPort, proxy);
            }
            return proxy;
        }
        catch (Exception ex) {
            if (port != 0) {
                this.proxies.remove(port);
            }
            try {
                proxy.stop();
            }
            catch (Exception ex2) {
                ex.addSuppressed(ex2);
            }
            throw ex;
        }
    }

    private synchronized int nextPort() {
        return this.lastPort < this.maxPort ? (this.lastPort = this.lastPort + 1) : (this.lastPort = this.minPort);
    }

    public Collection<BrowserUpProxyServer> get() {
        return this.proxies.values();
    }

    public void delete(int port) {
        BrowserUpProxyServer proxy = (BrowserUpProxyServer)this.proxies.remove(port);
        if (proxy == null) {
            return;
        }
        if (!proxy.isStopped()) {
            proxy.stop();
        }
    }

    private static class ProxyCleanupTask
    implements Runnable {
        private final WeakReference<Cache<Integer, BrowserUpProxyServer>> proxyCache;

        public ProxyCleanupTask(Cache<Integer, BrowserUpProxyServer> cache) {
            this.proxyCache = new WeakReference<Cache<Integer, BrowserUpProxyServer>>(cache);
        }

        @Override
        public void run() {
            Cache cache = (Cache)this.proxyCache.get();
            if (cache != null) {
                try {
                    cache.cleanUp();
                }
                catch (RuntimeException e) {
                    LOG.warn("Error occurred while attempting to clean up expired proxies", (Throwable)e);
                }
            } else {
                LOG.info("Proxy Cache was garbage collected. No longer cleaning up expired proxies for unused ProxyManager.");
                throw new RuntimeException("Exiting ProxyCleanupTask");
            }
        }
    }

    private static class ScheduledExecutorHolder {
        private static final ScheduledExecutorService expiredProxyCleanupExecutor = Executors.newSingleThreadScheduledExecutor(new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                Thread thread = Executors.defaultThreadFactory().newThread(r);
                thread.setName("expired-proxy-cleanup-thread");
                thread.setDaemon(true);
                return thread;
            }
        });

        private ScheduledExecutorHolder() {
        }
    }
}

