/*
 * Decompiled with CFR 0.152.
 */
package com.azure.cosmos.implementation.throughputControl.sdk;

import com.azure.cosmos.BridgeInternal;
import com.azure.cosmos.ConnectionMode;
import com.azure.cosmos.CosmosException;
import com.azure.cosmos.implementation.Exceptions;
import com.azure.cosmos.implementation.ResourceType;
import com.azure.cosmos.implementation.RxDocumentServiceRequest;
import com.azure.cosmos.implementation.Utils;
import com.azure.cosmos.implementation.apachecommons.lang.StringUtils;
import com.azure.cosmos.implementation.apachecommons.lang.tuple.Pair;
import com.azure.cosmos.implementation.caches.AsyncCache;
import com.azure.cosmos.implementation.caches.RxClientCollectionCache;
import com.azure.cosmos.implementation.caches.RxPartitionKeyRangeCache;
import com.azure.cosmos.implementation.guava25.base.Preconditions;
import com.azure.cosmos.implementation.throughputControl.EmptyThroughputContainerController;
import com.azure.cosmos.implementation.throughputControl.IThroughputContainerController;
import com.azure.cosmos.implementation.throughputControl.sdk.ContainerSDKThroughputControlGroupProperties;
import com.azure.cosmos.implementation.throughputControl.sdk.LinkedCancellationToken;
import com.azure.cosmos.implementation.throughputControl.sdk.LinkedCancellationTokenSource;
import com.azure.cosmos.implementation.throughputControl.sdk.config.SDKThroughputControlGroupInternal;
import com.azure.cosmos.implementation.throughputControl.sdk.controller.container.SDKThroughputContainerController;
import com.azure.cosmos.implementation.throughputControl.sdk.exceptions.ThroughputControlInitializationException;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public class SDKThroughputControlStore {
    private static final Logger logger = LoggerFactory.getLogger(SDKThroughputControlStore.class);
    private final RxClientCollectionCache collectionCache;
    private final ConnectionMode connectionMode;
    private final AsyncCache<String, IThroughputContainerController> containerControllerCache;
    private final ConcurrentHashMap<String, ContainerSDKThroughputControlGroupProperties> containerMap;
    private final RxPartitionKeyRangeCache partitionKeyRangeCache;
    private final LinkedCancellationTokenSource cancellationTokenSource;
    private final ConcurrentHashMap<String, LinkedCancellationToken> cancellationTokenMap;

    public SDKThroughputControlStore(RxClientCollectionCache collectionCache, ConnectionMode connectionMode, RxPartitionKeyRangeCache partitionKeyRangeCache) {
        Preconditions.checkNotNull(collectionCache, "RxClientCollectionCache can not be null");
        Preconditions.checkNotNull(partitionKeyRangeCache, "PartitionKeyRangeCache can not be null");
        this.collectionCache = collectionCache;
        this.connectionMode = connectionMode;
        this.containerControllerCache = new AsyncCache();
        this.containerMap = new ConcurrentHashMap();
        this.partitionKeyRangeCache = partitionKeyRangeCache;
        this.cancellationTokenSource = new LinkedCancellationTokenSource();
        this.cancellationTokenMap = new ConcurrentHashMap();
    }

    public void enableThroughputControlGroup(SDKThroughputControlGroupInternal group, Mono<Integer> throughputQueryMono) {
        Preconditions.checkNotNull(group, "Throughput control group cannot be null");
        String containerNameLink = Utils.trimBeginningAndEndingSlashes(BridgeInternal.extractContainerSelfLink(group.getTargetContainer()));
        this.containerMap.compute(containerNameLink, (key, throughputControlContainerProperties) -> {
            if (throughputControlContainerProperties == null) {
                throughputControlContainerProperties = new ContainerSDKThroughputControlGroupProperties(containerNameLink);
            }
            int groupSizeBefore = throughputControlContainerProperties.getThroughputControlGroups().size();
            Pair<Integer, Boolean> stateAfterEnabling = throughputControlContainerProperties.enableThroughputControlGroup(group, throughputQueryMono);
            int groupSizeAfter = stateAfterEnabling.getLeft();
            boolean wasGroupConfigUpdated = stateAfterEnabling.getRight();
            if (groupSizeAfter > groupSizeBefore && groupSizeAfter == 1 || wasGroupConfigUpdated) {
                this.containerControllerCache.remove(containerNameLink);
            }
            return throughputControlContainerProperties;
        });
    }

    public <T> Mono<T> processRequest(RxDocumentServiceRequest request, Mono<T> originalRequestMono) {
        Preconditions.checkNotNull(request, "Request can not be null");
        Preconditions.checkNotNull(originalRequestMono, "originalRequestMono can not be null");
        if (request.getResourceType() != ResourceType.Document && request.getResourceType() != ResourceType.StoredProcedure) {
            return originalRequestMono;
        }
        String collectionNameLink = Utils.getCollectionName(request.getResourceAddress());
        return this.resolveContainerController(collectionNameLink).flatMap(containerController -> {
            if (containerController.canHandleRequest(request)) {
                return containerController.processRequest(request, originalRequestMono).doOnError(throwable -> this.handleException(collectionNameLink, request, (Throwable)throwable));
            }
            return this.updateControllerAndRetry(collectionNameLink, request, originalRequestMono);
        }).onErrorResume(throwable -> {
            Exception unwrappedException = Utils.as(reactor.core.Exceptions.unwrap((Throwable)throwable), Exception.class);
            if (unwrappedException instanceof ThroughputControlInitializationException) {
                if (this.shouldContinueRequestOnInitError(request, collectionNameLink, unwrappedException)) {
                    return originalRequestMono;
                }
                return Mono.error((Throwable)unwrappedException.getCause());
            }
            return Mono.error((Throwable)throwable);
        });
    }

    private boolean shouldContinueRequestOnInitError(RxDocumentServiceRequest request, String collectionNameLink, Throwable throwable) {
        if (throwable instanceof ThroughputControlInitializationException) {
            ContainerSDKThroughputControlGroupProperties throughputControlContainerProperties = this.containerMap.get(collectionNameLink);
            Preconditions.checkNotNull(throughputControlContainerProperties, "Throughput control container properties should not be null");
            Preconditions.checkArgument(throughputControlContainerProperties.getThroughputControlGroups().size() > 0, "There should be more than one throughput control group");
            return throughputControlContainerProperties.allowRequestToContinueOnInitError(request);
        }
        return false;
    }

    private <T> Mono<T> updateControllerAndRetry(String containerNameLink, RxDocumentServiceRequest request, Mono<T> originalRequestMono) {
        return this.shouldRefreshContainerController(containerNameLink, request).flatMap(shouldRefresh -> {
            if (shouldRefresh.booleanValue()) {
                this.cancellationTokenMap.compute(containerNameLink, (key, cancellationToken) -> {
                    if (cancellationToken != null) {
                        cancellationToken.cancel();
                    }
                    return null;
                });
                this.containerControllerCache.refresh(containerNameLink, () -> this.createAndInitContainerController(containerNameLink));
                return this.resolveContainerController(containerNameLink).flatMap(updatedContainerController -> {
                    if (updatedContainerController.canHandleRequest(request)) {
                        return updatedContainerController.processRequest(request, originalRequestMono).doOnError(throwable -> this.handleException(containerNameLink, request, (Throwable)throwable));
                    }
                    logger.warn("Can not find container controller to process request {} with collectionRid {} ", (Object)request.getActivityId(), (Object)request.requestContext.resolvedCollectionRid);
                    return originalRequestMono;
                });
            }
            return originalRequestMono;
        });
    }

    private Mono<IThroughputContainerController> resolveContainerController(String containerNameLink) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(containerNameLink), "Container name link can not be null or empty");
        return this.containerControllerCache.getAsync(containerNameLink, null, () -> this.createAndInitContainerController(containerNameLink)).onErrorResume(throwable -> Mono.error((Throwable)new ThroughputControlInitializationException((Throwable)throwable)));
    }

    private Mono<IThroughputContainerController> createAndInitContainerController(String containerNameLink) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(containerNameLink), "Container link should not be null or empty");
        if (this.containerMap.containsKey(containerNameLink)) {
            return Mono.just((Object)this.containerMap.get(containerNameLink)).flatMap(throughputControlContainerProperties -> {
                LinkedCancellationToken parentToken = this.cancellationTokenMap.compute(containerNameLink, (key, cancellationToken) -> this.cancellationTokenSource.getToken());
                SDKThroughputContainerController containerController = new SDKThroughputContainerController(this.collectionCache, this.connectionMode, throughputControlContainerProperties.getThroughputControlGroups(), this.partitionKeyRangeCache, parentToken, throughputControlContainerProperties.getThroughputQueryMono());
                return containerController.init();
            });
        }
        return Mono.just((Object)new EmptyThroughputContainerController()).flatMap(EmptyThroughputContainerController::init);
    }

    private Mono<Boolean> shouldRefreshContainerController(String containerLink, RxDocumentServiceRequest request) {
        return this.collectionCache.resolveByNameAsync(null, containerLink, null).flatMap(documentCollection -> Mono.just((Object)StringUtils.equals(documentCollection.getResourceId(), request.requestContext.resolvedCollectionRid)));
    }

    private void handleException(String containerNameLink, RxDocumentServiceRequest request, Throwable throwable) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(containerNameLink), "Container name link can not be null nor empty");
        Preconditions.checkNotNull(request, "Request can not be null");
        Preconditions.checkNotNull(throwable, "Exception can not be null");
        CosmosException cosmosException = Utils.as(reactor.core.Exceptions.unwrap((Throwable)throwable), CosmosException.class);
        if (cosmosException != null && (Exceptions.isNameCacheStale(cosmosException) || Exceptions.isPartitionKeyMismatchException(cosmosException))) {
            this.cancellationTokenMap.compute(containerNameLink, (key, cancellationToken) -> {
                if (cancellationToken != null) {
                    cancellationToken.cancel();
                }
                return null;
            });
            String containerLink = Utils.getCollectionName(request.getResourceAddress());
            this.collectionCache.refresh(null, containerLink, null);
            this.containerControllerCache.refresh(containerLink, () -> this.createAndInitContainerController(containerLink));
        }
    }

    public boolean hasDefaultGroup(String containerNameLink) {
        if (this.containerMap.containsKey(containerNameLink)) {
            return this.containerMap.get(containerNameLink).hasDefaultGroup();
        }
        return false;
    }

    public boolean hasGroup(String containerNameLink, String throughputControlGroupName) {
        if (StringUtils.isEmpty(throughputControlGroupName)) {
            return false;
        }
        if (this.containerMap.containsKey(containerNameLink)) {
            return this.containerMap.get(containerNameLink).hasGroup(throughputControlGroupName);
        }
        return false;
    }

    public void close() {
        this.cancellationTokenSource.close();
    }
}

