// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.cosmos.models;

import com.azure.cosmos.implementation.ImplementationBridgeHelpers;
import com.azure.cosmos.implementation.JsonSerializable;
import com.azure.cosmos.implementation.PartitionKeyHelper;
import com.azure.cosmos.implementation.Utils;
import com.azure.cosmos.implementation.routing.PartitionKeyInternal;
import com.fasterxml.jackson.databind.node.ObjectNode;

import java.util.List;
import java.util.Map;

import static com.azure.cosmos.implementation.guava25.base.Preconditions.checkNotNull;

/**
 * Represents a partition key value in the Azure Cosmos DB database service. A
 * partition key identifies the partition where the item is stored in.
 */
public class PartitionKey {

    private final PartitionKeyInternal internalPartitionKey;
    private Object keyObject;

    PartitionKey(PartitionKeyInternal partitionKeyInternal) {
        this.internalPartitionKey = partitionKeyInternal;
    }

    /**
     * Constructor. CREATE a new instance of the PartitionKey object.
     *
     * @param key the value of the partition key.
     */
    @SuppressWarnings("serial")
    public PartitionKey(final Object key) {
        this.keyObject = key;
        this.internalPartitionKey = PartitionKeyInternal.fromObjectArray(new Object[] {key}, true);
    }

    /**
     * Create a new instance of the PartitionKey object from a serialized JSON
     * partition key.
     *
     * @param jsonString the JSON string representation of this PartitionKey object.
     * @return the PartitionKey instance.
     */
    static PartitionKey fromJsonString(String jsonString) {
        return new PartitionKey(PartitionKeyInternal.fromJsonString(jsonString));
    }

    /**
     * Partition key that represents no partition key.
     */
    public static final PartitionKey NONE = new PartitionKey(PartitionKeyInternal.None);

    /**
     * Serialize the PartitionKey object to a JSON string.
     *
     * @return the string representation of this PartitionKey object.
     */
    public String toString() {
        return this.internalPartitionKey.toJson();
    }

    PartitionKeyInternal getInternalPartitionKey() {
        return internalPartitionKey;
    }

    /**
     * Returns the PartitionKey from an array of objects that is generated by PartitionKeyInternal
     * @param values The object array of values to construct the PartitionKey with
     * @param strict The boolean value for if to use strict on object types
     * @return The PartitionKey
     */
    public static PartitionKey fromObjectArray(Object[] values, boolean strict) {
        checkNotNull(values, "Argument 'values' must not be null.");
        checkNotNull(strict, "Argument 'strict' must not be null.");
        if (values.length == 0) {
            throw new NullPointerException("Argument 'values' must not be null.");
        }
        return new PartitionKey(PartitionKeyInternal.fromObjectArray(values, strict));
    }


    /**
     * Returns the PartitionKey extracted from the item
     * @param item The JsonSerializable object to get the PartitionKey value from
     * @param partitionKeyDefinition The PartitionKeyDefinition to use to extract the PartitionKey value
     * @return The PartitionKey
     */
    public static PartitionKey fromItem(
        Map<String, Object> item,
        PartitionKeyDefinition partitionKeyDefinition) {
        checkNotNull(item, "Argument 'item' must not be null.");
        checkNotNull(partitionKeyDefinition, "Argument 'partitionKeyDefinition' must not be null.");

        return PartitionKeyHelper.extractPartitionKeyFromDocument(
            new JsonSerializable(Utils.getSimpleObjectMapper().convertValue(item, ObjectNode.class)),
            partitionKeyDefinition);
    }

    /**
     * Overrides the Equal operator for object comparisons between two instances of
     * {@link PartitionKey}
     *
     * @param other The object to compare with.
     * @return True if two object instance are considered equal.
     */
    @Override
    public boolean equals(Object other) {
        if (other == null) {
            return false;
        }

        if (this == other) {
            return true;
        }

        PartitionKey otherKey = Utils.as(other, PartitionKey.class);
        return otherKey != null && this.internalPartitionKey.equals(otherKey.internalPartitionKey);
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    ///////////////////////////////////////////////////////////////////////////////////////////
    // the following helper/accessor only helps to access this class outside of this package.//
    ///////////////////////////////////////////////////////////////////////////////////////////
    static void initialize() {
        ImplementationBridgeHelpers.PartitionKeyHelper.setPartitionKeyAccessor(
            new ImplementationBridgeHelpers.PartitionKeyHelper.PartitionKeyAccessor() {
                @Override
                public PartitionKey toPartitionKey(PartitionKeyInternal partitionKeyInternal) {
                    return new PartitionKey(partitionKeyInternal);
                }

                @Override
                public PartitionKey toPartitionKey(List<Object> values, boolean strict) {
                    PartitionKeyInternal partitionKeyInternal = PartitionKeyInternal.fromObjectArray(values, strict);
                    return new PartitionKey(partitionKeyInternal);
                }

                @Override
                public PartitionKeyInternal getPartitionKeyInternal(PartitionKey partitionKey) {
                    return partitionKey.internalPartitionKey;
                }
            }
        );
    }

    static { initialize(); }
}
