// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
// Code generated by Microsoft (R) AutoRest Code Generator.

package com.azure.communication.callautomation.models;

import com.azure.core.annotation.Fluent;
import com.azure.core.util.logging.ClientLogger;

import java.util.HashMap;
import java.util.Map;

/**
 * Custom calling context details.
 */
@Fluent
public final class CustomCallingContext {
    private final Map<String, String> sipHeaders;
    private final Map<String, String> voipHeaders;
    private final ClientLogger logger;

    /**
     * Creates an instance of CustomCallingContext class.
     */
    CustomCallingContext() {
        this.sipHeaders = new HashMap<String, String>();
        this.voipHeaders = new HashMap<String, String>();
        this.logger = new ClientLogger(CustomCallingContext.class);
    }

    /**
     * Create a CustomCallingContext object with SIP and VOIP headers
     *
     * @param sipHeaders custom context SIP headers
     * @param voipHeaders custom context VOIP headers
     */
    CustomCallingContext(Map<String, String> sipHeaders, Map<String, String> voipHeaders) {
        this.sipHeaders = sipHeaders;
        this.voipHeaders = voipHeaders;
        this.logger = new ClientLogger(CustomCallingContext.class);
    }

    /**
     * Get list of custom context SIP headers
     *
     * @return list of custom context SIP headers
     */
    public Map<String, String> getSipHeaders() {
        return sipHeaders;
    }

    /**
     * Get list of custom context VOIP headers
     *
     * @return list of custom context VOIP headers
     */
    public Map<String, String> getVoipHeaders() {
        return voipHeaders;
    }

    /**
     * Add a custom context sip UUI header. The Key always remains 'User-To-User'
     *
     * @param value custom context sip UUI header's value.
     * @throws IllegalStateException If sipHeaders is null
     */
    public void addSipUui(String value) {
        if (sipHeaders == null) {
            throw logger.logExceptionAsError(new IllegalStateException("Cannot add sip header, SipHeaders is null."));
        }
        sipHeaders.put("User-To-User", value);
    }

    /**
     * Add a custom context sip X header. The provided key is appended to 'X-MS-Custom-' in last.
     *
     * @param key custom context sip x header's key.
     * @param value custom context sip x header's value.
     */
    public void addSipX(String key, String value) {
        addSipX(key, value, SipHeaderPrefix.X_MS_CUSTOM);
    }

    /**
     * Add a custom context sip X header. The provided key is appended to 'X-' or "X-MS-Custom-" in last.
     *
     * @param key custom context sip x header's key.
     * @param value custom context sip x header's value.
     * @param prefix The prefix to use for the header.
     * @throws IllegalStateException If sipHeaders is null
     */
    public void addSipX(String key, String value, SipHeaderPrefix prefix) {
        if (sipHeaders == null) {
            throw logger.logExceptionAsError(new IllegalStateException("Cannot add sip header, SipHeaders is null."));
        }
        if (prefix == SipHeaderPrefix.X) {
            sipHeaders.put("X-" + key, value);
        } else {
            sipHeaders.put("X-MS-Custom-" + key, value);
        }
    }

    /**
     * Add a custom context voip header.
     *
     * @param key custom context voip header's key.
     * @param value custom context voip header's value.
     * @throws IllegalStateException If voipHeaders is null
     */
    public void addVoip(String key, String value) {
        if (sipHeaders == null) {
            throw logger.logExceptionAsError(new IllegalStateException("Cannot add sip header, VoipHeaders is null."));
        }
        voipHeaders.put(key, value);
    }
}
