/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.xray.XRayAsyncClient;
import software.amazon.awssdk.services.xray.internal.UserAgentUtils;
import software.amazon.awssdk.services.xray.model.BatchGetTracesRequest;
import software.amazon.awssdk.services.xray.model.BatchGetTracesResponse;
import software.amazon.awssdk.services.xray.model.Trace;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.xray.XRayAsyncClient#batchGetTracesPaginator(software.amazon.awssdk.services.xray.model.BatchGetTracesRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of {@link software.amazon.awssdk.services.xray.model.BatchGetTracesResponse} response
 * pages as per demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the subscribe helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.xray.paginators.BatchGetTracesPublisher publisher = client.batchGetTracesPaginator(request);
 * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.xray.paginators.BatchGetTracesPublisher publisher = client.batchGetTracesPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.xray.model.BatchGetTracesResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.xray.model.BatchGetTracesResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It only
 * limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #batchGetTraces(software.amazon.awssdk.services.xray.model.BatchGetTracesRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class BatchGetTracesPublisher implements SdkPublisher<BatchGetTracesResponse> {
    private final XRayAsyncClient client;

    private final BatchGetTracesRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public BatchGetTracesPublisher(XRayAsyncClient client, BatchGetTracesRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private BatchGetTracesPublisher(XRayAsyncClient client, BatchGetTracesRequest firstRequest, boolean isLastPage) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new BatchGetTracesResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super BatchGetTracesResponse> subscriber) {
        subscriber.onSubscribe(ResponsesSubscription.builder().subscriber(subscriber).nextPageFetcher(nextPageFetcher).build());
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<Trace> traces() {
        Function<BatchGetTracesResponse, Iterator<Trace>> getIterator = response -> {
            if (response != null && response.traces() != null) {
                return response.traces().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsPublisher.builder().nextPageFetcher(new BatchGetTracesResponseFetcher())
                .iteratorFunction(getIterator).isLastPage(isLastPage).build();
    }

    private class BatchGetTracesResponseFetcher implements AsyncPageFetcher<BatchGetTracesResponse> {
        @Override
        public boolean hasNextPage(final BatchGetTracesResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public CompletableFuture<BatchGetTracesResponse> nextPage(final BatchGetTracesResponse previousPage) {
            if (previousPage == null) {
                return client.batchGetTraces(firstRequest);
            }
            return client.batchGetTraces(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
