/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wisdom.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A recommendation trigger provides context on the event that produced the referenced recommendations. Recommendations
 * are only referenced in <code>recommendationIds</code> by a single RecommendationTrigger.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecommendationTrigger implements SdkPojo, Serializable,
        ToCopyableBuilder<RecommendationTrigger.Builder, RecommendationTrigger> {
    private static final SdkField<RecommendationTriggerData> DATA_FIELD = SdkField
            .<RecommendationTriggerData> builder(MarshallingType.SDK_POJO).memberName("data")
            .getter(getter(RecommendationTrigger::data)).setter(setter(Builder::data))
            .constructor(RecommendationTriggerData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("data").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(RecommendationTrigger::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<List<String>> RECOMMENDATION_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendationIds")
            .getter(getter(RecommendationTrigger::recommendationIds))
            .setter(setter(Builder::recommendationIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("source")
            .getter(getter(RecommendationTrigger::sourceAsString)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(RecommendationTrigger::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_FIELD, ID_FIELD,
            RECOMMENDATION_IDS_FIELD, SOURCE_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final RecommendationTriggerData data;

    private final String id;

    private final List<String> recommendationIds;

    private final String source;

    private final String type;

    private RecommendationTrigger(BuilderImpl builder) {
        this.data = builder.data;
        this.id = builder.id;
        this.recommendationIds = builder.recommendationIds;
        this.source = builder.source;
        this.type = builder.type;
    }

    /**
     * <p>
     * A union type containing information related to the trigger.
     * </p>
     * 
     * @return A union type containing information related to the trigger.
     */
    public final RecommendationTriggerData data() {
        return data;
    }

    /**
     * <p>
     * The identifier of the recommendation trigger.
     * </p>
     * 
     * @return The identifier of the recommendation trigger.
     */
    public final String id() {
        return id;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendationIds() {
        return recommendationIds != null && !(recommendationIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of the recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationIds} method.
     * </p>
     * 
     * @return The identifiers of the recommendations.
     */
    public final List<String> recommendationIds() {
        return recommendationIds;
    }

    /**
     * <p>
     * The source of the recommendation trigger.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
     * </p>
     * </li>
     * <li>
     * <p>
     * RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #source} will
     * return {@link RecommendationSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #sourceAsString}.
     * </p>
     * 
     * @return The source of the recommendation trigger.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
     *         </p>
     *         </li>
     * @see RecommendationSourceType
     */
    public final RecommendationSourceType source() {
        return RecommendationSourceType.fromValue(source);
    }

    /**
     * <p>
     * The source of the recommendation trigger.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
     * </p>
     * </li>
     * <li>
     * <p>
     * RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #source} will
     * return {@link RecommendationSourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #sourceAsString}.
     * </p>
     * 
     * @return The source of the recommendation trigger.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
     *         </p>
     *         </li>
     * @see RecommendationSourceType
     */
    public final String sourceAsString() {
        return source;
    }

    /**
     * <p>
     * The type of recommendation trigger.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link RecommendationTriggerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of recommendation trigger.
     * @see RecommendationTriggerType
     */
    public final RecommendationTriggerType type() {
        return RecommendationTriggerType.fromValue(type);
    }

    /**
     * <p>
     * The type of recommendation trigger.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link RecommendationTriggerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of recommendation trigger.
     * @see RecommendationTriggerType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationIds() ? recommendationIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecommendationTrigger)) {
            return false;
        }
        RecommendationTrigger other = (RecommendationTrigger) obj;
        return Objects.equals(data(), other.data()) && Objects.equals(id(), other.id())
                && hasRecommendationIds() == other.hasRecommendationIds()
                && Objects.equals(recommendationIds(), other.recommendationIds())
                && Objects.equals(sourceAsString(), other.sourceAsString())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecommendationTrigger").add("Data", data()).add("Id", id())
                .add("RecommendationIds", hasRecommendationIds() ? recommendationIds() : null).add("Source", sourceAsString())
                .add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "data":
            return Optional.ofNullable(clazz.cast(data()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "recommendationIds":
            return Optional.ofNullable(clazz.cast(recommendationIds()));
        case "source":
            return Optional.ofNullable(clazz.cast(sourceAsString()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecommendationTrigger, T> g) {
        return obj -> g.apply((RecommendationTrigger) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecommendationTrigger> {
        /**
         * <p>
         * A union type containing information related to the trigger.
         * </p>
         * 
         * @param data
         *        A union type containing information related to the trigger.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(RecommendationTriggerData data);

        /**
         * <p>
         * A union type containing information related to the trigger.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecommendationTriggerData.Builder}
         * avoiding the need to create one manually via {@link RecommendationTriggerData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecommendationTriggerData.Builder#build()} is called immediately
         * and its result is passed to {@link #data(RecommendationTriggerData)}.
         * 
         * @param data
         *        a consumer that will call methods on {@link RecommendationTriggerData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #data(RecommendationTriggerData)
         */
        default Builder data(Consumer<RecommendationTriggerData.Builder> data) {
            return data(RecommendationTriggerData.builder().applyMutation(data).build());
        }

        /**
         * <p>
         * The identifier of the recommendation trigger.
         * </p>
         * 
         * @param id
         *        The identifier of the recommendation trigger.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The identifiers of the recommendations.
         * </p>
         * 
         * @param recommendationIds
         *        The identifiers of the recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationIds(Collection<String> recommendationIds);

        /**
         * <p>
         * The identifiers of the recommendations.
         * </p>
         * 
         * @param recommendationIds
         *        The identifiers of the recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationIds(String... recommendationIds);

        /**
         * <p>
         * The source of the recommendation trigger.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
         * </p>
         * </li>
         * <li>
         * <p>
         * RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
         * </p>
         * </li>
         * </ul>
         * 
         * @param source
         *        The source of the recommendation trigger.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
         *        </p>
         *        </li>
         * @see RecommendationSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationSourceType
         */
        Builder source(String source);

        /**
         * <p>
         * The source of the recommendation trigger.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
         * </p>
         * </li>
         * <li>
         * <p>
         * RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
         * </p>
         * </li>
         * </ul>
         * 
         * @param source
         *        The source of the recommendation trigger.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ISSUE_DETECTION: The corresponding recommendations were triggered by a Contact Lens issue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RULE_EVALUATION: The corresponding recommendations were triggered by a Contact Lens rule.
         *        </p>
         *        </li>
         * @see RecommendationSourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationSourceType
         */
        Builder source(RecommendationSourceType source);

        /**
         * <p>
         * The type of recommendation trigger.
         * </p>
         * 
         * @param type
         *        The type of recommendation trigger.
         * @see RecommendationTriggerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationTriggerType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of recommendation trigger.
         * </p>
         * 
         * @param type
         *        The type of recommendation trigger.
         * @see RecommendationTriggerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationTriggerType
         */
        Builder type(RecommendationTriggerType type);
    }

    static final class BuilderImpl implements Builder {
        private RecommendationTriggerData data;

        private String id;

        private List<String> recommendationIds = DefaultSdkAutoConstructList.getInstance();

        private String source;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(RecommendationTrigger model) {
            data(model.data);
            id(model.id);
            recommendationIds(model.recommendationIds);
            source(model.source);
            type(model.type);
        }

        public final RecommendationTriggerData.Builder getData() {
            return data != null ? data.toBuilder() : null;
        }

        public final void setData(RecommendationTriggerData.BuilderImpl data) {
            this.data = data != null ? data.build() : null;
        }

        @Override
        public final Builder data(RecommendationTriggerData data) {
            this.data = data;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Collection<String> getRecommendationIds() {
            if (recommendationIds instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendationIds;
        }

        public final void setRecommendationIds(Collection<String> recommendationIds) {
            this.recommendationIds = RecommendationIdListCopier.copy(recommendationIds);
        }

        @Override
        public final Builder recommendationIds(Collection<String> recommendationIds) {
            this.recommendationIds = RecommendationIdListCopier.copy(recommendationIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationIds(String... recommendationIds) {
            recommendationIds(Arrays.asList(recommendationIds));
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        @Override
        public final Builder source(RecommendationSourceType source) {
            this.source(source == null ? null : source.toString());
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(RecommendationTriggerType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public RecommendationTrigger build() {
            return new RecommendationTrigger(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
