/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information for a release of the mobile SDK, including release notes and tags.
 * </p>
 * <p>
 * The mobile SDK is not generally available. Customers who have access to the mobile SDK can use it to establish and
 * manage WAF tokens for use in HTTP(S) requests from a mobile device to WAF. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-application-integration.html">WAF client application
 * integration</a> in the <i>WAF Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MobileSdkRelease implements SdkPojo, Serializable,
        ToCopyableBuilder<MobileSdkRelease.Builder, MobileSdkRelease> {
    private static final SdkField<String> RELEASE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReleaseVersion").getter(getter(MobileSdkRelease::releaseVersion))
            .setter(setter(Builder::releaseVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseVersion").build()).build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(MobileSdkRelease::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<String> RELEASE_NOTES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReleaseNotes").getter(getter(MobileSdkRelease::releaseNotes)).setter(setter(Builder::releaseNotes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseNotes").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(MobileSdkRelease::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RELEASE_VERSION_FIELD,
            TIMESTAMP_FIELD, RELEASE_NOTES_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String releaseVersion;

    private final Instant timestamp;

    private final String releaseNotes;

    private final List<Tag> tags;

    private MobileSdkRelease(BuilderImpl builder) {
        this.releaseVersion = builder.releaseVersion;
        this.timestamp = builder.timestamp;
        this.releaseNotes = builder.releaseNotes;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The release version.
     * </p>
     * 
     * @return The release version.
     */
    public final String releaseVersion() {
        return releaseVersion;
    }

    /**
     * <p>
     * The timestamp of the release.
     * </p>
     * 
     * @return The timestamp of the release.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * Notes describing the release.
     * </p>
     * 
     * @return Notes describing the release.
     */
    public final String releaseNotes() {
        return releaseNotes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags that are associated with the release.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags that are associated with the release.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(releaseVersion());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(releaseNotes());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MobileSdkRelease)) {
            return false;
        }
        MobileSdkRelease other = (MobileSdkRelease) obj;
        return Objects.equals(releaseVersion(), other.releaseVersion()) && Objects.equals(timestamp(), other.timestamp())
                && Objects.equals(releaseNotes(), other.releaseNotes()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MobileSdkRelease").add("ReleaseVersion", releaseVersion()).add("Timestamp", timestamp())
                .add("ReleaseNotes", releaseNotes()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReleaseVersion":
            return Optional.ofNullable(clazz.cast(releaseVersion()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "ReleaseNotes":
            return Optional.ofNullable(clazz.cast(releaseNotes()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MobileSdkRelease, T> g) {
        return obj -> g.apply((MobileSdkRelease) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MobileSdkRelease> {
        /**
         * <p>
         * The release version.
         * </p>
         * 
         * @param releaseVersion
         *        The release version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseVersion(String releaseVersion);

        /**
         * <p>
         * The timestamp of the release.
         * </p>
         * 
         * @param timestamp
         *        The timestamp of the release.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * Notes describing the release.
         * </p>
         * 
         * @param releaseNotes
         *        Notes describing the release.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseNotes(String releaseNotes);

        /**
         * <p>
         * Tags that are associated with the release.
         * </p>
         * 
         * @param tags
         *        Tags that are associated with the release.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Tags that are associated with the release.
         * </p>
         * 
         * @param tags
         *        Tags that are associated with the release.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Tags that are associated with the release.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.wafv2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.wafv2.model.Tag.Builder#build()}
         * is called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.wafv2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String releaseVersion;

        private Instant timestamp;

        private String releaseNotes;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MobileSdkRelease model) {
            releaseVersion(model.releaseVersion);
            timestamp(model.timestamp);
            releaseNotes(model.releaseNotes);
            tags(model.tags);
        }

        public final String getReleaseVersion() {
            return releaseVersion;
        }

        public final void setReleaseVersion(String releaseVersion) {
            this.releaseVersion = releaseVersion;
        }

        @Override
        public final Builder releaseVersion(String releaseVersion) {
            this.releaseVersion = releaseVersion;
            return this;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final String getReleaseNotes() {
            return releaseNotes;
        }

        public final void setReleaseNotes(String releaseNotes) {
            this.releaseNotes = releaseNotes;
        }

        @Override
        public final Builder releaseNotes(String releaseNotes) {
            this.releaseNotes = releaseNotes;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MobileSdkRelease build() {
            return new MobileSdkRelease(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
