/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
 * bypass detection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TextTransformation implements SdkPojo, Serializable,
        ToCopyableBuilder<TextTransformation.Builder, TextTransformation> {
    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(TextTransformation::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(TextTransformation::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRIORITY_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer priority;

    private final String type;

    private TextTransformation(BuilderImpl builder) {
        this.priority = builder.priority;
        this.type = builder.type;
    }

    /**
     * <p>
     * Sets the relative processing order for multiple transformations. WAF processes all transformations, from lowest
     * priority to highest, before inspecting the transformed content. The priorities don't need to be consecutive, but
     * they must all be different.
     * </p>
     * 
     * @return Sets the relative processing order for multiple transformations. WAF processes all transformations, from
     *         lowest priority to highest, before inspecting the transformed content. The priorities don't need to be
     *         consecutive, but they must all be different.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * You can specify the following transformation types:
     * </p>
     * <p>
     * <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
     * </p>
     * <p>
     * <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving implementation that
     * ignores characters that aren't valid.
     * </p>
     * <p>
     * <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of attackers who
     * inject an operating system command-line command and use unusual formatting to disguise some or all of the
     * command.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: <code>\ " ' ^</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: <code>/ (</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: <code>, ;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>\f</code>, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\t</code>, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\n</code>, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\r</code>, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\v</code>, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
     * <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so it can help
     * to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically be encoded. It's also
     * useful in countering evasion, which is a combination of a backslash and non-hexadecimal characters. For example,
     * <code>ja\vascript</code> for javascript.
     * </p>
     * <p>
     * <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>, <code>\b</code>,
     * <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>, <code>\\</code>,
     * <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal), <code>\0OOO</code> (octal).
     * Encodings that aren't valid remain in the output.
     * </p>
     * <p>
     * <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
     * <code>HTML_ENTITY_DECODE</code> performs these operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code> <code>HHHH</code> code
     * is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the higher byte is used to detect and
     * adjust the lower byte. If not, only the lower byte is used and the higher byte is zeroed, causing a possible loss
     * of information.
     * </p>
     * <p>
     * <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary form.
     * </p>
     * <p>
     * <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
     * </p>
     * <p>
     * <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory back-references that
     * are not at the beginning of the input from an input string.
     * </p>
     * <p>
     * <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts backslash
     * characters to forward slashes.
     * </p>
     * <p>
     * <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
     * </p>
     * <p>
     * <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>) with a single
     * space. Multiple consecutive occurrences are not compressed. Unterminated comments are also replaced with a space
     * (ASCII 0x20). However, a standalone termination of a comment (<code>*&#47;</code>) is not acted upon.
     * </p>
     * <p>
     * <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>).
     * </p>
     * <p>
     * <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (<code>ABC</code>
     * ).
     * </p>
     * <p>
     * <b>URL_DECODE</b> - Decode a URL-encoded value.
     * </p>
     * <p>
     * <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific <code>%u</code>
     * encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>, the higher byte is used to
     * detect and adjust the lower byte. Otherwise, only the lower byte is used and the higher byte is zeroed.
     * </p>
     * <p>
     * <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input normalization, and
     * minimizing false-positives and false-negatives for non-English languages.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TextTransformationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return You can specify the following transformation types:</p>
     *         <p>
     *         <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
     *         </p>
     *         <p>
     *         <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving
     *         implementation that ignores characters that aren't valid.
     *         </p>
     *         <p>
     *         <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of attackers
     *         who inject an operating system command-line command and use unusual formatting to disguise some or all of
     *         the command.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: <code>\ " ' ^</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: <code>/ (</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: <code>, ;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>\f</code>, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\t</code>, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\n</code>, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\r</code>, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\v</code>, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
     *         <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so it
     *         can help to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically be
     *         encoded. It's also useful in countering evasion, which is a combination of a backslash and
     *         non-hexadecimal characters. For example, <code>ja\vascript</code> for javascript.
     *         </p>
     *         <p>
     *         <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>, <code>\b</code>, <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>, <code>\\</code>,
     *         <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal), <code>\0OOO</code>
     *         (octal). Encodings that aren't valid remain in the output.
     *         </p>
     *         <p>
     *         <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs these operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code>
     *         <code>HHHH</code> code is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the higher
     *         byte is used to detect and adjust the lower byte. If not, only the lower byte is used and the higher byte
     *         is zeroed, causing a possible loss of information.
     *         </p>
     *         <p>
     *         <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary form.
     *         </p>
     *         <p>
     *         <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
     *         </p>
     *         <p>
     *         <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory back-references
     *         that are not at the beginning of the input from an input string.
     *         </p>
     *         <p>
     *         <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts backslash
     *         characters to forward slashes.
     *         </p>
     *         <p>
     *         <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
     *         </p>
     *         <p>
     *         <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>) with
     *         a single space. Multiple consecutive occurrences are not compressed. Unterminated comments are also
     *         replaced with a space (ASCII 0x20). However, a standalone termination of a comment (<code>*&#47;</code>)
     *         is not acted upon.
     *         </p>
     *         <p>
     *         <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>).
     *         </p>
     *         <p>
     *         <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (
     *         <code>ABC</code>).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b> - Decode a URL-encoded value.
     *         </p>
     *         <p>
     *         <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific
     *         <code>%u</code> encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>,
     *         the higher byte is used to detect and adjust the lower byte. Otherwise, only the lower byte is used and
     *         the higher byte is zeroed.
     *         </p>
     *         <p>
     *         <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input
     *         normalization, and minimizing false-positives and false-negatives for non-English languages.
     * @see TextTransformationType
     */
    public final TextTransformationType type() {
        return TextTransformationType.fromValue(type);
    }

    /**
     * <p>
     * You can specify the following transformation types:
     * </p>
     * <p>
     * <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
     * </p>
     * <p>
     * <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving implementation that
     * ignores characters that aren't valid.
     * </p>
     * <p>
     * <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of attackers who
     * inject an operating system command-line command and use unusual formatting to disguise some or all of the
     * command.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: <code>\ " ' ^</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: <code>/ (</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: <code>, ;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>\f</code>, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\t</code>, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\n</code>, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\r</code>, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>\v</code>, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
     * <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so it can help
     * to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically be encoded. It's also
     * useful in countering evasion, which is a combination of a backslash and non-hexadecimal characters. For example,
     * <code>ja\vascript</code> for javascript.
     * </p>
     * <p>
     * <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>, <code>\b</code>,
     * <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>, <code>\\</code>,
     * <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal), <code>\0OOO</code> (octal).
     * Encodings that aren't valid remain in the output.
     * </p>
     * <p>
     * <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
     * <code>HTML_ENTITY_DECODE</code> performs these operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code> <code>HHHH</code> code
     * is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the higher byte is used to detect and
     * adjust the lower byte. If not, only the lower byte is used and the higher byte is zeroed, causing a possible loss
     * of information.
     * </p>
     * <p>
     * <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary form.
     * </p>
     * <p>
     * <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
     * </p>
     * <p>
     * <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory back-references that
     * are not at the beginning of the input from an input string.
     * </p>
     * <p>
     * <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts backslash
     * characters to forward slashes.
     * </p>
     * <p>
     * <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
     * </p>
     * <p>
     * <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>) with a single
     * space. Multiple consecutive occurrences are not compressed. Unterminated comments are also replaced with a space
     * (ASCII 0x20). However, a standalone termination of a comment (<code>*&#47;</code>) is not acted upon.
     * </p>
     * <p>
     * <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>).
     * </p>
     * <p>
     * <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (<code>ABC</code>
     * ).
     * </p>
     * <p>
     * <b>URL_DECODE</b> - Decode a URL-encoded value.
     * </p>
     * <p>
     * <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific <code>%u</code>
     * encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>, the higher byte is used to
     * detect and adjust the lower byte. Otherwise, only the lower byte is used and the higher byte is zeroed.
     * </p>
     * <p>
     * <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input normalization, and
     * minimizing false-positives and false-negatives for non-English languages.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TextTransformationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return You can specify the following transformation types:</p>
     *         <p>
     *         <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
     *         </p>
     *         <p>
     *         <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving
     *         implementation that ignores characters that aren't valid.
     *         </p>
     *         <p>
     *         <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of attackers
     *         who inject an operating system command-line command and use unusual formatting to disguise some or all of
     *         the command.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: <code>\ " ' ^</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: <code>/ (</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: <code>, ;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>\f</code>, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\t</code>, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\n</code>, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\r</code>, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>\v</code>, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
     *         <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so it
     *         can help to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically be
     *         encoded. It's also useful in countering evasion, which is a combination of a backslash and
     *         non-hexadecimal characters. For example, <code>ja\vascript</code> for javascript.
     *         </p>
     *         <p>
     *         <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>, <code>\b</code>, <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>, <code>\\</code>,
     *         <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal), <code>\0OOO</code>
     *         (octal). Encodings that aren't valid remain in the output.
     *         </p>
     *         <p>
     *         <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs these operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code>
     *         <code>HHHH</code> code is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the higher
     *         byte is used to detect and adjust the lower byte. If not, only the lower byte is used and the higher byte
     *         is zeroed, causing a possible loss of information.
     *         </p>
     *         <p>
     *         <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary form.
     *         </p>
     *         <p>
     *         <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
     *         </p>
     *         <p>
     *         <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory back-references
     *         that are not at the beginning of the input from an input string.
     *         </p>
     *         <p>
     *         <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts backslash
     *         characters to forward slashes.
     *         </p>
     *         <p>
     *         <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
     *         </p>
     *         <p>
     *         <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>) with
     *         a single space. Multiple consecutive occurrences are not compressed. Unterminated comments are also
     *         replaced with a space (ASCII 0x20). However, a standalone termination of a comment (<code>*&#47;</code>)
     *         is not acted upon.
     *         </p>
     *         <p>
     *         <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>).
     *         </p>
     *         <p>
     *         <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (
     *         <code>ABC</code>).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b> - Decode a URL-encoded value.
     *         </p>
     *         <p>
     *         <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific
     *         <code>%u</code> encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>,
     *         the higher byte is used to detect and adjust the lower byte. Otherwise, only the lower byte is used and
     *         the higher byte is zeroed.
     *         </p>
     *         <p>
     *         <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input
     *         normalization, and minimizing false-positives and false-negatives for non-English languages.
     * @see TextTransformationType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TextTransformation)) {
            return false;
        }
        TextTransformation other = (TextTransformation) obj;
        return Objects.equals(priority(), other.priority()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TextTransformation").add("Priority", priority()).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TextTransformation, T> g) {
        return obj -> g.apply((TextTransformation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TextTransformation> {
        /**
         * <p>
         * Sets the relative processing order for multiple transformations. WAF processes all transformations, from
         * lowest priority to highest, before inspecting the transformed content. The priorities don't need to be
         * consecutive, but they must all be different.
         * </p>
         * 
         * @param priority
         *        Sets the relative processing order for multiple transformations. WAF processes all transformations,
         *        from lowest priority to highest, before inspecting the transformed content. The priorities don't need
         *        to be consecutive, but they must all be different.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * You can specify the following transformation types:
         * </p>
         * <p>
         * <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
         * </p>
         * <p>
         * <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving implementation
         * that ignores characters that aren't valid.
         * </p>
         * <p>
         * <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of attackers who
         * inject an operating system command-line command and use unusual formatting to disguise some or all of the
         * command.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: <code>\ " ' ^</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: <code>/ (</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: <code>, ;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>\f</code>, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\t</code>, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\n</code>, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\r</code>, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\v</code>, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
         * <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so it can
         * help to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically be encoded.
         * It's also useful in countering evasion, which is a combination of a backslash and non-hexadecimal characters.
         * For example, <code>ja\vascript</code> for javascript.
         * </p>
         * <p>
         * <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>, <code>\b</code>,
         * <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>, <code>\\</code>,
         * <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal), <code>\0OOO</code>
         * (octal). Encodings that aren't valid remain in the output.
         * </p>
         * <p>
         * <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
         * <code>HTML_ENTITY_DECODE</code> performs these operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code> <code>HHHH</code>
         * code is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the higher byte is used to detect
         * and adjust the lower byte. If not, only the lower byte is used and the higher byte is zeroed, causing a
         * possible loss of information.
         * </p>
         * <p>
         * <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary form.
         * </p>
         * <p>
         * <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
         * </p>
         * <p>
         * <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory back-references
         * that are not at the beginning of the input from an input string.
         * </p>
         * <p>
         * <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts backslash
         * characters to forward slashes.
         * </p>
         * <p>
         * <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
         * </p>
         * <p>
         * <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>) with a
         * single space. Multiple consecutive occurrences are not compressed. Unterminated comments are also replaced
         * with a space (ASCII 0x20). However, a standalone termination of a comment (<code>*&#47;</code>) is not acted
         * upon.
         * </p>
         * <p>
         * <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>).
         * </p>
         * <p>
         * <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (
         * <code>ABC</code>).
         * </p>
         * <p>
         * <b>URL_DECODE</b> - Decode a URL-encoded value.
         * </p>
         * <p>
         * <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific <code>%u</code>
         * encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>, the higher byte is
         * used to detect and adjust the lower byte. Otherwise, only the lower byte is used and the higher byte is
         * zeroed.
         * </p>
         * <p>
         * <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input normalization,
         * and minimizing false-positives and false-negatives for non-English languages.
         * </p>
         * 
         * @param type
         *        You can specify the following transformation types:</p>
         *        <p>
         *        <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
         *        </p>
         *        <p>
         *        <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving
         *        implementation that ignores characters that aren't valid.
         *        </p>
         *        <p>
         *        <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of
         *        attackers who inject an operating system command-line command and use unusual formatting to disguise
         *        some or all of the command.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: <code>\ " ' ^</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: <code>/ (</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: <code>, ;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>\f</code>, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\t</code>, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\n</code>, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\r</code>, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\v</code>, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
         *        <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so
         *        it can help to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically
         *        be encoded. It's also useful in countering evasion, which is a combination of a backslash and
         *        non-hexadecimal characters. For example, <code>ja\vascript</code> for javascript.
         *        </p>
         *        <p>
         *        <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>,
         *        <code>\b</code>, <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>,
         *        <code>\\</code>, <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal),
         *        <code>\0OOO</code> (octal). Encodings that aren't valid remain in the output.
         *        </p>
         *        <p>
         *        <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs these operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code>
         *        <code>HHHH</code> code is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the
         *        higher byte is used to detect and adjust the lower byte. If not, only the lower byte is used and the
         *        higher byte is zeroed, causing a possible loss of information.
         *        </p>
         *        <p>
         *        <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary
         *        form.
         *        </p>
         *        <p>
         *        <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
         *        </p>
         *        <p>
         *        <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory
         *        back-references that are not at the beginning of the input from an input string.
         *        </p>
         *        <p>
         *        <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts
         *        backslash characters to forward slashes.
         *        </p>
         *        <p>
         *        <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
         *        </p>
         *        <p>
         *        <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>)
         *        with a single space. Multiple consecutive occurrences are not compressed. Unterminated comments are
         *        also replaced with a space (ASCII 0x20). However, a standalone termination of a comment (
         *        <code>*&#47;</code>) is not acted upon.
         *        </p>
         *        <p>
         *        <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>
         *        ).
         *        </p>
         *        <p>
         *        <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (
         *        <code>ABC</code>).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b> - Decode a URL-encoded value.
         *        </p>
         *        <p>
         *        <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific
         *        <code>%u</code> encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>,
         *        the higher byte is used to detect and adjust the lower byte. Otherwise, only the lower byte is used
         *        and the higher byte is zeroed.
         *        </p>
         *        <p>
         *        <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input
         *        normalization, and minimizing false-positives and false-negatives for non-English languages.
         * @see TextTransformationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformationType
         */
        Builder type(String type);

        /**
         * <p>
         * You can specify the following transformation types:
         * </p>
         * <p>
         * <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
         * </p>
         * <p>
         * <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving implementation
         * that ignores characters that aren't valid.
         * </p>
         * <p>
         * <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of attackers who
         * inject an operating system command-line command and use unusual formatting to disguise some or all of the
         * command.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: <code>\ " ' ^</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: <code>/ (</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: <code>, ;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>\f</code>, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\t</code>, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\n</code>, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\r</code>, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>\v</code>, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
         * <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so it can
         * help to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically be encoded.
         * It's also useful in countering evasion, which is a combination of a backslash and non-hexadecimal characters.
         * For example, <code>ja\vascript</code> for javascript.
         * </p>
         * <p>
         * <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>, <code>\b</code>,
         * <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>, <code>\\</code>,
         * <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal), <code>\0OOO</code>
         * (octal). Encodings that aren't valid remain in the output.
         * </p>
         * <p>
         * <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
         * <code>HTML_ENTITY_DECODE</code> performs these operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code> <code>HHHH</code>
         * code is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the higher byte is used to detect
         * and adjust the lower byte. If not, only the lower byte is used and the higher byte is zeroed, causing a
         * possible loss of information.
         * </p>
         * <p>
         * <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary form.
         * </p>
         * <p>
         * <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
         * </p>
         * <p>
         * <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory back-references
         * that are not at the beginning of the input from an input string.
         * </p>
         * <p>
         * <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts backslash
         * characters to forward slashes.
         * </p>
         * <p>
         * <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
         * </p>
         * <p>
         * <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>) with a
         * single space. Multiple consecutive occurrences are not compressed. Unterminated comments are also replaced
         * with a space (ASCII 0x20). However, a standalone termination of a comment (<code>*&#47;</code>) is not acted
         * upon.
         * </p>
         * <p>
         * <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>).
         * </p>
         * <p>
         * <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (
         * <code>ABC</code>).
         * </p>
         * <p>
         * <b>URL_DECODE</b> - Decode a URL-encoded value.
         * </p>
         * <p>
         * <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific <code>%u</code>
         * encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>, the higher byte is
         * used to detect and adjust the lower byte. Otherwise, only the lower byte is used and the higher byte is
         * zeroed.
         * </p>
         * <p>
         * <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input normalization,
         * and minimizing false-positives and false-negatives for non-English languages.
         * </p>
         * 
         * @param type
         *        You can specify the following transformation types:</p>
         *        <p>
         *        <b>BASE64_DECODE</b> - Decode a <code>Base64</code>-encoded string.
         *        </p>
         *        <p>
         *        <b>BASE64_DECODE_EXT</b> - Decode a <code>Base64</code>-encoded string, but use a forgiving
         *        implementation that ignores characters that aren't valid.
         *        </p>
         *        <p>
         *        <b>CMD_LINE</b> - Command-line transformations. These are helpful in reducing effectiveness of
         *        attackers who inject an operating system command-line command and use unusual formatting to disguise
         *        some or all of the command.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: <code>\ " ' ^</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: <code>/ (</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: <code>, ;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b> - Replace these characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>\f</code>, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\t</code>, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\n</code>, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\r</code>, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>\v</code>, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>CSS_DECODE</b> - Decode characters that were encoded using CSS 2.x escape rules
         *        <code>syndata.html#characters</code>. This function uses up to two bytes in the decoding process, so
         *        it can help to uncover ASCII characters that were encoded using CSS encoding that wouldn’t typically
         *        be encoded. It's also useful in countering evasion, which is a combination of a backslash and
         *        non-hexadecimal characters. For example, <code>ja\vascript</code> for javascript.
         *        </p>
         *        <p>
         *        <b>ESCAPE_SEQ_DECODE</b> - Decode the following ANSI C escape sequences: <code>\a</code>,
         *        <code>\b</code>, <code>\f</code>, <code>\n</code>, <code>\r</code>, <code>\t</code>, <code>\v</code>,
         *        <code>\\</code>, <code>\?</code>, <code>\'</code>, <code>\"</code>, <code>\xHH</code> (hexadecimal),
         *        <code>\0OOO</code> (octal). Encodings that aren't valid remain in the output.
         *        </p>
         *        <p>
         *        <b>HEX_DECODE</b> - Decode a string of hexadecimal characters into a binary.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b> - Replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs these operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>JS_DECODE</b> - Decode JavaScript escape sequences. If a <code>\</code> <code>u</code>
         *        <code>HHHH</code> code is in the full-width ASCII code range of <code>FF01-FF5E</code>, then the
         *        higher byte is used to detect and adjust the lower byte. If not, only the lower byte is used and the
         *        higher byte is zeroed, causing a possible loss of information.
         *        </p>
         *        <p>
         *        <b>LOWERCASE</b> - Convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>MD5</b> - Calculate an MD5 hash from the data in the input. The computed hash is in a raw binary
         *        form.
         *        </p>
         *        <p>
         *        <b>NONE</b> - Specify <code>NONE</code> if you don't want any text transformations.
         *        </p>
         *        <p>
         *        <b>NORMALIZE_PATH</b> - Remove multiple slashes, directory self-references, and directory
         *        back-references that are not at the beginning of the input from an input string.
         *        </p>
         *        <p>
         *        <b>NORMALIZE_PATH_WIN</b> - This is the same as <code>NORMALIZE_PATH</code>, but first converts
         *        backslash characters to forward slashes.
         *        </p>
         *        <p>
         *        <b>REMOVE_NULLS</b> - Remove all <code>NULL</code> bytes from the input.
         *        </p>
         *        <p>
         *        <b>REPLACE_COMMENTS</b> - Replace each occurrence of a C-style comment (<code>/* ... *&#47;</code>)
         *        with a single space. Multiple consecutive occurrences are not compressed. Unterminated comments are
         *        also replaced with a space (ASCII 0x20). However, a standalone termination of a comment (
         *        <code>*&#47;</code>) is not acted upon.
         *        </p>
         *        <p>
         *        <b>REPLACE_NULLS</b> - Replace NULL bytes in the input with space characters (ASCII <code>0x20</code>
         *        ).
         *        </p>
         *        <p>
         *        <b>SQL_HEX_DECODE</b> - Decode SQL hex data. Example (<code>0x414243</code>) will be decoded to (
         *        <code>ABC</code>).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b> - Decode a URL-encoded value.
         *        </p>
         *        <p>
         *        <b>URL_DECODE_UNI</b> - Like <code>URL_DECODE</code>, but with support for Microsoft-specific
         *        <code>%u</code> encoding. If the code is in the full-width ASCII code range of <code>FF01-FF5E</code>,
         *        the higher byte is used to detect and adjust the lower byte. Otherwise, only the lower byte is used
         *        and the higher byte is zeroed.
         *        </p>
         *        <p>
         *        <b>UTF8_TO_UNICODE</b> - Convert all UTF-8 character sequences to Unicode. This helps input
         *        normalization, and minimizing false-positives and false-negatives for non-English languages.
         * @see TextTransformationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformationType
         */
        Builder type(TextTransformationType type);
    }

    static final class BuilderImpl implements Builder {
        private Integer priority;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(TextTransformation model) {
            priority(model.priority);
            type(model.type);
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(TextTransformationType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public TextTransformation build() {
            return new TextTransformation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
