/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.swf;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.swf.model.CountClosedWorkflowExecutionsRequest;
import software.amazon.awssdk.services.swf.model.CountClosedWorkflowExecutionsResponse;
import software.amazon.awssdk.services.swf.model.CountOpenWorkflowExecutionsRequest;
import software.amazon.awssdk.services.swf.model.CountOpenWorkflowExecutionsResponse;
import software.amazon.awssdk.services.swf.model.CountPendingActivityTasksRequest;
import software.amazon.awssdk.services.swf.model.CountPendingActivityTasksResponse;
import software.amazon.awssdk.services.swf.model.CountPendingDecisionTasksRequest;
import software.amazon.awssdk.services.swf.model.CountPendingDecisionTasksResponse;
import software.amazon.awssdk.services.swf.model.DefaultUndefinedException;
import software.amazon.awssdk.services.swf.model.DeprecateActivityTypeRequest;
import software.amazon.awssdk.services.swf.model.DeprecateActivityTypeResponse;
import software.amazon.awssdk.services.swf.model.DeprecateDomainRequest;
import software.amazon.awssdk.services.swf.model.DeprecateDomainResponse;
import software.amazon.awssdk.services.swf.model.DeprecateWorkflowTypeRequest;
import software.amazon.awssdk.services.swf.model.DeprecateWorkflowTypeResponse;
import software.amazon.awssdk.services.swf.model.DescribeActivityTypeRequest;
import software.amazon.awssdk.services.swf.model.DescribeActivityTypeResponse;
import software.amazon.awssdk.services.swf.model.DescribeDomainRequest;
import software.amazon.awssdk.services.swf.model.DescribeDomainResponse;
import software.amazon.awssdk.services.swf.model.DescribeWorkflowExecutionRequest;
import software.amazon.awssdk.services.swf.model.DescribeWorkflowExecutionResponse;
import software.amazon.awssdk.services.swf.model.DescribeWorkflowTypeRequest;
import software.amazon.awssdk.services.swf.model.DescribeWorkflowTypeResponse;
import software.amazon.awssdk.services.swf.model.DomainAlreadyExistsException;
import software.amazon.awssdk.services.swf.model.DomainDeprecatedException;
import software.amazon.awssdk.services.swf.model.GetWorkflowExecutionHistoryRequest;
import software.amazon.awssdk.services.swf.model.GetWorkflowExecutionHistoryResponse;
import software.amazon.awssdk.services.swf.model.LimitExceededException;
import software.amazon.awssdk.services.swf.model.ListActivityTypesRequest;
import software.amazon.awssdk.services.swf.model.ListActivityTypesResponse;
import software.amazon.awssdk.services.swf.model.ListClosedWorkflowExecutionsRequest;
import software.amazon.awssdk.services.swf.model.ListClosedWorkflowExecutionsResponse;
import software.amazon.awssdk.services.swf.model.ListDomainsRequest;
import software.amazon.awssdk.services.swf.model.ListDomainsResponse;
import software.amazon.awssdk.services.swf.model.ListOpenWorkflowExecutionsRequest;
import software.amazon.awssdk.services.swf.model.ListOpenWorkflowExecutionsResponse;
import software.amazon.awssdk.services.swf.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.swf.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.swf.model.ListWorkflowTypesRequest;
import software.amazon.awssdk.services.swf.model.ListWorkflowTypesResponse;
import software.amazon.awssdk.services.swf.model.OperationNotPermittedException;
import software.amazon.awssdk.services.swf.model.PollForActivityTaskRequest;
import software.amazon.awssdk.services.swf.model.PollForActivityTaskResponse;
import software.amazon.awssdk.services.swf.model.PollForDecisionTaskRequest;
import software.amazon.awssdk.services.swf.model.PollForDecisionTaskResponse;
import software.amazon.awssdk.services.swf.model.RecordActivityTaskHeartbeatRequest;
import software.amazon.awssdk.services.swf.model.RecordActivityTaskHeartbeatResponse;
import software.amazon.awssdk.services.swf.model.RegisterActivityTypeRequest;
import software.amazon.awssdk.services.swf.model.RegisterActivityTypeResponse;
import software.amazon.awssdk.services.swf.model.RegisterDomainRequest;
import software.amazon.awssdk.services.swf.model.RegisterDomainResponse;
import software.amazon.awssdk.services.swf.model.RegisterWorkflowTypeRequest;
import software.amazon.awssdk.services.swf.model.RegisterWorkflowTypeResponse;
import software.amazon.awssdk.services.swf.model.RequestCancelWorkflowExecutionRequest;
import software.amazon.awssdk.services.swf.model.RequestCancelWorkflowExecutionResponse;
import software.amazon.awssdk.services.swf.model.RespondActivityTaskCanceledRequest;
import software.amazon.awssdk.services.swf.model.RespondActivityTaskCanceledResponse;
import software.amazon.awssdk.services.swf.model.RespondActivityTaskCompletedRequest;
import software.amazon.awssdk.services.swf.model.RespondActivityTaskCompletedResponse;
import software.amazon.awssdk.services.swf.model.RespondActivityTaskFailedRequest;
import software.amazon.awssdk.services.swf.model.RespondActivityTaskFailedResponse;
import software.amazon.awssdk.services.swf.model.RespondDecisionTaskCompletedRequest;
import software.amazon.awssdk.services.swf.model.RespondDecisionTaskCompletedResponse;
import software.amazon.awssdk.services.swf.model.SignalWorkflowExecutionRequest;
import software.amazon.awssdk.services.swf.model.SignalWorkflowExecutionResponse;
import software.amazon.awssdk.services.swf.model.StartWorkflowExecutionRequest;
import software.amazon.awssdk.services.swf.model.StartWorkflowExecutionResponse;
import software.amazon.awssdk.services.swf.model.SwfException;
import software.amazon.awssdk.services.swf.model.SwfRequest;
import software.amazon.awssdk.services.swf.model.TagResourceRequest;
import software.amazon.awssdk.services.swf.model.TagResourceResponse;
import software.amazon.awssdk.services.swf.model.TerminateWorkflowExecutionRequest;
import software.amazon.awssdk.services.swf.model.TerminateWorkflowExecutionResponse;
import software.amazon.awssdk.services.swf.model.TooManyTagsException;
import software.amazon.awssdk.services.swf.model.TypeAlreadyExistsException;
import software.amazon.awssdk.services.swf.model.TypeDeprecatedException;
import software.amazon.awssdk.services.swf.model.UndeprecateActivityTypeRequest;
import software.amazon.awssdk.services.swf.model.UndeprecateActivityTypeResponse;
import software.amazon.awssdk.services.swf.model.UndeprecateDomainRequest;
import software.amazon.awssdk.services.swf.model.UndeprecateDomainResponse;
import software.amazon.awssdk.services.swf.model.UndeprecateWorkflowTypeRequest;
import software.amazon.awssdk.services.swf.model.UndeprecateWorkflowTypeResponse;
import software.amazon.awssdk.services.swf.model.UnknownResourceException;
import software.amazon.awssdk.services.swf.model.UntagResourceRequest;
import software.amazon.awssdk.services.swf.model.UntagResourceResponse;
import software.amazon.awssdk.services.swf.model.WorkflowExecutionAlreadyStartedException;
import software.amazon.awssdk.services.swf.paginators.GetWorkflowExecutionHistoryPublisher;
import software.amazon.awssdk.services.swf.paginators.ListActivityTypesPublisher;
import software.amazon.awssdk.services.swf.paginators.ListClosedWorkflowExecutionsPublisher;
import software.amazon.awssdk.services.swf.paginators.ListDomainsPublisher;
import software.amazon.awssdk.services.swf.paginators.ListOpenWorkflowExecutionsPublisher;
import software.amazon.awssdk.services.swf.paginators.ListWorkflowTypesPublisher;
import software.amazon.awssdk.services.swf.paginators.PollForDecisionTaskPublisher;
import software.amazon.awssdk.services.swf.transform.CountClosedWorkflowExecutionsRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.CountOpenWorkflowExecutionsRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.CountPendingActivityTasksRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.CountPendingDecisionTasksRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DeprecateActivityTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DeprecateDomainRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DeprecateWorkflowTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DescribeActivityTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DescribeWorkflowExecutionRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.DescribeWorkflowTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.GetWorkflowExecutionHistoryRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.ListActivityTypesRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.ListClosedWorkflowExecutionsRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.ListOpenWorkflowExecutionsRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.ListWorkflowTypesRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.PollForActivityTaskRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.PollForDecisionTaskRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RecordActivityTaskHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RegisterActivityTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RegisterDomainRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RegisterWorkflowTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RequestCancelWorkflowExecutionRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RespondActivityTaskCanceledRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RespondActivityTaskCompletedRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RespondActivityTaskFailedRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.RespondDecisionTaskCompletedRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.SignalWorkflowExecutionRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.StartWorkflowExecutionRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.TerminateWorkflowExecutionRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.UndeprecateActivityTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.UndeprecateDomainRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.UndeprecateWorkflowTypeRequestMarshaller;
import software.amazon.awssdk.services.swf.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SwfAsyncClient}.
 *
 * @see SwfAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSwfAsyncClient implements SwfAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSwfAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SwfServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSwfAsyncClient(SwfServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns the number of closed workflow executions within the given domain that meet the specified filtering
     * criteria.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagFilter.tag</code>: String constraint. The key is <code>swf:tagFilter.tag</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.name</code>: String constraint. The key is <code>swf:typeFilter.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.version</code>: String constraint. The key is <code>swf:typeFilter.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param countClosedWorkflowExecutionsRequest
     * @return A Java Future containing the result of the CountClosedWorkflowExecutions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.CountClosedWorkflowExecutions
     */
    @Override
    public CompletableFuture<CountClosedWorkflowExecutionsResponse> countClosedWorkflowExecutions(
            CountClosedWorkflowExecutionsRequest countClosedWorkflowExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                countClosedWorkflowExecutionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CountClosedWorkflowExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CountClosedWorkflowExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CountClosedWorkflowExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CountClosedWorkflowExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CountClosedWorkflowExecutionsRequest, CountClosedWorkflowExecutionsResponse>()
                            .withOperationName("CountClosedWorkflowExecutions")
                            .withMarshaller(new CountClosedWorkflowExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(countClosedWorkflowExecutionsRequest));
            CompletableFuture<CountClosedWorkflowExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of open workflow executions within the given domain that meet the specified filtering
     * criteria.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagFilter.tag</code>: String constraint. The key is <code>swf:tagFilter.tag</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.name</code>: String constraint. The key is <code>swf:typeFilter.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.version</code>: String constraint. The key is <code>swf:typeFilter.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param countOpenWorkflowExecutionsRequest
     * @return A Java Future containing the result of the CountOpenWorkflowExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.CountOpenWorkflowExecutions
     */
    @Override
    public CompletableFuture<CountOpenWorkflowExecutionsResponse> countOpenWorkflowExecutions(
            CountOpenWorkflowExecutionsRequest countOpenWorkflowExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, countOpenWorkflowExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CountOpenWorkflowExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CountOpenWorkflowExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CountOpenWorkflowExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CountOpenWorkflowExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CountOpenWorkflowExecutionsRequest, CountOpenWorkflowExecutionsResponse>()
                            .withOperationName("CountOpenWorkflowExecutions")
                            .withMarshaller(new CountOpenWorkflowExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(countOpenWorkflowExecutionsRequest));
            CompletableFuture<CountOpenWorkflowExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the estimated number of activity tasks in the specified task list. The count returned is an approximation
     * and isn't guaranteed to be exact. If you specify a task list that no activity task was ever scheduled in then
     * <code>0</code> is returned.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the <code>taskList.name</code> parameter by using a <code>Condition</code> element with the
     * <code>swf:taskList.name</code> key to allow the action to access only certain task lists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param countPendingActivityTasksRequest
     * @return A Java Future containing the result of the CountPendingActivityTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.CountPendingActivityTasks
     */
    @Override
    public CompletableFuture<CountPendingActivityTasksResponse> countPendingActivityTasks(
            CountPendingActivityTasksRequest countPendingActivityTasksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, countPendingActivityTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CountPendingActivityTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CountPendingActivityTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CountPendingActivityTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CountPendingActivityTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CountPendingActivityTasksRequest, CountPendingActivityTasksResponse>()
                            .withOperationName("CountPendingActivityTasks")
                            .withMarshaller(new CountPendingActivityTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(countPendingActivityTasksRequest));
            CompletableFuture<CountPendingActivityTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the estimated number of decision tasks in the specified task list. The count returned is an approximation
     * and isn't guaranteed to be exact. If you specify a task list that no decision task was ever scheduled in then
     * <code>0</code> is returned.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the <code>taskList.name</code> parameter by using a <code>Condition</code> element with the
     * <code>swf:taskList.name</code> key to allow the action to access only certain task lists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param countPendingDecisionTasksRequest
     * @return A Java Future containing the result of the CountPendingDecisionTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.CountPendingDecisionTasks
     */
    @Override
    public CompletableFuture<CountPendingDecisionTasksResponse> countPendingDecisionTasks(
            CountPendingDecisionTasksRequest countPendingDecisionTasksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, countPendingDecisionTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CountPendingDecisionTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CountPendingDecisionTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CountPendingDecisionTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CountPendingDecisionTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CountPendingDecisionTasksRequest, CountPendingDecisionTasksResponse>()
                            .withOperationName("CountPendingDecisionTasks")
                            .withMarshaller(new CountPendingDecisionTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(countPendingDecisionTasksRequest));
            CompletableFuture<CountPendingDecisionTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deprecates the specified <i>activity type</i>. After an activity type has been deprecated, you cannot create new
     * tasks of that activity type. Tasks of this type that were scheduled before the type was deprecated continue to
     * run.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>activityType.name</code>: String constraint. The key is <code>swf:activityType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>activityType.version</code>: String constraint. The key is <code>swf:activityType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param deprecateActivityTypeRequest
     * @return A Java Future containing the result of the DeprecateActivityType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>TypeDeprecatedException Returned when the specified activity or workflow type was already deprecated.
     *         </li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DeprecateActivityType
     */
    @Override
    public CompletableFuture<DeprecateActivityTypeResponse> deprecateActivityType(
            DeprecateActivityTypeRequest deprecateActivityTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deprecateActivityTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeprecateActivityType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeprecateActivityTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeprecateActivityTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeprecateActivityTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeprecateActivityTypeRequest, DeprecateActivityTypeResponse>()
                            .withOperationName("DeprecateActivityType")
                            .withMarshaller(new DeprecateActivityTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deprecateActivityTypeRequest));
            CompletableFuture<DeprecateActivityTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deprecates the specified domain. After a domain has been deprecated it cannot be used to create new workflow
     * executions or register new types. However, you can still use visibility actions on this domain. Deprecating a
     * domain also deprecates all activity and workflow types registered in the domain. Executions that were started
     * before the domain was deprecated continues to run.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param deprecateDomainRequest
     * @return A Java Future containing the result of the DeprecateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>DomainDeprecatedException Returned when the specified domain has been deprecated.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DeprecateDomain
     */
    @Override
    public CompletableFuture<DeprecateDomainResponse> deprecateDomain(DeprecateDomainRequest deprecateDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deprecateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeprecateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeprecateDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeprecateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeprecateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeprecateDomainRequest, DeprecateDomainResponse>()
                            .withOperationName("DeprecateDomain")
                            .withMarshaller(new DeprecateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deprecateDomainRequest));
            CompletableFuture<DeprecateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deprecates the specified <i>workflow type</i>. After a workflow type has been deprecated, you cannot create new
     * executions of that type. Executions that were started before the type was deprecated continues to run. A
     * deprecated workflow type may still be used when calling visibility actions.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>workflowType.name</code>: String constraint. The key is <code>swf:workflowType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>workflowType.version</code>: String constraint. The key is <code>swf:workflowType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param deprecateWorkflowTypeRequest
     * @return A Java Future containing the result of the DeprecateWorkflowType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>TypeDeprecatedException Returned when the specified activity or workflow type was already deprecated.
     *         </li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DeprecateWorkflowType
     */
    @Override
    public CompletableFuture<DeprecateWorkflowTypeResponse> deprecateWorkflowType(
            DeprecateWorkflowTypeRequest deprecateWorkflowTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deprecateWorkflowTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeprecateWorkflowType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeprecateWorkflowTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeprecateWorkflowTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeprecateWorkflowTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeprecateWorkflowTypeRequest, DeprecateWorkflowTypeResponse>()
                            .withOperationName("DeprecateWorkflowType")
                            .withMarshaller(new DeprecateWorkflowTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deprecateWorkflowTypeRequest));
            CompletableFuture<DeprecateWorkflowTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified activity type. This includes configuration settings provided when the
     * type was registered and other general information about the type.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>activityType.name</code>: String constraint. The key is <code>swf:activityType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>activityType.version</code>: String constraint. The key is <code>swf:activityType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param describeActivityTypeRequest
     * @return A Java Future containing the result of the DescribeActivityType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DescribeActivityType
     */
    @Override
    public CompletableFuture<DescribeActivityTypeResponse> describeActivityType(
            DescribeActivityTypeRequest describeActivityTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeActivityTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeActivityType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeActivityTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeActivityTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeActivityTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeActivityTypeRequest, DescribeActivityTypeResponse>()
                            .withOperationName("DescribeActivityType")
                            .withMarshaller(new DescribeActivityTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeActivityTypeRequest));
            CompletableFuture<DescribeActivityTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified domain, including description and status.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param describeDomainRequest
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DescribeDomain
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain")
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainRequest));
            CompletableFuture<DescribeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified workflow execution including its type and some statistics.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param describeWorkflowExecutionRequest
     * @return A Java Future containing the result of the DescribeWorkflowExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DescribeWorkflowExecution
     */
    @Override
    public CompletableFuture<DescribeWorkflowExecutionResponse> describeWorkflowExecution(
            DescribeWorkflowExecutionRequest describeWorkflowExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkflowExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkflowExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkflowExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkflowExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkflowExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkflowExecutionRequest, DescribeWorkflowExecutionResponse>()
                            .withOperationName("DescribeWorkflowExecution")
                            .withMarshaller(new DescribeWorkflowExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeWorkflowExecutionRequest));
            CompletableFuture<DescribeWorkflowExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified <i>workflow type</i>. This includes configuration settings specified when
     * the type was registered and other information such as creation date, current status, etc.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>workflowType.name</code>: String constraint. The key is <code>swf:workflowType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>workflowType.version</code>: String constraint. The key is <code>swf:workflowType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param describeWorkflowTypeRequest
     * @return A Java Future containing the result of the DescribeWorkflowType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.DescribeWorkflowType
     */
    @Override
    public CompletableFuture<DescribeWorkflowTypeResponse> describeWorkflowType(
            DescribeWorkflowTypeRequest describeWorkflowTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkflowTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkflowType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkflowTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkflowTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkflowTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkflowTypeRequest, DescribeWorkflowTypeResponse>()
                            .withOperationName("DescribeWorkflowType")
                            .withMarshaller(new DescribeWorkflowTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeWorkflowTypeRequest));
            CompletableFuture<DescribeWorkflowTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the history of the specified workflow execution. The results may be split into multiple pages. To
     * retrieve subsequent pages, make the call again using the <code>nextPageToken</code> returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param getWorkflowExecutionHistoryRequest
     * @return A Java Future containing the result of the GetWorkflowExecutionHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.GetWorkflowExecutionHistory
     */
    @Override
    public CompletableFuture<GetWorkflowExecutionHistoryResponse> getWorkflowExecutionHistory(
            GetWorkflowExecutionHistoryRequest getWorkflowExecutionHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkflowExecutionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkflowExecutionHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkflowExecutionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkflowExecutionHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWorkflowExecutionHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkflowExecutionHistoryRequest, GetWorkflowExecutionHistoryResponse>()
                            .withOperationName("GetWorkflowExecutionHistory")
                            .withMarshaller(new GetWorkflowExecutionHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getWorkflowExecutionHistoryRequest));
            CompletableFuture<GetWorkflowExecutionHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the history of the specified workflow execution. The results may be split into multiple pages. To
     * retrieve subsequent pages, make the call again using the <code>nextPageToken</code> returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getWorkflowExecutionHistory(software.amazon.awssdk.services.swf.model.GetWorkflowExecutionHistoryRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.GetWorkflowExecutionHistoryPublisher publisher = client.getWorkflowExecutionHistoryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.GetWorkflowExecutionHistoryPublisher publisher = client.getWorkflowExecutionHistoryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.GetWorkflowExecutionHistoryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.GetWorkflowExecutionHistoryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getWorkflowExecutionHistory(software.amazon.awssdk.services.swf.model.GetWorkflowExecutionHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getWorkflowExecutionHistoryRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.GetWorkflowExecutionHistory
     */
    public GetWorkflowExecutionHistoryPublisher getWorkflowExecutionHistoryPaginator(
            GetWorkflowExecutionHistoryRequest getWorkflowExecutionHistoryRequest) {
        return new GetWorkflowExecutionHistoryPublisher(this, applyPaginatorUserAgent(getWorkflowExecutionHistoryRequest));
    }

    /**
     * <p>
     * Returns information about all activities registered in the specified domain that match the specified name and
     * registration status. The result includes information like creation date, current status of the activity, etc. The
     * results may be split into multiple pages. To retrieve subsequent pages, make the call again using the
     * <code>nextPageToken</code> returned by the initial call.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param listActivityTypesRequest
     * @return A Java Future containing the result of the ListActivityTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListActivityTypes
     */
    @Override
    public CompletableFuture<ListActivityTypesResponse> listActivityTypes(ListActivityTypesRequest listActivityTypesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActivityTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActivityTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListActivityTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListActivityTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListActivityTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListActivityTypesRequest, ListActivityTypesResponse>()
                            .withOperationName("ListActivityTypes")
                            .withMarshaller(new ListActivityTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listActivityTypesRequest));
            CompletableFuture<ListActivityTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about all activities registered in the specified domain that match the specified name and
     * registration status. The result includes information like creation date, current status of the activity, etc. The
     * results may be split into multiple pages. To retrieve subsequent pages, make the call again using the
     * <code>nextPageToken</code> returned by the initial call.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listActivityTypes(software.amazon.awssdk.services.swf.model.ListActivityTypesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListActivityTypesPublisher publisher = client.listActivityTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListActivityTypesPublisher publisher = client.listActivityTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.ListActivityTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.ListActivityTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listActivityTypes(software.amazon.awssdk.services.swf.model.ListActivityTypesRequest)} operation.</b>
     * </p>
     *
     * @param listActivityTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListActivityTypes
     */
    public ListActivityTypesPublisher listActivityTypesPaginator(ListActivityTypesRequest listActivityTypesRequest) {
        return new ListActivityTypesPublisher(this, applyPaginatorUserAgent(listActivityTypesRequest));
    }

    /**
     * <p>
     * Returns a list of closed workflow executions in the specified domain that meet the filtering criteria. The
     * results may be split into multiple pages. To retrieve subsequent pages, make the call again using the
     * nextPageToken returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagFilter.tag</code>: String constraint. The key is <code>swf:tagFilter.tag</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.name</code>: String constraint. The key is <code>swf:typeFilter.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.version</code>: String constraint. The key is <code>swf:typeFilter.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param listClosedWorkflowExecutionsRequest
     * @return A Java Future containing the result of the ListClosedWorkflowExecutions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListClosedWorkflowExecutions
     */
    @Override
    public CompletableFuture<ListClosedWorkflowExecutionsResponse> listClosedWorkflowExecutions(
            ListClosedWorkflowExecutionsRequest listClosedWorkflowExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClosedWorkflowExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClosedWorkflowExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClosedWorkflowExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListClosedWorkflowExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListClosedWorkflowExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClosedWorkflowExecutionsRequest, ListClosedWorkflowExecutionsResponse>()
                            .withOperationName("ListClosedWorkflowExecutions")
                            .withMarshaller(new ListClosedWorkflowExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listClosedWorkflowExecutionsRequest));
            CompletableFuture<ListClosedWorkflowExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of closed workflow executions in the specified domain that meet the filtering criteria. The
     * results may be split into multiple pages. To retrieve subsequent pages, make the call again using the
     * nextPageToken returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagFilter.tag</code>: String constraint. The key is <code>swf:tagFilter.tag</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.name</code>: String constraint. The key is <code>swf:typeFilter.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.version</code>: String constraint. The key is <code>swf:typeFilter.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listClosedWorkflowExecutions(software.amazon.awssdk.services.swf.model.ListClosedWorkflowExecutionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListClosedWorkflowExecutionsPublisher publisher = client.listClosedWorkflowExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListClosedWorkflowExecutionsPublisher publisher = client.listClosedWorkflowExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.ListClosedWorkflowExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.ListClosedWorkflowExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClosedWorkflowExecutions(software.amazon.awssdk.services.swf.model.ListClosedWorkflowExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listClosedWorkflowExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListClosedWorkflowExecutions
     */
    public ListClosedWorkflowExecutionsPublisher listClosedWorkflowExecutionsPaginator(
            ListClosedWorkflowExecutionsRequest listClosedWorkflowExecutionsRequest) {
        return new ListClosedWorkflowExecutionsPublisher(this, applyPaginatorUserAgent(listClosedWorkflowExecutionsRequest));
    }

    /**
     * <p>
     * Returns the list of domains registered in the account. The results may be split into multiple pages. To retrieve
     * subsequent pages, make the call again using the nextPageToken returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains. The
     * element must be set to <code>arn:aws:swf::AccountID:domain/*</code>, where <i>AccountID</i> is the account ID,
     * with no dashes.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListDomains
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainsRequest));
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of domains registered in the account. The results may be split into multiple pages. To retrieve
     * subsequent pages, make the call again using the nextPageToken returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains. The
     * element must be set to <code>arn:aws:swf::AccountID:domain/*</code>, where <i>AccountID</i> is the account ID,
     * with no dashes.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDomains(software.amazon.awssdk.services.swf.model.ListDomainsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.swf.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListDomains
     */
    public ListDomainsPublisher listDomainsPaginator(ListDomainsRequest listDomainsRequest) {
        return new ListDomainsPublisher(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * Returns a list of open workflow executions in the specified domain that meet the filtering criteria. The results
     * may be split into multiple pages. To retrieve subsequent pages, make the call again using the nextPageToken
     * returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagFilter.tag</code>: String constraint. The key is <code>swf:tagFilter.tag</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.name</code>: String constraint. The key is <code>swf:typeFilter.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.version</code>: String constraint. The key is <code>swf:typeFilter.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param listOpenWorkflowExecutionsRequest
     * @return A Java Future containing the result of the ListOpenWorkflowExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListOpenWorkflowExecutions
     */
    @Override
    public CompletableFuture<ListOpenWorkflowExecutionsResponse> listOpenWorkflowExecutions(
            ListOpenWorkflowExecutionsRequest listOpenWorkflowExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOpenWorkflowExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOpenWorkflowExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOpenWorkflowExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOpenWorkflowExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOpenWorkflowExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOpenWorkflowExecutionsRequest, ListOpenWorkflowExecutionsResponse>()
                            .withOperationName("ListOpenWorkflowExecutions")
                            .withMarshaller(new ListOpenWorkflowExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOpenWorkflowExecutionsRequest));
            CompletableFuture<ListOpenWorkflowExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of open workflow executions in the specified domain that meet the filtering criteria. The results
     * may be split into multiple pages. To retrieve subsequent pages, make the call again using the nextPageToken
     * returned by the initial call.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagFilter.tag</code>: String constraint. The key is <code>swf:tagFilter.tag</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.name</code>: String constraint. The key is <code>swf:typeFilter.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>typeFilter.version</code>: String constraint. The key is <code>swf:typeFilter.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOpenWorkflowExecutions(software.amazon.awssdk.services.swf.model.ListOpenWorkflowExecutionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListOpenWorkflowExecutionsPublisher publisher = client.listOpenWorkflowExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListOpenWorkflowExecutionsPublisher publisher = client.listOpenWorkflowExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.ListOpenWorkflowExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.ListOpenWorkflowExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOpenWorkflowExecutions(software.amazon.awssdk.services.swf.model.ListOpenWorkflowExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOpenWorkflowExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListOpenWorkflowExecutions
     */
    public ListOpenWorkflowExecutionsPublisher listOpenWorkflowExecutionsPaginator(
            ListOpenWorkflowExecutionsRequest listOpenWorkflowExecutionsRequest) {
        return new ListOpenWorkflowExecutionsPublisher(this, applyPaginatorUserAgent(listOpenWorkflowExecutionsRequest));
    }

    /**
     * <p>
     * List tags for a given domain.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListTagsForResource
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about workflow types in the specified domain. The results may be split into multiple pages
     * that can be retrieved by making the call repeatedly.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param listWorkflowTypesRequest
     * @return A Java Future containing the result of the ListWorkflowTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListWorkflowTypes
     */
    @Override
    public CompletableFuture<ListWorkflowTypesResponse> listWorkflowTypes(ListWorkflowTypesRequest listWorkflowTypesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkflowTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkflowTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkflowTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkflowTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkflowTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkflowTypesRequest, ListWorkflowTypesResponse>()
                            .withOperationName("ListWorkflowTypes")
                            .withMarshaller(new ListWorkflowTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkflowTypesRequest));
            CompletableFuture<ListWorkflowTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about workflow types in the specified domain. The results may be split into multiple pages
     * that can be retrieved by making the call repeatedly.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkflowTypes(software.amazon.awssdk.services.swf.model.ListWorkflowTypesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListWorkflowTypesPublisher publisher = client.listWorkflowTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.ListWorkflowTypesPublisher publisher = client.listWorkflowTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.ListWorkflowTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.ListWorkflowTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkflowTypes(software.amazon.awssdk.services.swf.model.ListWorkflowTypesRequest)} operation.</b>
     * </p>
     *
     * @param listWorkflowTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.ListWorkflowTypes
     */
    public ListWorkflowTypesPublisher listWorkflowTypesPaginator(ListWorkflowTypesRequest listWorkflowTypesRequest) {
        return new ListWorkflowTypesPublisher(this, applyPaginatorUserAgent(listWorkflowTypesRequest));
    }

    /**
     * <p>
     * Used by workers to get an <a>ActivityTask</a> from the specified activity <code>taskList</code>. This initiates a
     * long poll, where the service holds the HTTP connection open and responds as soon as a task becomes available. The
     * maximum time the service holds on to the request before responding is 60 seconds. If no task is available within
     * 60 seconds, the poll returns an empty result. An empty result, in this context, means that an ActivityTask is
     * returned, but that the value of taskToken is an empty string. If a task is returned, the worker should use its
     * type to identify and process it correctly.
     * </p>
     * <important>
     * <p>
     * Workers should set their client side socket timeout to at least 70 seconds (10 seconds higher than the maximum
     * time service may hold the poll request).
     * </p>
     * </important>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the <code>taskList.name</code> parameter by using a <code>Condition</code> element with the
     * <code>swf:taskList.name</code> key to allow the action to access only certain task lists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param pollForActivityTaskRequest
     * @return A Java Future containing the result of the PollForActivityTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.PollForActivityTask
     */
    @Override
    public CompletableFuture<PollForActivityTaskResponse> pollForActivityTask(
            PollForActivityTaskRequest pollForActivityTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pollForActivityTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PollForActivityTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PollForActivityTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PollForActivityTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PollForActivityTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PollForActivityTaskRequest, PollForActivityTaskResponse>()
                            .withOperationName("PollForActivityTask")
                            .withMarshaller(new PollForActivityTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(pollForActivityTaskRequest));
            CompletableFuture<PollForActivityTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by deciders to get a <a>DecisionTask</a> from the specified decision <code>taskList</code>. A decision task
     * may be returned for any open workflow execution that is using the specified task list. The task includes a
     * paginated view of the history of the workflow execution. The decider should use the workflow type and the history
     * to determine how to properly handle the task.
     * </p>
     * <p>
     * This action initiates a long poll, where the service holds the HTTP connection open and responds as soon a task
     * becomes available. If no decision task is available in the specified task list before the timeout of 60 seconds
     * expires, an empty result is returned. An empty result, in this context, means that a DecisionTask is returned,
     * but that the value of taskToken is an empty string.
     * </p>
     * <important>
     * <p>
     * Deciders should set their client side socket timeout to at least 70 seconds (10 seconds higher than the timeout).
     * </p>
     * </important> <important>
     * <p>
     * Because the number of workflow history events for a single workflow execution might be very large, the result
     * returned might be split up across a number of pages. To retrieve subsequent pages, make additional calls to
     * <code>PollForDecisionTask</code> using the <code>nextPageToken</code> returned by the initial call. Note that you
     * do <i>not</i> call <code>GetWorkflowExecutionHistory</code> with this <code>nextPageToken</code>. Instead, call
     * <code>PollForDecisionTask</code> again.
     * </p>
     * </important>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the <code>taskList.name</code> parameter by using a <code>Condition</code> element with the
     * <code>swf:taskList.name</code> key to allow the action to access only certain task lists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param pollForDecisionTaskRequest
     * @return A Java Future containing the result of the PollForDecisionTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.PollForDecisionTask
     */
    @Override
    public CompletableFuture<PollForDecisionTaskResponse> pollForDecisionTask(
            PollForDecisionTaskRequest pollForDecisionTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pollForDecisionTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PollForDecisionTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PollForDecisionTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PollForDecisionTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PollForDecisionTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PollForDecisionTaskRequest, PollForDecisionTaskResponse>()
                            .withOperationName("PollForDecisionTask")
                            .withMarshaller(new PollForDecisionTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(pollForDecisionTaskRequest));
            CompletableFuture<PollForDecisionTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by deciders to get a <a>DecisionTask</a> from the specified decision <code>taskList</code>. A decision task
     * may be returned for any open workflow execution that is using the specified task list. The task includes a
     * paginated view of the history of the workflow execution. The decider should use the workflow type and the history
     * to determine how to properly handle the task.
     * </p>
     * <p>
     * This action initiates a long poll, where the service holds the HTTP connection open and responds as soon a task
     * becomes available. If no decision task is available in the specified task list before the timeout of 60 seconds
     * expires, an empty result is returned. An empty result, in this context, means that a DecisionTask is returned,
     * but that the value of taskToken is an empty string.
     * </p>
     * <important>
     * <p>
     * Deciders should set their client side socket timeout to at least 70 seconds (10 seconds higher than the timeout).
     * </p>
     * </important> <important>
     * <p>
     * Because the number of workflow history events for a single workflow execution might be very large, the result
     * returned might be split up across a number of pages. To retrieve subsequent pages, make additional calls to
     * <code>PollForDecisionTask</code> using the <code>nextPageToken</code> returned by the initial call. Note that you
     * do <i>not</i> call <code>GetWorkflowExecutionHistory</code> with this <code>nextPageToken</code>. Instead, call
     * <code>PollForDecisionTask</code> again.
     * </p>
     * </important>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the <code>taskList.name</code> parameter by using a <code>Condition</code> element with the
     * <code>swf:taskList.name</code> key to allow the action to access only certain task lists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #pollForDecisionTask(software.amazon.awssdk.services.swf.model.PollForDecisionTaskRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.PollForDecisionTaskPublisher publisher = client.pollForDecisionTaskPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.swf.paginators.PollForDecisionTaskPublisher publisher = client.pollForDecisionTaskPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.swf.model.PollForDecisionTaskResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.swf.model.PollForDecisionTaskResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maximumPageSize won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #pollForDecisionTask(software.amazon.awssdk.services.swf.model.PollForDecisionTaskRequest)} operation.</b>
     * </p>
     *
     * @param pollForDecisionTaskRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.PollForDecisionTask
     */
    public PollForDecisionTaskPublisher pollForDecisionTaskPaginator(PollForDecisionTaskRequest pollForDecisionTaskRequest) {
        return new PollForDecisionTaskPublisher(this, applyPaginatorUserAgent(pollForDecisionTaskRequest));
    }

    /**
     * <p>
     * Used by activity workers to report to the service that the <a>ActivityTask</a> represented by the specified
     * <code>taskToken</code> is still making progress. The worker can also specify details of the progress, for example
     * percent complete, using the <code>details</code> parameter. This action can also be used by the worker as a
     * mechanism to check if cancellation is being requested for the activity task. If a cancellation is being attempted
     * for the specified task, then the boolean <code>cancelRequested</code> flag returned by the service is set to
     * <code>true</code>.
     * </p>
     * <p>
     * This action resets the <code>taskHeartbeatTimeout</code> clock. The <code>taskHeartbeatTimeout</code> is
     * specified in <a>RegisterActivityType</a>.
     * </p>
     * <p>
     * This action doesn't in itself create an event in the workflow execution history. However, if the task times out,
     * the workflow execution history contains a <code>ActivityTaskTimedOut</code> event that contains the information
     * from the last heartbeat generated by the activity worker.
     * </p>
     * <note>
     * <p>
     * The <code>taskStartToCloseTimeout</code> of an activity type is the maximum duration of an activity task,
     * regardless of the number of <a>RecordActivityTaskHeartbeat</a> requests received. The
     * <code>taskStartToCloseTimeout</code> is also specified in <a>RegisterActivityType</a>.
     * </p>
     * </note> <note>
     * <p>
     * This operation is only useful for long-lived activities to report liveliness of the task and to determine if a
     * cancellation is being attempted.
     * </p>
     * </note> <important>
     * <p>
     * If the <code>cancelRequested</code> flag returns <code>true</code>, a cancellation is being attempted. If the
     * worker can cancel the activity, it should respond with <a>RespondActivityTaskCanceled</a>. Otherwise, it should
     * ignore the cancellation request.
     * </p>
     * </important>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param recordActivityTaskHeartbeatRequest
     * @return A Java Future containing the result of the RecordActivityTaskHeartbeat operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RecordActivityTaskHeartbeat
     */
    @Override
    public CompletableFuture<RecordActivityTaskHeartbeatResponse> recordActivityTaskHeartbeat(
            RecordActivityTaskHeartbeatRequest recordActivityTaskHeartbeatRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, recordActivityTaskHeartbeatRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecordActivityTaskHeartbeat");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RecordActivityTaskHeartbeatResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RecordActivityTaskHeartbeatResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RecordActivityTaskHeartbeatResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RecordActivityTaskHeartbeatRequest, RecordActivityTaskHeartbeatResponse>()
                            .withOperationName("RecordActivityTaskHeartbeat")
                            .withMarshaller(new RecordActivityTaskHeartbeatRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(recordActivityTaskHeartbeatRequest));
            CompletableFuture<RecordActivityTaskHeartbeatResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a new <i>activity type</i> along with its configuration settings in the specified domain.
     * </p>
     * <important>
     * <p>
     * A <code>TypeAlreadyExists</code> fault is returned if the type already exists in the domain. You cannot change
     * any configuration settings of the type after its registration, and it must be registered as a new version.
     * </p>
     * </important>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>defaultTaskList.name</code>: String constraint. The key is <code>swf:defaultTaskList.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>name</code>: String constraint. The key is <code>swf:name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>version</code>: String constraint. The key is <code>swf:version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param registerActivityTypeRequest
     * @return A Java Future containing the result of the RegisterActivityType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TypeAlreadyExistsException Returned if the type already exists in the specified domain. You may get
     *         this fault if you are registering a type that is either already registered or deprecated, or if you
     *         undeprecate a type that is currently registered.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RegisterActivityType
     */
    @Override
    public CompletableFuture<RegisterActivityTypeResponse> registerActivityType(
            RegisterActivityTypeRequest registerActivityTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerActivityTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterActivityType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterActivityTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterActivityTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterActivityTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterActivityTypeRequest, RegisterActivityTypeResponse>()
                            .withOperationName("RegisterActivityType")
                            .withMarshaller(new RegisterActivityTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerActivityTypeRequest));
            CompletableFuture<RegisterActivityTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a new domain.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You cannot use an IAM policy to control domain access for this action. The name of the domain being registered is
     * available as the resource of this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param registerDomainRequest
     * @return A Java Future containing the result of the RegisterDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DomainAlreadyExistsException Returned if the domain already exists. You may get this fault if you are
     *         registering a domain that is either already registered or deprecated, or if you undeprecate a domain that
     *         is currently registered.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>TooManyTagsException You've exceeded the number of tags allowed for a domain.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RegisterDomain
     */
    @Override
    public CompletableFuture<RegisterDomainResponse> registerDomain(RegisterDomainRequest registerDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterDomainRequest, RegisterDomainResponse>()
                            .withOperationName("RegisterDomain")
                            .withMarshaller(new RegisterDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerDomainRequest));
            CompletableFuture<RegisterDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a new <i>workflow type</i> and its configuration settings in the specified domain.
     * </p>
     * <p>
     * The retention period for the workflow history is set by the <a>RegisterDomain</a> action.
     * </p>
     * <important>
     * <p>
     * If the type already exists, then a <code>TypeAlreadyExists</code> fault is returned. You cannot change the
     * configuration settings of a workflow type once it is registered and it must be registered as a new version.
     * </p>
     * </important>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>defaultTaskList.name</code>: String constraint. The key is <code>swf:defaultTaskList.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>name</code>: String constraint. The key is <code>swf:name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>version</code>: String constraint. The key is <code>swf:version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param registerWorkflowTypeRequest
     * @return A Java Future containing the result of the RegisterWorkflowType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TypeAlreadyExistsException Returned if the type already exists in the specified domain. You may get
     *         this fault if you are registering a type that is either already registered or deprecated, or if you
     *         undeprecate a type that is currently registered.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RegisterWorkflowType
     */
    @Override
    public CompletableFuture<RegisterWorkflowTypeResponse> registerWorkflowType(
            RegisterWorkflowTypeRequest registerWorkflowTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerWorkflowTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterWorkflowType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterWorkflowTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterWorkflowTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterWorkflowTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterWorkflowTypeRequest, RegisterWorkflowTypeResponse>()
                            .withOperationName("RegisterWorkflowType")
                            .withMarshaller(new RegisterWorkflowTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerWorkflowTypeRequest));
            CompletableFuture<RegisterWorkflowTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Records a <code>WorkflowExecutionCancelRequested</code> event in the currently running workflow execution
     * identified by the given domain, workflowId, and runId. This logically requests the cancellation of the workflow
     * execution as a whole. It is up to the decider to take appropriate actions when it receives an execution history
     * with this event.
     * </p>
     * <note>
     * <p>
     * If the runId isn't specified, the <code>WorkflowExecutionCancelRequested</code> event is recorded in the history
     * of the current open workflow execution with the specified workflowId in the domain.
     * </p>
     * </note> <note>
     * <p>
     * Because this action allows the workflow to properly clean up and gracefully close, it should be used instead of
     * <a>TerminateWorkflowExecution</a> when possible.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param requestCancelWorkflowExecutionRequest
     * @return A Java Future containing the result of the RequestCancelWorkflowExecution operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RequestCancelWorkflowExecution
     */
    @Override
    public CompletableFuture<RequestCancelWorkflowExecutionResponse> requestCancelWorkflowExecution(
            RequestCancelWorkflowExecutionRequest requestCancelWorkflowExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                requestCancelWorkflowExecutionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RequestCancelWorkflowExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RequestCancelWorkflowExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RequestCancelWorkflowExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RequestCancelWorkflowExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RequestCancelWorkflowExecutionRequest, RequestCancelWorkflowExecutionResponse>()
                            .withOperationName("RequestCancelWorkflowExecution")
                            .withMarshaller(new RequestCancelWorkflowExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(requestCancelWorkflowExecutionRequest));
            CompletableFuture<RequestCancelWorkflowExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by workers to tell the service that the <a>ActivityTask</a> identified by the <code>taskToken</code> was
     * successfully canceled. Additional <code>details</code> can be provided using the <code>details</code> argument.
     * </p>
     * <p>
     * These <code>details</code> (if provided) appear in the <code>ActivityTaskCanceled</code> event added to the
     * workflow history.
     * </p>
     * <important>
     * <p>
     * Only use this operation if the <code>canceled</code> flag of a <a>RecordActivityTaskHeartbeat</a> request returns
     * <code>true</code> and if the activity can be safely undone or abandoned.
     * </p>
     * </important>
     * <p>
     * A task is considered open from the time that it is scheduled until it is closed. Therefore a task is reported as
     * open while a worker is processing it. A task is closed after it has been specified in a call to
     * <a>RespondActivityTaskCompleted</a>, RespondActivityTaskCanceled, <a>RespondActivityTaskFailed</a>, or the task
     * has <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dg-basic.html#swf-dev-timeout-types">timed
     * out</a>.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param respondActivityTaskCanceledRequest
     * @return A Java Future containing the result of the RespondActivityTaskCanceled operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RespondActivityTaskCanceled
     */
    @Override
    public CompletableFuture<RespondActivityTaskCanceledResponse> respondActivityTaskCanceled(
            RespondActivityTaskCanceledRequest respondActivityTaskCanceledRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, respondActivityTaskCanceledRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RespondActivityTaskCanceled");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RespondActivityTaskCanceledResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RespondActivityTaskCanceledResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RespondActivityTaskCanceledResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RespondActivityTaskCanceledRequest, RespondActivityTaskCanceledResponse>()
                            .withOperationName("RespondActivityTaskCanceled")
                            .withMarshaller(new RespondActivityTaskCanceledRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(respondActivityTaskCanceledRequest));
            CompletableFuture<RespondActivityTaskCanceledResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by workers to tell the service that the <a>ActivityTask</a> identified by the <code>taskToken</code>
     * completed successfully with a <code>result</code> (if provided). The <code>result</code> appears in the
     * <code>ActivityTaskCompleted</code> event in the workflow history.
     * </p>
     * <important>
     * <p>
     * If the requested task doesn't complete successfully, use <a>RespondActivityTaskFailed</a> instead. If the worker
     * finds that the task is canceled through the <code>canceled</code> flag returned by
     * <a>RecordActivityTaskHeartbeat</a>, it should cancel the task, clean up and then call
     * <a>RespondActivityTaskCanceled</a>.
     * </p>
     * </important>
     * <p>
     * A task is considered open from the time that it is scheduled until it is closed. Therefore a task is reported as
     * open while a worker is processing it. A task is closed after it has been specified in a call to
     * RespondActivityTaskCompleted, <a>RespondActivityTaskCanceled</a>, <a>RespondActivityTaskFailed</a>, or the task
     * has <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dg-basic.html#swf-dev-timeout-types">timed
     * out</a>.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param respondActivityTaskCompletedRequest
     * @return A Java Future containing the result of the RespondActivityTaskCompleted operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RespondActivityTaskCompleted
     */
    @Override
    public CompletableFuture<RespondActivityTaskCompletedResponse> respondActivityTaskCompleted(
            RespondActivityTaskCompletedRequest respondActivityTaskCompletedRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, respondActivityTaskCompletedRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RespondActivityTaskCompleted");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RespondActivityTaskCompletedResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RespondActivityTaskCompletedResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RespondActivityTaskCompletedResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RespondActivityTaskCompletedRequest, RespondActivityTaskCompletedResponse>()
                            .withOperationName("RespondActivityTaskCompleted")
                            .withMarshaller(new RespondActivityTaskCompletedRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(respondActivityTaskCompletedRequest));
            CompletableFuture<RespondActivityTaskCompletedResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by workers to tell the service that the <a>ActivityTask</a> identified by the <code>taskToken</code> has
     * failed with <code>reason</code> (if specified). The <code>reason</code> and <code>details</code> appear in the
     * <code>ActivityTaskFailed</code> event added to the workflow history.
     * </p>
     * <p>
     * A task is considered open from the time that it is scheduled until it is closed. Therefore a task is reported as
     * open while a worker is processing it. A task is closed after it has been specified in a call to
     * <a>RespondActivityTaskCompleted</a>, <a>RespondActivityTaskCanceled</a>, RespondActivityTaskFailed, or the task
     * has <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dg-basic.html#swf-dev-timeout-types">timed
     * out</a>.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param respondActivityTaskFailedRequest
     * @return A Java Future containing the result of the RespondActivityTaskFailed operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RespondActivityTaskFailed
     */
    @Override
    public CompletableFuture<RespondActivityTaskFailedResponse> respondActivityTaskFailed(
            RespondActivityTaskFailedRequest respondActivityTaskFailedRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, respondActivityTaskFailedRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RespondActivityTaskFailed");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RespondActivityTaskFailedResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RespondActivityTaskFailedResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RespondActivityTaskFailedResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RespondActivityTaskFailedRequest, RespondActivityTaskFailedResponse>()
                            .withOperationName("RespondActivityTaskFailed")
                            .withMarshaller(new RespondActivityTaskFailedRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(respondActivityTaskFailedRequest));
            CompletableFuture<RespondActivityTaskFailedResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by deciders to tell the service that the <a>DecisionTask</a> identified by the <code>taskToken</code> has
     * successfully completed. The <code>decisions</code> argument specifies the list of decisions made while processing
     * the task.
     * </p>
     * <p>
     * A <code>DecisionTaskCompleted</code> event is added to the workflow history. The <code>executionContext</code>
     * specified is attached to the event in the workflow execution history.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * If an IAM policy grants permission to use <code>RespondDecisionTaskCompleted</code>, it can express permissions
     * for the list of decisions in the <code>decisions</code> parameter. Each of the decisions has one or more
     * parameters, much like a regular API call. To allow for policies to be as readable as possible, you can express
     * permissions on decisions as if they were actual API calls, including applying conditions to some parameters. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param respondDecisionTaskCompletedRequest
     *        Input data for a TaskCompleted response to a decision task.
     * @return A Java Future containing the result of the RespondDecisionTaskCompleted operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.RespondDecisionTaskCompleted
     */
    @Override
    public CompletableFuture<RespondDecisionTaskCompletedResponse> respondDecisionTaskCompleted(
            RespondDecisionTaskCompletedRequest respondDecisionTaskCompletedRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, respondDecisionTaskCompletedRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RespondDecisionTaskCompleted");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RespondDecisionTaskCompletedResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RespondDecisionTaskCompletedResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RespondDecisionTaskCompletedResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RespondDecisionTaskCompletedRequest, RespondDecisionTaskCompletedResponse>()
                            .withOperationName("RespondDecisionTaskCompleted")
                            .withMarshaller(new RespondDecisionTaskCompletedRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(respondDecisionTaskCompletedRequest));
            CompletableFuture<RespondDecisionTaskCompletedResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Records a <code>WorkflowExecutionSignaled</code> event in the workflow execution history and creates a decision
     * task for the workflow execution identified by the given domain, workflowId and runId. The event is recorded with
     * the specified user defined signalName and input (if provided).
     * </p>
     * <note>
     * <p>
     * If a runId isn't specified, then the <code>WorkflowExecutionSignaled</code> event is recorded in the history of
     * the current open workflow with the matching workflowId in the domain.
     * </p>
     * </note> <note>
     * <p>
     * If the specified workflow execution isn't open, this method fails with <code>UnknownResource</code>.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param signalWorkflowExecutionRequest
     * @return A Java Future containing the result of the SignalWorkflowExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.SignalWorkflowExecution
     */
    @Override
    public CompletableFuture<SignalWorkflowExecutionResponse> signalWorkflowExecution(
            SignalWorkflowExecutionRequest signalWorkflowExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, signalWorkflowExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SignalWorkflowExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SignalWorkflowExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SignalWorkflowExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SignalWorkflowExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SignalWorkflowExecutionRequest, SignalWorkflowExecutionResponse>()
                            .withOperationName("SignalWorkflowExecution")
                            .withMarshaller(new SignalWorkflowExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(signalWorkflowExecutionRequest));
            CompletableFuture<SignalWorkflowExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an execution of the workflow type in the specified domain using the provided <code>workflowId</code> and
     * input data.
     * </p>
     * <p>
     * This action returns the newly started workflow execution.
     * </p>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tagList.member.0</code>: The key is <code>swf:tagList.member.0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tagList.member.1</code>: The key is <code>swf:tagList.member.1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tagList.member.2</code>: The key is <code>swf:tagList.member.2</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tagList.member.3</code>: The key is <code>swf:tagList.member.3</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tagList.member.4</code>: The key is <code>swf:tagList.member.4</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>taskList</code>: String constraint. The key is <code>swf:taskList.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>workflowType.name</code>: String constraint. The key is <code>swf:workflowType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>workflowType.version</code>: String constraint. The key is <code>swf:workflowType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param startWorkflowExecutionRequest
     * @return A Java Future containing the result of the StartWorkflowExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>TypeDeprecatedException Returned when the specified activity or workflow type was already deprecated.
     *         </li>
     *         <li>WorkflowExecutionAlreadyStartedException Returned by <a>StartWorkflowExecution</a> when an open
     *         execution with the same workflowId is already running in the specified domain.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>DefaultUndefinedException The <code>StartWorkflowExecution</code> API action was called without the
     *         required parameters set.</p>
     *         <p>
     *         Some workflow execution parameters, such as the decision <code>taskList</code>, must be set to start the
     *         execution. However, these parameters might have been set as defaults when the workflow type was
     *         registered. In this case, you can omit these parameters from the <code>StartWorkflowExecution</code> call
     *         and Amazon SWF uses the values defined in the workflow type.
     *         </p>
     *         <note>
     *         <p>
     *         If these parameters aren't set and no default parameters were defined in the workflow type, this error is
     *         displayed.
     *         </p>
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.StartWorkflowExecution
     */
    @Override
    public CompletableFuture<StartWorkflowExecutionResponse> startWorkflowExecution(
            StartWorkflowExecutionRequest startWorkflowExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startWorkflowExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartWorkflowExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartWorkflowExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartWorkflowExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartWorkflowExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartWorkflowExecutionRequest, StartWorkflowExecutionResponse>()
                            .withOperationName("StartWorkflowExecution")
                            .withMarshaller(new StartWorkflowExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startWorkflowExecutionRequest));
            CompletableFuture<StartWorkflowExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add a tag to a Amazon SWF domain.
     * </p>
     * <note>
     * <p>
     * Amazon SWF supports a maximum of 50 tags per resource.
     * </p>
     * </note>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>TooManyTagsException You've exceeded the number of tags allowed for a domain.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.TagResource
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Records a <code>WorkflowExecutionTerminated</code> event and forces closure of the workflow execution identified
     * by the given domain, runId, and workflowId. The child policy, registered with the workflow type or specified when
     * starting this execution, is applied to any open child workflow executions of this workflow execution.
     * </p>
     * <important>
     * <p>
     * If the identified workflow execution was in progress, it is terminated immediately.
     * </p>
     * </important> <note>
     * <p>
     * If a runId isn't specified, then the <code>WorkflowExecutionTerminated</code> event is recorded in the history of
     * the current open workflow with the matching workflowId in the domain.
     * </p>
     * </note> <note>
     * <p>
     * You should consider using <a>RequestCancelWorkflowExecution</a> action instead because it allows the workflow to
     * gracefully close while <a>TerminateWorkflowExecution</a> doesn't.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param terminateWorkflowExecutionRequest
     * @return A Java Future containing the result of the TerminateWorkflowExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.TerminateWorkflowExecution
     */
    @Override
    public CompletableFuture<TerminateWorkflowExecutionResponse> terminateWorkflowExecution(
            TerminateWorkflowExecutionRequest terminateWorkflowExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateWorkflowExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateWorkflowExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateWorkflowExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TerminateWorkflowExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TerminateWorkflowExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateWorkflowExecutionRequest, TerminateWorkflowExecutionResponse>()
                            .withOperationName("TerminateWorkflowExecution")
                            .withMarshaller(new TerminateWorkflowExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(terminateWorkflowExecutionRequest));
            CompletableFuture<TerminateWorkflowExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Undeprecates a previously deprecated <i>activity type</i>. After an activity type has been undeprecated, you can
     * create new tasks of that activity type.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>activityType.name</code>: String constraint. The key is <code>swf:activityType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>activityType.version</code>: String constraint. The key is <code>swf:activityType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param undeprecateActivityTypeRequest
     * @return A Java Future containing the result of the UndeprecateActivityType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>TypeAlreadyExistsException Returned if the type already exists in the specified domain. You may get
     *         this fault if you are registering a type that is either already registered or deprecated, or if you
     *         undeprecate a type that is currently registered.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.UndeprecateActivityType
     */
    @Override
    public CompletableFuture<UndeprecateActivityTypeResponse> undeprecateActivityType(
            UndeprecateActivityTypeRequest undeprecateActivityTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, undeprecateActivityTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UndeprecateActivityType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UndeprecateActivityTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UndeprecateActivityTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UndeprecateActivityTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UndeprecateActivityTypeRequest, UndeprecateActivityTypeResponse>()
                            .withOperationName("UndeprecateActivityType")
                            .withMarshaller(new UndeprecateActivityTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(undeprecateActivityTypeRequest));
            CompletableFuture<UndeprecateActivityTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Undeprecates a previously deprecated domain. After a domain has been undeprecated it can be used to create new
     * workflow executions or register new types.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot use an IAM policy to constrain this action's parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param undeprecateDomainRequest
     * @return A Java Future containing the result of the UndeprecateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>DomainAlreadyExistsException Returned if the domain already exists. You may get this fault if you are
     *         registering a domain that is either already registered or deprecated, or if you undeprecate a domain that
     *         is currently registered.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.UndeprecateDomain
     */
    @Override
    public CompletableFuture<UndeprecateDomainResponse> undeprecateDomain(UndeprecateDomainRequest undeprecateDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, undeprecateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UndeprecateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UndeprecateDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UndeprecateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UndeprecateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UndeprecateDomainRequest, UndeprecateDomainResponse>()
                            .withOperationName("UndeprecateDomain")
                            .withMarshaller(new UndeprecateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(undeprecateDomainRequest));
            CompletableFuture<UndeprecateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Undeprecates a previously deprecated <i>workflow type</i>. After a workflow type has been undeprecated, you can
     * create new executions of that type.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not exactly reflect recent updates
     * and changes.
     * </p>
     * </note>
     * <p>
     * <b>Access Control</b>
     * </p>
     * <p>
     * You can use IAM policies to control this action's access to Amazon SWF resources as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use a <code>Resource</code> element with the domain name to limit the action to only specified domains.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use an <code>Action</code> element to allow or deny permission to call this action.
     * </p>
     * </li>
     * <li>
     * <p>
     * Constrain the following parameters by using a <code>Condition</code> element with the appropriate keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>workflowType.name</code>: String constraint. The key is <code>swf:workflowType.name</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>workflowType.version</code>: String constraint. The key is <code>swf:workflowType.version</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the caller doesn't have sufficient permissions to invoke the action, or the parameter values fall outside the
     * specified constraints, the action fails. The associated event attribute's <code>cause</code> parameter is set to
     * <code>OPERATION_NOT_PERMITTED</code>. For details and example IAM policies, see <a
     * href="https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html">Using IAM to Manage Access to
     * Amazon SWF Workflows</a> in the <i>Amazon SWF Developer Guide</i>.
     * </p>
     *
     * @param undeprecateWorkflowTypeRequest
     * @return A Java Future containing the result of the UndeprecateWorkflowType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>TypeAlreadyExistsException Returned if the type already exists in the specified domain. You may get
     *         this fault if you are registering a type that is either already registered or deprecated, or if you
     *         undeprecate a type that is currently registered.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.UndeprecateWorkflowType
     */
    @Override
    public CompletableFuture<UndeprecateWorkflowTypeResponse> undeprecateWorkflowType(
            UndeprecateWorkflowTypeRequest undeprecateWorkflowTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, undeprecateWorkflowTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UndeprecateWorkflowType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UndeprecateWorkflowTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UndeprecateWorkflowTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UndeprecateWorkflowTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UndeprecateWorkflowTypeRequest, UndeprecateWorkflowTypeResponse>()
                            .withOperationName("UndeprecateWorkflowType")
                            .withMarshaller(new UndeprecateWorkflowTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(undeprecateWorkflowTypeRequest));
            CompletableFuture<UndeprecateWorkflowTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove a tag from a Amazon SWF domain.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnknownResourceException Returned when the named resource cannot be found with in the scope of this
     *         operation (region or domain). This could happen if the named resource was never created or is no longer
     *         available for this operation.</li>
     *         <li>LimitExceededException Returned by any operation if a system imposed limitation has been reached. To
     *         address this fault you should either clean up unused resources or increase the limit by contacting AWS.</li>
     *         <li>OperationNotPermittedException Returned when the caller doesn't have sufficient permissions to invoke
     *         the action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SwfException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SwfAsyncClient.UntagResource
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SWF");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SwfServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SwfException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TypeAlreadyExistsFault")
                                .exceptionBuilderSupplier(TypeAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TypeDeprecatedFault")
                                .exceptionBuilderSupplier(TypeDeprecatedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededFault")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainDeprecatedFault")
                                .exceptionBuilderSupplier(DomainDeprecatedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DefaultUndefinedFault")
                                .exceptionBuilderSupplier(DefaultUndefinedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainAlreadyExistsFault")
                                .exceptionBuilderSupplier(DomainAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedFault")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WorkflowExecutionAlreadyStartedFault")
                                .exceptionBuilderSupplier(WorkflowExecutionAlreadyStartedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsFault")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceFault")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SwfRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
