/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmcontacts;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmcontacts.internal.SsmContactsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ConflictException;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationOverrideRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationOverrideResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DataEncryptionException;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationOverrideRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationOverrideResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationOverrideRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationOverrideResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.InternalServerException;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageResolutionsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageResolutionsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPreviewRotationShiftsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPreviewRotationShiftsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationOverridesRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationOverridesResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationShiftsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationShiftsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeRequest;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ssmcontacts.model.SsmContactsException;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ThrottlingException;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ValidationException;
import software.amazon.awssdk.services.ssmcontacts.transform.AcceptPageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ActivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateRotationOverrideRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateRotationRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeactivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteRotationOverrideRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteRotationRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribeEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribePageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetRotationOverrideRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetRotationRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactChannelsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListEngagementsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPageReceiptsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPageResolutionsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPreviewRotationShiftsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListRotationOverridesRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListRotationShiftsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListRotationsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.PutContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.SendActivationCodeRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StartEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StopEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateRotationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsmContactsClient}.
 *
 * @see SsmContactsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmContactsClient implements SsmContactsClient {
    private static final Logger log = Logger.loggerFor(DefaultSsmContactsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SsmContactsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSsmContactsClient(SsmContactsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Used to acknowledge an engagement to a contact channel during an incident.
     * </p>
     *
     * @param acceptPageRequest
     * @return Result of the AcceptPage operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.AcceptPage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/AcceptPage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AcceptPageResponse acceptPage(AcceptPageRequest acceptPageRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptPageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AcceptPageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptPageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptPageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptPage");

            return clientHandler.execute(new ClientExecutionParams<AcceptPageRequest, AcceptPageResponse>()
                    .withOperationName("AcceptPage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(acceptPageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AcceptPageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates a contact's contact channel. Incident Manager can't engage a contact until the contact channel has been
     * activated.
     * </p>
     *
     * @param activateContactChannelRequest
     * @return Result of the ActivateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ActivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ActivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ActivateContactChannelResponse activateContactChannel(ActivateContactChannelRequest activateContactChannelRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ActivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ActivateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(activateContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateContactChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<ActivateContactChannelRequest, ActivateContactChannelResponse>()
                            .withOperationName("ActivateContactChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(activateContactChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ActivateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Contacts are either the contacts that Incident Manager engages during an incident or the escalation plans that
     * Incident Manager uses to engage contacts in phases during an incident.
     * </p>
     *
     * @param createContactRequest
     * @return Result of the CreateContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.CreateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateContactResponse createContact(CreateContactRequest createContactRequest) throws AccessDeniedException,
            ConflictException, InternalServerException, ServiceQuotaExceededException, ThrottlingException, ValidationException,
            DataEncryptionException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContact");

            return clientHandler.execute(new ClientExecutionParams<CreateContactRequest, CreateContactResponse>()
                    .withOperationName("CreateContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A contact channel is the method that Incident Manager uses to engage your contact.
     * </p>
     *
     * @param createContactChannelRequest
     * @return Result of the CreateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.CreateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateContactChannelResponse createContactChannel(CreateContactChannelRequest createContactChannelRequest)
            throws AccessDeniedException, ConflictException, DataEncryptionException, InternalServerException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContactChannel");

            return clientHandler.execute(new ClientExecutionParams<CreateContactChannelRequest, CreateContactChannelResponse>()
                    .withOperationName("CreateContactChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a rotation in an on-call schedule.
     * </p>
     *
     * @param createRotationRequest
     * @return Result of the CreateRotation operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.CreateRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateRotation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateRotationResponse createRotation(CreateRotationRequest createRotationRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRotationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRotationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRotation");

            return clientHandler.execute(new ClientExecutionParams<CreateRotationRequest, CreateRotationResponse>()
                    .withOperationName("CreateRotation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createRotationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRotationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an override for a rotation in an on-call schedule.
     * </p>
     *
     * @param createRotationOverrideRequest
     * @return Result of the CreateRotationOverride operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.CreateRotationOverride
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateRotationOverride"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRotationOverrideResponse createRotationOverride(CreateRotationOverrideRequest createRotationOverrideRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRotationOverrideResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRotationOverrideResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRotationOverrideRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRotationOverrideRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRotationOverride");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateRotationOverrideRequest, CreateRotationOverrideResponse>()
                            .withOperationName("CreateRotationOverride").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createRotationOverrideRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateRotationOverrideRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To no longer receive Incident Manager engagements to a contact channel, you can deactivate the channel.
     * </p>
     *
     * @param deactivateContactChannelRequest
     * @return Result of the DeactivateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeactivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeactivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeactivateContactChannelResponse deactivateContactChannel(
            DeactivateContactChannelRequest deactivateContactChannelRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeactivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeactivateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deactivateContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateContactChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<DeactivateContactChannelRequest, DeactivateContactChannelResponse>()
                            .withOperationName("DeactivateContactChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deactivateContactChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeactivateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To remove a contact from Incident Manager, you can delete the contact. Deleting a contact removes them from all
     * escalation plans and related response plans. Deleting an escalation plan removes it from all related response
     * plans. You will have to recreate the contact and its contact channels before you can use it again.
     * </p>
     *
     * @param deleteContactRequest
     * @return Result of the DeleteContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeleteContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteContactResponse deleteContact(DeleteContactRequest deleteContactRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ConflictException, ValidationException,
            AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContact");

            return clientHandler.execute(new ClientExecutionParams<DeleteContactRequest, DeleteContactResponse>()
                    .withOperationName("DeleteContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * To no longer receive engagements on a contact channel, you can delete the channel from a contact. Deleting the
     * contact channel removes it from the contact's engagement plan. If you delete the only contact channel for a
     * contact, you won't be able to engage that contact during an incident.
     * </p>
     *
     * @param deleteContactChannelRequest
     * @return Result of the DeleteContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeleteContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteContactChannelResponse deleteContactChannel(DeleteContactChannelRequest deleteContactChannelRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContactChannel");

            return clientHandler.execute(new ClientExecutionParams<DeleteContactChannelRequest, DeleteContactChannelResponse>()
                    .withOperationName("DeleteContactChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a rotation from the system. If a rotation belongs to more than one on-call schedule, this operation
     * deletes it from all of them.
     * </p>
     *
     * @param deleteRotationRequest
     * @return Result of the DeleteRotation operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeleteRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteRotation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRotationResponse deleteRotation(DeleteRotationRequest deleteRotationRequest) throws AccessDeniedException,
            ConflictException, InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRotationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRotationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRotation");

            return clientHandler.execute(new ClientExecutionParams<DeleteRotationRequest, DeleteRotationResponse>()
                    .withOperationName("DeleteRotation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRotationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRotationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing override for an on-call rotation.
     * </p>
     *
     * @param deleteRotationOverrideRequest
     * @return Result of the DeleteRotationOverride operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DeleteRotationOverride
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteRotationOverride"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRotationOverrideResponse deleteRotationOverride(DeleteRotationOverrideRequest deleteRotationOverrideRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRotationOverrideResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRotationOverrideResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRotationOverrideRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRotationOverrideRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRotationOverride");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRotationOverrideRequest, DeleteRotationOverrideResponse>()
                            .withOperationName("DeleteRotationOverride").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteRotationOverrideRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRotationOverrideRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Incident Manager uses engagements to engage contacts and escalation plans during an incident. Use this command to
     * describe the engagement that occurred during an incident.
     * </p>
     *
     * @param describeEngagementRequest
     * @return Result of the DescribeEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DescribeEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribeEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEngagementResponse describeEngagement(DescribeEngagementRequest describeEngagementRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEngagementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEngagement");

            return clientHandler.execute(new ClientExecutionParams<DescribeEngagementRequest, DescribeEngagementResponse>()
                    .withOperationName("DescribeEngagement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists details of the engagement to a contact channel.
     * </p>
     *
     * @param describePageRequest
     * @return Result of the DescribePage operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.DescribePage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribePage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribePageResponse describePage(DescribePageRequest describePageRequest) throws AccessDeniedException,
            DataEncryptionException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribePageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePage");

            return clientHandler.execute(new ClientExecutionParams<DescribePageRequest, DescribePageResponse>()
                    .withOperationName("DescribePage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describePageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified contact or escalation plan.
     * </p>
     *
     * @param getContactRequest
     * @return Result of the GetContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetContactResponse getContact(GetContactRequest getContactRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            DataEncryptionException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContact");

            return clientHandler.execute(new ClientExecutionParams<GetContactRequest, GetContactResponse>()
                    .withOperationName("GetContact").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getContactRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List details about a specific contact channel.
     * </p>
     *
     * @param getContactChannelRequest
     * @return Result of the GetContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetContactChannelResponse getContactChannel(GetContactChannelRequest getContactChannelRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactChannel");

            return clientHandler.execute(new ClientExecutionParams<GetContactChannelRequest, GetContactChannelResponse>()
                    .withOperationName("GetContactChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource policies attached to the specified contact or escalation plan.
     * </p>
     *
     * @param getContactPolicyRequest
     * @return Result of the GetContactPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetContactPolicyResponse getContactPolicy(GetContactPolicyRequest getContactPolicyRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContactPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetContactPolicyRequest, GetContactPolicyResponse>()
                    .withOperationName("GetContactPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getContactPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an on-call rotation.
     * </p>
     *
     * @param getRotationRequest
     * @return Result of the GetRotation operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetRotation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRotationResponse getRotation(GetRotationRequest getRotationRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRotationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRotationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRotation");

            return clientHandler.execute(new ClientExecutionParams<GetRotationRequest, GetRotationResponse>()
                    .withOperationName("GetRotation").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRotationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRotationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an override to an on-call rotation.
     * </p>
     *
     * @param getRotationOverrideRequest
     * @return Result of the GetRotationOverride operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.GetRotationOverride
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetRotationOverride"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRotationOverrideResponse getRotationOverride(GetRotationOverrideRequest getRotationOverrideRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRotationOverrideResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRotationOverrideResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRotationOverrideRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRotationOverrideRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRotationOverride");

            return clientHandler.execute(new ClientExecutionParams<GetRotationOverrideRequest, GetRotationOverrideResponse>()
                    .withOperationName("GetRotationOverride").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRotationOverrideRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRotationOverrideRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all contact channels for the specified contact.
     * </p>
     *
     * @param listContactChannelsRequest
     * @return Result of the ListContactChannels operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListContactChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContactChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListContactChannelsResponse listContactChannels(ListContactChannelsRequest listContactChannelsRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListContactChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContactChannelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContactChannels");

            return clientHandler.execute(new ClientExecutionParams<ListContactChannelsRequest, ListContactChannelsResponse>()
                    .withOperationName("ListContactChannels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listContactChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all contacts and escalation plans in Incident Manager.
     * </p>
     *
     * @param listContactsRequest
     * @return Result of the ListContacts operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContactsResponse listContacts(ListContactsRequest listContactsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContactsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContactsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");

            return clientHandler.execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                    .withOperationName("ListContacts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listContactsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all engagements that have happened in an incident.
     * </p>
     *
     * @param listEngagementsRequest
     * @return Result of the ListEngagements operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListEngagements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListEngagements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEngagementsResponse listEngagements(ListEngagementsRequest listEngagementsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEngagementsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEngagementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEngagementsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEngagementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEngagements");

            return clientHandler.execute(new ClientExecutionParams<ListEngagementsRequest, ListEngagementsResponse>()
                    .withOperationName("ListEngagements").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEngagementsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEngagementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the engagements to contact channels that have been acknowledged.
     * </p>
     *
     * @param listPageReceiptsRequest
     * @return Result of the ListPageReceipts operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPageReceipts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageReceipts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPageReceiptsResponse listPageReceipts(ListPageReceiptsRequest listPageReceiptsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPageReceiptsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPageReceiptsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPageReceiptsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPageReceiptsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPageReceipts");

            return clientHandler.execute(new ClientExecutionParams<ListPageReceiptsRequest, ListPageReceiptsResponse>()
                    .withOperationName("ListPageReceipts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPageReceiptsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPageReceiptsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the resolution path of an engagement. For example, the escalation plan engaged in an incident might
     * target an on-call schedule that includes several contacts in a rotation, but just one contact on-call when the
     * incident starts. The resolution path indicates the hierarchy of <i>escalation plan &gt; on-call schedule &gt;
     * contact</i>.
     * </p>
     *
     * @param listPageResolutionsRequest
     * @return Result of the ListPageResolutions operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPageResolutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageResolutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPageResolutionsResponse listPageResolutions(ListPageResolutionsRequest listPageResolutionsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPageResolutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPageResolutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPageResolutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPageResolutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPageResolutions");

            return clientHandler.execute(new ClientExecutionParams<ListPageResolutionsRequest, ListPageResolutionsResponse>()
                    .withOperationName("ListPageResolutions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPageResolutionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPageResolutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the engagements to a contact's contact channels.
     * </p>
     *
     * @param listPagesByContactRequest
     * @return Result of the ListPagesByContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPagesByContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPagesByContactResponse listPagesByContact(ListPagesByContactRequest listPagesByContactRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPagesByContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPagesByContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPagesByContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByContact");

            return clientHandler.execute(new ClientExecutionParams<ListPagesByContactRequest, ListPagesByContactResponse>()
                    .withOperationName("ListPagesByContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPagesByContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPagesByContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the engagements to contact channels that occurred by engaging a contact.
     * </p>
     *
     * @param listPagesByEngagementRequest
     * @return Result of the ListPagesByEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPagesByEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPagesByEngagementResponse listPagesByEngagement(ListPagesByEngagementRequest listPagesByEngagementRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPagesByEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPagesByEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPagesByEngagementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByEngagement");

            return clientHandler.execute(new ClientExecutionParams<ListPagesByEngagementRequest, ListPagesByEngagementResponse>()
                    .withOperationName("ListPagesByEngagement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPagesByEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPagesByEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of shifts based on rotation configuration parameters.
     * </p>
     * <note>
     * <p>
     * The Incident Manager primarily uses this operation to populate the <b>Preview</b> calendar. It is not typically
     * run by end users.
     * </p>
     * </note>
     *
     * @param listPreviewRotationShiftsRequest
     * @return Result of the ListPreviewRotationShifts operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListPreviewRotationShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPreviewRotationShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPreviewRotationShiftsResponse listPreviewRotationShifts(
            ListPreviewRotationShiftsRequest listPreviewRotationShiftsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPreviewRotationShiftsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPreviewRotationShiftsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPreviewRotationShiftsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPreviewRotationShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPreviewRotationShifts");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPreviewRotationShiftsRequest, ListPreviewRotationShiftsResponse>()
                            .withOperationName("ListPreviewRotationShifts").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPreviewRotationShiftsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPreviewRotationShiftsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of overrides currently specified for an on-call rotation.
     * </p>
     *
     * @param listRotationOverridesRequest
     * @return Result of the ListRotationOverrides operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListRotationOverrides
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListRotationOverrides"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRotationOverridesResponse listRotationOverrides(ListRotationOverridesRequest listRotationOverridesRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRotationOverridesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRotationOverridesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRotationOverridesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRotationOverridesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRotationOverrides");

            return clientHandler.execute(new ClientExecutionParams<ListRotationOverridesRequest, ListRotationOverridesResponse>()
                    .withOperationName("ListRotationOverrides").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRotationOverridesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRotationOverridesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of shifts generated by an existing rotation in the system.
     * </p>
     *
     * @param listRotationShiftsRequest
     * @return Result of the ListRotationShifts operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListRotationShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListRotationShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRotationShiftsResponse listRotationShifts(ListRotationShiftsRequest listRotationShiftsRequest)
            throws AccessDeniedException, ConflictException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRotationShiftsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRotationShiftsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRotationShiftsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRotationShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRotationShifts");

            return clientHandler.execute(new ClientExecutionParams<ListRotationShiftsRequest, ListRotationShiftsResponse>()
                    .withOperationName("ListRotationShifts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRotationShiftsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRotationShiftsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of on-call rotations.
     * </p>
     *
     * @param listRotationsRequest
     * @return Result of the ListRotations operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListRotations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListRotations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListRotationsResponse listRotations(ListRotationsRequest listRotationsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRotationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRotationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRotationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRotationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRotations");

            return clientHandler.execute(new ClientExecutionParams<ListRotationsRequest, ListRotationsResponse>()
                    .withOperationName("ListRotations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRotationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRotationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags of an escalation plan or contact.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AccessDeniedException, ThrottlingException, ResourceNotFoundException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a resource policy to the specified contact or escalation plan. The resource policy is used to share the
     * contact or escalation plan using Resource Access Manager (RAM). For more information about cross-account sharing,
     * see <a href="https://docs.aws.amazon.com/incident-manager/latest/userguide/xa.html">Setting up cross-account
     * functionality</a>.
     * </p>
     *
     * @param putContactPolicyRequest
     * @return Result of the PutContactPolicy operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.PutContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/PutContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutContactPolicyResponse putContactPolicy(PutContactPolicyRequest putContactPolicyRequest)
            throws AccessDeniedException, ConflictException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutContactPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putContactPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContactPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutContactPolicyRequest, PutContactPolicyResponse>()
                    .withOperationName("PutContactPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putContactPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutContactPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends an activation code to a contact channel. The contact can use this code to activate the contact channel in
     * the console or with the <code>ActivateChannel</code> operation. Incident Manager can't engage a contact channel
     * until it has been activated.
     * </p>
     *
     * @param sendActivationCodeRequest
     * @return Result of the SendActivationCode operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.SendActivationCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/SendActivationCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SendActivationCodeResponse sendActivationCode(SendActivationCodeRequest sendActivationCodeRequest)
            throws AccessDeniedException, DataEncryptionException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendActivationCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SendActivationCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendActivationCodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendActivationCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendActivationCode");

            return clientHandler.execute(new ClientExecutionParams<SendActivationCodeRequest, SendActivationCodeResponse>()
                    .withOperationName("SendActivationCode").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(sendActivationCodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendActivationCodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an engagement to a contact or escalation plan. The engagement engages each contact specified in the
     * incident.
     * </p>
     *
     * @param startEngagementRequest
     * @return Result of the StartEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.StartEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StartEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartEngagementResponse startEngagement(StartEngagementRequest startEngagementRequest) throws AccessDeniedException,
            DataEncryptionException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartEngagementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startEngagementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEngagement");

            return clientHandler.execute(new ClientExecutionParams<StartEngagementRequest, StartEngagementResponse>()
                    .withOperationName("StartEngagement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an engagement before it finishes the final stage of the escalation plan or engagement plan. Further
     * contacts aren't engaged.
     * </p>
     *
     * @param stopEngagementRequest
     * @return Result of the StopEngagement operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.StopEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StopEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopEngagementResponse stopEngagement(StopEngagementRequest stopEngagementRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopEngagementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopEngagementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopEngagementRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopEngagement");

            return clientHandler.execute(new ClientExecutionParams<StopEngagementRequest, StopEngagementResponse>()
                    .withOperationName("StopEngagement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopEngagementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopEngagementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags a contact or escalation plan. You can tag only contacts and escalation plans in the first region of your
     * replication set.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, InternalServerException, ServiceQuotaExceededException,
            ValidationException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the contact or escalation plan specified.
     * </p>
     *
     * @param updateContactRequest
     * @return Result of the UpdateContact operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UpdateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateContactResponse updateContact(UpdateContactRequest updateContactRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException,
            ValidationException, DataEncryptionException, AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContact");

            return clientHandler.execute(new ClientExecutionParams<UpdateContactRequest, UpdateContactResponse>()
                    .withOperationName("UpdateContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a contact's contact channel.
     * </p>
     *
     * @param updateContactChannelRequest
     * @return Result of the UpdateContactChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws DataEncryptionException
     *         The operation failed to due an encryption key error.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UpdateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContactChannelResponse updateContactChannel(UpdateContactChannelRequest updateContactChannelRequest)
            throws AccessDeniedException, ConflictException, DataEncryptionException, InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContactChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContactChannel");

            return clientHandler.execute(new ClientExecutionParams<UpdateContactChannelRequest, UpdateContactChannelResponse>()
                    .withOperationName("UpdateContactChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateContactChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateContactChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the information specified for an on-call rotation.
     * </p>
     *
     * @param updateRotationRequest
     * @return Result of the UpdateRotation operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this operation.
     * @throws ConflictException
     *         Updating or deleting a resource causes an inconsistent state.
     * @throws InternalServerException
     *         Unexpected error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmContactsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmContactsClient.UpdateRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateRotation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateRotationResponse updateRotation(UpdateRotationRequest updateRotationRequest) throws AccessDeniedException,
            ConflictException, InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, SsmContactsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRotationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateRotationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRotation");

            return clientHandler.execute(new ClientExecutionParams<UpdateRotationRequest, UpdateRotationResponse>()
                    .withOperationName("UpdateRotation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateRotationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRotationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SsmContactsServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = SsmContactsServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmContactsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataEncryptionException")
                                .exceptionBuilderSupplier(DataEncryptionException::builder).httpStatusCode(400).build());
    }

    @Override
    public final SsmContactsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
