/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmcontacts;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmcontacts.internal.SsmContactsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.AcceptPageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.AccessDeniedException;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ActivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ConflictException;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationOverrideRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationOverrideResponse;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.CreateRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DataEncryptionException;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeactivateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationOverrideRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationOverrideResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DeleteRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribeEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageRequest;
import software.amazon.awssdk.services.ssmcontacts.model.DescribePageResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationOverrideRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationOverrideResponse;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.GetRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.InternalServerException;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactChannelsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListContactsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListEngagementsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageReceiptsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageResolutionsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPageResolutionsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPagesByEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListPreviewRotationShiftsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListPreviewRotationShiftsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationOverridesRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationOverridesResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationShiftsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationShiftsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationsRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListRotationsResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyRequest;
import software.amazon.awssdk.services.ssmcontacts.model.PutContactPolicyResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeRequest;
import software.amazon.awssdk.services.ssmcontacts.model.SendActivationCodeResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ssmcontacts.model.SsmContactsException;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StartEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementRequest;
import software.amazon.awssdk.services.ssmcontacts.model.StopEngagementResponse;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ThrottlingException;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactChannelResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateContactResponse;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateRotationRequest;
import software.amazon.awssdk.services.ssmcontacts.model.UpdateRotationResponse;
import software.amazon.awssdk.services.ssmcontacts.model.ValidationException;
import software.amazon.awssdk.services.ssmcontacts.transform.AcceptPageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ActivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateRotationOverrideRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.CreateRotationRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeactivateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteRotationOverrideRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DeleteRotationRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribeEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.DescribePageRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetRotationOverrideRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.GetRotationRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactChannelsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListEngagementsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPageReceiptsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPageResolutionsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPagesByEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListPreviewRotationShiftsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListRotationOverridesRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListRotationShiftsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListRotationsRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.PutContactPolicyRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.SendActivationCodeRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StartEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.StopEngagementRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactChannelRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateContactRequestMarshaller;
import software.amazon.awssdk.services.ssmcontacts.transform.UpdateRotationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SsmContactsAsyncClient}.
 *
 * @see SsmContactsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmContactsAsyncClient implements SsmContactsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSsmContactsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SsmContactsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSsmContactsAsyncClient(SsmContactsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Used to acknowledge an engagement to a contact channel during an incident.
     * </p>
     *
     * @param acceptPageRequest
     * @return A Java Future containing the result of the AcceptPage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.AcceptPage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/AcceptPage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptPageResponse> acceptPage(AcceptPageRequest acceptPageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptPageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptPageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptPage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptPageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AcceptPageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptPageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptPageRequest, AcceptPageResponse>().withOperationName("AcceptPage")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcceptPageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptPageRequest));
            CompletableFuture<AcceptPageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates a contact's contact channel. Incident Manager can't engage a contact until the contact channel has been
     * activated.
     * </p>
     *
     * @param activateContactChannelRequest
     * @return A Java Future containing the result of the ActivateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ActivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ActivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateContactChannelResponse> activateContactChannel(
            ActivateContactChannelRequest activateContactChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(activateContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateContactChannelRequest, ActivateContactChannelResponse>()
                            .withOperationName("ActivateContactChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ActivateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(activateContactChannelRequest));
            CompletableFuture<ActivateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Contacts are either the contacts that Incident Manager engages during an incident or the escalation plans that
     * Incident Manager uses to engage contacts in phases during an incident.
     * </p>
     *
     * @param createContactRequest
     * @return A Java Future containing the result of the CreateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.CreateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateContactResponse> createContact(CreateContactRequest createContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateContactRequest, CreateContactResponse>()
                            .withOperationName("CreateContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createContactRequest));
            CompletableFuture<CreateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A contact channel is the method that Incident Manager uses to engage your contact.
     * </p>
     *
     * @param createContactChannelRequest
     * @return A Java Future containing the result of the CreateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.CreateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateContactChannelResponse> createContactChannel(
            CreateContactChannelRequest createContactChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateContactChannelRequest, CreateContactChannelResponse>()
                            .withOperationName("CreateContactChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createContactChannelRequest));
            CompletableFuture<CreateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a rotation in an on-call schedule.
     * </p>
     *
     * @param createRotationRequest
     * @return A Java Future containing the result of the CreateRotation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.CreateRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateRotation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRotationResponse> createRotation(CreateRotationRequest createRotationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRotation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRotationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRotationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRotationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRotationRequest, CreateRotationResponse>()
                            .withOperationName("CreateRotation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRotationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRotationRequest));
            CompletableFuture<CreateRotationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an override for a rotation in an on-call schedule.
     * </p>
     *
     * @param createRotationOverrideRequest
     * @return A Java Future containing the result of the CreateRotationOverride operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.CreateRotationOverride
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/CreateRotationOverride"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRotationOverrideResponse> createRotationOverride(
            CreateRotationOverrideRequest createRotationOverrideRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRotationOverrideRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRotationOverrideRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRotationOverride");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRotationOverrideResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRotationOverrideResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRotationOverrideResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRotationOverrideRequest, CreateRotationOverrideResponse>()
                            .withOperationName("CreateRotationOverride").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRotationOverrideRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRotationOverrideRequest));
            CompletableFuture<CreateRotationOverrideResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * To no longer receive Incident Manager engagements to a contact channel, you can deactivate the channel.
     * </p>
     *
     * @param deactivateContactChannelRequest
     * @return A Java Future containing the result of the DeactivateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeactivateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeactivateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateContactChannelResponse> deactivateContactChannel(
            DeactivateContactChannelRequest deactivateContactChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deactivateContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateContactChannelRequest, DeactivateContactChannelResponse>()
                            .withOperationName("DeactivateContactChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeactivateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deactivateContactChannelRequest));
            CompletableFuture<DeactivateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * To remove a contact from Incident Manager, you can delete the contact. Deleting a contact removes them from all
     * escalation plans and related response plans. Deleting an escalation plan removes it from all related response
     * plans. You will have to recreate the contact and its contact channels before you can use it again.
     * </p>
     *
     * @param deleteContactRequest
     * @return A Java Future containing the result of the DeleteContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeleteContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteContactResponse> deleteContact(DeleteContactRequest deleteContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteContactRequest, DeleteContactResponse>()
                            .withOperationName("DeleteContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteContactRequest));
            CompletableFuture<DeleteContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * To no longer receive engagements on a contact channel, you can delete the channel from a contact. Deleting the
     * contact channel removes it from the contact's engagement plan. If you delete the only contact channel for a
     * contact, you won't be able to engage that contact during an incident.
     * </p>
     *
     * @param deleteContactChannelRequest
     * @return A Java Future containing the result of the DeleteContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeleteContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteContactChannelResponse> deleteContactChannel(
            DeleteContactChannelRequest deleteContactChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteContactChannelRequest, DeleteContactChannelResponse>()
                            .withOperationName("DeleteContactChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteContactChannelRequest));
            CompletableFuture<DeleteContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a rotation from the system. If a rotation belongs to more than one on-call schedule, this operation
     * deletes it from all of them.
     * </p>
     *
     * @param deleteRotationRequest
     * @return A Java Future containing the result of the DeleteRotation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeleteRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteRotation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRotationResponse> deleteRotation(DeleteRotationRequest deleteRotationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRotation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRotationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRotationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRotationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRotationRequest, DeleteRotationResponse>()
                            .withOperationName("DeleteRotation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRotationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRotationRequest));
            CompletableFuture<DeleteRotationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing override for an on-call rotation.
     * </p>
     *
     * @param deleteRotationOverrideRequest
     * @return A Java Future containing the result of the DeleteRotationOverride operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DeleteRotationOverride
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DeleteRotationOverride"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRotationOverrideResponse> deleteRotationOverride(
            DeleteRotationOverrideRequest deleteRotationOverrideRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRotationOverrideRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRotationOverrideRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRotationOverride");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRotationOverrideResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRotationOverrideResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRotationOverrideResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRotationOverrideRequest, DeleteRotationOverrideResponse>()
                            .withOperationName("DeleteRotationOverride").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRotationOverrideRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRotationOverrideRequest));
            CompletableFuture<DeleteRotationOverrideResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Incident Manager uses engagements to engage contacts and escalation plans during an incident. Use this command to
     * describe the engagement that occurred during an incident.
     * </p>
     *
     * @param describeEngagementRequest
     * @return A Java Future containing the result of the DescribeEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DescribeEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribeEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEngagementResponse> describeEngagement(DescribeEngagementRequest describeEngagementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEngagementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEngagementRequest, DescribeEngagementResponse>()
                            .withOperationName("DescribeEngagement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEngagementRequest));
            CompletableFuture<DescribeEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists details of the engagement to a contact channel.
     * </p>
     *
     * @param describePageRequest
     * @return A Java Future containing the result of the DescribePage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.DescribePage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/DescribePage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePageResponse> describePage(DescribePageRequest describePageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribePageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePageRequest, DescribePageResponse>()
                            .withOperationName("DescribePage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePageRequest));
            CompletableFuture<DescribePageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified contact or escalation plan.
     * </p>
     *
     * @param getContactRequest
     * @return A Java Future containing the result of the GetContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactResponse> getContact(GetContactRequest getContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactRequest, GetContactResponse>().withOperationName("GetContact")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getContactRequest));
            CompletableFuture<GetContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List details about a specific contact channel.
     * </p>
     *
     * @param getContactChannelRequest
     * @return A Java Future containing the result of the GetContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactChannelResponse> getContactChannel(GetContactChannelRequest getContactChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactChannelRequest, GetContactChannelResponse>()
                            .withOperationName("GetContactChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getContactChannelRequest));
            CompletableFuture<GetContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource policies attached to the specified contact or escalation plan.
     * </p>
     *
     * @param getContactPolicyRequest
     * @return A Java Future containing the result of the GetContactPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactPolicyResponse> getContactPolicy(GetContactPolicyRequest getContactPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContactPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContactPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactPolicyRequest, GetContactPolicyResponse>()
                            .withOperationName("GetContactPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetContactPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getContactPolicyRequest));
            CompletableFuture<GetContactPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an on-call rotation.
     * </p>
     *
     * @param getRotationRequest
     * @return A Java Future containing the result of the GetRotation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetRotation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRotationResponse> getRotation(GetRotationRequest getRotationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRotation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRotationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRotationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRotationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRotationRequest, GetRotationResponse>()
                            .withOperationName("GetRotation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRotationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRotationRequest));
            CompletableFuture<GetRotationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an override to an on-call rotation.
     * </p>
     *
     * @param getRotationOverrideRequest
     * @return A Java Future containing the result of the GetRotationOverride operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.GetRotationOverride
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/GetRotationOverride"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRotationOverrideResponse> getRotationOverride(
            GetRotationOverrideRequest getRotationOverrideRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRotationOverrideRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRotationOverrideRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRotationOverride");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRotationOverrideResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRotationOverrideResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRotationOverrideResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRotationOverrideRequest, GetRotationOverrideResponse>()
                            .withOperationName("GetRotationOverride").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRotationOverrideRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRotationOverrideRequest));
            CompletableFuture<GetRotationOverrideResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all contact channels for the specified contact.
     * </p>
     *
     * @param listContactChannelsRequest
     * @return A Java Future containing the result of the ListContactChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListContactChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContactChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListContactChannelsResponse> listContactChannels(
            ListContactChannelsRequest listContactChannelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContactChannelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContactChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContactChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListContactChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContactChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContactChannelsRequest, ListContactChannelsResponse>()
                            .withOperationName("ListContactChannels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListContactChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listContactChannelsRequest));
            CompletableFuture<ListContactChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all contacts and escalation plans in Incident Manager.
     * </p>
     *
     * @param listContactsRequest
     * @return A Java Future containing the result of the ListContacts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListContacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListContactsResponse> listContacts(ListContactsRequest listContactsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContactsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListContactsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContactsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                            .withOperationName("ListContacts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListContactsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listContactsRequest));
            CompletableFuture<ListContactsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all engagements that have happened in an incident.
     * </p>
     *
     * @param listEngagementsRequest
     * @return A Java Future containing the result of the ListEngagements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListEngagements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListEngagements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEngagementsResponse> listEngagements(ListEngagementsRequest listEngagementsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEngagementsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEngagementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEngagements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEngagementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEngagementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEngagementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEngagementsRequest, ListEngagementsResponse>()
                            .withOperationName("ListEngagements").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEngagementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEngagementsRequest));
            CompletableFuture<ListEngagementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the engagements to contact channels that have been acknowledged.
     * </p>
     *
     * @param listPageReceiptsRequest
     * @return A Java Future containing the result of the ListPageReceipts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPageReceipts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageReceipts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPageReceiptsResponse> listPageReceipts(ListPageReceiptsRequest listPageReceiptsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPageReceiptsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPageReceiptsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPageReceipts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPageReceiptsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPageReceiptsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPageReceiptsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPageReceiptsRequest, ListPageReceiptsResponse>()
                            .withOperationName("ListPageReceipts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPageReceiptsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPageReceiptsRequest));
            CompletableFuture<ListPageReceiptsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the resolution path of an engagement. For example, the escalation plan engaged in an incident might
     * target an on-call schedule that includes several contacts in a rotation, but just one contact on-call when the
     * incident starts. The resolution path indicates the hierarchy of <i>escalation plan &gt; on-call schedule &gt;
     * contact</i>.
     * </p>
     *
     * @param listPageResolutionsRequest
     * @return A Java Future containing the result of the ListPageResolutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPageResolutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPageResolutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPageResolutionsResponse> listPageResolutions(
            ListPageResolutionsRequest listPageResolutionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPageResolutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPageResolutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPageResolutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPageResolutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPageResolutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPageResolutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPageResolutionsRequest, ListPageResolutionsResponse>()
                            .withOperationName("ListPageResolutions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPageResolutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPageResolutionsRequest));
            CompletableFuture<ListPageResolutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the engagements to a contact's contact channels.
     * </p>
     *
     * @param listPagesByContactRequest
     * @return A Java Future containing the result of the ListPagesByContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPagesByContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPagesByContactResponse> listPagesByContact(ListPagesByContactRequest listPagesByContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPagesByContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPagesByContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPagesByContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPagesByContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPagesByContactRequest, ListPagesByContactResponse>()
                            .withOperationName("ListPagesByContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPagesByContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPagesByContactRequest));
            CompletableFuture<ListPagesByContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the engagements to contact channels that occurred by engaging a contact.
     * </p>
     *
     * @param listPagesByEngagementRequest
     * @return A Java Future containing the result of the ListPagesByEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPagesByEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPagesByEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPagesByEngagementResponse> listPagesByEngagement(
            ListPagesByEngagementRequest listPagesByEngagementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPagesByEngagementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPagesByEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPagesByEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPagesByEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPagesByEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPagesByEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPagesByEngagementRequest, ListPagesByEngagementResponse>()
                            .withOperationName("ListPagesByEngagement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPagesByEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPagesByEngagementRequest));
            CompletableFuture<ListPagesByEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of shifts based on rotation configuration parameters.
     * </p>
     * <note>
     * <p>
     * The Incident Manager primarily uses this operation to populate the <b>Preview</b> calendar. It is not typically
     * run by end users.
     * </p>
     * </note>
     *
     * @param listPreviewRotationShiftsRequest
     * @return A Java Future containing the result of the ListPreviewRotationShifts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListPreviewRotationShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListPreviewRotationShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPreviewRotationShiftsResponse> listPreviewRotationShifts(
            ListPreviewRotationShiftsRequest listPreviewRotationShiftsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPreviewRotationShiftsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPreviewRotationShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPreviewRotationShifts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPreviewRotationShiftsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPreviewRotationShiftsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPreviewRotationShiftsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPreviewRotationShiftsRequest, ListPreviewRotationShiftsResponse>()
                            .withOperationName("ListPreviewRotationShifts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPreviewRotationShiftsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPreviewRotationShiftsRequest));
            CompletableFuture<ListPreviewRotationShiftsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of overrides currently specified for an on-call rotation.
     * </p>
     *
     * @param listRotationOverridesRequest
     * @return A Java Future containing the result of the ListRotationOverrides operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListRotationOverrides
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListRotationOverrides"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRotationOverridesResponse> listRotationOverrides(
            ListRotationOverridesRequest listRotationOverridesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRotationOverridesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRotationOverridesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRotationOverrides");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRotationOverridesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRotationOverridesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRotationOverridesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRotationOverridesRequest, ListRotationOverridesResponse>()
                            .withOperationName("ListRotationOverrides").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRotationOverridesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRotationOverridesRequest));
            CompletableFuture<ListRotationOverridesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of shifts generated by an existing rotation in the system.
     * </p>
     *
     * @param listRotationShiftsRequest
     * @return A Java Future containing the result of the ListRotationShifts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListRotationShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListRotationShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRotationShiftsResponse> listRotationShifts(ListRotationShiftsRequest listRotationShiftsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRotationShiftsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRotationShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRotationShifts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRotationShiftsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRotationShiftsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRotationShiftsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRotationShiftsRequest, ListRotationShiftsResponse>()
                            .withOperationName("ListRotationShifts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRotationShiftsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRotationShiftsRequest));
            CompletableFuture<ListRotationShiftsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of on-call rotations.
     * </p>
     *
     * @param listRotationsRequest
     * @return A Java Future containing the result of the ListRotations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListRotations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListRotations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRotationsResponse> listRotations(ListRotationsRequest listRotationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRotationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRotationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRotations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRotationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRotationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRotationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRotationsRequest, ListRotationsResponse>()
                            .withOperationName("ListRotations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRotationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRotationsRequest));
            CompletableFuture<ListRotationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags of an escalation plan or contact.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a resource policy to the specified contact or escalation plan. The resource policy is used to share the
     * contact or escalation plan using Resource Access Manager (RAM). For more information about cross-account sharing,
     * see <a href="https://docs.aws.amazon.com/incident-manager/latest/userguide/xa.html">Setting up cross-account
     * functionality</a>.
     * </p>
     *
     * @param putContactPolicyRequest
     * @return A Java Future containing the result of the PutContactPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.PutContactPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/PutContactPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutContactPolicyResponse> putContactPolicy(PutContactPolicyRequest putContactPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putContactPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContactPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContactPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutContactPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutContactPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutContactPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutContactPolicyRequest, PutContactPolicyResponse>()
                            .withOperationName("PutContactPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutContactPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putContactPolicyRequest));
            CompletableFuture<PutContactPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends an activation code to a contact channel. The contact can use this code to activate the contact channel in
     * the console or with the <code>ActivateChannel</code> operation. Incident Manager can't engage a contact channel
     * until it has been activated.
     * </p>
     *
     * @param sendActivationCodeRequest
     * @return A Java Future containing the result of the SendActivationCode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.SendActivationCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/SendActivationCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendActivationCodeResponse> sendActivationCode(SendActivationCodeRequest sendActivationCodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendActivationCodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendActivationCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendActivationCode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendActivationCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendActivationCodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendActivationCodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendActivationCodeRequest, SendActivationCodeResponse>()
                            .withOperationName("SendActivationCode").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendActivationCodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendActivationCodeRequest));
            CompletableFuture<SendActivationCodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an engagement to a contact or escalation plan. The engagement engages each contact specified in the
     * incident.
     * </p>
     *
     * @param startEngagementRequest
     * @return A Java Future containing the result of the StartEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.StartEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StartEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartEngagementResponse> startEngagement(StartEngagementRequest startEngagementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startEngagementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartEngagementRequest, StartEngagementResponse>()
                            .withOperationName("StartEngagement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startEngagementRequest));
            CompletableFuture<StartEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an engagement before it finishes the final stage of the escalation plan or engagement plan. Further
     * contacts aren't engaged.
     * </p>
     *
     * @param stopEngagementRequest
     * @return A Java Future containing the result of the StopEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.StopEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/StopEngagement" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopEngagementResponse> stopEngagement(StopEngagementRequest stopEngagementRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopEngagementRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopEngagementRequest, StopEngagementResponse>()
                            .withOperationName("StopEngagement").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopEngagementRequest));
            CompletableFuture<StopEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a contact or escalation plan. You can tag only contacts and escalation plans in the first region of your
     * replication set.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the contact or escalation plan specified.
     * </p>
     *
     * @param updateContactRequest
     * @return A Java Future containing the result of the UpdateContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UpdateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateContactResponse> updateContact(UpdateContactRequest updateContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateContactRequest, UpdateContactResponse>()
                            .withOperationName("UpdateContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateContactRequest));
            CompletableFuture<UpdateContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a contact's contact channel.
     * </p>
     *
     * @param updateContactChannelRequest
     * @return A Java Future containing the result of the UpdateContactChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>DataEncryptionException The operation failed to due an encryption key error.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UpdateContactChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateContactChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateContactChannelResponse> updateContactChannel(
            UpdateContactChannelRequest updateContactChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContactChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContactChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateContactChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateContactChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateContactChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateContactChannelRequest, UpdateContactChannelResponse>()
                            .withOperationName("UpdateContactChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateContactChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateContactChannelRequest));
            CompletableFuture<UpdateContactChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the information specified for an on-call rotation.
     * </p>
     *
     * @param updateRotationRequest
     * @return A Java Future containing the result of the UpdateRotation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have sufficient access to perform this operation.</li>
     *         <li>ConflictException Updating or deleting a resource causes an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error occurred while processing the request.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SsmContactsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SsmContactsAsyncClient.UpdateRotation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-contacts-2021-05-03/UpdateRotation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRotationResponse> updateRotation(UpdateRotationRequest updateRotationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRotationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRotationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SSM Contacts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRotation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRotationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRotationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRotationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRotationRequest, UpdateRotationResponse>()
                            .withOperationName("UpdateRotation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRotationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRotationRequest));
            CompletableFuture<UpdateRotationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SsmContactsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmContactsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataEncryptionException")
                                .exceptionBuilderSupplier(DataEncryptionException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SsmContactsServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = SsmContactsServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
