/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartSessionRequest extends SsmRequest implements
        ToCopyableBuilder<StartSessionRequest.Builder, StartSessionRequest> {
    private static final SdkField<String> TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Target")
            .getter(getter(StartSessionRequest::target)).setter(setter(Builder::target))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Target").build()).build();

    private static final SdkField<String> DOCUMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentName").getter(getter(StartSessionRequest::documentName)).setter(setter(Builder::documentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentName").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Reason")
            .getter(getter(StartSessionRequest::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Reason").build()).build();

    private static final SdkField<Map<String, List<String>>> PARAMETERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(StartSessionRequest::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_FIELD,
            DOCUMENT_NAME_FIELD, REASON_FIELD, PARAMETERS_FIELD));

    private final String target;

    private final String documentName;

    private final String reason;

    private final Map<String, List<String>> parameters;

    private StartSessionRequest(BuilderImpl builder) {
        super(builder);
        this.target = builder.target;
        this.documentName = builder.documentName;
        this.reason = builder.reason;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * The managed node to connect to for the session.
     * </p>
     * 
     * @return The managed node to connect to for the session.
     */
    public final String target() {
        return target;
    }

    /**
     * <p>
     * The name of the SSM document you want to use to define the type of session, input parameters, or preferences for
     * the session. For example, <code>SSM-SessionManagerRunShell</code>. You can call the <a>GetDocument</a> API to
     * verify the document exists before attempting to start a session. If no document name is provided, a shell to the
     * managed node is launched by default. For more information, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-sessions-start.html"
     * >Start a session</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * 
     * @return The name of the SSM document you want to use to define the type of session, input parameters, or
     *         preferences for the session. For example, <code>SSM-SessionManagerRunShell</code>. You can call the
     *         <a>GetDocument</a> API to verify the document exists before attempting to start a session. If no document
     *         name is provided, a shell to the managed node is launched by default. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-sessions-start.html"
     *         >Start a session</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     */
    public final String documentName() {
        return documentName;
    }

    /**
     * <p>
     * The reason for connecting to the instance. This value is included in the details for the Amazon CloudWatch Events
     * event created when you start the session.
     * </p>
     * 
     * @return The reason for connecting to the instance. This value is included in the details for the Amazon
     *         CloudWatch Events event created when you start the session.
     */
    public final String reason() {
        return reason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The values you want to specify for the parameters defined in the Session document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The values you want to specify for the parameters defined in the Session document.
     */
    public final Map<String, List<String>> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(target());
        hashCode = 31 * hashCode + Objects.hashCode(documentName());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartSessionRequest)) {
            return false;
        }
        StartSessionRequest other = (StartSessionRequest) obj;
        return Objects.equals(target(), other.target()) && Objects.equals(documentName(), other.documentName())
                && Objects.equals(reason(), other.reason()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartSessionRequest").add("Target", target()).add("DocumentName", documentName())
                .add("Reason", reason()).add("Parameters", hasParameters() ? parameters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Target":
            return Optional.ofNullable(clazz.cast(target()));
        case "DocumentName":
            return Optional.ofNullable(clazz.cast(documentName()));
        case "Reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartSessionRequest, T> g) {
        return obj -> g.apply((StartSessionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartSessionRequest> {
        /**
         * <p>
         * The managed node to connect to for the session.
         * </p>
         * 
         * @param target
         *        The managed node to connect to for the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder target(String target);

        /**
         * <p>
         * The name of the SSM document you want to use to define the type of session, input parameters, or preferences
         * for the session. For example, <code>SSM-SessionManagerRunShell</code>. You can call the <a>GetDocument</a>
         * API to verify the document exists before attempting to start a session. If no document name is provided, a
         * shell to the managed node is launched by default. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-sessions-start.html"
         * >Start a session</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param documentName
         *        The name of the SSM document you want to use to define the type of session, input parameters, or
         *        preferences for the session. For example, <code>SSM-SessionManagerRunShell</code>. You can call the
         *        <a>GetDocument</a> API to verify the document exists before attempting to start a session. If no
         *        document name is provided, a shell to the managed node is launched by default. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-sessions-start.html"
         *        >Start a session</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentName(String documentName);

        /**
         * <p>
         * The reason for connecting to the instance. This value is included in the details for the Amazon CloudWatch
         * Events event created when you start the session.
         * </p>
         * 
         * @param reason
         *        The reason for connecting to the instance. This value is included in the details for the Amazon
         *        CloudWatch Events event created when you start the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The values you want to specify for the parameters defined in the Session document.
         * </p>
         * 
         * @param parameters
         *        The values you want to specify for the parameters defined in the Session document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ? extends Collection<String>> parameters);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String target;

        private String documentName;

        private String reason;

        private Map<String, List<String>> parameters = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StartSessionRequest model) {
            super(model);
            target(model.target);
            documentName(model.documentName);
            reason(model.reason);
            parameters(model.parameters);
        }

        public final String getTarget() {
            return target;
        }

        public final void setTarget(String target) {
            this.target = target;
        }

        @Override
        public final Builder target(String target) {
            this.target = target;
            return this;
        }

        public final String getDocumentName() {
            return documentName;
        }

        public final void setDocumentName(String documentName) {
            this.documentName = documentName;
        }

        @Override
        public final Builder documentName(String documentName) {
            this.documentName = documentName;
            return this;
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = SessionManagerParametersCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, ? extends Collection<String>> parameters) {
            this.parameters = SessionManagerParametersCopier.copy(parameters);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartSessionRequest build() {
            return new StartSessionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
