/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sfn.internal.SfnServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.sfn.model.ActivityDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.ActivityLimitExceededException;
import software.amazon.awssdk.services.sfn.model.ActivityWorkerLimitExceededException;
import software.amazon.awssdk.services.sfn.model.ConflictException;
import software.amazon.awssdk.services.sfn.model.CreateActivityRequest;
import software.amazon.awssdk.services.sfn.model.CreateActivityResponse;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineAliasRequest;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineAliasResponse;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.CreateStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.DeleteActivityRequest;
import software.amazon.awssdk.services.sfn.model.DeleteActivityResponse;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineAliasRequest;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineAliasResponse;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineVersionRequest;
import software.amazon.awssdk.services.sfn.model.DeleteStateMachineVersionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeActivityRequest;
import software.amazon.awssdk.services.sfn.model.DescribeActivityResponse;
import software.amazon.awssdk.services.sfn.model.DescribeExecutionRequest;
import software.amazon.awssdk.services.sfn.model.DescribeExecutionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeMapRunRequest;
import software.amazon.awssdk.services.sfn.model.DescribeMapRunResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineAliasRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineAliasResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineForExecutionRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineForExecutionResponse;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.DescribeStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.ExecutionAlreadyExistsException;
import software.amazon.awssdk.services.sfn.model.ExecutionDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.ExecutionLimitExceededException;
import software.amazon.awssdk.services.sfn.model.GetActivityTaskRequest;
import software.amazon.awssdk.services.sfn.model.GetActivityTaskResponse;
import software.amazon.awssdk.services.sfn.model.GetExecutionHistoryRequest;
import software.amazon.awssdk.services.sfn.model.GetExecutionHistoryResponse;
import software.amazon.awssdk.services.sfn.model.InvalidArnException;
import software.amazon.awssdk.services.sfn.model.InvalidDefinitionException;
import software.amazon.awssdk.services.sfn.model.InvalidExecutionInputException;
import software.amazon.awssdk.services.sfn.model.InvalidLoggingConfigurationException;
import software.amazon.awssdk.services.sfn.model.InvalidNameException;
import software.amazon.awssdk.services.sfn.model.InvalidOutputException;
import software.amazon.awssdk.services.sfn.model.InvalidTokenException;
import software.amazon.awssdk.services.sfn.model.InvalidTracingConfigurationException;
import software.amazon.awssdk.services.sfn.model.ListActivitiesRequest;
import software.amazon.awssdk.services.sfn.model.ListActivitiesResponse;
import software.amazon.awssdk.services.sfn.model.ListExecutionsRequest;
import software.amazon.awssdk.services.sfn.model.ListExecutionsResponse;
import software.amazon.awssdk.services.sfn.model.ListMapRunsRequest;
import software.amazon.awssdk.services.sfn.model.ListMapRunsResponse;
import software.amazon.awssdk.services.sfn.model.ListStateMachineAliasesRequest;
import software.amazon.awssdk.services.sfn.model.ListStateMachineAliasesResponse;
import software.amazon.awssdk.services.sfn.model.ListStateMachineVersionsRequest;
import software.amazon.awssdk.services.sfn.model.ListStateMachineVersionsResponse;
import software.amazon.awssdk.services.sfn.model.ListStateMachinesRequest;
import software.amazon.awssdk.services.sfn.model.ListStateMachinesResponse;
import software.amazon.awssdk.services.sfn.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sfn.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sfn.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.sfn.model.PublishStateMachineVersionRequest;
import software.amazon.awssdk.services.sfn.model.PublishStateMachineVersionResponse;
import software.amazon.awssdk.services.sfn.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sfn.model.SendTaskFailureRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskFailureResponse;
import software.amazon.awssdk.services.sfn.model.SendTaskHeartbeatRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskHeartbeatResponse;
import software.amazon.awssdk.services.sfn.model.SendTaskSuccessRequest;
import software.amazon.awssdk.services.sfn.model.SendTaskSuccessResponse;
import software.amazon.awssdk.services.sfn.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.sfn.model.SfnException;
import software.amazon.awssdk.services.sfn.model.StartExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StartExecutionResponse;
import software.amazon.awssdk.services.sfn.model.StartSyncExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StartSyncExecutionResponse;
import software.amazon.awssdk.services.sfn.model.StateMachineAlreadyExistsException;
import software.amazon.awssdk.services.sfn.model.StateMachineDeletingException;
import software.amazon.awssdk.services.sfn.model.StateMachineDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.StateMachineLimitExceededException;
import software.amazon.awssdk.services.sfn.model.StateMachineTypeNotSupportedException;
import software.amazon.awssdk.services.sfn.model.StopExecutionRequest;
import software.amazon.awssdk.services.sfn.model.StopExecutionResponse;
import software.amazon.awssdk.services.sfn.model.TagResourceRequest;
import software.amazon.awssdk.services.sfn.model.TagResourceResponse;
import software.amazon.awssdk.services.sfn.model.TaskDoesNotExistException;
import software.amazon.awssdk.services.sfn.model.TaskTimedOutException;
import software.amazon.awssdk.services.sfn.model.TooManyTagsException;
import software.amazon.awssdk.services.sfn.model.UntagResourceRequest;
import software.amazon.awssdk.services.sfn.model.UntagResourceResponse;
import software.amazon.awssdk.services.sfn.model.UpdateMapRunRequest;
import software.amazon.awssdk.services.sfn.model.UpdateMapRunResponse;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineAliasRequest;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineAliasResponse;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineRequest;
import software.amazon.awssdk.services.sfn.model.UpdateStateMachineResponse;
import software.amazon.awssdk.services.sfn.model.ValidationException;
import software.amazon.awssdk.services.sfn.transform.CreateActivityRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.CreateStateMachineAliasRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.CreateStateMachineRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DeleteActivityRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DeleteStateMachineAliasRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DeleteStateMachineRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DeleteStateMachineVersionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeActivityRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeMapRunRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeStateMachineAliasRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeStateMachineForExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.DescribeStateMachineRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.GetActivityTaskRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.GetExecutionHistoryRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListActivitiesRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListExecutionsRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListMapRunsRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListStateMachineAliasesRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListStateMachineVersionsRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListStateMachinesRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.PublishStateMachineVersionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.SendTaskFailureRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.SendTaskHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.SendTaskSuccessRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.StartExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.StartSyncExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.StopExecutionRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.UpdateMapRunRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.UpdateStateMachineAliasRequestMarshaller;
import software.amazon.awssdk.services.sfn.transform.UpdateStateMachineRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SfnAsyncClient}.
 *
 * @see SfnAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSfnAsyncClient implements SfnAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSfnAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSfnAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an activity. An activity is a task that you write in any programming language and host on any machine
     * that has access to Step Functions. Activities must poll Step Functions using the <code>GetActivityTask</code> API
     * action and respond using <code>SendTask*</code> API actions. This function lets Step Functions know the existence
     * of your activity and returns an identifier for use in a state machine and when polling from the activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateActivity</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it was
     * already created. <code>CreateActivity</code>'s idempotency check is based on the activity <code>name</code>. If a
     * following request has different <code>tags</code> values, Step Functions will ignore these differences and treat
     * it as an idempotent request of the previous. In this case, <code>tags</code> will not be updated, even if they
     * are different.
     * </p>
     * </note>
     *
     * @param createActivityRequest
     * @return A Java Future containing the result of the CreateActivity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ActivityLimitExceededException The maximum number of activities has been reached. Existing activities
     *         must be deleted before a new activity can be created.</li>
     *         <li>InvalidNameException The provided name is not valid.</li>
     *         <li>TooManyTagsException You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the Step
     *         Functions Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.CreateActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateActivity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateActivityResponse> createActivity(CreateActivityRequest createActivityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createActivityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateActivity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateActivityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateActivityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateActivityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateActivityRequest, CreateActivityResponse>()
                            .withOperationName("CreateActivity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateActivityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createActivityRequest));
            CompletableFuture<CreateActivityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a state machine. A state machine consists of a collection of states that can do work (<code>Task</code>
     * states), determine to which states to transition next (<code>Choice</code> states), stop an execution with an
     * error (<code>Fail</code> states), and so on. State machines are specified using a JSON-based, structured
     * language. For more information, see <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-amazon-states-language.html">Amazon States
     * Language</a> in the Step Functions User Guide.
     * </p>
     * <p>
     * If you set the <code>publish</code> parameter of this API action to <code>true</code>, it publishes version
     * <code>1</code> as the first revision of the state machine.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note> <note>
     * <p>
     * <code>CreateStateMachine</code> is an idempotent API. Subsequent requests won’t create a duplicate resource if it
     * was already created. <code>CreateStateMachine</code>'s idempotency check is based on the state machine
     * <code>name</code>, <code>definition</code>, <code>type</code>, <code>LoggingConfiguration</code>, and
     * <code>TracingConfiguration</code>. The check is also based on the <code>publish</code> and
     * <code>versionDescription</code> parameters. If a following request has a different <code>roleArn</code> or
     * <code>tags</code>, Step Functions will ignore these differences and treat it as an idempotent request of the
     * previous. In this case, <code>roleArn</code> and <code>tags</code> will not be updated, even if they are
     * different.
     * </p>
     * </note>
     *
     * @param createStateMachineRequest
     * @return A Java Future containing the result of the CreateStateMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidDefinitionException The provided Amazon States Language definition is not valid.</li>
     *         <li>InvalidNameException The provided name is not valid.</li>
     *         <li>InvalidLoggingConfigurationException</li>
     *         <li>InvalidTracingConfigurationException Your <code>tracingConfiguration</code> key does not match, or
     *         <code>enabled</code> has not been set to <code>true</code> or <code>false</code>.</li>
     *         <li>StateMachineAlreadyExistsException A state machine with the same name but a different definition or
     *         role ARN already exists.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>StateMachineLimitExceededException The maximum number of state machines has been reached. Existing
     *         state machines must be deleted before a new state machine can be created.</li>
     *         <li>StateMachineTypeNotSupportedException</li>
     *         <li>TooManyTagsException You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the Step
     *         Functions Developer Guide.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.CreateStateMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStateMachineResponse> createStateMachine(CreateStateMachineRequest createStateMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStateMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStateMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateStateMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStateMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStateMachineRequest, CreateStateMachineResponse>()
                            .withOperationName("CreateStateMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStateMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStateMachineRequest));
            CompletableFuture<CreateStateMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a> for a
     * state machine that points to one or two <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">versions</a> of
     * the same state machine. You can set your application to call <a>StartExecution</a> with an alias and update the
     * version the alias uses without changing the client's code.
     * </p>
     * <p>
     * You can also map an alias to split <a>StartExecution</a> requests between two versions of a state machine. To do
     * this, add a second <code>RoutingConfig</code> object in the <code>routingConfiguration</code> parameter. You must
     * also specify the percentage of execution run requests each version should receive in both
     * <code>RoutingConfig</code> objects. Step Functions randomly chooses which version runs a given execution based on
     * the percentage you specify.
     * </p>
     * <p>
     * To create an alias that points to a single version, specify a single <code>RoutingConfig</code> object with a
     * <code>weight</code> set to 100.
     * </p>
     * <p>
     * You can create up to 100 aliases for each state machine. You must delete unused aliases using the
     * <a>DeleteStateMachineAlias</a> API action.
     * </p>
     * <p>
     * <code>CreateStateMachineAlias</code> is an idempotent API. Step Functions bases the idempotency check on the
     * <code>stateMachineArn</code>, <code>description</code>, <code>name</code>, and <code>routingConfiguration</code>
     * parameters. Requests that contain the same values for these parameters return a successful idempotent response
     * without creating a duplicate resource.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListStateMachineAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteStateMachineAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createStateMachineAliasRequest
     * @return A Java Future containing the result of the CreateStateMachineAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidNameException The provided name is not valid.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</p>
     *         <p>
     *         HTTP Status Code: 402</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.CreateStateMachineAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/CreateStateMachineAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStateMachineAliasResponse> createStateMachineAlias(
            CreateStateMachineAliasRequest createStateMachineAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStateMachineAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStateMachineAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStateMachineAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStateMachineAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateStateMachineAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStateMachineAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStateMachineAliasRequest, CreateStateMachineAliasResponse>()
                            .withOperationName("CreateStateMachineAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStateMachineAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStateMachineAliasRequest));
            CompletableFuture<CreateStateMachineAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an activity.
     * </p>
     *
     * @param deleteActivityRequest
     * @return A Java Future containing the result of the DeleteActivity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DeleteActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteActivity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteActivityResponse> deleteActivity(DeleteActivityRequest deleteActivityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteActivityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteActivity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteActivityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteActivityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteActivityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteActivityRequest, DeleteActivityResponse>()
                            .withOperationName("DeleteActivity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteActivityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteActivityRequest));
            CompletableFuture<DeleteActivityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a state machine. This is an asynchronous operation: It sets the state machine's status to
     * <code>DELETING</code> and begins the deletion process.
     * </p>
     * <p>
     * A qualified state machine ARN can either refer to a <i>Distributed Map state</i> defined within a state machine,
     * a version ARN, or an alias ARN.
     * </p>
     * <p>
     * The following are some examples of qualified and unqualified state machine ARNs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to a <i>Distributed Map state</i> with a label
     * <code>mapStateLabel</code> in a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:partition:states:region:account-id:stateMachine:myStateMachine/mapStateLabel</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a <i>Distributed Map state</i>, the request fails
     * with <code>ValidationException</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following unqualified state machine ARN refers to a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:partition:states:region:account-id:stateMachine:myStateMachine</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This API action also deletes all <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">versions</a> and
     * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">aliases</a>
     * associated with a state machine.
     * </p>
     * <note>
     * <p>
     * For <code>EXPRESS</code> state machines, the deletion happens eventually (usually in less than a minute). Running
     * executions may emit logs after <code>DeleteStateMachine</code> API is called.
     * </p>
     * </note>
     *
     * @param deleteStateMachineRequest
     * @return A Java Future containing the result of the DeleteStateMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DeleteStateMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStateMachineResponse> deleteStateMachine(DeleteStateMachineRequest deleteStateMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStateMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStateMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteStateMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStateMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStateMachineRequest, DeleteStateMachineResponse>()
                            .withOperationName("DeleteStateMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStateMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStateMachineRequest));
            CompletableFuture<DeleteStateMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a>.
     * </p>
     * <p>
     * After you delete a state machine alias, you can't use it to start executions. When you delete a state machine
     * alias, Step Functions doesn't delete the state machine versions that alias references.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListStateMachineAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateStateMachineAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteStateMachineAliasRequest
     * @return A Java Future containing the result of the DeleteStateMachineAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DeleteStateMachineAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteStateMachineAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStateMachineAliasResponse> deleteStateMachineAlias(
            DeleteStateMachineAliasRequest deleteStateMachineAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStateMachineAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStateMachineAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStateMachineAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStateMachineAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteStateMachineAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStateMachineAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStateMachineAliasRequest, DeleteStateMachineAliasResponse>()
                            .withOperationName("DeleteStateMachineAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStateMachineAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStateMachineAliasRequest));
            CompletableFuture<DeleteStateMachineAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">version</a>.
     * After you delete a version, you can't call <a>StartExecution</a> using that version's ARN or use the version with
     * a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a>.
     * </p>
     * <note>
     * <p>
     * Deleting a state machine version won't terminate its in-progress executions.
     * </p>
     * </note> <note>
     * <p>
     * You can't delete a state machine version currently referenced by one or more aliases. Before you delete a
     * version, you must either delete the aliases or update them to point to another state machine version.
     * </p>
     * </note>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>PublishStateMachineVersion</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListStateMachineVersions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteStateMachineVersionRequest
     * @return A Java Future containing the result of the DeleteStateMachineVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DeleteStateMachineVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DeleteStateMachineVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStateMachineVersionResponse> deleteStateMachineVersion(
            DeleteStateMachineVersionRequest deleteStateMachineVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStateMachineVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStateMachineVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStateMachineVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStateMachineVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteStateMachineVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStateMachineVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStateMachineVersionRequest, DeleteStateMachineVersionResponse>()
                            .withOperationName("DeleteStateMachineVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStateMachineVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStateMachineVersionRequest));
            CompletableFuture<DeleteStateMachineVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an activity.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param describeActivityRequest
     * @return A Java Future containing the result of the DescribeActivity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ActivityDoesNotExistException The specified activity does not exist.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DescribeActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeActivity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeActivityResponse> describeActivity(DescribeActivityRequest describeActivityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeActivityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeActivity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeActivityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeActivityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeActivityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeActivityRequest, DescribeActivityResponse>()
                            .withOperationName("DescribeActivity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeActivityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeActivityRequest));
            CompletableFuture<DescribeActivityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a state machine execution, such as the state machine associated with the execution,
     * the execution input and output, and relevant execution metadata. Use this API action to return the Map Run Amazon
     * Resource Name (ARN) if the execution was dispatched by a Map Run.
     * </p>
     * <p>
     * If you specify a version or alias ARN when you call the <a>StartExecution</a> API action,
     * <code>DescribeExecution</code> returns that ARN.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * Executions of an <code>EXPRESS</code> state machinearen't supported by <code>DescribeExecution</code> unless a
     * Map Run dispatched them.
     * </p>
     *
     * @param describeExecutionRequest
     * @return A Java Future containing the result of the DescribeExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExecutionDoesNotExistException The specified execution does not exist.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DescribeExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExecutionResponse> describeExecution(DescribeExecutionRequest describeExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExecutionRequest, DescribeExecutionResponse>()
                            .withOperationName("DescribeExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeExecutionRequest));
            CompletableFuture<DescribeExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a Map Run's configuration, progress, and results. For more information, see <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-examine-map-run.html">Examining Map Run</a>
     * in the <i>Step Functions Developer Guide</i>.
     * </p>
     *
     * @param describeMapRunRequest
     * @return A Java Future containing the result of the DescribeMapRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DescribeMapRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeMapRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMapRunResponse> describeMapRun(DescribeMapRunRequest describeMapRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMapRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMapRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMapRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMapRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMapRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMapRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMapRunRequest, DescribeMapRunResponse>()
                            .withOperationName("DescribeMapRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeMapRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMapRunRequest));
            CompletableFuture<DescribeMapRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a state machine's definition, its IAM role Amazon Resource Name (ARN), and
     * configuration.
     * </p>
     * <p>
     * A qualified state machine ARN can either refer to a <i>Distributed Map state</i> defined within a state machine,
     * a version ARN, or an alias ARN.
     * </p>
     * <p>
     * The following are some examples of qualified and unqualified state machine ARNs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to a <i>Distributed Map state</i> with a label
     * <code>mapStateLabel</code> in a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:partition:states:region:account-id:stateMachine:myStateMachine/mapStateLabel</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a <i>Distributed Map state</i>, the request fails
     * with <code>ValidationException</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to an alias named <code>PROD</code>.
     * </p>
     * <p>
     * <code>arn:&lt;partition&gt;:states:&lt;region&gt;:&lt;account-id&gt;:stateMachine:&lt;myStateMachine:PROD&gt;</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a version ARN or an alias ARN, the request starts
     * execution for that version or alias.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following unqualified state machine ARN refers to a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:&lt;partition&gt;:states:&lt;region&gt;:&lt;account-id&gt;:stateMachine:&lt;myStateMachine&gt;</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This API action returns the details for a state machine version if the <code>stateMachineArn</code> you specify
     * is a state machine version ARN.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param describeStateMachineRequest
     * @return A Java Future containing the result of the DescribeStateMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DescribeStateMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachine" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStateMachineResponse> describeStateMachine(
            DescribeStateMachineRequest describeStateMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStateMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStateMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStateMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStateMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStateMachineRequest, DescribeStateMachineResponse>()
                            .withOperationName("DescribeStateMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeStateMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeStateMachineRequest));
            CompletableFuture<DescribeStateMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a>.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListStateMachineAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteStateMachineAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeStateMachineAliasRequest
     * @return A Java Future containing the result of the DescribeStateMachineAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DescribeStateMachineAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachineAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStateMachineAliasResponse> describeStateMachineAlias(
            DescribeStateMachineAliasRequest describeStateMachineAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStateMachineAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStateMachineAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStateMachineAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStateMachineAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStateMachineAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStateMachineAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStateMachineAliasRequest, DescribeStateMachineAliasResponse>()
                            .withOperationName("DescribeStateMachineAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeStateMachineAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeStateMachineAliasRequest));
            CompletableFuture<DescribeStateMachineAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a state machine's definition, its execution role ARN, and configuration. If a Map Run
     * dispatched the execution, this action returns the Map Run Amazon Resource Name (ARN) in the response. The state
     * machine returned is the state machine associated with the Map Run.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param describeStateMachineForExecutionRequest
     * @return A Java Future containing the result of the DescribeStateMachineForExecution operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExecutionDoesNotExistException The specified execution does not exist.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.DescribeStateMachineForExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/DescribeStateMachineForExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStateMachineForExecutionResponse> describeStateMachineForExecution(
            DescribeStateMachineForExecutionRequest describeStateMachineForExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStateMachineForExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeStateMachineForExecutionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStateMachineForExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStateMachineForExecutionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeStateMachineForExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStateMachineForExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStateMachineForExecutionRequest, DescribeStateMachineForExecutionResponse>()
                            .withOperationName("DescribeStateMachineForExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeStateMachineForExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeStateMachineForExecutionRequest));
            CompletableFuture<DescribeStateMachineForExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by workers to retrieve a task (with the specified activity ARN) which has been scheduled for execution by a
     * running state machine. This initiates a long poll, where the service holds the HTTP connection open and responds
     * as soon as a task becomes available (i.e. an execution of a task of this type is needed.) The maximum time the
     * service holds on to the request before responding is 60 seconds. If no task is available within 60 seconds, the
     * poll returns a <code>taskToken</code> with a null string.
     * </p>
     * <note>
     * <p>
     * This API action isn't logged in CloudTrail.
     * </p>
     * </note> <important>
     * <p>
     * Workers should set their client side socket timeout to at least 65 seconds (5 seconds higher than the maximum
     * time the service may hold the poll request).
     * </p>
     * <p>
     * Polling with <code>GetActivityTask</code> can cause latency in some implementations. See <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/bp-activity-pollers.html">Avoid Latency When Polling
     * for Activity Tasks</a> in the Step Functions Developer Guide.
     * </p>
     * </important>
     *
     * @param getActivityTaskRequest
     * @return A Java Future containing the result of the GetActivityTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ActivityDoesNotExistException The specified activity does not exist.</li>
     *         <li>ActivityWorkerLimitExceededException The maximum number of workers concurrently polling for activity
     *         tasks has been reached.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.GetActivityTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetActivityTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetActivityTaskResponse> getActivityTask(GetActivityTaskRequest getActivityTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getActivityTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getActivityTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetActivityTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetActivityTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetActivityTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetActivityTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetActivityTaskRequest, GetActivityTaskResponse>()
                            .withOperationName("GetActivityTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetActivityTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getActivityTaskRequest));
            CompletableFuture<GetActivityTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the history of the specified execution as a list of events. By default, the results are returned in
     * ascending order of the <code>timeStamp</code> of the events. Use the <code>reverseOrder</code> parameter to get
     * the latest events first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param getExecutionHistoryRequest
     * @return A Java Future containing the result of the GetExecutionHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExecutionDoesNotExistException The specified execution does not exist.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.GetExecutionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/GetExecutionHistory" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetExecutionHistoryResponse> getExecutionHistory(
            GetExecutionHistoryRequest getExecutionHistoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExecutionHistoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExecutionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExecutionHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExecutionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetExecutionHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExecutionHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExecutionHistoryRequest, GetExecutionHistoryResponse>()
                            .withOperationName("GetExecutionHistory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetExecutionHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getExecutionHistoryRequest));
            CompletableFuture<GetExecutionHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the existing activities.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param listActivitiesRequest
     * @return A Java Future containing the result of the ListActivities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListActivities" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListActivitiesResponse> listActivities(ListActivitiesRequest listActivitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listActivitiesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActivities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListActivitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListActivitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListActivitiesRequest, ListActivitiesResponse>()
                            .withOperationName("ListActivities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListActivitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listActivitiesRequest));
            CompletableFuture<ListActivitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all executions of a state machine or a Map Run. You can list all executions related to a state machine by
     * specifying a state machine Amazon Resource Name (ARN), or those related to a Map Run by specifying a Map Run ARN.
     * </p>
     * <p>
     * You can also provide a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a> ARN or <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">version</a> ARN
     * to list the executions associated with a specific alias or version.
     * </p>
     * <p>
     * Results are sorted by time, with the most recent execution first.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param listExecutionsRequest
     * @return A Java Future containing the result of the ListExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>StateMachineTypeNotSupportedException</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExecutionsResponse> listExecutions(ListExecutionsRequest listExecutionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExecutionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExecutionsRequest, ListExecutionsResponse>()
                            .withOperationName("ListExecutions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listExecutionsRequest));
            CompletableFuture<ListExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Map Runs that were started by a given state machine execution. Use this API action to obtain Map Run
     * ARNs, and then call <code>DescribeMapRun</code> to obtain more information, if needed.
     * </p>
     *
     * @param listMapRunsRequest
     * @return A Java Future containing the result of the ListMapRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExecutionDoesNotExistException The specified execution does not exist.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListMapRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListMapRuns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMapRunsResponse> listMapRuns(ListMapRunsRequest listMapRunsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMapRunsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMapRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMapRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMapRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMapRunsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMapRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMapRunsRequest, ListMapRunsResponse>()
                            .withOperationName("ListMapRuns").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMapRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMapRunsRequest));
            CompletableFuture<ListMapRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">aliases</a> for a
     * specified state machine ARN. Results are sorted by time, with the most recently created aliases listed first.
     * </p>
     * <p>
     * To list aliases that reference a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">version</a>, you
     * can specify the version ARN in the <code>stateMachineArn</code> parameter.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteStateMachineAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listStateMachineAliasesRequest
     * @return A Java Future containing the result of the ListStateMachineAliases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListStateMachineAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachineAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStateMachineAliasesResponse> listStateMachineAliases(
            ListStateMachineAliasesRequest listStateMachineAliasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStateMachineAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStateMachineAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStateMachineAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStateMachineAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStateMachineAliasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStateMachineAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStateMachineAliasesRequest, ListStateMachineAliasesResponse>()
                            .withOperationName("ListStateMachineAliases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStateMachineAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStateMachineAliasesRequest));
            CompletableFuture<ListStateMachineAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">versions</a> for
     * the specified state machine Amazon Resource Name (ARN).
     * </p>
     * <p>
     * The results are sorted in descending order of the version creation time.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>PublishStateMachineVersion</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteStateMachineVersion</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listStateMachineVersionsRequest
     * @return A Java Future containing the result of the ListStateMachineVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListStateMachineVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachineVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStateMachineVersionsResponse> listStateMachineVersions(
            ListStateMachineVersionsRequest listStateMachineVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStateMachineVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStateMachineVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStateMachineVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStateMachineVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStateMachineVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStateMachineVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStateMachineVersionsRequest, ListStateMachineVersionsResponse>()
                            .withOperationName("ListStateMachineVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStateMachineVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStateMachineVersionsRequest));
            CompletableFuture<ListStateMachineVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the existing state machines.
     * </p>
     * <p>
     * If <code>nextToken</code> is returned, there are more results available. The value of <code>nextToken</code> is a
     * unique pagination token for each page. Make the call again using the returned token to retrieve the next page.
     * Keep all other arguments unchanged. Each pagination token expires after 24 hours. Using an expired pagination
     * token will return an <i>HTTP 400 InvalidToken</i> error.
     * </p>
     * <note>
     * <p>
     * This operation is eventually consistent. The results are best effort and may not reflect very recent updates and
     * changes.
     * </p>
     * </note>
     *
     * @param listStateMachinesRequest
     * @return A Java Future containing the result of the ListStateMachines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListStateMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListStateMachines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStateMachinesResponse> listStateMachines(ListStateMachinesRequest listStateMachinesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStateMachinesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStateMachinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStateMachines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStateMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStateMachinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStateMachinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStateMachinesRequest, ListStateMachinesResponse>()
                            .withOperationName("ListStateMachines").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStateMachinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStateMachinesRequest));
            CompletableFuture<ListStateMachinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List tags for a given resource.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">version</a> from
     * the current revision of a state machine. Use versions to create immutable snapshots of your state machine. You
     * can start executions from versions either directly or with an alias. To create an alias, use
     * <a>CreateStateMachineAlias</a>.
     * </p>
     * <p>
     * You can publish up to 1000 versions for each state machine. You must manually delete unused versions using the
     * <a>DeleteStateMachineVersion</a> API action.
     * </p>
     * <p>
     * <code>PublishStateMachineVersion</code> is an idempotent API. It doesn't create a duplicate state machine version
     * if it already exists for the current revision. Step Functions bases <code>PublishStateMachineVersion</code>'s
     * idempotency check on the <code>stateMachineArn</code>, <code>name</code>, and <code>revisionId</code> parameters.
     * Requests with the same parameters return a successful idempotent response. If you don't specify a
     * <code>revisionId</code>, Step Functions checks for a previously published version of the state machine's current
     * revision.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteStateMachineVersion</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListStateMachineVersions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param publishStateMachineVersionRequest
     * @return A Java Future containing the result of the PublishStateMachineVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</p>
     *         <p>
     *         HTTP Status Code: 402</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.PublishStateMachineVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/PublishStateMachineVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PublishStateMachineVersionResponse> publishStateMachineVersion(
            PublishStateMachineVersionRequest publishStateMachineVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(publishStateMachineVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishStateMachineVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PublishStateMachineVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PublishStateMachineVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PublishStateMachineVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PublishStateMachineVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PublishStateMachineVersionRequest, PublishStateMachineVersionResponse>()
                            .withOperationName("PublishStateMachineVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PublishStateMachineVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(publishStateMachineVersionRequest));
            CompletableFuture<PublishStateMachineVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> failed.
     * </p>
     *
     * @param sendTaskFailureRequest
     * @return A Java Future containing the result of the SendTaskFailure operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TaskDoesNotExistException</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>TaskTimedOutException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.SendTaskFailure
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskFailure" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendTaskFailureResponse> sendTaskFailure(SendTaskFailureRequest sendTaskFailureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendTaskFailureRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTaskFailureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTaskFailure");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendTaskFailureResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendTaskFailureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendTaskFailureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendTaskFailureRequest, SendTaskFailureResponse>()
                            .withOperationName("SendTaskFailure").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendTaskFailureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendTaskFailureRequest));
            CompletableFuture<SendTaskFailureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report to Step Functions that the task represented by the specified
     * <code>taskToken</code> is still making progress. This action resets the <code>Heartbeat</code> clock. The
     * <code>Heartbeat</code> threshold is specified in the state machine's Amazon States Language definition (
     * <code>HeartbeatSeconds</code>). This action does not in itself create an event in the execution history. However,
     * if the task times out, the execution history contains an <code>ActivityTimedOut</code> entry for activities, or a
     * <code>TaskTimedOut</code> entry for for tasks using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-sync">job run</a> or
     * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">
     * callback</a> pattern.
     * </p>
     * <note>
     * <p>
     * The <code>Timeout</code> of a task, defined in the state machine's Amazon States Language definition, is its
     * maximum allowed duration, regardless of the number of <a>SendTaskHeartbeat</a> requests received. Use
     * <code>HeartbeatSeconds</code> to configure the timeout interval for heartbeats.
     * </p>
     * </note>
     *
     * @param sendTaskHeartbeatRequest
     * @return A Java Future containing the result of the SendTaskHeartbeat operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TaskDoesNotExistException</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>TaskTimedOutException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.SendTaskHeartbeat
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskHeartbeat" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendTaskHeartbeatResponse> sendTaskHeartbeat(SendTaskHeartbeatRequest sendTaskHeartbeatRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendTaskHeartbeatRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTaskHeartbeatRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTaskHeartbeat");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendTaskHeartbeatResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendTaskHeartbeatResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendTaskHeartbeatResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendTaskHeartbeatRequest, SendTaskHeartbeatResponse>()
                            .withOperationName("SendTaskHeartbeat").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendTaskHeartbeatRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendTaskHeartbeatRequest));
            CompletableFuture<SendTaskHeartbeatResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by activity workers and task states using the <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token"
     * >callback</a> pattern to report that the task identified by the <code>taskToken</code> completed successfully.
     * </p>
     *
     * @param sendTaskSuccessRequest
     * @return A Java Future containing the result of the SendTaskSuccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TaskDoesNotExistException</li>
     *         <li>InvalidOutputException The provided JSON output data is not valid.</li>
     *         <li>InvalidTokenException The provided token is not valid.</li>
     *         <li>TaskTimedOutException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.SendTaskSuccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/SendTaskSuccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendTaskSuccessResponse> sendTaskSuccess(SendTaskSuccessRequest sendTaskSuccessRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendTaskSuccessRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTaskSuccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTaskSuccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendTaskSuccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendTaskSuccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendTaskSuccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendTaskSuccessRequest, SendTaskSuccessResponse>()
                            .withOperationName("SendTaskSuccess").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendTaskSuccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendTaskSuccessRequest));
            CompletableFuture<SendTaskSuccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a state machine execution.
     * </p>
     * <p>
     * A qualified state machine ARN can either refer to a <i>Distributed Map state</i> defined within a state machine,
     * a version ARN, or an alias ARN.
     * </p>
     * <p>
     * The following are some examples of qualified and unqualified state machine ARNs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to a <i>Distributed Map state</i> with a label
     * <code>mapStateLabel</code> in a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:partition:states:region:account-id:stateMachine:myStateMachine/mapStateLabel</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a <i>Distributed Map state</i>, the request fails
     * with <code>ValidationException</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to an alias named <code>PROD</code>.
     * </p>
     * <p>
     * <code>arn:&lt;partition&gt;:states:&lt;region&gt;:&lt;account-id&gt;:stateMachine:&lt;myStateMachine:PROD&gt;</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a version ARN or an alias ARN, the request starts
     * execution for that version or alias.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following unqualified state machine ARN refers to a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:&lt;partition&gt;:states:&lt;region&gt;:&lt;account-id&gt;:stateMachine:&lt;myStateMachine&gt;</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you start an execution with an unqualified state machine ARN, Step Functions uses the latest revision of the
     * state machine for the execution.
     * </p>
     * <p>
     * To start executions of a state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">version</a>, call
     * <code>StartExecution</code> and provide the version ARN or the ARN of an <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a> that
     * points to the version.
     * </p>
     * <note>
     * <p>
     * <code>StartExecution</code> is idempotent for <code>STANDARD</code> workflows. For a <code>STANDARD</code>
     * workflow, if you call <code>StartExecution</code> with the same name and input as a running execution, the call
     * succeeds and return the same response as the original request. If the execution is closed or if the input is
     * different, it returns a <code>400 ExecutionAlreadyExists</code> error. You can reuse names after 90 days.
     * </p>
     * <p>
     * <code>StartExecution</code> isn't idempotent for <code>EXPRESS</code> workflows.
     * </p>
     * </note>
     *
     * @param startExecutionRequest
     * @return A Java Future containing the result of the StartExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExecutionLimitExceededException The maximum number of running executions has been reached. Running
     *         executions must end or be stopped before a new execution can be started.</li>
     *         <li>ExecutionAlreadyExistsException The execution has the same <code>name</code> as another execution
     *         (but a different <code>input</code>).</p> <note>
     *         <p>
     *         Executions with the same <code>name</code> and <code>input</code> are considered idempotent.
     *         </p>
     *         </li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidExecutionInputException The provided JSON input data is not valid.</li>
     *         <li>InvalidNameException The provided name is not valid.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.StartExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StartExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartExecutionResponse> startExecution(StartExecutionRequest startExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startExecutionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartExecutionRequest, StartExecutionResponse>()
                            .withOperationName("StartExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startExecutionRequest));
            CompletableFuture<StartExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a Synchronous Express state machine execution. <code>StartSyncExecution</code> is not available for
     * <code>STANDARD</code> workflows.
     * </p>
     * <note>
     * <p>
     * <code>StartSyncExecution</code> will return a <code>200 OK</code> response, even if your execution fails, because
     * the status code in the API response doesn't reflect function errors. Error codes are reserved for errors that
     * prevent your execution from running, such as permissions errors, limit errors, or issues with your state machine
     * code and configuration.
     * </p>
     * </note> <note>
     * <p>
     * This API action isn't logged in CloudTrail.
     * </p>
     * </note>
     *
     * @param startSyncExecutionRequest
     * @return A Java Future containing the result of the StartSyncExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidExecutionInputException The provided JSON input data is not valid.</li>
     *         <li>InvalidNameException The provided name is not valid.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>StateMachineTypeNotSupportedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.StartSyncExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StartSyncExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartSyncExecutionResponse> startSyncExecution(StartSyncExecutionRequest startSyncExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSyncExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSyncExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSyncExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSyncExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSyncExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "sync-";
            String resolvedHostExpression = "sync-";

            CompletableFuture<StartSyncExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSyncExecutionRequest, StartSyncExecutionResponse>()
                            .withOperationName("StartSyncExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSyncExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(startSyncExecutionRequest));
            CompletableFuture<StartSyncExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an execution.
     * </p>
     * <p>
     * This API action is not supported by <code>EXPRESS</code> state machines.
     * </p>
     *
     * @param stopExecutionRequest
     * @return A Java Future containing the result of the StopExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExecutionDoesNotExistException The specified execution does not exist.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.StopExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/StopExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopExecutionResponse> stopExecution(StopExecutionRequest stopExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopExecutionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopExecutionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopExecutionRequest, StopExecutionResponse>()
                            .withOperationName("StopExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopExecutionRequest));
            CompletableFuture<StopExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add a tag to a Step Functions resource.
     * </p>
     * <p>
     * An array of key-value pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling Access Using IAM
     * Tags</a>.
     * </p>
     * <p>
     * Tags may only contain Unicode letters, digits, white space, or these symbols: <code>_ . : / = + - @</code>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>TooManyTagsException You've exceeded the number of tags allowed for a resource. See the <a
     *         href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html"> Limits Topic</a> in the Step
     *         Functions Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove a tag from a Step Functions resource
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an in-progress Map Run's configuration to include changes to the settings that control maximum
     * concurrency and Map Run failure.
     * </p>
     *
     * @param updateMapRunRequest
     * @return A Java Future containing the result of the UpdateMapRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.UpdateMapRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UpdateMapRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMapRunResponse> updateMapRun(UpdateMapRunRequest updateMapRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMapRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMapRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMapRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMapRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateMapRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMapRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMapRunRequest, UpdateMapRunResponse>()
                            .withOperationName("UpdateMapRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMapRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMapRunRequest));
            CompletableFuture<UpdateMapRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing state machine by modifying its <code>definition</code>, <code>roleArn</code>, or
     * <code>loggingConfiguration</code>. Running executions will continue to use the previous <code>definition</code>
     * and <code>roleArn</code>. You must include at least one of <code>definition</code> or <code>roleArn</code> or you
     * will receive a <code>MissingRequiredParameter</code> error.
     * </p>
     * <p>
     * A qualified state machine ARN refers to a <i>Distributed Map state</i> defined within a state machine. For
     * example, the qualified state machine ARN
     * <code>arn:partition:states:region:account-id:stateMachine:stateMachineName/mapStateLabel</code> refers to a
     * <i>Distributed Map state</i> with a label <code>mapStateLabel</code> in the state machine named
     * <code>stateMachineName</code>.
     * </p>
     * <p>
     * A qualified state machine ARN can either refer to a <i>Distributed Map state</i> defined within a state machine,
     * a version ARN, or an alias ARN.
     * </p>
     * <p>
     * The following are some examples of qualified and unqualified state machine ARNs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to a <i>Distributed Map state</i> with a label
     * <code>mapStateLabel</code> in a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:partition:states:region:account-id:stateMachine:myStateMachine/mapStateLabel</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a <i>Distributed Map state</i>, the request fails
     * with <code>ValidationException</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following qualified state machine ARN refers to an alias named <code>PROD</code>.
     * </p>
     * <p>
     * <code>arn:&lt;partition&gt;:states:&lt;region&gt;:&lt;account-id&gt;:stateMachine:&lt;myStateMachine:PROD&gt;</code>
     * </p>
     * <note>
     * <p>
     * If you provide a qualified state machine ARN that refers to a version ARN or an alias ARN, the request starts
     * execution for that version or alias.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * The following unqualified state machine ARN refers to a state machine named <code>myStateMachine</code>.
     * </p>
     * <p>
     * <code>arn:&lt;partition&gt;:states:&lt;region&gt;:&lt;account-id&gt;:stateMachine:&lt;myStateMachine&gt;</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you update your state machine, you can set the <code>publish</code> parameter to <code>true</code> in the
     * same action to publish a new <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-version.html">version</a>. This
     * way, you can opt-in to strict versioning of your state machine.
     * </p>
     * <note>
     * <p>
     * Step Functions assigns monotonically increasing integers for state machine versions, starting at version number
     * 1.
     * </p>
     * </note> <note>
     * <p>
     * All <code>StartExecution</code> calls within a few seconds use the updated <code>definition</code> and
     * <code>roleArn</code>. Executions started immediately after you call <code>UpdateStateMachine</code> may use the
     * previous state machine <code>definition</code> and <code>roleArn</code>.
     * </p>
     * </note>
     *
     * @param updateStateMachineRequest
     * @return A Java Future containing the result of the UpdateStateMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>InvalidDefinitionException The provided Amazon States Language definition is not valid.</li>
     *         <li>InvalidLoggingConfigurationException</li>
     *         <li>InvalidTracingConfigurationException Your <code>tracingConfiguration</code> key does not match, or
     *         <code>enabled</code> has not been set to <code>true</code> or <code>false</code>.</li>
     *         <li>MissingRequiredParameterException Request is missing a required parameter. This error occurs if both
     *         <code>definition</code> and <code>roleArn</code> are not specified.</li>
     *         <li>StateMachineDeletingException The specified state machine is being deleted.</li>
     *         <li>StateMachineDoesNotExistException The specified state machine does not exist.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</p>
     *         <p>
     *         HTTP Status Code: 402</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.UpdateStateMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UpdateStateMachine" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStateMachineResponse> updateStateMachine(UpdateStateMachineRequest updateStateMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStateMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStateMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStateMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStateMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateStateMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStateMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStateMachineRequest, UpdateStateMachineResponse>()
                            .withOperationName("UpdateStateMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateStateMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateStateMachineRequest));
            CompletableFuture<UpdateStateMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing state machine <a
     * href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-alias.html">alias</a> by
     * modifying its <code>description</code> or <code>routingConfiguration</code>.
     * </p>
     * <p>
     * You must specify at least one of the <code>description</code> or <code>routingConfiguration</code> parameters to
     * update a state machine alias.
     * </p>
     * <note>
     * <p>
     * <code>UpdateStateMachineAlias</code> is an idempotent API. Step Functions bases the idempotency check on the
     * <code>stateMachineAliasArn</code>, <code>description</code>, and <code>routingConfiguration</code> parameters.
     * Requests with the same parameters return an idempotent response.
     * </p>
     * </note> <note>
     * <p>
     * This operation is eventually consistent. All <a>StartExecution</a> requests made within a few seconds use the
     * latest alias configuration. Executions started immediately after calling <code>UpdateStateMachineAlias</code> may
     * use the previous routing configuration.
     * </p>
     * </note>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeStateMachineAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListStateMachineAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteStateMachineAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateStateMachineAliasRequest
     * @return A Java Future containing the result of the UpdateStateMachineAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InvalidArnException The provided Amazon Resource Name (ARN) is not valid.</li>
     *         <li>ResourceNotFoundException Could not find the referenced resource.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state. This error occurs
     *         when there're concurrent requests for <a>DeleteStateMachineVersion</a>,
     *         <a>PublishStateMachineVersion</a>, or <a>UpdateStateMachine</a> with the <code>publish</code> parameter
     *         set to <code>true</code>.</p>
     *         <p>
     *         HTTP Status Code: 409</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SfnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SfnAsyncClient.UpdateStateMachineAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/states-2016-11-23/UpdateStateMachineAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStateMachineAliasResponse> updateStateMachineAlias(
            UpdateStateMachineAliasRequest updateStateMachineAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStateMachineAliasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStateMachineAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SFN");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStateMachineAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStateMachineAliasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateStateMachineAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStateMachineAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStateMachineAliasRequest, UpdateStateMachineAliasResponse>()
                            .withOperationName("UpdateStateMachineAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateStateMachineAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateStateMachineAliasRequest));
            CompletableFuture<UpdateStateMachineAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SfnServiceClientConfiguration serviceClientConfiguration() {
        return new SfnServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SfnException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActivityDoesNotExist")
                                .exceptionBuilderSupplier(ActivityDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineTypeNotSupported")
                                .exceptionBuilderSupplier(StateMachineTypeNotSupportedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDefinition")
                                .exceptionBuilderSupplier(InvalidDefinitionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidName")
                                .exceptionBuilderSupplier(InvalidNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLoggingConfiguration")
                                .exceptionBuilderSupplier(InvalidLoggingConfigurationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineAlreadyExists")
                                .exceptionBuilderSupplier(StateMachineAlreadyExistsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecutionLimitExceeded")
                                .exceptionBuilderSupplier(ExecutionLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArn")
                                .exceptionBuilderSupplier(InvalidArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineDeleting")
                                .exceptionBuilderSupplier(StateMachineDeletingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActivityWorkerLimitExceeded")
                                .exceptionBuilderSupplier(ActivityWorkerLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaskTimedOut")
                                .exceptionBuilderSupplier(TaskTimedOutException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecutionAlreadyExists")
                                .exceptionBuilderSupplier(ExecutionAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameter")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExecutionDoesNotExist")
                                .exceptionBuilderSupplier(ExecutionDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineLimitExceeded")
                                .exceptionBuilderSupplier(StateMachineLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidToken")
                                .exceptionBuilderSupplier(InvalidTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTags")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOutput")
                                .exceptionBuilderSupplier(InvalidOutputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTracingConfiguration")
                                .exceptionBuilderSupplier(InvalidTracingConfigurationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActivityLimitExceeded")
                                .exceptionBuilderSupplier(ActivityLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExecutionInput")
                                .exceptionBuilderSupplier(InvalidExecutionInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaskDoesNotExist")
                                .exceptionBuilderSupplier(TaskDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StateMachineDoesNotExist")
                                .exceptionBuilderSupplier(StateMachineDoesNotExistException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SfnServiceClientConfigurationBuilder serviceConfigBuilder = new SfnServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
