/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Recipient-related information to include in the Delivery Status Notification (DSN) when an email that Amazon SES
 * receives on your behalf bounces.
 * </p>
 * <p>
 * For information about receiving email through Amazon SES, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BouncedRecipientInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<BouncedRecipientInfo.Builder, BouncedRecipientInfo> {
    private static final SdkField<String> RECIPIENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Recipient").getter(getter(BouncedRecipientInfo::recipient)).setter(setter(Builder::recipient))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recipient").build()).build();

    private static final SdkField<String> RECIPIENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecipientArn").getter(getter(BouncedRecipientInfo::recipientArn)).setter(setter(Builder::recipientArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecipientArn").build()).build();

    private static final SdkField<String> BOUNCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BounceType").getter(getter(BouncedRecipientInfo::bounceTypeAsString))
            .setter(setter(Builder::bounceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BounceType").build()).build();

    private static final SdkField<RecipientDsnFields> RECIPIENT_DSN_FIELDS_FIELD = SdkField
            .<RecipientDsnFields> builder(MarshallingType.SDK_POJO).memberName("RecipientDsnFields")
            .getter(getter(BouncedRecipientInfo::recipientDsnFields)).setter(setter(Builder::recipientDsnFields))
            .constructor(RecipientDsnFields::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecipientDsnFields").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RECIPIENT_FIELD,
            RECIPIENT_ARN_FIELD, BOUNCE_TYPE_FIELD, RECIPIENT_DSN_FIELDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String recipient;

    private final String recipientArn;

    private final String bounceType;

    private final RecipientDsnFields recipientDsnFields;

    private BouncedRecipientInfo(BuilderImpl builder) {
        this.recipient = builder.recipient;
        this.recipientArn = builder.recipientArn;
        this.bounceType = builder.bounceType;
        this.recipientDsnFields = builder.recipientDsnFields;
    }

    /**
     * <p>
     * The email address of the recipient of the bounced email.
     * </p>
     * 
     * @return The email address of the recipient of the bounced email.
     */
    public final String recipient() {
        return recipient;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to receive email for the recipient of the bounced email. For more
     * information about sending authorization, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to receive email for the recipient of the bounced
     *         email. For more information about sending authorization, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer
     *         Guide</a>.
     */
    public final String recipientArn() {
        return recipientArn;
    }

    /**
     * <p>
     * The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bounceType} will
     * return {@link BounceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bounceTypeAsString}.
     * </p>
     * 
     * @return The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
     * @see BounceType
     */
    public final BounceType bounceType() {
        return BounceType.fromValue(bounceType);
    }

    /**
     * <p>
     * The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bounceType} will
     * return {@link BounceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bounceTypeAsString}.
     * </p>
     * 
     * @return The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
     * @see BounceType
     */
    public final String bounceTypeAsString() {
        return bounceType;
    }

    /**
     * <p>
     * Recipient-related DSN fields, most of which would normally be filled in automatically when provided with a
     * <code>BounceType</code>. You must provide either this parameter or <code>BounceType</code>.
     * </p>
     * 
     * @return Recipient-related DSN fields, most of which would normally be filled in automatically when provided with
     *         a <code>BounceType</code>. You must provide either this parameter or <code>BounceType</code>.
     */
    public final RecipientDsnFields recipientDsnFields() {
        return recipientDsnFields;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(recipient());
        hashCode = 31 * hashCode + Objects.hashCode(recipientArn());
        hashCode = 31 * hashCode + Objects.hashCode(bounceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(recipientDsnFields());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BouncedRecipientInfo)) {
            return false;
        }
        BouncedRecipientInfo other = (BouncedRecipientInfo) obj;
        return Objects.equals(recipient(), other.recipient()) && Objects.equals(recipientArn(), other.recipientArn())
                && Objects.equals(bounceTypeAsString(), other.bounceTypeAsString())
                && Objects.equals(recipientDsnFields(), other.recipientDsnFields());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BouncedRecipientInfo").add("Recipient", recipient()).add("RecipientArn", recipientArn())
                .add("BounceType", bounceTypeAsString()).add("RecipientDsnFields", recipientDsnFields()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Recipient":
            return Optional.ofNullable(clazz.cast(recipient()));
        case "RecipientArn":
            return Optional.ofNullable(clazz.cast(recipientArn()));
        case "BounceType":
            return Optional.ofNullable(clazz.cast(bounceTypeAsString()));
        case "RecipientDsnFields":
            return Optional.ofNullable(clazz.cast(recipientDsnFields()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BouncedRecipientInfo, T> g) {
        return obj -> g.apply((BouncedRecipientInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BouncedRecipientInfo> {
        /**
         * <p>
         * The email address of the recipient of the bounced email.
         * </p>
         * 
         * @param recipient
         *        The email address of the recipient of the bounced email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipient(String recipient);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to receive email for the recipient of the bounced email.
         * For more information about sending authorization, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer Guide</a>.
         * </p>
         * 
         * @param recipientArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to receive email for the recipient
         *        of the bounced email. For more information about sending authorization, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/dg/sending-authorization.html">Amazon SES Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipientArn(String recipientArn);

        /**
         * <p>
         * The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
         * </p>
         * 
         * @param bounceType
         *        The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
         * @see BounceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BounceType
         */
        Builder bounceType(String bounceType);

        /**
         * <p>
         * The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
         * </p>
         * 
         * @param bounceType
         *        The reason for the bounce. You must provide either this parameter or <code>RecipientDsnFields</code>.
         * @see BounceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BounceType
         */
        Builder bounceType(BounceType bounceType);

        /**
         * <p>
         * Recipient-related DSN fields, most of which would normally be filled in automatically when provided with a
         * <code>BounceType</code>. You must provide either this parameter or <code>BounceType</code>.
         * </p>
         * 
         * @param recipientDsnFields
         *        Recipient-related DSN fields, most of which would normally be filled in automatically when provided
         *        with a <code>BounceType</code>. You must provide either this parameter or <code>BounceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipientDsnFields(RecipientDsnFields recipientDsnFields);

        /**
         * <p>
         * Recipient-related DSN fields, most of which would normally be filled in automatically when provided with a
         * <code>BounceType</code>. You must provide either this parameter or <code>BounceType</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecipientDsnFields.Builder} avoiding the
         * need to create one manually via {@link RecipientDsnFields#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecipientDsnFields.Builder#build()} is called immediately and its
         * result is passed to {@link #recipientDsnFields(RecipientDsnFields)}.
         * 
         * @param recipientDsnFields
         *        a consumer that will call methods on {@link RecipientDsnFields.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recipientDsnFields(RecipientDsnFields)
         */
        default Builder recipientDsnFields(Consumer<RecipientDsnFields.Builder> recipientDsnFields) {
            return recipientDsnFields(RecipientDsnFields.builder().applyMutation(recipientDsnFields).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String recipient;

        private String recipientArn;

        private String bounceType;

        private RecipientDsnFields recipientDsnFields;

        private BuilderImpl() {
        }

        private BuilderImpl(BouncedRecipientInfo model) {
            recipient(model.recipient);
            recipientArn(model.recipientArn);
            bounceType(model.bounceType);
            recipientDsnFields(model.recipientDsnFields);
        }

        public final String getRecipient() {
            return recipient;
        }

        public final void setRecipient(String recipient) {
            this.recipient = recipient;
        }

        @Override
        public final Builder recipient(String recipient) {
            this.recipient = recipient;
            return this;
        }

        public final String getRecipientArn() {
            return recipientArn;
        }

        public final void setRecipientArn(String recipientArn) {
            this.recipientArn = recipientArn;
        }

        @Override
        public final Builder recipientArn(String recipientArn) {
            this.recipientArn = recipientArn;
            return this;
        }

        public final String getBounceType() {
            return bounceType;
        }

        public final void setBounceType(String bounceType) {
            this.bounceType = bounceType;
        }

        @Override
        public final Builder bounceType(String bounceType) {
            this.bounceType = bounceType;
            return this;
        }

        @Override
        public final Builder bounceType(BounceType bounceType) {
            this.bounceType(bounceType == null ? null : bounceType.toString());
            return this;
        }

        public final RecipientDsnFields.Builder getRecipientDsnFields() {
            return recipientDsnFields != null ? recipientDsnFields.toBuilder() : null;
        }

        public final void setRecipientDsnFields(RecipientDsnFields.BuilderImpl recipientDsnFields) {
            this.recipientDsnFields = recipientDsnFields != null ? recipientDsnFields.build() : null;
        }

        @Override
        public final Builder recipientDsnFields(RecipientDsnFields recipientDsnFields) {
            this.recipientDsnFields = recipientDsnFields;
            return this;
        }

        @Override
        public BouncedRecipientInfo build() {
            return new BouncedRecipientInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
