/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicediscovery.internal.ServiceDiscoveryServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CustomHealthNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesRevisionRequest;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesRevisionResponse;
import software.amazon.awssdk.services.servicediscovery.model.DuplicateRequestException;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.InstanceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.InvalidInputException;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.OperationNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.RequestLimitExceededException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceInUseException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceDiscoveryException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.TagResourceRequest;
import software.amazon.awssdk.services.servicediscovery.model.TagResourceResponse;
import software.amazon.awssdk.services.servicediscovery.model.TooManyTagsException;
import software.amazon.awssdk.services.servicediscovery.model.UntagResourceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UntagResourceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateHttpNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateHttpNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePrivateDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePrivateDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePublicDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdatePublicDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.transform.CreateHttpNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePrivateDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePublicDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeregisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DiscoverInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DiscoverInstancesRevisionRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstancesHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetOperationRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListNamespacesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.RegisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateHttpNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateInstanceCustomHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdatePrivateDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdatePublicDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateServiceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ServiceDiscoveryAsyncClient}.
 *
 * @see ServiceDiscoveryAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceDiscoveryAsyncClient implements ServiceDiscoveryAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultServiceDiscoveryAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceDiscoveryAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an HTTP namespace. Service instances registered using an HTTP namespace can be discovered using a
     * <code>DiscoverInstances</code> request but can't be discovered using DNS.
     * </p>
     * <p>
     * For the current quota on the number of namespaces that you can create using the same Amazon Web Services account,
     * see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map quotas</a> in the
     * <i>Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createHttpNamespaceRequest
     * @return A Java Future containing the result of the CreateHttpNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceAlreadyExistsException The namespace that you're trying to create already exists.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the quota on the
     *         number of resources.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>TooManyTagsException The list of tags on the resource is over the quota. The maximum number of tags
     *         that can be applied to a resource is 50.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreateHttpNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateHttpNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHttpNamespaceResponse> createHttpNamespace(
            CreateHttpNamespaceRequest createHttpNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHttpNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHttpNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHttpNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHttpNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateHttpNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHttpNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHttpNamespaceRequest, CreateHttpNamespaceResponse>()
                            .withOperationName("CreateHttpNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHttpNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHttpNamespaceRequest));
            CompletableFuture<CreateHttpNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a private namespace based on DNS, which is visible only inside a specified Amazon VPC. The namespace
     * defines your service naming scheme. For example, if you name your namespace <code>example.com</code> and name
     * your service <code>backend</code>, the resulting DNS name for the service is <code>backend.example.com</code>.
     * Service instances that are registered using a private DNS namespace can be discovered using either a
     * <code>DiscoverInstances</code> request or using DNS. For the current quota on the number of namespaces that you
     * can create using the same Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map quotas</a> in the <i>Cloud
     * Map Developer Guide</i>.
     * </p>
     *
     * @param createPrivateDnsNamespaceRequest
     * @return A Java Future containing the result of the CreatePrivateDnsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceAlreadyExistsException The namespace that you're trying to create already exists.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the quota on the
     *         number of resources.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>TooManyTagsException The list of tags on the resource is over the quota. The maximum number of tags
     *         that can be applied to a resource is 50.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreatePrivateDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePrivateDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePrivateDnsNamespaceResponse> createPrivateDnsNamespace(
            CreatePrivateDnsNamespaceRequest createPrivateDnsNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPrivateDnsNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPrivateDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePrivateDnsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePrivateDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePrivateDnsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePrivateDnsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePrivateDnsNamespaceRequest, CreatePrivateDnsNamespaceResponse>()
                            .withOperationName("CreatePrivateDnsNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePrivateDnsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPrivateDnsNamespaceRequest));
            CompletableFuture<CreatePrivateDnsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a public namespace based on DNS, which is visible on the internet. The namespace defines your service
     * naming scheme. For example, if you name your namespace <code>example.com</code> and name your service
     * <code>backend</code>, the resulting DNS name for the service is <code>backend.example.com</code>. You can
     * discover instances that were registered with a public DNS namespace by using either a
     * <code>DiscoverInstances</code> request or using DNS. For the current quota on the number of namespaces that you
     * can create using the same Amazon Web Services account, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map quotas</a> in the <i>Cloud
     * Map Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * The <code>CreatePublicDnsNamespace</code> API operation is not supported in the Amazon Web Services GovCloud (US)
     * Regions.
     * </p>
     * </important>
     *
     * @param createPublicDnsNamespaceRequest
     * @return A Java Future containing the result of the CreatePublicDnsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceAlreadyExistsException The namespace that you're trying to create already exists.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the quota on the
     *         number of resources.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>TooManyTagsException The list of tags on the resource is over the quota. The maximum number of tags
     *         that can be applied to a resource is 50.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreatePublicDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePublicDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePublicDnsNamespaceResponse> createPublicDnsNamespace(
            CreatePublicDnsNamespaceRequest createPublicDnsNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPublicDnsNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPublicDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePublicDnsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePublicDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePublicDnsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePublicDnsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePublicDnsNamespaceRequest, CreatePublicDnsNamespaceResponse>()
                            .withOperationName("CreatePublicDnsNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePublicDnsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPublicDnsNamespaceRequest));
            CompletableFuture<CreatePublicDnsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a service. This action defines the configuration for the following entities:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For public and private DNS namespaces, one of the following combinations of DNS records in Amazon Route 53:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>A</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AAAA</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>A</code> and <code>AAAA</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SRV</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CNAME</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Optionally, a health check
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you create the service, you can submit a <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a> request,
     * and Cloud Map uses the values in the configuration to create the specified entities.
     * </p>
     * <p>
     * For the current quota on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map
     * quotas</a> in the <i>Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createServiceRequest
     * @return A Java Future containing the result of the CreateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the quota on the
     *         number of resources.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ServiceAlreadyExistsException The service can't be created because a service with the same name
     *         already exists.</li>
     *         <li>TooManyTagsException The list of tags on the resource is over the quota. The maximum number of tags
     *         that can be applied to a resource is 50.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServiceResponse> createService(CreateServiceRequest createServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                            .withOperationName("CreateService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createServiceRequest));
            CompletableFuture<CreateServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a namespace from the current account. If the namespace still contains one or more services, the request
     * fails.
     * </p>
     *
     * @param deleteNamespaceRequest
     * @return A Java Future containing the result of the DeleteNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DeleteNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNamespaceResponse> deleteNamespace(DeleteNamespaceRequest deleteNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNamespaceRequest, DeleteNamespaceResponse>()
                            .withOperationName("DeleteNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNamespaceRequest));
            CompletableFuture<DeleteNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified service. If the service still contains one or more registered instances, the request fails.
     * </p>
     *
     * @param deleteServiceRequest
     * @return A Java Future containing the result of the DeleteService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DeleteService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceResponse> deleteService(DeleteServiceRequest deleteServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                            .withOperationName("DeleteService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServiceRequest));
            CompletableFuture<DeleteServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon Route 53 DNS records and health check, if any, that Cloud Map created for the specified
     * instance.
     * </p>
     *
     * @param deregisterInstanceRequest
     * @return A Java Future containing the result of the DeregisterInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DeregisterInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeregisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterInstanceResponse> deregisterInstance(DeregisterInstanceRequest deregisterInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterInstanceRequest, DeregisterInstanceResponse>()
                            .withOperationName("DeregisterInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterInstanceRequest));
            CompletableFuture<DeregisterInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Discovers registered instances for a specified namespace and service. You can use <code>DiscoverInstances</code>
     * to discover instances for any type of namespace. <code>DiscoverInstances</code> returns a randomized list of
     * instances allowing customers to distribute traffic evenly across instances. For public and private DNS
     * namespaces, you can also use DNS queries to discover instances.
     * </p>
     *
     * @param discoverInstancesRequest
     * @return A Java Future containing the result of the DiscoverInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>RequestLimitExceededException The operation can't be completed because you've reached the quota for
     *         the number of requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/dg/throttling.html">Cloud Map API request throttling
     *         quota</a> in the <i>Cloud Map Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DiscoverInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DiscoverInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DiscoverInstancesResponse> discoverInstances(DiscoverInstancesRequest discoverInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(discoverInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, discoverInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DiscoverInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DiscoverInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DiscoverInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "data-";
            String resolvedHostExpression = "data-";

            CompletableFuture<DiscoverInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DiscoverInstancesRequest, DiscoverInstancesResponse>()
                            .withOperationName("DiscoverInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DiscoverInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(discoverInstancesRequest));
            CompletableFuture<DiscoverInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Discovers the increasing revision associated with an instance.
     * </p>
     *
     * @param discoverInstancesRevisionRequest
     * @return A Java Future containing the result of the DiscoverInstancesRevision operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>RequestLimitExceededException The operation can't be completed because you've reached the quota for
     *         the number of requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/cloud-map/latest/dg/throttling.html">Cloud Map API request throttling
     *         quota</a> in the <i>Cloud Map Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DiscoverInstancesRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DiscoverInstancesRevision"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DiscoverInstancesRevisionResponse> discoverInstancesRevision(
            DiscoverInstancesRevisionRequest discoverInstancesRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(discoverInstancesRevisionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, discoverInstancesRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DiscoverInstancesRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DiscoverInstancesRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DiscoverInstancesRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "data-";
            String resolvedHostExpression = "data-";

            CompletableFuture<DiscoverInstancesRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DiscoverInstancesRevisionRequest, DiscoverInstancesRevisionResponse>()
                            .withOperationName("DiscoverInstancesRevision").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DiscoverInstancesRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(discoverInstancesRevisionRequest));
            CompletableFuture<DiscoverInstancesRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified instance.
     * </p>
     *
     * @param getInstanceRequest
     * @return A Java Future containing the result of the GetInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInstanceResponse> getInstance(GetInstanceRequest getInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInstanceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInstanceRequest, GetInstanceResponse>()
                            .withOperationName("GetInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInstanceRequest));
            CompletableFuture<GetInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the current health status (<code>Healthy</code>, <code>Unhealthy</code>, or <code>Unknown</code>) of one or
     * more instances that are associated with a specified service.
     * </p>
     * <note>
     * <p>
     * There's a brief delay between when you register an instance and when the health status for the instance is
     * available.
     * </p>
     * </note>
     *
     * @param getInstancesHealthStatusRequest
     * @return A Java Future containing the result of the GetInstancesHealthStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetInstancesHealthStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstancesHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInstancesHealthStatusResponse> getInstancesHealthStatus(
            GetInstancesHealthStatusRequest getInstancesHealthStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInstancesHealthStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInstancesHealthStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstancesHealthStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInstancesHealthStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInstancesHealthStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInstancesHealthStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInstancesHealthStatusRequest, GetInstancesHealthStatusResponse>()
                            .withOperationName("GetInstancesHealthStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInstancesHealthStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInstancesHealthStatusRequest));
            CompletableFuture<GetInstancesHealthStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a namespace.
     * </p>
     *
     * @param getNamespaceRequest
     * @return A Java Future containing the result of the GetNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNamespaceResponse> getNamespace(GetNamespaceRequest getNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNamespaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNamespaceRequest, GetNamespaceResponse>()
                            .withOperationName("GetNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNamespaceRequest));
            CompletableFuture<GetNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about any operation that returns an operation ID in the response, such as a
     * <code>CreateHttpNamespace</code> request.
     * </p>
     * <note>
     * <p>
     * To get a list of operations that match specified criteria, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html">ListOperations</a>.
     * </p>
     * </note>
     *
     * @param getOperationRequest
     * @return A Java Future containing the result of the GetOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>OperationNotFoundException No operation exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOperationResponse> getOperation(GetOperationRequest getOperationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOperationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOperation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOperationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOperationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOperationRequest, GetOperationResponse>()
                            .withOperationName("GetOperation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOperationRequest));
            CompletableFuture<GetOperationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the settings for a specified service.
     * </p>
     *
     * @param getServiceRequest
     * @return A Java Future containing the result of the GetService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceResponse> getService(GetServiceRequest getServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>().withOperationName("GetService")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceRequest));
            CompletableFuture<GetServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information about the instances that you registered by using a specified service.
     * </p>
     *
     * @param listInstancesRequest
     * @return A Java Future containing the result of the ListInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInstancesResponse> listInstances(ListInstancesRequest listInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstancesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                            .withOperationName("ListInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInstancesRequest));
            CompletableFuture<ListInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information about the namespaces that were created by the current Amazon Web Services account.
     * </p>
     *
     * @param listNamespacesRequest
     * @return A Java Future containing the result of the ListNamespaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListNamespaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListNamespaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNamespacesResponse> listNamespaces(ListNamespacesRequest listNamespacesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNamespacesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNamespacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNamespaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNamespacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNamespacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNamespacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNamespacesRequest, ListNamespacesResponse>()
                            .withOperationName("ListNamespaces").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNamespacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNamespacesRequest));
            CompletableFuture<ListNamespacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists operations that match the criteria that you specify.
     * </p>
     *
     * @param listOperationsRequest
     * @return A Java Future containing the result of the ListOperations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOperationsResponse> listOperations(ListOperationsRequest listOperationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOperationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                            .withOperationName("ListOperations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOperationsRequest));
            CompletableFuture<ListOperationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information for all the services that are associated with one or more specified namespaces.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                            .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicesRequest));
            CompletableFuture<ListServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation can't be completed because the resource was not found.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates one or more records and, optionally, creates a health check based on the settings in a
     * specified service. When you submit a <code>RegisterInstance</code> request, the following occurs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For each DNS record that you define in the service that's specified by <code>ServiceId</code>, a record is
     * created or updated in the hosted zone that's associated with the corresponding namespace.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the service includes <code>HealthCheckConfig</code>, a health check is created based on the settings in the
     * health check configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * The health check, if any, is associated with each of the new or updated records.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * One <code>RegisterInstance</code> request must complete before you can submit another request and specify the
     * same service ID and instance ID.
     * </p>
     * </important>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html">CreateService</a>.
     * </p>
     * <p>
     * When Cloud Map receives a DNS query for the specified DNS name, it returns the applicable value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>If the health check is healthy</b>: returns all the records
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If the health check is unhealthy</b>: returns the applicable value for the last healthy instance
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you didn't specify a health check configuration</b>: returns all the records
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the current quota on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">Cloud Map
     * quotas</a> in the <i>Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param registerInstanceRequest
     * @return A Java Future containing the result of the RegisterInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the quota on the
     *         number of resources.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.RegisterInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/RegisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterInstanceResponse> registerInstance(RegisterInstanceRequest registerInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterInstanceRequest, RegisterInstanceResponse>()
                            .withOperationName("RegisterInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerInstanceRequest));
            CompletableFuture<RegisterInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation can't be completed because the resource was not found.</li>
     *         <li>TooManyTagsException The list of tags on the resource is over the quota. The maximum number of tags
     *         that can be applied to a resource is 50.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation can't be completed because the resource was not found.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an HTTP namespace.
     * </p>
     *
     * @param updateHttpNamespaceRequest
     * @return A Java Future containing the result of the UpdateHttpNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdateHttpNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateHttpNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateHttpNamespaceResponse> updateHttpNamespace(
            UpdateHttpNamespaceRequest updateHttpNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHttpNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHttpNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHttpNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateHttpNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateHttpNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateHttpNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateHttpNamespaceRequest, UpdateHttpNamespaceResponse>()
                            .withOperationName("UpdateHttpNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateHttpNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateHttpNamespaceRequest));
            CompletableFuture<UpdateHttpNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits a request to change the health status of a custom health check to healthy or unhealthy.
     * </p>
     * <p>
     * You can use <code>UpdateInstanceCustomHealthStatus</code> to change the status only for custom health checks,
     * which you define using <code>HealthCheckCustomConfig</code> when you create a service. You can't use it to change
     * the status for Route 53 health checks, which you define using <code>HealthCheckConfig</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_HealthCheckCustomConfig.html"
     * >HealthCheckCustomConfig</a>.
     * </p>
     *
     * @param updateInstanceCustomHealthStatusRequest
     * @return A Java Future containing the result of the UpdateInstanceCustomHealthStatus operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>CustomHealthNotFoundException The health check for the instance that's specified by
     *         <code>ServiceId</code> and <code>InstanceId</code> isn't a custom health check.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdateInstanceCustomHealthStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateInstanceCustomHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInstanceCustomHealthStatusResponse> updateInstanceCustomHealthStatus(
            UpdateInstanceCustomHealthStatusRequest updateInstanceCustomHealthStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInstanceCustomHealthStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateInstanceCustomHealthStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInstanceCustomHealthStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInstanceCustomHealthStatusResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateInstanceCustomHealthStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInstanceCustomHealthStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInstanceCustomHealthStatusRequest, UpdateInstanceCustomHealthStatusResponse>()
                            .withOperationName("UpdateInstanceCustomHealthStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateInstanceCustomHealthStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateInstanceCustomHealthStatusRequest));
            CompletableFuture<UpdateInstanceCustomHealthStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a private DNS namespace.
     * </p>
     *
     * @param updatePrivateDnsNamespaceRequest
     * @return A Java Future containing the result of the UpdatePrivateDnsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdatePrivateDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdatePrivateDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePrivateDnsNamespaceResponse> updatePrivateDnsNamespace(
            UpdatePrivateDnsNamespaceRequest updatePrivateDnsNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePrivateDnsNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePrivateDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePrivateDnsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePrivateDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePrivateDnsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePrivateDnsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePrivateDnsNamespaceRequest, UpdatePrivateDnsNamespaceResponse>()
                            .withOperationName("UpdatePrivateDnsNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePrivateDnsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePrivateDnsNamespaceRequest));
            CompletableFuture<UpdatePrivateDnsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a public DNS namespace.
     * </p>
     *
     * @param updatePublicDnsNamespaceRequest
     * @return A Java Future containing the result of the UpdatePublicDnsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdatePublicDnsNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdatePublicDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePublicDnsNamespaceResponse> updatePublicDnsNamespace(
            UpdatePublicDnsNamespaceRequest updatePublicDnsNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePublicDnsNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePublicDnsNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePublicDnsNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePublicDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePublicDnsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePublicDnsNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePublicDnsNamespaceRequest, UpdatePublicDnsNamespaceResponse>()
                            .withOperationName("UpdatePublicDnsNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePublicDnsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePublicDnsNamespaceRequest));
            CompletableFuture<UpdatePublicDnsNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits a request to perform the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Update the TTL setting for existing <code>DnsRecords</code> configurations
     * </p>
     * </li>
     * <li>
     * <p>
     * Add, update, or delete <code>HealthCheckConfig</code> for a specified service
     * </p>
     * <note>
     * <p>
     * You can't add, update, or delete a <code>HealthCheckCustomConfig</code> configuration.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * For public and private DNS namespaces, note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you omit any existing <code>DnsRecords</code> or <code>HealthCheckConfig</code> configurations from an
     * <code>UpdateService</code> request, the configurations are deleted from the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you omit an existing <code>HealthCheckCustomConfig</code> configuration from an <code>UpdateService</code>
     * request, the configuration isn't deleted from the service.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you update settings for a service, Cloud Map also updates the corresponding settings in all the records and
     * health checks that were created by using the specified service.
     * </p>
     *
     * @param updateServiceRequest
     * @return A Java Future containing the result of the UpdateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceResponse> updateService(UpdateServiceRequest updateServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ServiceDiscovery");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                            .withOperationName("UpdateService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceRequest));
            CompletableFuture<UpdateServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ServiceDiscoveryServiceClientConfiguration serviceClientConfiguration() {
        return new ServiceDiscoveryServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceDiscoveryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestLimitExceeded")
                                .exceptionBuilderSupplier(RequestLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUse")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceAlreadyExists")
                                .exceptionBuilderSupplier(ServiceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceNotFound")
                                .exceptionBuilderSupplier(ServiceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InstanceNotFound")
                                .exceptionBuilderSupplier(InstanceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotFound")
                                .exceptionBuilderSupplier(OperationNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceAlreadyExists")
                                .exceptionBuilderSupplier(NamespaceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceNotFound")
                                .exceptionBuilderSupplier(NamespaceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomHealthNotFound")
                                .exceptionBuilderSupplier(CustomHealthNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceeded")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ServiceDiscoveryServiceClientConfigurationBuilder serviceConfigBuilder = new ServiceDiscoveryServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
