/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides details about a control's enablement status in a specified standard.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StandardsControlAssociationDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<StandardsControlAssociationDetail.Builder, StandardsControlAssociationDetail> {
    private static final SdkField<String> STANDARDS_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StandardsArn").getter(getter(StandardsControlAssociationDetail::standardsArn))
            .setter(setter(Builder::standardsArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardsArn").build()).build();

    private static final SdkField<String> SECURITY_CONTROL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlId").getter(getter(StandardsControlAssociationDetail::securityControlId))
            .setter(setter(Builder::securityControlId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlId").build()).build();

    private static final SdkField<String> SECURITY_CONTROL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlArn").getter(getter(StandardsControlAssociationDetail::securityControlArn))
            .setter(setter(Builder::securityControlArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlArn").build())
            .build();

    private static final SdkField<String> ASSOCIATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationStatus").getter(getter(StandardsControlAssociationDetail::associationStatusAsString))
            .setter(setter(Builder::associationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationStatus").build()).build();

    private static final SdkField<List<String>> RELATED_REQUIREMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RelatedRequirements")
            .getter(getter(StandardsControlAssociationDetail::relatedRequirements))
            .setter(setter(Builder::relatedRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt")
            .getter(getter(StandardsControlAssociationDetail::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> UPDATED_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpdatedReason").getter(getter(StandardsControlAssociationDetail::updatedReason))
            .setter(setter(Builder::updatedReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedReason").build()).build();

    private static final SdkField<String> STANDARDS_CONTROL_TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StandardsControlTitle").getter(getter(StandardsControlAssociationDetail::standardsControlTitle))
            .setter(setter(Builder::standardsControlTitle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardsControlTitle").build())
            .build();

    private static final SdkField<String> STANDARDS_CONTROL_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StandardsControlDescription")
            .getter(getter(StandardsControlAssociationDetail::standardsControlDescription))
            .setter(setter(Builder::standardsControlDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardsControlDescription")
                    .build()).build();

    private static final SdkField<List<String>> STANDARDS_CONTROL_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StandardsControlArns")
            .getter(getter(StandardsControlAssociationDetail::standardsControlArns))
            .setter(setter(Builder::standardsControlArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardsControlArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STANDARDS_ARN_FIELD,
            SECURITY_CONTROL_ID_FIELD, SECURITY_CONTROL_ARN_FIELD, ASSOCIATION_STATUS_FIELD, RELATED_REQUIREMENTS_FIELD,
            UPDATED_AT_FIELD, UPDATED_REASON_FIELD, STANDARDS_CONTROL_TITLE_FIELD, STANDARDS_CONTROL_DESCRIPTION_FIELD,
            STANDARDS_CONTROL_ARNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String standardsArn;

    private final String securityControlId;

    private final String securityControlArn;

    private final String associationStatus;

    private final List<String> relatedRequirements;

    private final Instant updatedAt;

    private final String updatedReason;

    private final String standardsControlTitle;

    private final String standardsControlDescription;

    private final List<String> standardsControlArns;

    private StandardsControlAssociationDetail(BuilderImpl builder) {
        this.standardsArn = builder.standardsArn;
        this.securityControlId = builder.securityControlId;
        this.securityControlArn = builder.securityControlArn;
        this.associationStatus = builder.associationStatus;
        this.relatedRequirements = builder.relatedRequirements;
        this.updatedAt = builder.updatedAt;
        this.updatedReason = builder.updatedReason;
        this.standardsControlTitle = builder.standardsControlTitle;
        this.standardsControlDescription = builder.standardsControlDescription;
        this.standardsControlArns = builder.standardsControlArns;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of a security standard.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of a security standard.
     */
    public final String standardsArn() {
        return standardsArn;
    }

    /**
     * <p>
     * The unique identifier of a security control across standards. Values for this field typically consist of an
     * Amazon Web Service name and a number, such as APIGateway.3.
     * </p>
     * 
     * @return The unique identifier of a security control across standards. Values for this field typically consist of
     *         an Amazon Web Service name and a number, such as APIGateway.3.
     */
    public final String securityControlId() {
        return securityControlId;
    }

    /**
     * <p>
     * The ARN of a security control across standards, such as
     * <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter doesn't mention
     * a specific standard.
     * </p>
     * 
     * @return The ARN of a security control across standards, such as
     *         <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter doesn't
     *         mention a specific standard.
     */
    public final String securityControlArn() {
        return securityControlArn;
    }

    /**
     * <p>
     * Specifies whether a control is enabled or disabled in a specified standard.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #associationStatus}
     * will return {@link AssociationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #associationStatusAsString}.
     * </p>
     * 
     * @return Specifies whether a control is enabled or disabled in a specified standard.
     * @see AssociationStatus
     */
    public final AssociationStatus associationStatus() {
        return AssociationStatus.fromValue(associationStatus);
    }

    /**
     * <p>
     * Specifies whether a control is enabled or disabled in a specified standard.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #associationStatus}
     * will return {@link AssociationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #associationStatusAsString}.
     * </p>
     * 
     * @return Specifies whether a control is enabled or disabled in a specified standard.
     * @see AssociationStatus
     */
    public final String associationStatusAsString() {
        return associationStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedRequirements() {
        return relatedRequirements != null && !(relatedRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The requirement that underlies a control in the compliance framework related to the standard.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedRequirements} method.
     * </p>
     * 
     * @return The requirement that underlies a control in the compliance framework related to the standard.
     */
    public final List<String> relatedRequirements() {
        return relatedRequirements;
    }

    /**
     * <p>
     * The time at which the enablement status of the control in the specified standard was last updated.
     * </p>
     * 
     * @return The time at which the enablement status of the control in the specified standard was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The reason for updating the enablement status of a control in a specified standard.
     * </p>
     * 
     * @return The reason for updating the enablement status of a control in a specified standard.
     */
    public final String updatedReason() {
        return updatedReason;
    }

    /**
     * <p>
     * The title of a control. This field may reference a specific standard.
     * </p>
     * 
     * @return The title of a control. This field may reference a specific standard.
     */
    public final String standardsControlTitle() {
        return standardsControlTitle;
    }

    /**
     * <p>
     * The description of a control. This typically summarizes how Security Hub evaluates the control and the conditions
     * under which it produces a failed finding. This parameter may reference a specific standard.
     * </p>
     * 
     * @return The description of a control. This typically summarizes how Security Hub evaluates the control and the
     *         conditions under which it produces a failed finding. This parameter may reference a specific standard.
     */
    public final String standardsControlDescription() {
        return standardsControlDescription;
    }

    /**
     * For responses, this returns true if the service returned a value for the StandardsControlArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStandardsControlArns() {
        return standardsControlArns != null && !(standardsControlArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides the input parameter that Security Hub uses to call the <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html"
     * >UpdateStandardsControl</a> API. This API can be used to enable or disable a control in a specified standard.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStandardsControlArns} method.
     * </p>
     * 
     * @return Provides the input parameter that Security Hub uses to call the <a
     *         href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html"
     *         >UpdateStandardsControl</a> API. This API can be used to enable or disable a control in a specified
     *         standard.
     */
    public final List<String> standardsControlArns() {
        return standardsControlArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(standardsArn());
        hashCode = 31 * hashCode + Objects.hashCode(securityControlId());
        hashCode = 31 * hashCode + Objects.hashCode(securityControlArn());
        hashCode = 31 * hashCode + Objects.hashCode(associationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedRequirements() ? relatedRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedReason());
        hashCode = 31 * hashCode + Objects.hashCode(standardsControlTitle());
        hashCode = 31 * hashCode + Objects.hashCode(standardsControlDescription());
        hashCode = 31 * hashCode + Objects.hashCode(hasStandardsControlArns() ? standardsControlArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StandardsControlAssociationDetail)) {
            return false;
        }
        StandardsControlAssociationDetail other = (StandardsControlAssociationDetail) obj;
        return Objects.equals(standardsArn(), other.standardsArn())
                && Objects.equals(securityControlId(), other.securityControlId())
                && Objects.equals(securityControlArn(), other.securityControlArn())
                && Objects.equals(associationStatusAsString(), other.associationStatusAsString())
                && hasRelatedRequirements() == other.hasRelatedRequirements()
                && Objects.equals(relatedRequirements(), other.relatedRequirements())
                && Objects.equals(updatedAt(), other.updatedAt()) && Objects.equals(updatedReason(), other.updatedReason())
                && Objects.equals(standardsControlTitle(), other.standardsControlTitle())
                && Objects.equals(standardsControlDescription(), other.standardsControlDescription())
                && hasStandardsControlArns() == other.hasStandardsControlArns()
                && Objects.equals(standardsControlArns(), other.standardsControlArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StandardsControlAssociationDetail").add("StandardsArn", standardsArn())
                .add("SecurityControlId", securityControlId()).add("SecurityControlArn", securityControlArn())
                .add("AssociationStatus", associationStatusAsString())
                .add("RelatedRequirements", hasRelatedRequirements() ? relatedRequirements() : null)
                .add("UpdatedAt", updatedAt()).add("UpdatedReason", updatedReason())
                .add("StandardsControlTitle", standardsControlTitle())
                .add("StandardsControlDescription", standardsControlDescription())
                .add("StandardsControlArns", hasStandardsControlArns() ? standardsControlArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StandardsArn":
            return Optional.ofNullable(clazz.cast(standardsArn()));
        case "SecurityControlId":
            return Optional.ofNullable(clazz.cast(securityControlId()));
        case "SecurityControlArn":
            return Optional.ofNullable(clazz.cast(securityControlArn()));
        case "AssociationStatus":
            return Optional.ofNullable(clazz.cast(associationStatusAsString()));
        case "RelatedRequirements":
            return Optional.ofNullable(clazz.cast(relatedRequirements()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "UpdatedReason":
            return Optional.ofNullable(clazz.cast(updatedReason()));
        case "StandardsControlTitle":
            return Optional.ofNullable(clazz.cast(standardsControlTitle()));
        case "StandardsControlDescription":
            return Optional.ofNullable(clazz.cast(standardsControlDescription()));
        case "StandardsControlArns":
            return Optional.ofNullable(clazz.cast(standardsControlArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StandardsControlAssociationDetail, T> g) {
        return obj -> g.apply((StandardsControlAssociationDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StandardsControlAssociationDetail> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of a security standard.
         * </p>
         * 
         * @param standardsArn
         *        The Amazon Resource Name (ARN) of a security standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardsArn(String standardsArn);

        /**
         * <p>
         * The unique identifier of a security control across standards. Values for this field typically consist of an
         * Amazon Web Service name and a number, such as APIGateway.3.
         * </p>
         * 
         * @param securityControlId
         *        The unique identifier of a security control across standards. Values for this field typically consist
         *        of an Amazon Web Service name and a number, such as APIGateway.3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlId(String securityControlId);

        /**
         * <p>
         * The ARN of a security control across standards, such as
         * <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter doesn't
         * mention a specific standard.
         * </p>
         * 
         * @param securityControlArn
         *        The ARN of a security control across standards, such as
         *        <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter
         *        doesn't mention a specific standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlArn(String securityControlArn);

        /**
         * <p>
         * Specifies whether a control is enabled or disabled in a specified standard.
         * </p>
         * 
         * @param associationStatus
         *        Specifies whether a control is enabled or disabled in a specified standard.
         * @see AssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationStatus
         */
        Builder associationStatus(String associationStatus);

        /**
         * <p>
         * Specifies whether a control is enabled or disabled in a specified standard.
         * </p>
         * 
         * @param associationStatus
         *        Specifies whether a control is enabled or disabled in a specified standard.
         * @see AssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssociationStatus
         */
        Builder associationStatus(AssociationStatus associationStatus);

        /**
         * <p>
         * The requirement that underlies a control in the compliance framework related to the standard.
         * </p>
         * 
         * @param relatedRequirements
         *        The requirement that underlies a control in the compliance framework related to the standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedRequirements(Collection<String> relatedRequirements);

        /**
         * <p>
         * The requirement that underlies a control in the compliance framework related to the standard.
         * </p>
         * 
         * @param relatedRequirements
         *        The requirement that underlies a control in the compliance framework related to the standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedRequirements(String... relatedRequirements);

        /**
         * <p>
         * The time at which the enablement status of the control in the specified standard was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The time at which the enablement status of the control in the specified standard was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The reason for updating the enablement status of a control in a specified standard.
         * </p>
         * 
         * @param updatedReason
         *        The reason for updating the enablement status of a control in a specified standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedReason(String updatedReason);

        /**
         * <p>
         * The title of a control. This field may reference a specific standard.
         * </p>
         * 
         * @param standardsControlTitle
         *        The title of a control. This field may reference a specific standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardsControlTitle(String standardsControlTitle);

        /**
         * <p>
         * The description of a control. This typically summarizes how Security Hub evaluates the control and the
         * conditions under which it produces a failed finding. This parameter may reference a specific standard.
         * </p>
         * 
         * @param standardsControlDescription
         *        The description of a control. This typically summarizes how Security Hub evaluates the control and the
         *        conditions under which it produces a failed finding. This parameter may reference a specific standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardsControlDescription(String standardsControlDescription);

        /**
         * <p>
         * Provides the input parameter that Security Hub uses to call the <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html"
         * >UpdateStandardsControl</a> API. This API can be used to enable or disable a control in a specified standard.
         * </p>
         * 
         * @param standardsControlArns
         *        Provides the input parameter that Security Hub uses to call the <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html"
         *        >UpdateStandardsControl</a> API. This API can be used to enable or disable a control in a specified
         *        standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardsControlArns(Collection<String> standardsControlArns);

        /**
         * <p>
         * Provides the input parameter that Security Hub uses to call the <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html"
         * >UpdateStandardsControl</a> API. This API can be used to enable or disable a control in a specified standard.
         * </p>
         * 
         * @param standardsControlArns
         *        Provides the input parameter that Security Hub uses to call the <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html"
         *        >UpdateStandardsControl</a> API. This API can be used to enable or disable a control in a specified
         *        standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardsControlArns(String... standardsControlArns);
    }

    static final class BuilderImpl implements Builder {
        private String standardsArn;

        private String securityControlId;

        private String securityControlArn;

        private String associationStatus;

        private List<String> relatedRequirements = DefaultSdkAutoConstructList.getInstance();

        private Instant updatedAt;

        private String updatedReason;

        private String standardsControlTitle;

        private String standardsControlDescription;

        private List<String> standardsControlArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StandardsControlAssociationDetail model) {
            standardsArn(model.standardsArn);
            securityControlId(model.securityControlId);
            securityControlArn(model.securityControlArn);
            associationStatus(model.associationStatus);
            relatedRequirements(model.relatedRequirements);
            updatedAt(model.updatedAt);
            updatedReason(model.updatedReason);
            standardsControlTitle(model.standardsControlTitle);
            standardsControlDescription(model.standardsControlDescription);
            standardsControlArns(model.standardsControlArns);
        }

        public final String getStandardsArn() {
            return standardsArn;
        }

        public final void setStandardsArn(String standardsArn) {
            this.standardsArn = standardsArn;
        }

        @Override
        public final Builder standardsArn(String standardsArn) {
            this.standardsArn = standardsArn;
            return this;
        }

        public final String getSecurityControlId() {
            return securityControlId;
        }

        public final void setSecurityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
        }

        @Override
        public final Builder securityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
            return this;
        }

        public final String getSecurityControlArn() {
            return securityControlArn;
        }

        public final void setSecurityControlArn(String securityControlArn) {
            this.securityControlArn = securityControlArn;
        }

        @Override
        public final Builder securityControlArn(String securityControlArn) {
            this.securityControlArn = securityControlArn;
            return this;
        }

        public final String getAssociationStatus() {
            return associationStatus;
        }

        public final void setAssociationStatus(String associationStatus) {
            this.associationStatus = associationStatus;
        }

        @Override
        public final Builder associationStatus(String associationStatus) {
            this.associationStatus = associationStatus;
            return this;
        }

        @Override
        public final Builder associationStatus(AssociationStatus associationStatus) {
            this.associationStatus(associationStatus == null ? null : associationStatus.toString());
            return this;
        }

        public final Collection<String> getRelatedRequirements() {
            if (relatedRequirements instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedRequirements;
        }

        public final void setRelatedRequirements(Collection<String> relatedRequirements) {
            this.relatedRequirements = RelatedRequirementsListCopier.copy(relatedRequirements);
        }

        @Override
        public final Builder relatedRequirements(Collection<String> relatedRequirements) {
            this.relatedRequirements = RelatedRequirementsListCopier.copy(relatedRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedRequirements(String... relatedRequirements) {
            relatedRequirements(Arrays.asList(relatedRequirements));
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final String getUpdatedReason() {
            return updatedReason;
        }

        public final void setUpdatedReason(String updatedReason) {
            this.updatedReason = updatedReason;
        }

        @Override
        public final Builder updatedReason(String updatedReason) {
            this.updatedReason = updatedReason;
            return this;
        }

        public final String getStandardsControlTitle() {
            return standardsControlTitle;
        }

        public final void setStandardsControlTitle(String standardsControlTitle) {
            this.standardsControlTitle = standardsControlTitle;
        }

        @Override
        public final Builder standardsControlTitle(String standardsControlTitle) {
            this.standardsControlTitle = standardsControlTitle;
            return this;
        }

        public final String getStandardsControlDescription() {
            return standardsControlDescription;
        }

        public final void setStandardsControlDescription(String standardsControlDescription) {
            this.standardsControlDescription = standardsControlDescription;
        }

        @Override
        public final Builder standardsControlDescription(String standardsControlDescription) {
            this.standardsControlDescription = standardsControlDescription;
            return this;
        }

        public final Collection<String> getStandardsControlArns() {
            if (standardsControlArns instanceof SdkAutoConstructList) {
                return null;
            }
            return standardsControlArns;
        }

        public final void setStandardsControlArns(Collection<String> standardsControlArns) {
            this.standardsControlArns = StandardsControlArnListCopier.copy(standardsControlArns);
        }

        @Override
        public final Builder standardsControlArns(Collection<String> standardsControlArns) {
            this.standardsControlArns = StandardsControlArnListCopier.copy(standardsControlArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder standardsControlArns(String... standardsControlArns) {
            standardsControlArns(Arrays.asList(standardsControlArns));
            return this;
        }

        @Override
        public StandardsControlAssociationDetail build() {
            return new StandardsControlAssociationDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
