/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A vulnerability associated with a finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Vulnerability implements SdkPojo, Serializable, ToCopyableBuilder<Vulnerability.Builder, Vulnerability> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Vulnerability::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<List<SoftwarePackage>> VULNERABLE_PACKAGES_FIELD = SdkField
            .<List<SoftwarePackage>> builder(MarshallingType.LIST)
            .memberName("VulnerablePackages")
            .getter(getter(Vulnerability::vulnerablePackages))
            .setter(setter(Builder::vulnerablePackages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VulnerablePackages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SoftwarePackage> builder(MarshallingType.SDK_POJO)
                                            .constructor(SoftwarePackage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Cvss>> CVSS_FIELD = SdkField
            .<List<Cvss>> builder(MarshallingType.LIST)
            .memberName("Cvss")
            .getter(getter(Vulnerability::cvss))
            .setter(setter(Builder::cvss))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cvss").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Cvss> builder(MarshallingType.SDK_POJO)
                                            .constructor(Cvss::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RELATED_VULNERABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RelatedVulnerabilities")
            .getter(getter(Vulnerability::relatedVulnerabilities))
            .setter(setter(Builder::relatedVulnerabilities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedVulnerabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VulnerabilityVendor> VENDOR_FIELD = SdkField
            .<VulnerabilityVendor> builder(MarshallingType.SDK_POJO).memberName("Vendor").getter(getter(Vulnerability::vendor))
            .setter(setter(Builder::vendor)).constructor(VulnerabilityVendor::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Vendor").build()).build();

    private static final SdkField<List<String>> REFERENCE_URLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReferenceUrls")
            .getter(getter(Vulnerability::referenceUrls))
            .setter(setter(Builder::referenceUrls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferenceUrls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FIX_AVAILABLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FixAvailable").getter(getter(Vulnerability::fixAvailableAsString)).setter(setter(Builder::fixAvailable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FixAvailable").build()).build();

    private static final SdkField<Double> EPSS_SCORE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("EpssScore").getter(getter(Vulnerability::epssScore)).setter(setter(Builder::epssScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EpssScore").build()).build();

    private static final SdkField<String> EXPLOIT_AVAILABLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExploitAvailable").getter(getter(Vulnerability::exploitAvailableAsString))
            .setter(setter(Builder::exploitAvailable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExploitAvailable").build()).build();

    private static final SdkField<List<VulnerabilityCodeVulnerabilities>> CODE_VULNERABILITIES_FIELD = SdkField
            .<List<VulnerabilityCodeVulnerabilities>> builder(MarshallingType.LIST)
            .memberName("CodeVulnerabilities")
            .getter(getter(Vulnerability::codeVulnerabilities))
            .setter(setter(Builder::codeVulnerabilities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CodeVulnerabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VulnerabilityCodeVulnerabilities> builder(MarshallingType.SDK_POJO)
                                            .constructor(VulnerabilityCodeVulnerabilities::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            VULNERABLE_PACKAGES_FIELD, CVSS_FIELD, RELATED_VULNERABILITIES_FIELD, VENDOR_FIELD, REFERENCE_URLS_FIELD,
            FIX_AVAILABLE_FIELD, EPSS_SCORE_FIELD, EXPLOIT_AVAILABLE_FIELD, CODE_VULNERABILITIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final List<SoftwarePackage> vulnerablePackages;

    private final List<Cvss> cvss;

    private final List<String> relatedVulnerabilities;

    private final VulnerabilityVendor vendor;

    private final List<String> referenceUrls;

    private final String fixAvailable;

    private final Double epssScore;

    private final String exploitAvailable;

    private final List<VulnerabilityCodeVulnerabilities> codeVulnerabilities;

    private Vulnerability(BuilderImpl builder) {
        this.id = builder.id;
        this.vulnerablePackages = builder.vulnerablePackages;
        this.cvss = builder.cvss;
        this.relatedVulnerabilities = builder.relatedVulnerabilities;
        this.vendor = builder.vendor;
        this.referenceUrls = builder.referenceUrls;
        this.fixAvailable = builder.fixAvailable;
        this.epssScore = builder.epssScore;
        this.exploitAvailable = builder.exploitAvailable;
        this.codeVulnerabilities = builder.codeVulnerabilities;
    }

    /**
     * <p>
     * The identifier of the vulnerability.
     * </p>
     * 
     * @return The identifier of the vulnerability.
     */
    public final String id() {
        return id;
    }

    /**
     * For responses, this returns true if the service returned a value for the VulnerablePackages property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVulnerablePackages() {
        return vulnerablePackages != null && !(vulnerablePackages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of software packages that have the vulnerability.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVulnerablePackages} method.
     * </p>
     * 
     * @return List of software packages that have the vulnerability.
     */
    public final List<SoftwarePackage> vulnerablePackages() {
        return vulnerablePackages;
    }

    /**
     * For responses, this returns true if the service returned a value for the Cvss property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasCvss() {
        return cvss != null && !(cvss instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * CVSS scores from the advisory related to the vulnerability.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCvss} method.
     * </p>
     * 
     * @return CVSS scores from the advisory related to the vulnerability.
     */
    public final List<Cvss> cvss() {
        return cvss;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedVulnerabilities property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRelatedVulnerabilities() {
        return relatedVulnerabilities != null && !(relatedVulnerabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of vulnerabilities that are related to this vulnerability.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedVulnerabilities} method.
     * </p>
     * 
     * @return List of vulnerabilities that are related to this vulnerability.
     */
    public final List<String> relatedVulnerabilities() {
        return relatedVulnerabilities;
    }

    /**
     * <p>
     * Information about the vendor that generates the vulnerability report.
     * </p>
     * 
     * @return Information about the vendor that generates the vulnerability report.
     */
    public final VulnerabilityVendor vendor() {
        return vendor;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReferenceUrls property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReferenceUrls() {
        return referenceUrls != null && !(referenceUrls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of URLs that provide additional information about the vulnerability.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReferenceUrls} method.
     * </p>
     * 
     * @return A list of URLs that provide additional information about the vulnerability.
     */
    public final List<String> referenceUrls() {
        return referenceUrls;
    }

    /**
     * <p>
     * Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
     * <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the number of
     * vulnerable packages that have a value for both <code>FixedInVersion</code> and <code>Remediation</code>. Valid
     * values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
     * <code>Remediation</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
     * <code>Remediation</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PARTIAL</code> otherwise
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fixAvailable} will
     * return {@link VulnerabilityFixAvailable#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #fixAvailableAsString}.
     * </p>
     * 
     * @return Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
     *         <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the
     *         number of vulnerable packages that have a value for both <code>FixedInVersion</code> and
     *         <code>Remediation</code>. Valid values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
     *         <code>Remediation</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
     *         <code>Remediation</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PARTIAL</code> otherwise
     *         </p>
     *         </li>
     * @see VulnerabilityFixAvailable
     */
    public final VulnerabilityFixAvailable fixAvailable() {
        return VulnerabilityFixAvailable.fromValue(fixAvailable);
    }

    /**
     * <p>
     * Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
     * <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the number of
     * vulnerable packages that have a value for both <code>FixedInVersion</code> and <code>Remediation</code>. Valid
     * values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
     * <code>Remediation</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
     * <code>Remediation</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PARTIAL</code> otherwise
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fixAvailable} will
     * return {@link VulnerabilityFixAvailable#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #fixAvailableAsString}.
     * </p>
     * 
     * @return Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
     *         <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the
     *         number of vulnerable packages that have a value for both <code>FixedInVersion</code> and
     *         <code>Remediation</code>. Valid values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
     *         <code>Remediation</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
     *         <code>Remediation</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PARTIAL</code> otherwise
     *         </p>
     *         </li>
     * @see VulnerabilityFixAvailable
     */
    public final String fixAvailableAsString() {
        return fixAvailable;
    }

    /**
     * <p>
     * The Exploit Prediction Scoring System (EPSS) score for a finding.
     * </p>
     * 
     * @return The Exploit Prediction Scoring System (EPSS) score for a finding.
     */
    public final Double epssScore() {
        return epssScore;
    }

    /**
     * <p>
     * Whether an exploit is available for a finding.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #exploitAvailable}
     * will return {@link VulnerabilityExploitAvailable#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #exploitAvailableAsString}.
     * </p>
     * 
     * @return Whether an exploit is available for a finding.
     * @see VulnerabilityExploitAvailable
     */
    public final VulnerabilityExploitAvailable exploitAvailable() {
        return VulnerabilityExploitAvailable.fromValue(exploitAvailable);
    }

    /**
     * <p>
     * Whether an exploit is available for a finding.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #exploitAvailable}
     * will return {@link VulnerabilityExploitAvailable#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #exploitAvailableAsString}.
     * </p>
     * 
     * @return Whether an exploit is available for a finding.
     * @see VulnerabilityExploitAvailable
     */
    public final String exploitAvailableAsString() {
        return exploitAvailable;
    }

    /**
     * For responses, this returns true if the service returned a value for the CodeVulnerabilities property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCodeVulnerabilities() {
        return codeVulnerabilities != null && !(codeVulnerabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The vulnerabilities found in your Lambda function code. This field pertains to findings that Security Hub
     * receives from Amazon Inspector.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCodeVulnerabilities} method.
     * </p>
     * 
     * @return The vulnerabilities found in your Lambda function code. This field pertains to findings that Security Hub
     *         receives from Amazon Inspector.
     */
    public final List<VulnerabilityCodeVulnerabilities> codeVulnerabilities() {
        return codeVulnerabilities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(hasVulnerablePackages() ? vulnerablePackages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCvss() ? cvss() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedVulnerabilities() ? relatedVulnerabilities() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vendor());
        hashCode = 31 * hashCode + Objects.hashCode(hasReferenceUrls() ? referenceUrls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(fixAvailableAsString());
        hashCode = 31 * hashCode + Objects.hashCode(epssScore());
        hashCode = 31 * hashCode + Objects.hashCode(exploitAvailableAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCodeVulnerabilities() ? codeVulnerabilities() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Vulnerability)) {
            return false;
        }
        Vulnerability other = (Vulnerability) obj;
        return Objects.equals(id(), other.id()) && hasVulnerablePackages() == other.hasVulnerablePackages()
                && Objects.equals(vulnerablePackages(), other.vulnerablePackages()) && hasCvss() == other.hasCvss()
                && Objects.equals(cvss(), other.cvss()) && hasRelatedVulnerabilities() == other.hasRelatedVulnerabilities()
                && Objects.equals(relatedVulnerabilities(), other.relatedVulnerabilities())
                && Objects.equals(vendor(), other.vendor()) && hasReferenceUrls() == other.hasReferenceUrls()
                && Objects.equals(referenceUrls(), other.referenceUrls())
                && Objects.equals(fixAvailableAsString(), other.fixAvailableAsString())
                && Objects.equals(epssScore(), other.epssScore())
                && Objects.equals(exploitAvailableAsString(), other.exploitAvailableAsString())
                && hasCodeVulnerabilities() == other.hasCodeVulnerabilities()
                && Objects.equals(codeVulnerabilities(), other.codeVulnerabilities());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Vulnerability").add("Id", id())
                .add("VulnerablePackages", hasVulnerablePackages() ? vulnerablePackages() : null)
                .add("Cvss", hasCvss() ? cvss() : null)
                .add("RelatedVulnerabilities", hasRelatedVulnerabilities() ? relatedVulnerabilities() : null)
                .add("Vendor", vendor()).add("ReferenceUrls", hasReferenceUrls() ? referenceUrls() : null)
                .add("FixAvailable", fixAvailableAsString()).add("EpssScore", epssScore())
                .add("ExploitAvailable", exploitAvailableAsString())
                .add("CodeVulnerabilities", hasCodeVulnerabilities() ? codeVulnerabilities() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "VulnerablePackages":
            return Optional.ofNullable(clazz.cast(vulnerablePackages()));
        case "Cvss":
            return Optional.ofNullable(clazz.cast(cvss()));
        case "RelatedVulnerabilities":
            return Optional.ofNullable(clazz.cast(relatedVulnerabilities()));
        case "Vendor":
            return Optional.ofNullable(clazz.cast(vendor()));
        case "ReferenceUrls":
            return Optional.ofNullable(clazz.cast(referenceUrls()));
        case "FixAvailable":
            return Optional.ofNullable(clazz.cast(fixAvailableAsString()));
        case "EpssScore":
            return Optional.ofNullable(clazz.cast(epssScore()));
        case "ExploitAvailable":
            return Optional.ofNullable(clazz.cast(exploitAvailableAsString()));
        case "CodeVulnerabilities":
            return Optional.ofNullable(clazz.cast(codeVulnerabilities()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Vulnerability, T> g) {
        return obj -> g.apply((Vulnerability) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Vulnerability> {
        /**
         * <p>
         * The identifier of the vulnerability.
         * </p>
         * 
         * @param id
         *        The identifier of the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * List of software packages that have the vulnerability.
         * </p>
         * 
         * @param vulnerablePackages
         *        List of software packages that have the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vulnerablePackages(Collection<SoftwarePackage> vulnerablePackages);

        /**
         * <p>
         * List of software packages that have the vulnerability.
         * </p>
         * 
         * @param vulnerablePackages
         *        List of software packages that have the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vulnerablePackages(SoftwarePackage... vulnerablePackages);

        /**
         * <p>
         * List of software packages that have the vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.SoftwarePackage.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.securityhub.model.SoftwarePackage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.SoftwarePackage.Builder#build()} is called
         * immediately and its result is passed to {@link #vulnerablePackages(List<SoftwarePackage>)}.
         * 
         * @param vulnerablePackages
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.SoftwarePackage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vulnerablePackages(java.util.Collection<SoftwarePackage>)
         */
        Builder vulnerablePackages(Consumer<SoftwarePackage.Builder>... vulnerablePackages);

        /**
         * <p>
         * CVSS scores from the advisory related to the vulnerability.
         * </p>
         * 
         * @param cvss
         *        CVSS scores from the advisory related to the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cvss(Collection<Cvss> cvss);

        /**
         * <p>
         * CVSS scores from the advisory related to the vulnerability.
         * </p>
         * 
         * @param cvss
         *        CVSS scores from the advisory related to the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cvss(Cvss... cvss);

        /**
         * <p>
         * CVSS scores from the advisory related to the vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.Cvss.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.Cvss#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.Cvss.Builder#build()} is called immediately and its
         * result is passed to {@link #cvss(List<Cvss>)}.
         * 
         * @param cvss
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.Cvss.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cvss(java.util.Collection<Cvss>)
         */
        Builder cvss(Consumer<Cvss.Builder>... cvss);

        /**
         * <p>
         * List of vulnerabilities that are related to this vulnerability.
         * </p>
         * 
         * @param relatedVulnerabilities
         *        List of vulnerabilities that are related to this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedVulnerabilities(Collection<String> relatedVulnerabilities);

        /**
         * <p>
         * List of vulnerabilities that are related to this vulnerability.
         * </p>
         * 
         * @param relatedVulnerabilities
         *        List of vulnerabilities that are related to this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedVulnerabilities(String... relatedVulnerabilities);

        /**
         * <p>
         * Information about the vendor that generates the vulnerability report.
         * </p>
         * 
         * @param vendor
         *        Information about the vendor that generates the vulnerability report.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vendor(VulnerabilityVendor vendor);

        /**
         * <p>
         * Information about the vendor that generates the vulnerability report.
         * </p>
         * This is a convenience method that creates an instance of the {@link VulnerabilityVendor.Builder} avoiding the
         * need to create one manually via {@link VulnerabilityVendor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VulnerabilityVendor.Builder#build()} is called immediately and
         * its result is passed to {@link #vendor(VulnerabilityVendor)}.
         * 
         * @param vendor
         *        a consumer that will call methods on {@link VulnerabilityVendor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vendor(VulnerabilityVendor)
         */
        default Builder vendor(Consumer<VulnerabilityVendor.Builder> vendor) {
            return vendor(VulnerabilityVendor.builder().applyMutation(vendor).build());
        }

        /**
         * <p>
         * A list of URLs that provide additional information about the vulnerability.
         * </p>
         * 
         * @param referenceUrls
         *        A list of URLs that provide additional information about the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceUrls(Collection<String> referenceUrls);

        /**
         * <p>
         * A list of URLs that provide additional information about the vulnerability.
         * </p>
         * 
         * @param referenceUrls
         *        A list of URLs that provide additional information about the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceUrls(String... referenceUrls);

        /**
         * <p>
         * Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
         * <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the number
         * of vulnerable packages that have a value for both <code>FixedInVersion</code> and <code>Remediation</code>.
         * Valid values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
         * <code>Remediation</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
         * <code>Remediation</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PARTIAL</code> otherwise
         * </p>
         * </li>
         * </ul>
         * 
         * @param fixAvailable
         *        Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
         *        <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the
         *        number of vulnerable packages that have a value for both <code>FixedInVersion</code> and
         *        <code>Remediation</code>. Valid values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
         *        <code>Remediation</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
         *        <code>Remediation</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PARTIAL</code> otherwise
         *        </p>
         *        </li>
         * @see VulnerabilityFixAvailable
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VulnerabilityFixAvailable
         */
        Builder fixAvailable(String fixAvailable);

        /**
         * <p>
         * Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
         * <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the number
         * of vulnerable packages that have a value for both <code>FixedInVersion</code> and <code>Remediation</code>.
         * Valid values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
         * <code>Remediation</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
         * <code>Remediation</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PARTIAL</code> otherwise
         * </p>
         * </li>
         * </ul>
         * 
         * @param fixAvailable
         *        Specifies if all vulnerable packages in a finding have a value for <code>FixedInVersion</code> and
         *        <code>Remediation</code>. This field is evaluated for each vulnerability <code>Id</code> based on the
         *        number of vulnerable packages that have a value for both <code>FixedInVersion</code> and
         *        <code>Remediation</code>. Valid values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>YES</code> if all vulnerable packages have a value for both <code>FixedInVersion</code> and
         *        <code>Remediation</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO</code> if no vulnerable packages have a value for <code>FixedInVersion</code> and
         *        <code>Remediation</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PARTIAL</code> otherwise
         *        </p>
         *        </li>
         * @see VulnerabilityFixAvailable
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VulnerabilityFixAvailable
         */
        Builder fixAvailable(VulnerabilityFixAvailable fixAvailable);

        /**
         * <p>
         * The Exploit Prediction Scoring System (EPSS) score for a finding.
         * </p>
         * 
         * @param epssScore
         *        The Exploit Prediction Scoring System (EPSS) score for a finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder epssScore(Double epssScore);

        /**
         * <p>
         * Whether an exploit is available for a finding.
         * </p>
         * 
         * @param exploitAvailable
         *        Whether an exploit is available for a finding.
         * @see VulnerabilityExploitAvailable
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VulnerabilityExploitAvailable
         */
        Builder exploitAvailable(String exploitAvailable);

        /**
         * <p>
         * Whether an exploit is available for a finding.
         * </p>
         * 
         * @param exploitAvailable
         *        Whether an exploit is available for a finding.
         * @see VulnerabilityExploitAvailable
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VulnerabilityExploitAvailable
         */
        Builder exploitAvailable(VulnerabilityExploitAvailable exploitAvailable);

        /**
         * <p>
         * The vulnerabilities found in your Lambda function code. This field pertains to findings that Security Hub
         * receives from Amazon Inspector.
         * </p>
         * 
         * @param codeVulnerabilities
         *        The vulnerabilities found in your Lambda function code. This field pertains to findings that Security
         *        Hub receives from Amazon Inspector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codeVulnerabilities(Collection<VulnerabilityCodeVulnerabilities> codeVulnerabilities);

        /**
         * <p>
         * The vulnerabilities found in your Lambda function code. This field pertains to findings that Security Hub
         * receives from Amazon Inspector.
         * </p>
         * 
         * @param codeVulnerabilities
         *        The vulnerabilities found in your Lambda function code. This field pertains to findings that Security
         *        Hub receives from Amazon Inspector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codeVulnerabilities(VulnerabilityCodeVulnerabilities... codeVulnerabilities);

        /**
         * <p>
         * The vulnerabilities found in your Lambda function code. This field pertains to findings that Security Hub
         * receives from Amazon Inspector.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.VulnerabilityCodeVulnerabilities.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.securityhub.model.VulnerabilityCodeVulnerabilities#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.VulnerabilityCodeVulnerabilities.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #codeVulnerabilities(List<VulnerabilityCodeVulnerabilities>)}.
         * 
         * @param codeVulnerabilities
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.VulnerabilityCodeVulnerabilities.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #codeVulnerabilities(java.util.Collection<VulnerabilityCodeVulnerabilities>)
         */
        Builder codeVulnerabilities(Consumer<VulnerabilityCodeVulnerabilities.Builder>... codeVulnerabilities);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private List<SoftwarePackage> vulnerablePackages = DefaultSdkAutoConstructList.getInstance();

        private List<Cvss> cvss = DefaultSdkAutoConstructList.getInstance();

        private List<String> relatedVulnerabilities = DefaultSdkAutoConstructList.getInstance();

        private VulnerabilityVendor vendor;

        private List<String> referenceUrls = DefaultSdkAutoConstructList.getInstance();

        private String fixAvailable;

        private Double epssScore;

        private String exploitAvailable;

        private List<VulnerabilityCodeVulnerabilities> codeVulnerabilities = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Vulnerability model) {
            id(model.id);
            vulnerablePackages(model.vulnerablePackages);
            cvss(model.cvss);
            relatedVulnerabilities(model.relatedVulnerabilities);
            vendor(model.vendor);
            referenceUrls(model.referenceUrls);
            fixAvailable(model.fixAvailable);
            epssScore(model.epssScore);
            exploitAvailable(model.exploitAvailable);
            codeVulnerabilities(model.codeVulnerabilities);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final List<SoftwarePackage.Builder> getVulnerablePackages() {
            List<SoftwarePackage.Builder> result = SoftwarePackageListCopier.copyToBuilder(this.vulnerablePackages);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVulnerablePackages(Collection<SoftwarePackage.BuilderImpl> vulnerablePackages) {
            this.vulnerablePackages = SoftwarePackageListCopier.copyFromBuilder(vulnerablePackages);
        }

        @Override
        public final Builder vulnerablePackages(Collection<SoftwarePackage> vulnerablePackages) {
            this.vulnerablePackages = SoftwarePackageListCopier.copy(vulnerablePackages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vulnerablePackages(SoftwarePackage... vulnerablePackages) {
            vulnerablePackages(Arrays.asList(vulnerablePackages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vulnerablePackages(Consumer<SoftwarePackage.Builder>... vulnerablePackages) {
            vulnerablePackages(Stream.of(vulnerablePackages).map(c -> SoftwarePackage.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Cvss.Builder> getCvss() {
            List<Cvss.Builder> result = CvssListCopier.copyToBuilder(this.cvss);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCvss(Collection<Cvss.BuilderImpl> cvss) {
            this.cvss = CvssListCopier.copyFromBuilder(cvss);
        }

        @Override
        public final Builder cvss(Collection<Cvss> cvss) {
            this.cvss = CvssListCopier.copy(cvss);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cvss(Cvss... cvss) {
            cvss(Arrays.asList(cvss));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cvss(Consumer<Cvss.Builder>... cvss) {
            cvss(Stream.of(cvss).map(c -> Cvss.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getRelatedVulnerabilities() {
            if (relatedVulnerabilities instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedVulnerabilities;
        }

        public final void setRelatedVulnerabilities(Collection<String> relatedVulnerabilities) {
            this.relatedVulnerabilities = StringListCopier.copy(relatedVulnerabilities);
        }

        @Override
        public final Builder relatedVulnerabilities(Collection<String> relatedVulnerabilities) {
            this.relatedVulnerabilities = StringListCopier.copy(relatedVulnerabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedVulnerabilities(String... relatedVulnerabilities) {
            relatedVulnerabilities(Arrays.asList(relatedVulnerabilities));
            return this;
        }

        public final VulnerabilityVendor.Builder getVendor() {
            return vendor != null ? vendor.toBuilder() : null;
        }

        public final void setVendor(VulnerabilityVendor.BuilderImpl vendor) {
            this.vendor = vendor != null ? vendor.build() : null;
        }

        @Override
        public final Builder vendor(VulnerabilityVendor vendor) {
            this.vendor = vendor;
            return this;
        }

        public final Collection<String> getReferenceUrls() {
            if (referenceUrls instanceof SdkAutoConstructList) {
                return null;
            }
            return referenceUrls;
        }

        public final void setReferenceUrls(Collection<String> referenceUrls) {
            this.referenceUrls = StringListCopier.copy(referenceUrls);
        }

        @Override
        public final Builder referenceUrls(Collection<String> referenceUrls) {
            this.referenceUrls = StringListCopier.copy(referenceUrls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceUrls(String... referenceUrls) {
            referenceUrls(Arrays.asList(referenceUrls));
            return this;
        }

        public final String getFixAvailable() {
            return fixAvailable;
        }

        public final void setFixAvailable(String fixAvailable) {
            this.fixAvailable = fixAvailable;
        }

        @Override
        public final Builder fixAvailable(String fixAvailable) {
            this.fixAvailable = fixAvailable;
            return this;
        }

        @Override
        public final Builder fixAvailable(VulnerabilityFixAvailable fixAvailable) {
            this.fixAvailable(fixAvailable == null ? null : fixAvailable.toString());
            return this;
        }

        public final Double getEpssScore() {
            return epssScore;
        }

        public final void setEpssScore(Double epssScore) {
            this.epssScore = epssScore;
        }

        @Override
        public final Builder epssScore(Double epssScore) {
            this.epssScore = epssScore;
            return this;
        }

        public final String getExploitAvailable() {
            return exploitAvailable;
        }

        public final void setExploitAvailable(String exploitAvailable) {
            this.exploitAvailable = exploitAvailable;
        }

        @Override
        public final Builder exploitAvailable(String exploitAvailable) {
            this.exploitAvailable = exploitAvailable;
            return this;
        }

        @Override
        public final Builder exploitAvailable(VulnerabilityExploitAvailable exploitAvailable) {
            this.exploitAvailable(exploitAvailable == null ? null : exploitAvailable.toString());
            return this;
        }

        public final List<VulnerabilityCodeVulnerabilities.Builder> getCodeVulnerabilities() {
            List<VulnerabilityCodeVulnerabilities.Builder> result = VulnerabilityCodeVulnerabilitiesListCopier
                    .copyToBuilder(this.codeVulnerabilities);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCodeVulnerabilities(Collection<VulnerabilityCodeVulnerabilities.BuilderImpl> codeVulnerabilities) {
            this.codeVulnerabilities = VulnerabilityCodeVulnerabilitiesListCopier.copyFromBuilder(codeVulnerabilities);
        }

        @Override
        public final Builder codeVulnerabilities(Collection<VulnerabilityCodeVulnerabilities> codeVulnerabilities) {
            this.codeVulnerabilities = VulnerabilityCodeVulnerabilitiesListCopier.copy(codeVulnerabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder codeVulnerabilities(VulnerabilityCodeVulnerabilities... codeVulnerabilities) {
            codeVulnerabilities(Arrays.asList(codeVulnerabilities));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder codeVulnerabilities(Consumer<VulnerabilityCodeVulnerabilities.Builder>... codeVulnerabilities) {
            codeVulnerabilities(Stream.of(codeVulnerabilities)
                    .map(c -> VulnerabilityCodeVulnerabilities.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Vulnerability build() {
            return new Vulnerability(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
