/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains finding details that are specific to control-based findings. Only returned for findings generated from
 * controls.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Compliance implements SdkPojo, Serializable, ToCopyableBuilder<Compliance.Builder, Compliance> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Compliance::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> RELATED_REQUIREMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RelatedRequirements")
            .getter(getter(Compliance::relatedRequirements))
            .setter(setter(Builder::relatedRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<StatusReason>> STATUS_REASONS_FIELD = SdkField
            .<List<StatusReason>> builder(MarshallingType.LIST)
            .memberName("StatusReasons")
            .getter(getter(Compliance::statusReasons))
            .setter(setter(Builder::statusReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StatusReason> builder(MarshallingType.SDK_POJO)
                                            .constructor(StatusReason::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SECURITY_CONTROL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlId").getter(getter(Compliance::securityControlId))
            .setter(setter(Builder::securityControlId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlId").build()).build();

    private static final SdkField<List<AssociatedStandard>> ASSOCIATED_STANDARDS_FIELD = SdkField
            .<List<AssociatedStandard>> builder(MarshallingType.LIST)
            .memberName("AssociatedStandards")
            .getter(getter(Compliance::associatedStandards))
            .setter(setter(Builder::associatedStandards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedStandards").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AssociatedStandard> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssociatedStandard::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            RELATED_REQUIREMENTS_FIELD, STATUS_REASONS_FIELD, SECURITY_CONTROL_ID_FIELD, ASSOCIATED_STANDARDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String status;

    private final List<String> relatedRequirements;

    private final List<StatusReason> statusReasons;

    private final String securityControlId;

    private final List<AssociatedStandard> associatedStandards;

    private Compliance(BuilderImpl builder) {
        this.status = builder.status;
        this.relatedRequirements = builder.relatedRequirements;
        this.statusReasons = builder.statusReasons;
        this.securityControlId = builder.securityControlId;
        this.associatedStandards = builder.associatedStandards;
    }

    /**
     * <p>
     * The result of a standards check.
     * </p>
     * <p>
     * The valid values for <code>Status</code> are as follows.
     * </p>
     * <ul>
     * <li>
     * <ul>
     * <li>
     * <p>
     * <code>PASSED</code> - Standards check passed for all evaluated resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WARNING</code> - Some information is missing or this check is not supported for your configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Standards check failed for at least one evaluated resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or because the
     * result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation result was
     * <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the finding.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The result of a standards check.</p>
     *         <p>
     *         The valid values for <code>Status</code> are as follows.
     *         </p>
     *         <ul>
     *         <li>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PASSED</code> - Standards check passed for all evaluated resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WARNING</code> - Some information is missing or this check is not supported for your configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Standards check failed for at least one evaluated resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or because
     *         the result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation result was
     *         <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the finding.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see ComplianceStatus
     */
    public final ComplianceStatus status() {
        return ComplianceStatus.fromValue(status);
    }

    /**
     * <p>
     * The result of a standards check.
     * </p>
     * <p>
     * The valid values for <code>Status</code> are as follows.
     * </p>
     * <ul>
     * <li>
     * <ul>
     * <li>
     * <p>
     * <code>PASSED</code> - Standards check passed for all evaluated resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WARNING</code> - Some information is missing or this check is not supported for your configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Standards check failed for at least one evaluated resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or because the
     * result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation result was
     * <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the finding.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComplianceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The result of a standards check.</p>
     *         <p>
     *         The valid values for <code>Status</code> are as follows.
     *         </p>
     *         <ul>
     *         <li>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PASSED</code> - Standards check passed for all evaluated resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WARNING</code> - Some information is missing or this check is not supported for your configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Standards check failed for at least one evaluated resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or because
     *         the result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation result was
     *         <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the finding.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see ComplianceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedRequirements() {
        return relatedRequirements != null && !(relatedRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For a control, the industry or regulatory framework requirements that are related to the control. The check for
     * that control is aligned with these requirements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedRequirements} method.
     * </p>
     * 
     * @return For a control, the industry or regulatory framework requirements that are related to the control. The
     *         check for that control is aligned with these requirements.
     */
    public final List<String> relatedRequirements() {
        return relatedRequirements;
    }

    /**
     * For responses, this returns true if the service returned a value for the StatusReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStatusReasons() {
        return statusReasons != null && !(statusReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For the list of
     * status reason codes and their meanings, see <a href=
     * "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
     * >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatusReasons} method.
     * </p>
     * 
     * @return For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For the
     *         list of status reason codes and their meanings, see <a href=
     *         "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
     *         >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
     */
    public final List<StatusReason> statusReasons() {
        return statusReasons;
    }

    /**
     * <p>
     * The unique identifier of a control across standards. Values for this field typically consist of an Amazon Web
     * Service and a number, such as APIGateway.5.
     * </p>
     * 
     * @return The unique identifier of a control across standards. Values for this field typically consist of an Amazon
     *         Web Service and a number, such as APIGateway.5.
     */
    public final String securityControlId() {
        return securityControlId;
    }

    /**
     * For responses, this returns true if the service returned a value for the AssociatedStandards property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAssociatedStandards() {
        return associatedStandards != null && !(associatedStandards instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The enabled security standards in which a security control is currently enabled.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssociatedStandards} method.
     * </p>
     * 
     * @return The enabled security standards in which a security control is currently enabled.
     */
    public final List<AssociatedStandard> associatedStandards() {
        return associatedStandards;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedRequirements() ? relatedRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStatusReasons() ? statusReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(securityControlId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssociatedStandards() ? associatedStandards() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Compliance)) {
            return false;
        }
        Compliance other = (Compliance) obj;
        return Objects.equals(statusAsString(), other.statusAsString())
                && hasRelatedRequirements() == other.hasRelatedRequirements()
                && Objects.equals(relatedRequirements(), other.relatedRequirements())
                && hasStatusReasons() == other.hasStatusReasons() && Objects.equals(statusReasons(), other.statusReasons())
                && Objects.equals(securityControlId(), other.securityControlId())
                && hasAssociatedStandards() == other.hasAssociatedStandards()
                && Objects.equals(associatedStandards(), other.associatedStandards());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Compliance").add("Status", statusAsString())
                .add("RelatedRequirements", hasRelatedRequirements() ? relatedRequirements() : null)
                .add("StatusReasons", hasStatusReasons() ? statusReasons() : null).add("SecurityControlId", securityControlId())
                .add("AssociatedStandards", hasAssociatedStandards() ? associatedStandards() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "RelatedRequirements":
            return Optional.ofNullable(clazz.cast(relatedRequirements()));
        case "StatusReasons":
            return Optional.ofNullable(clazz.cast(statusReasons()));
        case "SecurityControlId":
            return Optional.ofNullable(clazz.cast(securityControlId()));
        case "AssociatedStandards":
            return Optional.ofNullable(clazz.cast(associatedStandards()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Compliance, T> g) {
        return obj -> g.apply((Compliance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Compliance> {
        /**
         * <p>
         * The result of a standards check.
         * </p>
         * <p>
         * The valid values for <code>Status</code> are as follows.
         * </p>
         * <ul>
         * <li>
         * <ul>
         * <li>
         * <p>
         * <code>PASSED</code> - Standards check passed for all evaluated resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WARNING</code> - Some information is missing or this check is not supported for your configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Standards check failed for at least one evaluated resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or because the
         * result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation result was
         * <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the finding.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param status
         *        The result of a standards check.</p>
         *        <p>
         *        The valid values for <code>Status</code> are as follows.
         *        </p>
         *        <ul>
         *        <li>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PASSED</code> - Standards check passed for all evaluated resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WARNING</code> - Some information is missing or this check is not supported for your
         *        configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Standards check failed for at least one evaluated resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or
         *        because the result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation
         *        result was <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the
         *        finding.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see ComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The result of a standards check.
         * </p>
         * <p>
         * The valid values for <code>Status</code> are as follows.
         * </p>
         * <ul>
         * <li>
         * <ul>
         * <li>
         * <p>
         * <code>PASSED</code> - Standards check passed for all evaluated resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WARNING</code> - Some information is missing or this check is not supported for your configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Standards check failed for at least one evaluated resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or because the
         * result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation result was
         * <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the finding.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param status
         *        The result of a standards check.</p>
         *        <p>
         *        The valid values for <code>Status</code> are as follows.
         *        </p>
         *        <ul>
         *        <li>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PASSED</code> - Standards check passed for all evaluated resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WARNING</code> - Some information is missing or this check is not supported for your
         *        configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Standards check failed for at least one evaluated resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_AVAILABLE</code> - Check could not be performed due to a service outage, API error, or
         *        because the result of the Config evaluation was <code>NOT_APPLICABLE</code>. If the Config evaluation
         *        result was <code>NOT_APPLICABLE</code>, then after 3 days, Security Hub automatically archives the
         *        finding.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see ComplianceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceStatus
         */
        Builder status(ComplianceStatus status);

        /**
         * <p>
         * For a control, the industry or regulatory framework requirements that are related to the control. The check
         * for that control is aligned with these requirements.
         * </p>
         * 
         * @param relatedRequirements
         *        For a control, the industry or regulatory framework requirements that are related to the control. The
         *        check for that control is aligned with these requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedRequirements(Collection<String> relatedRequirements);

        /**
         * <p>
         * For a control, the industry or regulatory framework requirements that are related to the control. The check
         * for that control is aligned with these requirements.
         * </p>
         * 
         * @param relatedRequirements
         *        For a control, the industry or regulatory framework requirements that are related to the control. The
         *        check for that control is aligned with these requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedRequirements(String... relatedRequirements);

        /**
         * <p>
         * For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For the list
         * of status reason codes and their meanings, see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
         * >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param statusReasons
         *        For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For
         *        the list of status reason codes and their meanings, see <a href=
         *        "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
         *        >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasons(Collection<StatusReason> statusReasons);

        /**
         * <p>
         * For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For the list
         * of status reason codes and their meanings, see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
         * >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param statusReasons
         *        For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For
         *        the list of status reason codes and their meanings, see <a href=
         *        "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
         *        >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasons(StatusReason... statusReasons);

        /**
         * <p>
         * For findings generated from controls, a list of reasons behind the value of <code>Status</code>. For the list
         * of status reason codes and their meanings, see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-results.html#securityhub-standards-results-asff"
         * >Standards-related information in the ASFF</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.StatusReason.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.securityhub.model.StatusReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.StatusReason.Builder#build()} is called immediately
         * and its result is passed to {@link #statusReasons(List<StatusReason>)}.
         * 
         * @param statusReasons
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.StatusReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusReasons(java.util.Collection<StatusReason>)
         */
        Builder statusReasons(Consumer<StatusReason.Builder>... statusReasons);

        /**
         * <p>
         * The unique identifier of a control across standards. Values for this field typically consist of an Amazon Web
         * Service and a number, such as APIGateway.5.
         * </p>
         * 
         * @param securityControlId
         *        The unique identifier of a control across standards. Values for this field typically consist of an
         *        Amazon Web Service and a number, such as APIGateway.5.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlId(String securityControlId);

        /**
         * <p>
         * The enabled security standards in which a security control is currently enabled.
         * </p>
         * 
         * @param associatedStandards
         *        The enabled security standards in which a security control is currently enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedStandards(Collection<AssociatedStandard> associatedStandards);

        /**
         * <p>
         * The enabled security standards in which a security control is currently enabled.
         * </p>
         * 
         * @param associatedStandards
         *        The enabled security standards in which a security control is currently enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedStandards(AssociatedStandard... associatedStandards);

        /**
         * <p>
         * The enabled security standards in which a security control is currently enabled.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.AssociatedStandard.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.securityhub.model.AssociatedStandard#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.AssociatedStandard.Builder#build()} is called
         * immediately and its result is passed to {@link #associatedStandards(List<AssociatedStandard>)}.
         * 
         * @param associatedStandards
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.AssociatedStandard.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associatedStandards(java.util.Collection<AssociatedStandard>)
         */
        Builder associatedStandards(Consumer<AssociatedStandard.Builder>... associatedStandards);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private List<String> relatedRequirements = DefaultSdkAutoConstructList.getInstance();

        private List<StatusReason> statusReasons = DefaultSdkAutoConstructList.getInstance();

        private String securityControlId;

        private List<AssociatedStandard> associatedStandards = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Compliance model) {
            status(model.status);
            relatedRequirements(model.relatedRequirements);
            statusReasons(model.statusReasons);
            securityControlId(model.securityControlId);
            associatedStandards(model.associatedStandards);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ComplianceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Collection<String> getRelatedRequirements() {
            if (relatedRequirements instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedRequirements;
        }

        public final void setRelatedRequirements(Collection<String> relatedRequirements) {
            this.relatedRequirements = RelatedRequirementsListCopier.copy(relatedRequirements);
        }

        @Override
        public final Builder relatedRequirements(Collection<String> relatedRequirements) {
            this.relatedRequirements = RelatedRequirementsListCopier.copy(relatedRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedRequirements(String... relatedRequirements) {
            relatedRequirements(Arrays.asList(relatedRequirements));
            return this;
        }

        public final List<StatusReason.Builder> getStatusReasons() {
            List<StatusReason.Builder> result = StatusReasonsListCopier.copyToBuilder(this.statusReasons);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStatusReasons(Collection<StatusReason.BuilderImpl> statusReasons) {
            this.statusReasons = StatusReasonsListCopier.copyFromBuilder(statusReasons);
        }

        @Override
        public final Builder statusReasons(Collection<StatusReason> statusReasons) {
            this.statusReasons = StatusReasonsListCopier.copy(statusReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusReasons(StatusReason... statusReasons) {
            statusReasons(Arrays.asList(statusReasons));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusReasons(Consumer<StatusReason.Builder>... statusReasons) {
            statusReasons(Stream.of(statusReasons).map(c -> StatusReason.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getSecurityControlId() {
            return securityControlId;
        }

        public final void setSecurityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
        }

        @Override
        public final Builder securityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
            return this;
        }

        public final List<AssociatedStandard.Builder> getAssociatedStandards() {
            List<AssociatedStandard.Builder> result = AssociatedStandardsListCopier.copyToBuilder(this.associatedStandards);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAssociatedStandards(Collection<AssociatedStandard.BuilderImpl> associatedStandards) {
            this.associatedStandards = AssociatedStandardsListCopier.copyFromBuilder(associatedStandards);
        }

        @Override
        public final Builder associatedStandards(Collection<AssociatedStandard> associatedStandards) {
            this.associatedStandards = AssociatedStandardsListCopier.copy(associatedStandards);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedStandards(AssociatedStandard... associatedStandards) {
            associatedStandards(Arrays.asList(associatedStandards));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedStandards(Consumer<AssociatedStandard.Builder>... associatedStandards) {
            associatedStandards(Stream.of(associatedStandards).map(c -> AssociatedStandard.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Compliance build() {
            return new Compliance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
