/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A string filter for filtering Security Hub findings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StringFilter implements SdkPojo, Serializable, ToCopyableBuilder<StringFilter.Builder, StringFilter> {
    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Value")
            .getter(getter(StringFilter::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<String> COMPARISON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Comparison").getter(getter(StringFilter::comparisonAsString)).setter(setter(Builder::comparison))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comparison").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(VALUE_FIELD, COMPARISON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String value;

    private final String comparison;

    private StringFilter(BuilderImpl builder) {
        this.value = builder.value;
        this.comparison = builder.comparison;
    }

    /**
     * <p>
     * The string filter value. Filter values are case sensitive. For example, the product name for control-based
     * findings is <code>Security Hub</code>. If you provide <code>security hub</code> as the filter value, there's no
     * match.
     * </p>
     * 
     * @return The string filter value. Filter values are case sensitive. For example, the product name for
     *         control-based findings is <code>Security Hub</code>. If you provide <code>security hub</code> as the
     *         filter value, there's no match.
     */
    public final String value() {
        return value;
    }

    /**
     * <p>
     * The condition to apply to a string value when filtering Security Hub findings.
     * </p>
     * <p>
     * To search for values that have the filter value, use one of the following comparison operators:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
     * <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes the string
     * CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the filter
     * <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
     * <code>123456789012</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values that start with the filter value, use <code>PREFIX</code>. For example, the filter
     * <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code> that starts with
     * <code>us</code>. A <code>ResourceRegion</code> that starts with a different value, such as <code>af</code>,
     * <code>ap</code>, or <code>ca</code>, doesn't match.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are joined by
     * <code>OR</code>. A finding matches if it matches any one of those filters. For example, the filters
     * <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that includes either
     * <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
     * </p>
     * <p>
     * To search for values that don’t have the filter value, use one of the following comparison operators:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the filter
     * <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that excludes the
     * string CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
     * <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other than
     * <code>123456789012</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For example, the
     * filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a <code>ResourceRegion</code> that
     * starts with a value other than <code>us</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the same field
     * are joined by <code>AND</code>. A finding matches only if it matches all of those filters. For example, the
     * filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code> match a finding that
     * excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
     * </p>
     * <p>
     * You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same field.
     * Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code> or
     * <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an error.
     * <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
     * <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
     * </p>
     * <p>
     * You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
     * filters for the same field. Security Hub first processes the <code>PREFIX</code> filters, and then the
     * <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code> filters.
     * </p>
     * <p>
     * For example, for the following filters, Security Hub first identifies findings that have resource types that
     * start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings that have a resource type
     * of <code>AwsIamPolicy</code> and findings that have a resource type of <code>AwsEc2NetworkInterface</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ResourceType PREFIX AwsIam</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType PREFIX AwsEc2</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation rules</a> in the
     * <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #comparison} will
     * return {@link StringFilterComparison#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #comparisonAsString}.
     * </p>
     * 
     * @return The condition to apply to a string value when filtering Security Hub findings.</p>
     *         <p>
     *         To search for values that have the filter value, use one of the following comparison operators:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
     *         <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes the
     *         string CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the
     *         filter <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
     *         <code>123456789012</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values that start with the filter value, use <code>PREFIX</code>. For example, the filter
     *         <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code> that
     *         starts with <code>us</code>. A <code>ResourceRegion</code> that starts with a different value, such as
     *         <code>af</code>, <code>ap</code>, or <code>ca</code>, doesn't match.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are joined
     *         by <code>OR</code>. A finding matches if it matches any one of those filters. For example, the filters
     *         <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that includes either
     *         <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
     *         </p>
     *         <p>
     *         To search for values that don’t have the filter value, use one of the following comparison operators:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the
     *         filter <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that
     *         excludes the string CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
     *         <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other
     *         than <code>123456789012</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For
     *         example, the filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a
     *         <code>ResourceRegion</code> that starts with a value other than <code>us</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the
     *         same field are joined by <code>AND</code>. A finding matches only if it matches all of those filters. For
     *         example, the filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code> match a
     *         finding that excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
     *         </p>
     *         <p>
     *         You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same
     *         field. Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code> or
     *         <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an error.
     *         <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
     *         <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
     *         </p>
     *         <p>
     *         You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or
     *         <code>PREFIX_NOT_EQUALS</code> filters for the same field. Security Hub first processes the
     *         <code>PREFIX</code> filters, and then the <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
     *         filters.
     *         </p>
     *         <p>
     *         For example, for the following filters, Security Hub first identifies findings that have resource types
     *         that start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings that have a
     *         resource type of <code>AwsIamPolicy</code> and findings that have a resource type of
     *         <code>AwsEc2NetworkInterface</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ResourceType PREFIX AwsIam</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType PREFIX AwsEc2</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation
     *         rules</a> in the <i>Security Hub User Guide</i>.
     * @see StringFilterComparison
     */
    public final StringFilterComparison comparison() {
        return StringFilterComparison.fromValue(comparison);
    }

    /**
     * <p>
     * The condition to apply to a string value when filtering Security Hub findings.
     * </p>
     * <p>
     * To search for values that have the filter value, use one of the following comparison operators:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
     * <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes the string
     * CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the filter
     * <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
     * <code>123456789012</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values that start with the filter value, use <code>PREFIX</code>. For example, the filter
     * <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code> that starts with
     * <code>us</code>. A <code>ResourceRegion</code> that starts with a different value, such as <code>af</code>,
     * <code>ap</code>, or <code>ca</code>, doesn't match.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are joined by
     * <code>OR</code>. A finding matches if it matches any one of those filters. For example, the filters
     * <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that includes either
     * <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
     * </p>
     * <p>
     * To search for values that don’t have the filter value, use one of the following comparison operators:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the filter
     * <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that excludes the
     * string CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
     * <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other than
     * <code>123456789012</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For example, the
     * filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a <code>ResourceRegion</code> that
     * starts with a value other than <code>us</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the same field
     * are joined by <code>AND</code>. A finding matches only if it matches all of those filters. For example, the
     * filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code> match a finding that
     * excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
     * </p>
     * <p>
     * You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same field.
     * Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code> or
     * <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an error.
     * <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
     * <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
     * </p>
     * <p>
     * You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
     * filters for the same field. Security Hub first processes the <code>PREFIX</code> filters, and then the
     * <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code> filters.
     * </p>
     * <p>
     * For example, for the following filters, Security Hub first identifies findings that have resource types that
     * start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings that have a resource type
     * of <code>AwsIamPolicy</code> and findings that have a resource type of <code>AwsEc2NetworkInterface</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ResourceType PREFIX AwsIam</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType PREFIX AwsEc2</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation rules</a> in the
     * <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #comparison} will
     * return {@link StringFilterComparison#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #comparisonAsString}.
     * </p>
     * 
     * @return The condition to apply to a string value when filtering Security Hub findings.</p>
     *         <p>
     *         To search for values that have the filter value, use one of the following comparison operators:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
     *         <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes the
     *         string CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the
     *         filter <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
     *         <code>123456789012</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values that start with the filter value, use <code>PREFIX</code>. For example, the filter
     *         <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code> that
     *         starts with <code>us</code>. A <code>ResourceRegion</code> that starts with a different value, such as
     *         <code>af</code>, <code>ap</code>, or <code>ca</code>, doesn't match.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are joined
     *         by <code>OR</code>. A finding matches if it matches any one of those filters. For example, the filters
     *         <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that includes either
     *         <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
     *         </p>
     *         <p>
     *         To search for values that don’t have the filter value, use one of the following comparison operators:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the
     *         filter <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that
     *         excludes the string CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
     *         <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other
     *         than <code>123456789012</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For
     *         example, the filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a
     *         <code>ResourceRegion</code> that starts with a value other than <code>us</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the
     *         same field are joined by <code>AND</code>. A finding matches only if it matches all of those filters. For
     *         example, the filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code> match a
     *         finding that excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
     *         </p>
     *         <p>
     *         You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same
     *         field. Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code> or
     *         <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an error.
     *         <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
     *         <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
     *         </p>
     *         <p>
     *         You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or
     *         <code>PREFIX_NOT_EQUALS</code> filters for the same field. Security Hub first processes the
     *         <code>PREFIX</code> filters, and then the <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
     *         filters.
     *         </p>
     *         <p>
     *         For example, for the following filters, Security Hub first identifies findings that have resource types
     *         that start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings that have a
     *         resource type of <code>AwsIamPolicy</code> and findings that have a resource type of
     *         <code>AwsEc2NetworkInterface</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ResourceType PREFIX AwsIam</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType PREFIX AwsEc2</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation
     *         rules</a> in the <i>Security Hub User Guide</i>.
     * @see StringFilterComparison
     */
    public final String comparisonAsString() {
        return comparison;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StringFilter)) {
            return false;
        }
        StringFilter other = (StringFilter) obj;
        return Objects.equals(value(), other.value()) && Objects.equals(comparisonAsString(), other.comparisonAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StringFilter").add("Value", value()).add("Comparison", comparisonAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Comparison":
            return Optional.ofNullable(clazz.cast(comparisonAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StringFilter, T> g) {
        return obj -> g.apply((StringFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StringFilter> {
        /**
         * <p>
         * The string filter value. Filter values are case sensitive. For example, the product name for control-based
         * findings is <code>Security Hub</code>. If you provide <code>security hub</code> as the filter value, there's
         * no match.
         * </p>
         * 
         * @param value
         *        The string filter value. Filter values are case sensitive. For example, the product name for
         *        control-based findings is <code>Security Hub</code>. If you provide <code>security hub</code> as the
         *        filter value, there's no match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The condition to apply to a string value when filtering Security Hub findings.
         * </p>
         * <p>
         * To search for values that have the filter value, use one of the following comparison operators:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
         * <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes the
         * string CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the filter
         * <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
         * <code>123456789012</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values that start with the filter value, use <code>PREFIX</code>. For example, the filter
         * <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code> that starts
         * with <code>us</code>. A <code>ResourceRegion</code> that starts with a different value, such as
         * <code>af</code>, <code>ap</code>, or <code>ca</code>, doesn't match.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are joined by
         * <code>OR</code>. A finding matches if it matches any one of those filters. For example, the filters
         * <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that includes either
         * <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
         * </p>
         * <p>
         * To search for values that don’t have the filter value, use one of the following comparison operators:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the filter
         * <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that excludes the
         * string CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
         * <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other than
         * <code>123456789012</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For example,
         * the filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a
         * <code>ResourceRegion</code> that starts with a value other than <code>us</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the same
         * field are joined by <code>AND</code>. A finding matches only if it matches all of those filters. For example,
         * the filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code> match a finding that
         * excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
         * </p>
         * <p>
         * You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same field.
         * Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code> or
         * <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an error.
         * <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
         * <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
         * </p>
         * <p>
         * You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
         * filters for the same field. Security Hub first processes the <code>PREFIX</code> filters, and then the
         * <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code> filters.
         * </p>
         * <p>
         * For example, for the following filters, Security Hub first identifies findings that have resource types that
         * start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings that have a resource
         * type of <code>AwsIamPolicy</code> and findings that have a resource type of
         * <code>AwsEc2NetworkInterface</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ResourceType PREFIX AwsIam</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType PREFIX AwsEc2</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation rules</a> in
         * the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param comparison
         *        The condition to apply to a string value when filtering Security Hub findings.</p>
         *        <p>
         *        To search for values that have the filter value, use one of the following comparison operators:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
         *        <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes
         *        the string CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the
         *        filter <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
         *        <code>123456789012</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values that start with the filter value, use <code>PREFIX</code>. For example, the
         *        filter <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code>
         *        that starts with <code>us</code>. A <code>ResourceRegion</code> that starts with a different value,
         *        such as <code>af</code>, <code>ap</code>, or <code>ca</code>, doesn't match.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are
         *        joined by <code>OR</code>. A finding matches if it matches any one of those filters. For example, the
         *        filters <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that
         *        includes either <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
         *        </p>
         *        <p>
         *        To search for values that don’t have the filter value, use one of the following comparison operators:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the
         *        filter <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that
         *        excludes the string CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
         *        <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other
         *        than <code>123456789012</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For
         *        example, the filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a
         *        <code>ResourceRegion</code> that starts with a value other than <code>us</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the
         *        same field are joined by <code>AND</code>. A finding matches only if it matches all of those filters.
         *        For example, the filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code>
         *        match a finding that excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
         *        </p>
         *        <p>
         *        You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same
         *        field. Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code>
         *        or <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an
         *        error. <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
         *        <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
         *        </p>
         *        <p>
         *        You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or
         *        <code>PREFIX_NOT_EQUALS</code> filters for the same field. Security Hub first processes the
         *        <code>PREFIX</code> filters, and then the <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
         *        filters.
         *        </p>
         *        <p>
         *        For example, for the following filters, Security Hub first identifies findings that have resource
         *        types that start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings
         *        that have a resource type of <code>AwsIamPolicy</code> and findings that have a resource type of
         *        <code>AwsEc2NetworkInterface</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ResourceType PREFIX AwsIam</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType PREFIX AwsEc2</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation
         *        rules</a> in the <i>Security Hub User Guide</i>.
         * @see StringFilterComparison
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StringFilterComparison
         */
        Builder comparison(String comparison);

        /**
         * <p>
         * The condition to apply to a string value when filtering Security Hub findings.
         * </p>
         * <p>
         * To search for values that have the filter value, use one of the following comparison operators:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
         * <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes the
         * string CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the filter
         * <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
         * <code>123456789012</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values that start with the filter value, use <code>PREFIX</code>. For example, the filter
         * <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code> that starts
         * with <code>us</code>. A <code>ResourceRegion</code> that starts with a different value, such as
         * <code>af</code>, <code>ap</code>, or <code>ca</code>, doesn't match.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are joined by
         * <code>OR</code>. A finding matches if it matches any one of those filters. For example, the filters
         * <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that includes either
         * <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
         * </p>
         * <p>
         * To search for values that don’t have the filter value, use one of the following comparison operators:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the filter
         * <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that excludes the
         * string CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
         * <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other than
         * <code>123456789012</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For example,
         * the filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a
         * <code>ResourceRegion</code> that starts with a value other than <code>us</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the same
         * field are joined by <code>AND</code>. A finding matches only if it matches all of those filters. For example,
         * the filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code> match a finding that
         * excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
         * </p>
         * <p>
         * You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same field.
         * Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code> or
         * <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an error.
         * <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
         * <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
         * </p>
         * <p>
         * You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
         * filters for the same field. Security Hub first processes the <code>PREFIX</code> filters, and then the
         * <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code> filters.
         * </p>
         * <p>
         * For example, for the following filters, Security Hub first identifies findings that have resource types that
         * start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings that have a resource
         * type of <code>AwsIamPolicy</code> and findings that have a resource type of
         * <code>AwsEc2NetworkInterface</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ResourceType PREFIX AwsIam</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType PREFIX AwsEc2</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation rules</a> in
         * the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param comparison
         *        The condition to apply to a string value when filtering Security Hub findings.</p>
         *        <p>
         *        To search for values that have the filter value, use one of the following comparison operators:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To search for values that include the filter value, use <code>CONTAINS</code>. For example, the filter
         *        <code>Title CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that includes
         *        the string CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values that exactly match the filter value, use <code>EQUALS</code>. For example, the
         *        filter <code>AwsAccountId EQUALS 123456789012</code> only matches findings that have an account ID of
         *        <code>123456789012</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values that start with the filter value, use <code>PREFIX</code>. For example, the
         *        filter <code>ResourceRegion PREFIX us</code> matches findings that have a <code>ResourceRegion</code>
         *        that starts with <code>us</code>. A <code>ResourceRegion</code> that starts with a different value,
         *        such as <code>af</code>, <code>ap</code>, or <code>ca</code>, doesn't match.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>CONTAINS</code>, <code>EQUALS</code>, and <code>PREFIX</code> filters on the same field are
         *        joined by <code>OR</code>. A finding matches if it matches any one of those filters. For example, the
         *        filters <code>Title CONTAINS CloudFront OR Title CONTAINS CloudWatch</code> match a finding that
         *        includes either <code>CloudFront</code>, <code>CloudWatch</code>, or both strings in the title.
         *        </p>
         *        <p>
         *        To search for values that don’t have the filter value, use one of the following comparison operators:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To search for values that exclude the filter value, use <code>NOT_CONTAINS</code>. For example, the
         *        filter <code>Title NOT_CONTAINS CloudFront</code> matches findings that have a <code>Title</code> that
         *        excludes the string CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values other than the filter value, use <code>NOT_EQUALS</code>. For example, the filter
         *        <code>AwsAccountId NOT_EQUALS 123456789012</code> only matches findings that have an account ID other
         *        than <code>123456789012</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To search for values that don't start with the filter value, use <code>PREFIX_NOT_EQUALS</code>. For
         *        example, the filter <code>ResourceRegion PREFIX_NOT_EQUALS us</code> matches findings with a
         *        <code>ResourceRegion</code> that starts with a value other than <code>us</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>NOT_CONTAINS</code>, <code>NOT_EQUALS</code>, and <code>PREFIX_NOT_EQUALS</code> filters on the
         *        same field are joined by <code>AND</code>. A finding matches only if it matches all of those filters.
         *        For example, the filters <code>Title NOT_CONTAINS CloudFront AND Title NOT_CONTAINS CloudWatch</code>
         *        match a finding that excludes both <code>CloudFront</code> and <code>CloudWatch</code> in the title.
         *        </p>
         *        <p>
         *        You can’t have both a <code>CONTAINS</code> filter and a <code>NOT_CONTAINS</code> filter on the same
         *        field. Similarly, you can't provide both an <code>EQUALS</code> filter and a <code>NOT_EQUALS</code>
         *        or <code>PREFIX_NOT_EQUALS</code> filter on the same field. Combining filters in this way returns an
         *        error. <code>CONTAINS</code> filters can only be used with other <code>CONTAINS</code> filters.
         *        <code>NOT_CONTAINS</code> filters can only be used with other <code>NOT_CONTAINS</code> filters.
         *        </p>
         *        <p>
         *        You can combine <code>PREFIX</code> filters with <code>NOT_EQUALS</code> or
         *        <code>PREFIX_NOT_EQUALS</code> filters for the same field. Security Hub first processes the
         *        <code>PREFIX</code> filters, and then the <code>NOT_EQUALS</code> or <code>PREFIX_NOT_EQUALS</code>
         *        filters.
         *        </p>
         *        <p>
         *        For example, for the following filters, Security Hub first identifies findings that have resource
         *        types that start with either <code>AwsIam</code> or <code>AwsEc2</code>. It then excludes findings
         *        that have a resource type of <code>AwsIamPolicy</code> and findings that have a resource type of
         *        <code>AwsEc2NetworkInterface</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ResourceType PREFIX AwsIam</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType PREFIX AwsEc2</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType NOT_EQUALS AwsIamPolicy</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType NOT_EQUALS AwsEc2NetworkInterface</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>CONTAINS</code> and <code>NOT_CONTAINS</code> operators can be used only with automation rules.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules.html">Automation
         *        rules</a> in the <i>Security Hub User Guide</i>.
         * @see StringFilterComparison
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StringFilterComparison
         */
        Builder comparison(StringFilterComparison comparison);
    }

    static final class BuilderImpl implements Builder {
        private String value;

        private String comparison;

        private BuilderImpl() {
        }

        private BuilderImpl(StringFilter model) {
            value(model.value);
            comparison(model.comparison);
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final String getComparison() {
            return comparison;
        }

        public final void setComparison(String comparison) {
            this.comparison = comparison;
        }

        @Override
        public final Builder comparison(String comparison) {
            this.comparison = comparison;
            return this;
        }

        @Override
        public final Builder comparison(StringFilterComparison comparison) {
            this.comparison(comparison == null ? null : comparison.toString());
            return this;
        }

        @Override
        public StringFilter build() {
            return new StringFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
