/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.sagemaker.SageMakerClient;
import software.amazon.awssdk.services.sagemaker.model.DescribeEndpointRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeEndpointResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageVersionRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageVersionResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeNotebookInstanceRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeNotebookInstanceResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeProcessingJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeProcessingJobResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeTrainingJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeTrainingJobResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeTransformJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeTransformJobResponse;
import software.amazon.awssdk.services.sagemaker.model.SageMakerRequest;
import software.amazon.awssdk.services.sagemaker.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultSageMakerWaiter implements SageMakerWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final SageMakerClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeNotebookInstanceResponse> notebookInstanceInServiceWaiter;

    private final Waiter<DescribeNotebookInstanceResponse> notebookInstanceStoppedWaiter;

    private final Waiter<DescribeNotebookInstanceResponse> notebookInstanceDeletedWaiter;

    private final Waiter<DescribeTrainingJobResponse> trainingJobCompletedOrStoppedWaiter;

    private final Waiter<DescribeEndpointResponse> endpointInServiceWaiter;

    private final Waiter<DescribeEndpointResponse> endpointDeletedWaiter;

    private final Waiter<DescribeTransformJobResponse> transformJobCompletedOrStoppedWaiter;

    private final Waiter<DescribeProcessingJobResponse> processingJobCompletedOrStoppedWaiter;

    private final Waiter<DescribeImageResponse> imageCreatedWaiter;

    private final Waiter<DescribeImageResponse> imageUpdatedWaiter;

    private final Waiter<DescribeImageResponse> imageDeletedWaiter;

    private final Waiter<DescribeImageVersionResponse> imageVersionCreatedWaiter;

    private final Waiter<DescribeImageVersionResponse> imageVersionDeletedWaiter;

    private DefaultSageMakerWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = SageMakerClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.notebookInstanceInServiceWaiter = Waiter.builder(DescribeNotebookInstanceResponse.class)
                .acceptors(notebookInstanceInServiceWaiterAcceptors())
                .overrideConfiguration(notebookInstanceInServiceWaiterConfig(builder.overrideConfiguration)).build();
        this.notebookInstanceStoppedWaiter = Waiter.builder(DescribeNotebookInstanceResponse.class)
                .acceptors(notebookInstanceStoppedWaiterAcceptors())
                .overrideConfiguration(notebookInstanceStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.notebookInstanceDeletedWaiter = Waiter.builder(DescribeNotebookInstanceResponse.class)
                .acceptors(notebookInstanceDeletedWaiterAcceptors())
                .overrideConfiguration(notebookInstanceDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.trainingJobCompletedOrStoppedWaiter = Waiter.builder(DescribeTrainingJobResponse.class)
                .acceptors(trainingJobCompletedOrStoppedWaiterAcceptors())
                .overrideConfiguration(trainingJobCompletedOrStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.endpointInServiceWaiter = Waiter.builder(DescribeEndpointResponse.class)
                .acceptors(endpointInServiceWaiterAcceptors())
                .overrideConfiguration(endpointInServiceWaiterConfig(builder.overrideConfiguration)).build();
        this.endpointDeletedWaiter = Waiter.builder(DescribeEndpointResponse.class).acceptors(endpointDeletedWaiterAcceptors())
                .overrideConfiguration(endpointDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.transformJobCompletedOrStoppedWaiter = Waiter.builder(DescribeTransformJobResponse.class)
                .acceptors(transformJobCompletedOrStoppedWaiterAcceptors())
                .overrideConfiguration(transformJobCompletedOrStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.processingJobCompletedOrStoppedWaiter = Waiter.builder(DescribeProcessingJobResponse.class)
                .acceptors(processingJobCompletedOrStoppedWaiterAcceptors())
                .overrideConfiguration(processingJobCompletedOrStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.imageCreatedWaiter = Waiter.builder(DescribeImageResponse.class).acceptors(imageCreatedWaiterAcceptors())
                .overrideConfiguration(imageCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.imageUpdatedWaiter = Waiter.builder(DescribeImageResponse.class).acceptors(imageUpdatedWaiterAcceptors())
                .overrideConfiguration(imageUpdatedWaiterConfig(builder.overrideConfiguration)).build();
        this.imageDeletedWaiter = Waiter.builder(DescribeImageResponse.class).acceptors(imageDeletedWaiterAcceptors())
                .overrideConfiguration(imageDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.imageVersionCreatedWaiter = Waiter.builder(DescribeImageVersionResponse.class)
                .acceptors(imageVersionCreatedWaiterAcceptors())
                .overrideConfiguration(imageVersionCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.imageVersionDeletedWaiter = Waiter.builder(DescribeImageVersionResponse.class)
                .acceptors(imageVersionDeletedWaiterAcceptors())
                .overrideConfiguration(imageVersionDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeEndpointResponse> waitUntilEndpointDeleted(DescribeEndpointRequest describeEndpointRequest) {
        return endpointDeletedWaiter.run(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)));
    }

    @Override
    public WaiterResponse<DescribeEndpointResponse> waitUntilEndpointDeleted(DescribeEndpointRequest describeEndpointRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return endpointDeletedWaiter.run(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)),
                endpointDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeEndpointResponse> waitUntilEndpointInService(DescribeEndpointRequest describeEndpointRequest) {
        return endpointInServiceWaiter.run(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)));
    }

    @Override
    public WaiterResponse<DescribeEndpointResponse> waitUntilEndpointInService(DescribeEndpointRequest describeEndpointRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return endpointInServiceWaiter.run(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)),
                endpointInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeImageResponse> waitUntilImageCreated(DescribeImageRequest describeImageRequest) {
        return imageCreatedWaiter.run(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)));
    }

    @Override
    public WaiterResponse<DescribeImageResponse> waitUntilImageCreated(DescribeImageRequest describeImageRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return imageCreatedWaiter.run(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)),
                imageCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeImageResponse> waitUntilImageDeleted(DescribeImageRequest describeImageRequest) {
        return imageDeletedWaiter.run(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)));
    }

    @Override
    public WaiterResponse<DescribeImageResponse> waitUntilImageDeleted(DescribeImageRequest describeImageRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return imageDeletedWaiter.run(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)),
                imageDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeImageResponse> waitUntilImageUpdated(DescribeImageRequest describeImageRequest) {
        return imageUpdatedWaiter.run(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)));
    }

    @Override
    public WaiterResponse<DescribeImageResponse> waitUntilImageUpdated(DescribeImageRequest describeImageRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return imageUpdatedWaiter.run(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)),
                imageUpdatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeImageVersionResponse> waitUntilImageVersionCreated(
            DescribeImageVersionRequest describeImageVersionRequest) {
        return imageVersionCreatedWaiter.run(() -> client
                .describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)));
    }

    @Override
    public WaiterResponse<DescribeImageVersionResponse> waitUntilImageVersionCreated(
            DescribeImageVersionRequest describeImageVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageVersionCreatedWaiter.run(
                () -> client.describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)),
                imageVersionCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeImageVersionResponse> waitUntilImageVersionDeleted(
            DescribeImageVersionRequest describeImageVersionRequest) {
        return imageVersionDeletedWaiter.run(() -> client
                .describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)));
    }

    @Override
    public WaiterResponse<DescribeImageVersionResponse> waitUntilImageVersionDeleted(
            DescribeImageVersionRequest describeImageVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageVersionDeletedWaiter.run(
                () -> client.describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)),
                imageVersionDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNotebookInstanceResponse> waitUntilNotebookInstanceDeleted(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        return notebookInstanceDeletedWaiter.run(() -> client
                .describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)));
    }

    @Override
    public WaiterResponse<DescribeNotebookInstanceResponse> waitUntilNotebookInstanceDeleted(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return notebookInstanceDeletedWaiter.run(
                () -> client.describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)),
                notebookInstanceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNotebookInstanceResponse> waitUntilNotebookInstanceInService(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        return notebookInstanceInServiceWaiter.run(() -> client
                .describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)));
    }

    @Override
    public WaiterResponse<DescribeNotebookInstanceResponse> waitUntilNotebookInstanceInService(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return notebookInstanceInServiceWaiter.run(
                () -> client.describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)),
                notebookInstanceInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNotebookInstanceResponse> waitUntilNotebookInstanceStopped(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        return notebookInstanceStoppedWaiter.run(() -> client
                .describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)));
    }

    @Override
    public WaiterResponse<DescribeNotebookInstanceResponse> waitUntilNotebookInstanceStopped(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return notebookInstanceStoppedWaiter.run(
                () -> client.describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)),
                notebookInstanceStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeProcessingJobResponse> waitUntilProcessingJobCompletedOrStopped(
            DescribeProcessingJobRequest describeProcessingJobRequest) {
        return processingJobCompletedOrStoppedWaiter.run(() -> client
                .describeProcessingJob(applyWaitersUserAgent(describeProcessingJobRequest)));
    }

    @Override
    public WaiterResponse<DescribeProcessingJobResponse> waitUntilProcessingJobCompletedOrStopped(
            DescribeProcessingJobRequest describeProcessingJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return processingJobCompletedOrStoppedWaiter.run(
                () -> client.describeProcessingJob(applyWaitersUserAgent(describeProcessingJobRequest)),
                processingJobCompletedOrStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTrainingJobResponse> waitUntilTrainingJobCompletedOrStopped(
            DescribeTrainingJobRequest describeTrainingJobRequest) {
        return trainingJobCompletedOrStoppedWaiter.run(() -> client
                .describeTrainingJob(applyWaitersUserAgent(describeTrainingJobRequest)));
    }

    @Override
    public WaiterResponse<DescribeTrainingJobResponse> waitUntilTrainingJobCompletedOrStopped(
            DescribeTrainingJobRequest describeTrainingJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return trainingJobCompletedOrStoppedWaiter.run(
                () -> client.describeTrainingJob(applyWaitersUserAgent(describeTrainingJobRequest)),
                trainingJobCompletedOrStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTransformJobResponse> waitUntilTransformJobCompletedOrStopped(
            DescribeTransformJobRequest describeTransformJobRequest) {
        return transformJobCompletedOrStoppedWaiter.run(() -> client
                .describeTransformJob(applyWaitersUserAgent(describeTransformJobRequest)));
    }

    @Override
    public WaiterResponse<DescribeTransformJobResponse> waitUntilTransformJobCompletedOrStopped(
            DescribeTransformJobRequest describeTransformJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return transformJobCompletedOrStoppedWaiter.run(
                () -> client.describeTransformJob(applyWaitersUserAgent(describeTransformJobRequest)),
                transformJobCompletedOrStoppedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> notebookInstanceInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "InService");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> notebookInstanceStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> notebookInstanceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTrainingJobResponse>> trainingJobCompletedOrStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTrainingJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("TrainingJobStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("TrainingJobStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("TrainingJobStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEndpointResponse>> endpointInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("EndpointStatus").value(), "InService");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("EndpointStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEndpointResponse>> endpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("EndpointStatus").value(), "Failed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTransformJobResponse>> transformJobCompletedOrStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTransformJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("TransformJobStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("TransformJobStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("TransformJobStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeProcessingJobResponse>> processingJobCompletedOrStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeProcessingJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ProcessingJobStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ProcessingJobStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ProcessingJobStatus").value(), "Failed");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageResponse>> imageCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "CREATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "CREATE_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageResponse>> imageUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "CREATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "UPDATE_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageResponse>> imageDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "DELETE_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageVersionResponse>> imageVersionCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageVersionStatus").value(), "CREATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageVersionStatus").value(), "CREATE_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageVersionResponse>> imageVersionDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("ImageVersionStatus").value(), "DELETE_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration notebookInstanceInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration notebookInstanceStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration notebookInstanceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration trainingJobCompletedOrStoppedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration endpointInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration endpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration transformJobCompletedOrStoppedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration processingJobCompletedOrStoppedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageVersionCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageVersionDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static SageMakerWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends SageMakerRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements SageMakerWaiter.Builder {
        private SageMakerClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public SageMakerWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public SageMakerWaiter.Builder client(SageMakerClient client) {
            this.client = client;
            return this;
        }

        public SageMakerWaiter build() {
            return new DefaultSageMakerWaiter(this);
        }
    }
}
