/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * For Resolver list operations (<a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html"
 * >ListResolverEndpoints</a>, <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html"
 * >ListResolverRules</a>, <a href=
 * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html"
 * >ListResolverRuleAssociations</a>, <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html"
 * >ListResolverQueryLogConfigs</a>, <a href=
 * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html"
 * >ListResolverQueryLogConfigAssociations</a>), and <a
 * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverDnssecConfigs.html"
 * >ListResolverDnssecConfigs</a>), an optional specification to return a subset of objects.
 * </p>
 * <p>
 * To filter objects, such as Resolver endpoints or Resolver rules, you specify <code>Name</code> and
 * <code>Values</code>. For example, to list only inbound Resolver endpoints, specify <code>Direction</code> for
 * <code>Name</code> and specify <code>INBOUND</code> for <code>Values</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Filter implements SdkPojo, Serializable, ToCopyableBuilder<Filter.Builder, Filter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Filter::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(Filter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> values;

    private Filter(BuilderImpl builder) {
        this.name = builder.name;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of the parameter that you want to use to filter objects.
     * </p>
     * <p>
     * The valid values for <code>Name</code> depend on the action that you're including the filter in, <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html"
     * >ListResolverEndpoints</a>, <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html"
     * >ListResolverRules</a>, <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html"
     * >ListResolverRuleAssociations</a>, <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html"
     * >ListResolverQueryLogConfigs</a>, or <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html"
     * >ListResolverQueryLogConfigAssociations</a>.
     * </p>
     * <note>
     * <p>
     * In early versions of Resolver, values for <code>Name</code> were listed as uppercase, with underscore (_)
     * delimiters. For example, <code>CreatorRequestId</code> was originally listed as <code>CREATOR_REQUEST_ID</code>.
     * Uppercase values for <code>Name</code> are still supported.
     * </p>
     * </note>
     * <p>
     * <b>ListResolverEndpoints</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreatorRequestId</code>: The value that you specified when you created the Resolver endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Direction</code>: Whether you want to return inbound or outbound Resolver endpoints. If you specify
     * <code>DIRECTION</code> for <code>Name</code>, specify <code>INBOUND</code> or <code>OUTBOUND</code> for
     * <code>Values</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HostVPCId</code>: The ID of the VPC that inbound DNS queries pass through on the way from your network to
     * your VPCs in a region, or the VPC that outbound queries pass through on the way from your VPCs to your network.
     * In a <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
     * >CreateResolverEndpoint</a> request, <code>SubnetId</code> indirectly identifies the VPC. In a <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html"
     * >GetResolverEndpoint</a> request, the VPC ID for a Resolver endpoint is returned in the <code>HostVPCId</code>
     * element.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IpAddressCount</code>: The number of IP addresses that you have associated with the Resolver endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Name</code>: The name of the Resolver endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SecurityGroupIds</code>: The IDs of the VPC security groups that you specified when you created the
     * Resolver endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>: The status of the Resolver endpoint. If you specify <code>Status</code> for
     * <code>Name</code>, specify one of the following status codes for <code>Values</code>: <code>CREATING</code>,
     * <code>OPERATIONAL</code>, <code>UPDATING</code>, <code>AUTO_RECOVERING</code>, <code>ACTION_NEEDED</code>, or
     * <code>DELETING</code>. For more information, see <code>Status</code> in <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverEndpoint.html"
     * >ResolverEndpoint</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>ListResolverRules</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreatorRequestId</code>: The value that you specified when you created the Resolver rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DomainName</code>: The domain name for which Resolver is forwarding DNS queries to your network. In the
     * value that you specify for <code>Values</code>, include a trailing dot (.) after the domain name. For example, if
     * the domain name is example.com, specify the following value. Note the "." after <code>com</code>:
     * </p>
     * <p>
     * <code>example.com.</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Name</code>: The name of the Resolver rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResolverEndpointId</code>: The ID of the Resolver endpoint that the Resolver rule is associated with.
     * </p>
     * <note>
     * <p>
     * You can filter on the Resolver endpoint only for rules that have a value of <code>FORWARD</code> for
     * <code>RuleType</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>Status</code>: The status of the Resolver rule. If you specify <code>Status</code> for <code>Name</code>,
     * specify one of the following status codes for <code>Values</code>: <code>COMPLETE</code>, <code>DELETING</code>,
     * <code>UPDATING</code>, or <code>FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Type</code>: The type of the Resolver rule. If you specify <code>TYPE</code> for <code>Name</code>, specify
     * <code>FORWARD</code> or <code>SYSTEM</code> for <code>Values</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>ListResolverRuleAssociations</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Name</code>: The name of the Resolver rule association.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResolverRuleId</code>: The ID of the Resolver rule that is associated with one or more VPCs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>: The status of the Resolver rule association. If you specify <code>Status</code> for
     * <code>Name</code>, specify one of the following status codes for <code>Values</code>: <code>CREATING</code>,
     * <code>COMPLETE</code>, <code>DELETING</code>, or <code>FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VPCId</code>: The ID of the VPC that the Resolver rule is associated with.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>ListResolverQueryLogConfigs</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Arn</code>: The ARN for the query logging configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AssociationCount</code>: The number of VPCs that are associated with the query logging configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>: The date and time that the query logging configuration was created, in Unix time
     * format and Coordinated Universal Time (UTC).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreatorRequestId</code>: A unique string that identifies the request that created the query logging
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Destination</code>: The Amazon Web Services service that you want to forward query logs to. Valid values
     * include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>S3</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudWatchLogs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KinesisFirehose</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>DestinationArn</code>: The ARN of the location that Resolver is sending query logs to. This value can be
     * the ARN for an S3 bucket, a CloudWatch Logs log group, or a Kinesis Data Firehose delivery stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Id</code>: The ID of the query logging configuration
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Name</code>: The name of the query logging configuration
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OwnerId</code>: The Amazon Web Services account ID for the account that created the query logging
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ShareStatus</code>: An indication of whether the query logging configuration is shared with other Amazon
     * Web Services accounts, or was shared with the current account by another Amazon Web Services account. Valid
     * values include: <code>NOT_SHARED</code>, <code>SHARED_WITH_ME</code>, or <code>SHARED_BY_ME</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>: The status of the query logging configuration. If you specify <code>Status</code> for
     * <code>Name</code>, specify the applicable status code for <code>Values</code>: <code>CREATING</code>,
     * <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfig.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfig-Status"
     * >Status</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>ListResolverQueryLogConfigAssociations</b>
     * </p>
     * <p>
     * Valid values for <code>Name</code> include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>: The date and time that the VPC was associated with the query logging configuration, in
     * Unix time format and Coordinated Universal Time (UTC).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Error</code>: If the value of <code>Status</code> is <code>FAILED</code>, specify the cause:
     * <code>DESTINATION_NOT_FOUND</code> or <code>ACCESS_DENIED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Id</code>: The ID of the query logging association.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResolverQueryLogConfigId</code>: The ID of the query logging configuration that a VPC is associated with.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceId</code>: The ID of the Amazon VPC that is associated with the query logging configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>: The status of the query logging association. If you specify <code>Status</code> for
     * <code>Name</code>, specify the applicable status code for <code>Values</code>: <code>CREATING</code>,
     * <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfigAssociation.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfigAssociation-Status"
     * >Status</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the parameter that you want to use to filter objects.</p>
     *         <p>
     *         The valid values for <code>Name</code> depend on the action that you're including the filter in, <a
     *         href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html"
     *         >ListResolverEndpoints</a>, <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html"
     *         >ListResolverRules</a>, <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html"
     *         >ListResolverRuleAssociations</a>, <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html"
     *         >ListResolverQueryLogConfigs</a>, or <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html"
     *         >ListResolverQueryLogConfigAssociations</a>.
     *         </p>
     *         <note>
     *         <p>
     *         In early versions of Resolver, values for <code>Name</code> were listed as uppercase, with underscore (_)
     *         delimiters. For example, <code>CreatorRequestId</code> was originally listed as
     *         <code>CREATOR_REQUEST_ID</code>. Uppercase values for <code>Name</code> are still supported.
     *         </p>
     *         </note>
     *         <p>
     *         <b>ListResolverEndpoints</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CreatorRequestId</code>: The value that you specified when you created the Resolver endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Direction</code>: Whether you want to return inbound or outbound Resolver endpoints. If you specify
     *         <code>DIRECTION</code> for <code>Name</code>, specify <code>INBOUND</code> or <code>OUTBOUND</code> for
     *         <code>Values</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HostVPCId</code>: The ID of the VPC that inbound DNS queries pass through on the way from your
     *         network to your VPCs in a region, or the VPC that outbound queries pass through on the way from your VPCs
     *         to your network. In a <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
     *         >CreateResolverEndpoint</a> request, <code>SubnetId</code> indirectly identifies the VPC. In a <a
     *         href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html"
     *         >GetResolverEndpoint</a> request, the VPC ID for a Resolver endpoint is returned in the
     *         <code>HostVPCId</code> element.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IpAddressCount</code>: The number of IP addresses that you have associated with the Resolver
     *         endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Name</code>: The name of the Resolver endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SecurityGroupIds</code>: The IDs of the VPC security groups that you specified when you created the
     *         Resolver endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Status</code>: The status of the Resolver endpoint. If you specify <code>Status</code> for
     *         <code>Name</code>, specify one of the following status codes for <code>Values</code>:
     *         <code>CREATING</code>, <code>OPERATIONAL</code>, <code>UPDATING</code>, <code>AUTO_RECOVERING</code>,
     *         <code>ACTION_NEEDED</code>, or <code>DELETING</code>. For more information, see <code>Status</code> in <a
     *         href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverEndpoint.html">
     *         ResolverEndpoint</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>ListResolverRules</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CreatorRequestId</code>: The value that you specified when you created the Resolver rule.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DomainName</code>: The domain name for which Resolver is forwarding DNS queries to your network. In
     *         the value that you specify for <code>Values</code>, include a trailing dot (.) after the domain name. For
     *         example, if the domain name is example.com, specify the following value. Note the "." after
     *         <code>com</code>:
     *         </p>
     *         <p>
     *         <code>example.com.</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Name</code>: The name of the Resolver rule.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResolverEndpointId</code>: The ID of the Resolver endpoint that the Resolver rule is associated
     *         with.
     *         </p>
     *         <note>
     *         <p>
     *         You can filter on the Resolver endpoint only for rules that have a value of <code>FORWARD</code> for
     *         <code>RuleType</code>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>Status</code>: The status of the Resolver rule. If you specify <code>Status</code> for
     *         <code>Name</code>, specify one of the following status codes for <code>Values</code>:
     *         <code>COMPLETE</code>, <code>DELETING</code>, <code>UPDATING</code>, or <code>FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Type</code>: The type of the Resolver rule. If you specify <code>TYPE</code> for <code>Name</code>,
     *         specify <code>FORWARD</code> or <code>SYSTEM</code> for <code>Values</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>ListResolverRuleAssociations</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Name</code>: The name of the Resolver rule association.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResolverRuleId</code>: The ID of the Resolver rule that is associated with one or more VPCs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Status</code>: The status of the Resolver rule association. If you specify <code>Status</code> for
     *         <code>Name</code>, specify one of the following status codes for <code>Values</code>:
     *         <code>CREATING</code>, <code>COMPLETE</code>, <code>DELETING</code>, or <code>FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VPCId</code>: The ID of the VPC that the Resolver rule is associated with.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>ListResolverQueryLogConfigs</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Arn</code>: The ARN for the query logging configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AssociationCount</code>: The number of VPCs that are associated with the query logging
     *         configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CreationTime</code>: The date and time that the query logging configuration was created, in Unix
     *         time format and Coordinated Universal Time (UTC).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CreatorRequestId</code>: A unique string that identifies the request that created the query logging
     *         configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Destination</code>: The Amazon Web Services service that you want to forward query logs to. Valid
     *         values include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>S3</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudWatchLogs</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KinesisFirehose</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DestinationArn</code>: The ARN of the location that Resolver is sending query logs to. This value
     *         can be the ARN for an S3 bucket, a CloudWatch Logs log group, or a Kinesis Data Firehose delivery stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Id</code>: The ID of the query logging configuration
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Name</code>: The name of the query logging configuration
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OwnerId</code>: The Amazon Web Services account ID for the account that created the query logging
     *         configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ShareStatus</code>: An indication of whether the query logging configuration is shared with other
     *         Amazon Web Services accounts, or was shared with the current account by another Amazon Web Services
     *         account. Valid values include: <code>NOT_SHARED</code>, <code>SHARED_WITH_ME</code>, or
     *         <code>SHARED_BY_ME</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Status</code>: The status of the query logging configuration. If you specify <code>Status</code>
     *         for <code>Name</code>, specify the applicable status code for <code>Values</code>: <code>CREATING</code>,
     *         <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfig.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfig-Status"
     *         >Status</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>ListResolverQueryLogConfigAssociations</b>
     *         </p>
     *         <p>
     *         Valid values for <code>Name</code> include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CreationTime</code>: The date and time that the VPC was associated with the query logging
     *         configuration, in Unix time format and Coordinated Universal Time (UTC).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Error</code>: If the value of <code>Status</code> is <code>FAILED</code>, specify the cause:
     *         <code>DESTINATION_NOT_FOUND</code> or <code>ACCESS_DENIED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Id</code>: The ID of the query logging association.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResolverQueryLogConfigId</code>: The ID of the query logging configuration that a VPC is associated
     *         with.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceId</code>: The ID of the Amazon VPC that is associated with the query logging
     *         configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Status</code>: The status of the query logging association. If you specify <code>Status</code> for
     *         <code>Name</code>, specify the applicable status code for <code>Values</code>: <code>CREATING</code>,
     *         <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfigAssociation.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfigAssociation-Status"
     *         >Status</a>.
     *         </p>
     *         </li>
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * When you're using a <code>List</code> operation and you want the operation to return a subset of objects, such as
     * Resolver endpoints or Resolver rules, the value of the parameter that you want to use to filter objects. For
     * example, to list only inbound Resolver endpoints, specify <code>Direction</code> for <code>Name</code> and
     * specify <code>INBOUND</code> for <code>Values</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return When you're using a <code>List</code> operation and you want the operation to return a subset of objects,
     *         such as Resolver endpoints or Resolver rules, the value of the parameter that you want to use to filter
     *         objects. For example, to list only inbound Resolver endpoints, specify <code>Direction</code> for
     *         <code>Name</code> and specify <code>INBOUND</code> for <code>Values</code>.
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Filter)) {
            return false;
        }
        Filter other = (Filter) obj;
        return Objects.equals(name(), other.name()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Filter").add("Name", name()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Filter, T> g) {
        return obj -> g.apply((Filter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Filter> {
        /**
         * <p>
         * The name of the parameter that you want to use to filter objects.
         * </p>
         * <p>
         * The valid values for <code>Name</code> depend on the action that you're including the filter in, <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html"
         * >ListResolverEndpoints</a>, <a
         * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html"
         * >ListResolverRules</a>, <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html"
         * >ListResolverRuleAssociations</a>, <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html"
         * >ListResolverQueryLogConfigs</a>, or <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html"
         * >ListResolverQueryLogConfigAssociations</a>.
         * </p>
         * <note>
         * <p>
         * In early versions of Resolver, values for <code>Name</code> were listed as uppercase, with underscore (_)
         * delimiters. For example, <code>CreatorRequestId</code> was originally listed as
         * <code>CREATOR_REQUEST_ID</code>. Uppercase values for <code>Name</code> are still supported.
         * </p>
         * </note>
         * <p>
         * <b>ListResolverEndpoints</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CreatorRequestId</code>: The value that you specified when you created the Resolver endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Direction</code>: Whether you want to return inbound or outbound Resolver endpoints. If you specify
         * <code>DIRECTION</code> for <code>Name</code>, specify <code>INBOUND</code> or <code>OUTBOUND</code> for
         * <code>Values</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HostVPCId</code>: The ID of the VPC that inbound DNS queries pass through on the way from your network
         * to your VPCs in a region, or the VPC that outbound queries pass through on the way from your VPCs to your
         * network. In a <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
         * >CreateResolverEndpoint</a> request, <code>SubnetId</code> indirectly identifies the VPC. In a <a
         * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html"
         * >GetResolverEndpoint</a> request, the VPC ID for a Resolver endpoint is returned in the
         * <code>HostVPCId</code> element.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IpAddressCount</code>: The number of IP addresses that you have associated with the Resolver endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Name</code>: The name of the Resolver endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SecurityGroupIds</code>: The IDs of the VPC security groups that you specified when you created the
         * Resolver endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Status</code>: The status of the Resolver endpoint. If you specify <code>Status</code> for
         * <code>Name</code>, specify one of the following status codes for <code>Values</code>: <code>CREATING</code>,
         * <code>OPERATIONAL</code>, <code>UPDATING</code>, <code>AUTO_RECOVERING</code>, <code>ACTION_NEEDED</code>, or
         * <code>DELETING</code>. For more information, see <code>Status</code> in <a
         * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverEndpoint.html"
         * >ResolverEndpoint</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>ListResolverRules</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CreatorRequestId</code>: The value that you specified when you created the Resolver rule.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DomainName</code>: The domain name for which Resolver is forwarding DNS queries to your network. In the
         * value that you specify for <code>Values</code>, include a trailing dot (.) after the domain name. For
         * example, if the domain name is example.com, specify the following value. Note the "." after <code>com</code>:
         * </p>
         * <p>
         * <code>example.com.</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Name</code>: The name of the Resolver rule.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResolverEndpointId</code>: The ID of the Resolver endpoint that the Resolver rule is associated with.
         * </p>
         * <note>
         * <p>
         * You can filter on the Resolver endpoint only for rules that have a value of <code>FORWARD</code> for
         * <code>RuleType</code>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>Status</code>: The status of the Resolver rule. If you specify <code>Status</code> for
         * <code>Name</code>, specify one of the following status codes for <code>Values</code>: <code>COMPLETE</code>,
         * <code>DELETING</code>, <code>UPDATING</code>, or <code>FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Type</code>: The type of the Resolver rule. If you specify <code>TYPE</code> for <code>Name</code>,
         * specify <code>FORWARD</code> or <code>SYSTEM</code> for <code>Values</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>ListResolverRuleAssociations</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Name</code>: The name of the Resolver rule association.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResolverRuleId</code>: The ID of the Resolver rule that is associated with one or more VPCs.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Status</code>: The status of the Resolver rule association. If you specify <code>Status</code> for
         * <code>Name</code>, specify one of the following status codes for <code>Values</code>: <code>CREATING</code>,
         * <code>COMPLETE</code>, <code>DELETING</code>, or <code>FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VPCId</code>: The ID of the VPC that the Resolver rule is associated with.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>ListResolverQueryLogConfigs</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Arn</code>: The ARN for the query logging configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AssociationCount</code>: The number of VPCs that are associated with the query logging configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CreationTime</code>: The date and time that the query logging configuration was created, in Unix time
         * format and Coordinated Universal Time (UTC).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CreatorRequestId</code>: A unique string that identifies the request that created the query logging
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Destination</code>: The Amazon Web Services service that you want to forward query logs to. Valid
         * values include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>S3</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudWatchLogs</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KinesisFirehose</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>DestinationArn</code>: The ARN of the location that Resolver is sending query logs to. This value can
         * be the ARN for an S3 bucket, a CloudWatch Logs log group, or a Kinesis Data Firehose delivery stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Id</code>: The ID of the query logging configuration
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Name</code>: The name of the query logging configuration
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OwnerId</code>: The Amazon Web Services account ID for the account that created the query logging
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ShareStatus</code>: An indication of whether the query logging configuration is shared with other
         * Amazon Web Services accounts, or was shared with the current account by another Amazon Web Services account.
         * Valid values include: <code>NOT_SHARED</code>, <code>SHARED_WITH_ME</code>, or <code>SHARED_BY_ME</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Status</code>: The status of the query logging configuration. If you specify <code>Status</code> for
         * <code>Name</code>, specify the applicable status code for <code>Values</code>: <code>CREATING</code>,
         * <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more information, see <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfig.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfig-Status"
         * >Status</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>ListResolverQueryLogConfigAssociations</b>
         * </p>
         * <p>
         * Valid values for <code>Name</code> include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CreationTime</code>: The date and time that the VPC was associated with the query logging
         * configuration, in Unix time format and Coordinated Universal Time (UTC).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Error</code>: If the value of <code>Status</code> is <code>FAILED</code>, specify the cause:
         * <code>DESTINATION_NOT_FOUND</code> or <code>ACCESS_DENIED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Id</code>: The ID of the query logging association.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResolverQueryLogConfigId</code>: The ID of the query logging configuration that a VPC is associated
         * with.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceId</code>: The ID of the Amazon VPC that is associated with the query logging configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Status</code>: The status of the query logging association. If you specify <code>Status</code> for
         * <code>Name</code>, specify the applicable status code for <code>Values</code>: <code>CREATING</code>,
         * <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more information, see <a href=
         * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfigAssociation.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfigAssociation-Status"
         * >Status</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the parameter that you want to use to filter objects.</p>
         *        <p>
         *        The valid values for <code>Name</code> depend on the action that you're including the filter in, <a
         *        href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverEndpoints.html"
         *        >ListResolverEndpoints</a>, <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRules.html"
         *        >ListResolverRules</a>, <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverRuleAssociations.html"
         *        >ListResolverRuleAssociations</a>, <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigs.html"
         *        >ListResolverQueryLogConfigs</a>, or <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ListResolverQueryLogConfigAssociations.html"
         *        >ListResolverQueryLogConfigAssociations</a>.
         *        </p>
         *        <note>
         *        <p>
         *        In early versions of Resolver, values for <code>Name</code> were listed as uppercase, with underscore
         *        (_) delimiters. For example, <code>CreatorRequestId</code> was originally listed as
         *        <code>CREATOR_REQUEST_ID</code>. Uppercase values for <code>Name</code> are still supported.
         *        </p>
         *        </note>
         *        <p>
         *        <b>ListResolverEndpoints</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreatorRequestId</code>: The value that you specified when you created the Resolver endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Direction</code>: Whether you want to return inbound or outbound Resolver endpoints. If you
         *        specify <code>DIRECTION</code> for <code>Name</code>, specify <code>INBOUND</code> or
         *        <code>OUTBOUND</code> for <code>Values</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HostVPCId</code>: The ID of the VPC that inbound DNS queries pass through on the way from your
         *        network to your VPCs in a region, or the VPC that outbound queries pass through on the way from your
         *        VPCs to your network. In a <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverEndpoint.html"
         *        >CreateResolverEndpoint</a> request, <code>SubnetId</code> indirectly identifies the VPC. In a <a
         *        href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_GetResolverEndpoint.html"
         *        >GetResolverEndpoint</a> request, the VPC ID for a Resolver endpoint is returned in the
         *        <code>HostVPCId</code> element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IpAddressCount</code>: The number of IP addresses that you have associated with the Resolver
         *        endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Name</code>: The name of the Resolver endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SecurityGroupIds</code>: The IDs of the VPC security groups that you specified when you created
         *        the Resolver endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Status</code>: The status of the Resolver endpoint. If you specify <code>Status</code> for
         *        <code>Name</code>, specify one of the following status codes for <code>Values</code>:
         *        <code>CREATING</code>, <code>OPERATIONAL</code>, <code>UPDATING</code>, <code>AUTO_RECOVERING</code>,
         *        <code>ACTION_NEEDED</code>, or <code>DELETING</code>. For more information, see <code>Status</code> in
         *        <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverEndpoint.html"
         *        >ResolverEndpoint</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>ListResolverRules</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreatorRequestId</code>: The value that you specified when you created the Resolver rule.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DomainName</code>: The domain name for which Resolver is forwarding DNS queries to your network.
         *        In the value that you specify for <code>Values</code>, include a trailing dot (.) after the domain
         *        name. For example, if the domain name is example.com, specify the following value. Note the "." after
         *        <code>com</code>:
         *        </p>
         *        <p>
         *        <code>example.com.</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Name</code>: The name of the Resolver rule.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResolverEndpointId</code>: The ID of the Resolver endpoint that the Resolver rule is associated
         *        with.
         *        </p>
         *        <note>
         *        <p>
         *        You can filter on the Resolver endpoint only for rules that have a value of <code>FORWARD</code> for
         *        <code>RuleType</code>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>Status</code>: The status of the Resolver rule. If you specify <code>Status</code> for
         *        <code>Name</code>, specify one of the following status codes for <code>Values</code>:
         *        <code>COMPLETE</code>, <code>DELETING</code>, <code>UPDATING</code>, or <code>FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Type</code>: The type of the Resolver rule. If you specify <code>TYPE</code> for
         *        <code>Name</code>, specify <code>FORWARD</code> or <code>SYSTEM</code> for <code>Values</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>ListResolverRuleAssociations</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Name</code>: The name of the Resolver rule association.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResolverRuleId</code>: The ID of the Resolver rule that is associated with one or more VPCs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Status</code>: The status of the Resolver rule association. If you specify <code>Status</code>
         *        for <code>Name</code>, specify one of the following status codes for <code>Values</code>:
         *        <code>CREATING</code>, <code>COMPLETE</code>, <code>DELETING</code>, or <code>FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VPCId</code>: The ID of the VPC that the Resolver rule is associated with.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>ListResolverQueryLogConfigs</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Arn</code>: The ARN for the query logging configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AssociationCount</code>: The number of VPCs that are associated with the query logging
         *        configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CreationTime</code>: The date and time that the query logging configuration was created, in Unix
         *        time format and Coordinated Universal Time (UTC).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CreatorRequestId</code>: A unique string that identifies the request that created the query
         *        logging configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Destination</code>: The Amazon Web Services service that you want to forward query logs to.
         *        Valid values include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>S3</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudWatchLogs</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KinesisFirehose</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DestinationArn</code>: The ARN of the location that Resolver is sending query logs to. This
         *        value can be the ARN for an S3 bucket, a CloudWatch Logs log group, or a Kinesis Data Firehose
         *        delivery stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Id</code>: The ID of the query logging configuration
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Name</code>: The name of the query logging configuration
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OwnerId</code>: The Amazon Web Services account ID for the account that created the query
         *        logging configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ShareStatus</code>: An indication of whether the query logging configuration is shared with
         *        other Amazon Web Services accounts, or was shared with the current account by another Amazon Web
         *        Services account. Valid values include: <code>NOT_SHARED</code>, <code>SHARED_WITH_ME</code>, or
         *        <code>SHARED_BY_ME</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Status</code>: The status of the query logging configuration. If you specify <code>Status</code>
         *        for <code>Name</code>, specify the applicable status code for <code>Values</code>:
         *        <code>CREATING</code>, <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfig.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfig-Status"
         *        >Status</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>ListResolverQueryLogConfigAssociations</b>
         *        </p>
         *        <p>
         *        Valid values for <code>Name</code> include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreationTime</code>: The date and time that the VPC was associated with the query logging
         *        configuration, in Unix time format and Coordinated Universal Time (UTC).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Error</code>: If the value of <code>Status</code> is <code>FAILED</code>, specify the cause:
         *        <code>DESTINATION_NOT_FOUND</code> or <code>ACCESS_DENIED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Id</code>: The ID of the query logging association.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResolverQueryLogConfigId</code>: The ID of the query logging configuration that a VPC is
         *        associated with.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceId</code>: The ID of the Amazon VPC that is associated with the query logging
         *        configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Status</code>: The status of the query logging association. If you specify <code>Status</code>
         *        for <code>Name</code>, specify the applicable status code for <code>Values</code>:
         *        <code>CREATING</code>, <code>CREATED</code>, <code>DELETING</code>, or <code>FAILED</code>. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_ResolverQueryLogConfigAssociation.html#Route53Resolver-Type-route53resolver_ResolverQueryLogConfigAssociation-Status"
         *        >Status</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * When you're using a <code>List</code> operation and you want the operation to return a subset of objects,
         * such as Resolver endpoints or Resolver rules, the value of the parameter that you want to use to filter
         * objects. For example, to list only inbound Resolver endpoints, specify <code>Direction</code> for
         * <code>Name</code> and specify <code>INBOUND</code> for <code>Values</code>.
         * </p>
         * 
         * @param values
         *        When you're using a <code>List</code> operation and you want the operation to return a subset of
         *        objects, such as Resolver endpoints or Resolver rules, the value of the parameter that you want to use
         *        to filter objects. For example, to list only inbound Resolver endpoints, specify
         *        <code>Direction</code> for <code>Name</code> and specify <code>INBOUND</code> for <code>Values</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * When you're using a <code>List</code> operation and you want the operation to return a subset of objects,
         * such as Resolver endpoints or Resolver rules, the value of the parameter that you want to use to filter
         * objects. For example, to list only inbound Resolver endpoints, specify <code>Direction</code> for
         * <code>Name</code> and specify <code>INBOUND</code> for <code>Values</code>.
         * </p>
         * 
         * @param values
         *        When you're using a <code>List</code> operation and you want the operation to return a subset of
         *        objects, such as Resolver endpoints or Resolver rules, the value of the parameter that you want to use
         *        to filter objects. For example, to list only inbound Resolver endpoints, specify
         *        <code>Direction</code> for <code>Name</code> and specify <code>INBOUND</code> for <code>Values</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Filter model) {
            name(model.name);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = FilterValuesCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = FilterValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public Filter build() {
            return new Filter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
