/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshiftserverless.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an Amazon Redshift Serverless VPC endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EndpointAccess implements SdkPojo, Serializable, ToCopyableBuilder<EndpointAccess.Builder, EndpointAccess> {
    private static final SdkField<String> ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("address")
            .getter(getter(EndpointAccess::address)).setter(setter(Builder::address))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("address").build()).build();

    private static final SdkField<String> ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endpointArn").getter(getter(EndpointAccess::endpointArn)).setter(setter(Builder::endpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpointArn").build()).build();

    private static final SdkField<Instant> ENDPOINT_CREATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("endpointCreateTime")
            .getter(getter(EndpointAccess::endpointCreateTime))
            .setter(setter(Builder::endpointCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpointCreateTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> ENDPOINT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endpointName").getter(getter(EndpointAccess::endpointName)).setter(setter(Builder::endpointName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpointName").build()).build();

    private static final SdkField<String> ENDPOINT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endpointStatus").getter(getter(EndpointAccess::endpointStatus)).setter(setter(Builder::endpointStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpointStatus").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("port")
            .getter(getter(EndpointAccess::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("port").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("subnetIds")
            .getter(getter(EndpointAccess::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VpcEndpoint> VPC_ENDPOINT_FIELD = SdkField.<VpcEndpoint> builder(MarshallingType.SDK_POJO)
            .memberName("vpcEndpoint").getter(getter(EndpointAccess::vpcEndpoint)).setter(setter(Builder::vpcEndpoint))
            .constructor(VpcEndpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcEndpoint").build()).build();

    private static final SdkField<List<VpcSecurityGroupMembership>> VPC_SECURITY_GROUPS_FIELD = SdkField
            .<List<VpcSecurityGroupMembership>> builder(MarshallingType.LIST)
            .memberName("vpcSecurityGroups")
            .getter(getter(EndpointAccess::vpcSecurityGroups))
            .setter(setter(Builder::vpcSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcSecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VpcSecurityGroupMembership> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpcSecurityGroupMembership::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WORKGROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("workgroupName").getter(getter(EndpointAccess::workgroupName)).setter(setter(Builder::workgroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workgroupName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADDRESS_FIELD,
            ENDPOINT_ARN_FIELD, ENDPOINT_CREATE_TIME_FIELD, ENDPOINT_NAME_FIELD, ENDPOINT_STATUS_FIELD, PORT_FIELD,
            SUBNET_IDS_FIELD, VPC_ENDPOINT_FIELD, VPC_SECURITY_GROUPS_FIELD, WORKGROUP_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String address;

    private final String endpointArn;

    private final Instant endpointCreateTime;

    private final String endpointName;

    private final String endpointStatus;

    private final Integer port;

    private final List<String> subnetIds;

    private final VpcEndpoint vpcEndpoint;

    private final List<VpcSecurityGroupMembership> vpcSecurityGroups;

    private final String workgroupName;

    private EndpointAccess(BuilderImpl builder) {
        this.address = builder.address;
        this.endpointArn = builder.endpointArn;
        this.endpointCreateTime = builder.endpointCreateTime;
        this.endpointName = builder.endpointName;
        this.endpointStatus = builder.endpointStatus;
        this.port = builder.port;
        this.subnetIds = builder.subnetIds;
        this.vpcEndpoint = builder.vpcEndpoint;
        this.vpcSecurityGroups = builder.vpcSecurityGroups;
        this.workgroupName = builder.workgroupName;
    }

    /**
     * <p>
     * The DNS address of the endpoint.
     * </p>
     * 
     * @return The DNS address of the endpoint.
     */
    public final String address() {
        return address;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the VPC endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the VPC endpoint.
     */
    public final String endpointArn() {
        return endpointArn;
    }

    /**
     * <p>
     * The time that the endpoint was created.
     * </p>
     * 
     * @return The time that the endpoint was created.
     */
    public final Instant endpointCreateTime() {
        return endpointCreateTime;
    }

    /**
     * <p>
     * The name of the VPC endpoint.
     * </p>
     * 
     * @return The name of the VPC endpoint.
     */
    public final String endpointName() {
        return endpointName;
    }

    /**
     * <p>
     * The status of the VPC endpoint.
     * </p>
     * 
     * @return The status of the VPC endpoint.
     */
    public final String endpointStatus() {
        return endpointStatus;
    }

    /**
     * <p>
     * The port number on which Amazon Redshift Serverless accepts incoming connections.
     * </p>
     * 
     * @return The port number on which Amazon Redshift Serverless accepts incoming connections.
     */
    public final Integer port() {
        return port;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The unique identifier of subnets where Amazon Redshift Serverless choose to deploy the VPC endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return The unique identifier of subnets where Amazon Redshift Serverless choose to deploy the VPC endpoint.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * <p>
     * The connection endpoint for connecting to Amazon Redshift Serverless.
     * </p>
     * 
     * @return The connection endpoint for connecting to Amazon Redshift Serverless.
     */
    public final VpcEndpoint vpcEndpoint() {
        return vpcEndpoint;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcSecurityGroups property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVpcSecurityGroups() {
        return vpcSecurityGroups != null && !(vpcSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups associated with the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcSecurityGroups} method.
     * </p>
     * 
     * @return The security groups associated with the endpoint.
     */
    public final List<VpcSecurityGroupMembership> vpcSecurityGroups() {
        return vpcSecurityGroups;
    }

    /**
     * <p>
     * The name of the workgroup associated with the endpoint.
     * </p>
     * 
     * @return The name of the workgroup associated with the endpoint.
     */
    public final String workgroupName() {
        return workgroupName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(address());
        hashCode = 31 * hashCode + Objects.hashCode(endpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpointCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(endpointName());
        hashCode = 31 * hashCode + Objects.hashCode(endpointStatus());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcSecurityGroups() ? vpcSecurityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(workgroupName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EndpointAccess)) {
            return false;
        }
        EndpointAccess other = (EndpointAccess) obj;
        return Objects.equals(address(), other.address()) && Objects.equals(endpointArn(), other.endpointArn())
                && Objects.equals(endpointCreateTime(), other.endpointCreateTime())
                && Objects.equals(endpointName(), other.endpointName())
                && Objects.equals(endpointStatus(), other.endpointStatus()) && Objects.equals(port(), other.port())
                && hasSubnetIds() == other.hasSubnetIds() && Objects.equals(subnetIds(), other.subnetIds())
                && Objects.equals(vpcEndpoint(), other.vpcEndpoint()) && hasVpcSecurityGroups() == other.hasVpcSecurityGroups()
                && Objects.equals(vpcSecurityGroups(), other.vpcSecurityGroups())
                && Objects.equals(workgroupName(), other.workgroupName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EndpointAccess").add("Address", address()).add("EndpointArn", endpointArn())
                .add("EndpointCreateTime", endpointCreateTime()).add("EndpointName", endpointName())
                .add("EndpointStatus", endpointStatus()).add("Port", port())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null).add("VpcEndpoint", vpcEndpoint())
                .add("VpcSecurityGroups", hasVpcSecurityGroups() ? vpcSecurityGroups() : null)
                .add("WorkgroupName", workgroupName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "address":
            return Optional.ofNullable(clazz.cast(address()));
        case "endpointArn":
            return Optional.ofNullable(clazz.cast(endpointArn()));
        case "endpointCreateTime":
            return Optional.ofNullable(clazz.cast(endpointCreateTime()));
        case "endpointName":
            return Optional.ofNullable(clazz.cast(endpointName()));
        case "endpointStatus":
            return Optional.ofNullable(clazz.cast(endpointStatus()));
        case "port":
            return Optional.ofNullable(clazz.cast(port()));
        case "subnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "vpcEndpoint":
            return Optional.ofNullable(clazz.cast(vpcEndpoint()));
        case "vpcSecurityGroups":
            return Optional.ofNullable(clazz.cast(vpcSecurityGroups()));
        case "workgroupName":
            return Optional.ofNullable(clazz.cast(workgroupName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EndpointAccess, T> g) {
        return obj -> g.apply((EndpointAccess) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EndpointAccess> {
        /**
         * <p>
         * The DNS address of the endpoint.
         * </p>
         * 
         * @param address
         *        The DNS address of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder address(String address);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the VPC endpoint.
         * </p>
         * 
         * @param endpointArn
         *        The Amazon Resource Name (ARN) of the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointArn(String endpointArn);

        /**
         * <p>
         * The time that the endpoint was created.
         * </p>
         * 
         * @param endpointCreateTime
         *        The time that the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointCreateTime(Instant endpointCreateTime);

        /**
         * <p>
         * The name of the VPC endpoint.
         * </p>
         * 
         * @param endpointName
         *        The name of the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointName(String endpointName);

        /**
         * <p>
         * The status of the VPC endpoint.
         * </p>
         * 
         * @param endpointStatus
         *        The status of the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointStatus(String endpointStatus);

        /**
         * <p>
         * The port number on which Amazon Redshift Serverless accepts incoming connections.
         * </p>
         * 
         * @param port
         *        The port number on which Amazon Redshift Serverless accepts incoming connections.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The unique identifier of subnets where Amazon Redshift Serverless choose to deploy the VPC endpoint.
         * </p>
         * 
         * @param subnetIds
         *        The unique identifier of subnets where Amazon Redshift Serverless choose to deploy the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * The unique identifier of subnets where Amazon Redshift Serverless choose to deploy the VPC endpoint.
         * </p>
         * 
         * @param subnetIds
         *        The unique identifier of subnets where Amazon Redshift Serverless choose to deploy the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The connection endpoint for connecting to Amazon Redshift Serverless.
         * </p>
         * 
         * @param vpcEndpoint
         *        The connection endpoint for connecting to Amazon Redshift Serverless.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpoint(VpcEndpoint vpcEndpoint);

        /**
         * <p>
         * The connection endpoint for connecting to Amazon Redshift Serverless.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcEndpoint.Builder} avoiding the need to
         * create one manually via {@link VpcEndpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcEndpoint.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcEndpoint(VpcEndpoint)}.
         * 
         * @param vpcEndpoint
         *        a consumer that will call methods on {@link VpcEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcEndpoint(VpcEndpoint)
         */
        default Builder vpcEndpoint(Consumer<VpcEndpoint.Builder> vpcEndpoint) {
            return vpcEndpoint(VpcEndpoint.builder().applyMutation(vpcEndpoint).build());
        }

        /**
         * <p>
         * The security groups associated with the endpoint.
         * </p>
         * 
         * @param vpcSecurityGroups
         *        The security groups associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroups(Collection<VpcSecurityGroupMembership> vpcSecurityGroups);

        /**
         * <p>
         * The security groups associated with the endpoint.
         * </p>
         * 
         * @param vpcSecurityGroups
         *        The security groups associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroups(VpcSecurityGroupMembership... vpcSecurityGroups);

        /**
         * <p>
         * The security groups associated with the endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.redshiftserverless.model.VpcSecurityGroupMembership.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.redshiftserverless.model.VpcSecurityGroupMembership#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.redshiftserverless.model.VpcSecurityGroupMembership.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #vpcSecurityGroups(List<VpcSecurityGroupMembership>)}.
         * 
         * @param vpcSecurityGroups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.redshiftserverless.model.VpcSecurityGroupMembership.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcSecurityGroups(java.util.Collection<VpcSecurityGroupMembership>)
         */
        Builder vpcSecurityGroups(Consumer<VpcSecurityGroupMembership.Builder>... vpcSecurityGroups);

        /**
         * <p>
         * The name of the workgroup associated with the endpoint.
         * </p>
         * 
         * @param workgroupName
         *        The name of the workgroup associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workgroupName(String workgroupName);
    }

    static final class BuilderImpl implements Builder {
        private String address;

        private String endpointArn;

        private Instant endpointCreateTime;

        private String endpointName;

        private String endpointStatus;

        private Integer port;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private VpcEndpoint vpcEndpoint;

        private List<VpcSecurityGroupMembership> vpcSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private String workgroupName;

        private BuilderImpl() {
        }

        private BuilderImpl(EndpointAccess model) {
            address(model.address);
            endpointArn(model.endpointArn);
            endpointCreateTime(model.endpointCreateTime);
            endpointName(model.endpointName);
            endpointStatus(model.endpointStatus);
            port(model.port);
            subnetIds(model.subnetIds);
            vpcEndpoint(model.vpcEndpoint);
            vpcSecurityGroups(model.vpcSecurityGroups);
            workgroupName(model.workgroupName);
        }

        public final String getAddress() {
            return address;
        }

        public final void setAddress(String address) {
            this.address = address;
        }

        @Override
        public final Builder address(String address) {
            this.address = address;
            return this;
        }

        public final String getEndpointArn() {
            return endpointArn;
        }

        public final void setEndpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
        }

        @Override
        public final Builder endpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
            return this;
        }

        public final Instant getEndpointCreateTime() {
            return endpointCreateTime;
        }

        public final void setEndpointCreateTime(Instant endpointCreateTime) {
            this.endpointCreateTime = endpointCreateTime;
        }

        @Override
        public final Builder endpointCreateTime(Instant endpointCreateTime) {
            this.endpointCreateTime = endpointCreateTime;
            return this;
        }

        public final String getEndpointName() {
            return endpointName;
        }

        public final void setEndpointName(String endpointName) {
            this.endpointName = endpointName;
        }

        @Override
        public final Builder endpointName(String endpointName) {
            this.endpointName = endpointName;
            return this;
        }

        public final String getEndpointStatus() {
            return endpointStatus;
        }

        public final void setEndpointStatus(String endpointStatus) {
            this.endpointStatus = endpointStatus;
        }

        @Override
        public final Builder endpointStatus(String endpointStatus) {
            this.endpointStatus = endpointStatus;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdListCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdListCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final VpcEndpoint.Builder getVpcEndpoint() {
            return vpcEndpoint != null ? vpcEndpoint.toBuilder() : null;
        }

        public final void setVpcEndpoint(VpcEndpoint.BuilderImpl vpcEndpoint) {
            this.vpcEndpoint = vpcEndpoint != null ? vpcEndpoint.build() : null;
        }

        @Override
        public final Builder vpcEndpoint(VpcEndpoint vpcEndpoint) {
            this.vpcEndpoint = vpcEndpoint;
            return this;
        }

        public final List<VpcSecurityGroupMembership.Builder> getVpcSecurityGroups() {
            List<VpcSecurityGroupMembership.Builder> result = VpcSecurityGroupMembershipListCopier
                    .copyToBuilder(this.vpcSecurityGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVpcSecurityGroups(Collection<VpcSecurityGroupMembership.BuilderImpl> vpcSecurityGroups) {
            this.vpcSecurityGroups = VpcSecurityGroupMembershipListCopier.copyFromBuilder(vpcSecurityGroups);
        }

        @Override
        public final Builder vpcSecurityGroups(Collection<VpcSecurityGroupMembership> vpcSecurityGroups) {
            this.vpcSecurityGroups = VpcSecurityGroupMembershipListCopier.copy(vpcSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSecurityGroups(VpcSecurityGroupMembership... vpcSecurityGroups) {
            vpcSecurityGroups(Arrays.asList(vpcSecurityGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSecurityGroups(Consumer<VpcSecurityGroupMembership.Builder>... vpcSecurityGroups) {
            vpcSecurityGroups(Stream.of(vpcSecurityGroups)
                    .map(c -> VpcSecurityGroupMembership.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getWorkgroupName() {
            return workgroupName;
        }

        public final void setWorkgroupName(String workgroupName) {
            this.workgroupName = workgroupName;
        }

        @Override
        public final Builder workgroupName(String workgroupName) {
            this.workgroupName = workgroupName;
            return this;
        }

        @Override
        public EndpointAccess build() {
            return new EndpointAccess(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
