/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An automated backup of a DB cluster. It consists of system backups, transaction logs, and the database cluster
 * properties that existed at the time you deleted the source cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBClusterAutomatedBackup implements SdkPojo, Serializable,
        ToCopyableBuilder<DBClusterAutomatedBackup.Builder, DBClusterAutomatedBackup> {
    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(DBClusterAutomatedBackup::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(DBClusterAutomatedBackup::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<String> DB_CLUSTER_AUTOMATED_BACKUPS_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DBClusterAutomatedBackupsArn")
            .getter(getter(DBClusterAutomatedBackup::dbClusterAutomatedBackupsArn))
            .setter(setter(Builder::dbClusterAutomatedBackupsArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterAutomatedBackupsArn")
                    .build()).build();

    private static final SdkField<String> DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterIdentifier").getter(getter(DBClusterAutomatedBackup::dbClusterIdentifier))
            .setter(setter(Builder::dbClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterIdentifier").build())
            .build();

    private static final SdkField<RestoreWindow> RESTORE_WINDOW_FIELD = SdkField
            .<RestoreWindow> builder(MarshallingType.SDK_POJO).memberName("RestoreWindow")
            .getter(getter(DBClusterAutomatedBackup::restoreWindow)).setter(setter(Builder::restoreWindow))
            .constructor(RestoreWindow::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestoreWindow").build()).build();

    private static final SdkField<String> MASTER_USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterUsername").getter(getter(DBClusterAutomatedBackup::masterUsername))
            .setter(setter(Builder::masterUsername))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUsername").build()).build();

    private static final SdkField<String> DB_CLUSTER_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DbClusterResourceId").getter(getter(DBClusterAutomatedBackup::dbClusterResourceId))
            .setter(setter(Builder::dbClusterResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DbClusterResourceId").build())
            .build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(DBClusterAutomatedBackup::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<String> LICENSE_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LicenseModel").getter(getter(DBClusterAutomatedBackup::licenseModel))
            .setter(setter(Builder::licenseModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseModel").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DBClusterAutomatedBackup::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Boolean> IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IAMDatabaseAuthenticationEnabled")
            .getter(getter(DBClusterAutomatedBackup::iamDatabaseAuthenticationEnabled))
            .setter(setter(Builder::iamDatabaseAuthenticationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IAMDatabaseAuthenticationEnabled")
                    .build()).build();

    private static final SdkField<Instant> CLUSTER_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ClusterCreateTime").getter(getter(DBClusterAutomatedBackup::clusterCreateTime))
            .setter(setter(Builder::clusterCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterCreateTime").build()).build();

    private static final SdkField<Boolean> STORAGE_ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("StorageEncrypted").getter(getter(DBClusterAutomatedBackup::storageEncrypted))
            .setter(setter(Builder::storageEncrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageEncrypted").build()).build();

    private static final SdkField<Integer> ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocatedStorage").getter(getter(DBClusterAutomatedBackup::allocatedStorage))
            .setter(setter(Builder::allocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedStorage").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(DBClusterAutomatedBackup::engineVersion))
            .setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> DB_CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterArn").getter(getter(DBClusterAutomatedBackup::dbClusterArn))
            .setter(setter(Builder::dbClusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterArn").build()).build();

    private static final SdkField<Integer> BACKUP_RETENTION_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BackupRetentionPeriod").getter(getter(DBClusterAutomatedBackup::backupRetentionPeriod))
            .setter(setter(Builder::backupRetentionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupRetentionPeriod").build())
            .build();

    private static final SdkField<String> ENGINE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineMode").getter(getter(DBClusterAutomatedBackup::engineMode)).setter(setter(Builder::engineMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineMode").build()).build();

    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(DBClusterAutomatedBackup::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(DBClusterAutomatedBackup::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(DBClusterAutomatedBackup::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(DBClusterAutomatedBackup::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Iops")
            .getter(getter(DBClusterAutomatedBackup::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENGINE_FIELD, VPC_ID_FIELD,
            DB_CLUSTER_AUTOMATED_BACKUPS_ARN_FIELD, DB_CLUSTER_IDENTIFIER_FIELD, RESTORE_WINDOW_FIELD, MASTER_USERNAME_FIELD,
            DB_CLUSTER_RESOURCE_ID_FIELD, REGION_FIELD, LICENSE_MODEL_FIELD, STATUS_FIELD,
            IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD, CLUSTER_CREATE_TIME_FIELD, STORAGE_ENCRYPTED_FIELD,
            ALLOCATED_STORAGE_FIELD, ENGINE_VERSION_FIELD, DB_CLUSTER_ARN_FIELD, BACKUP_RETENTION_PERIOD_FIELD,
            ENGINE_MODE_FIELD, AVAILABILITY_ZONES_FIELD, PORT_FIELD, KMS_KEY_ID_FIELD, STORAGE_TYPE_FIELD, IOPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String engine;

    private final String vpcId;

    private final String dbClusterAutomatedBackupsArn;

    private final String dbClusterIdentifier;

    private final RestoreWindow restoreWindow;

    private final String masterUsername;

    private final String dbClusterResourceId;

    private final String region;

    private final String licenseModel;

    private final String status;

    private final Boolean iamDatabaseAuthenticationEnabled;

    private final Instant clusterCreateTime;

    private final Boolean storageEncrypted;

    private final Integer allocatedStorage;

    private final String engineVersion;

    private final String dbClusterArn;

    private final Integer backupRetentionPeriod;

    private final String engineMode;

    private final List<String> availabilityZones;

    private final Integer port;

    private final String kmsKeyId;

    private final String storageType;

    private final Integer iops;

    private DBClusterAutomatedBackup(BuilderImpl builder) {
        this.engine = builder.engine;
        this.vpcId = builder.vpcId;
        this.dbClusterAutomatedBackupsArn = builder.dbClusterAutomatedBackupsArn;
        this.dbClusterIdentifier = builder.dbClusterIdentifier;
        this.restoreWindow = builder.restoreWindow;
        this.masterUsername = builder.masterUsername;
        this.dbClusterResourceId = builder.dbClusterResourceId;
        this.region = builder.region;
        this.licenseModel = builder.licenseModel;
        this.status = builder.status;
        this.iamDatabaseAuthenticationEnabled = builder.iamDatabaseAuthenticationEnabled;
        this.clusterCreateTime = builder.clusterCreateTime;
        this.storageEncrypted = builder.storageEncrypted;
        this.allocatedStorage = builder.allocatedStorage;
        this.engineVersion = builder.engineVersion;
        this.dbClusterArn = builder.dbClusterArn;
        this.backupRetentionPeriod = builder.backupRetentionPeriod;
        this.engineMode = builder.engineMode;
        this.availabilityZones = builder.availabilityZones;
        this.port = builder.port;
        this.kmsKeyId = builder.kmsKeyId;
        this.storageType = builder.storageType;
        this.iops = builder.iops;
    }

    /**
     * <p>
     * The name of the database engine for this automated backup.
     * </p>
     * 
     * @return The name of the database engine for this automated backup.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The VPC ID associated with the DB cluster.
     * </p>
     * 
     * @return The VPC ID associated with the DB cluster.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the automated backups.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the automated backups.
     */
    public final String dbClusterAutomatedBackupsArn() {
        return dbClusterAutomatedBackupsArn;
    }

    /**
     * <p>
     * The identifier for the source DB cluster, which can't be changed and which is unique to an Amazon Web Services
     * Region.
     * </p>
     * 
     * @return The identifier for the source DB cluster, which can't be changed and which is unique to an Amazon Web
     *         Services Region.
     */
    public final String dbClusterIdentifier() {
        return dbClusterIdentifier;
    }

    /**
     * Returns the value of the RestoreWindow property for this object.
     * 
     * @return The value of the RestoreWindow property for this object.
     */
    public final RestoreWindow restoreWindow() {
        return restoreWindow;
    }

    /**
     * <p>
     * The master user name of the automated backup.
     * </p>
     * 
     * @return The master user name of the automated backup.
     */
    public final String masterUsername() {
        return masterUsername;
    }

    /**
     * <p>
     * The resource ID for the source DB cluster, which can't be changed and which is unique to an Amazon Web Services
     * Region.
     * </p>
     * 
     * @return The resource ID for the source DB cluster, which can't be changed and which is unique to an Amazon Web
     *         Services Region.
     */
    public final String dbClusterResourceId() {
        return dbClusterResourceId;
    }

    /**
     * <p>
     * The Amazon Web Services Region associated with the automated backup.
     * </p>
     * 
     * @return The Amazon Web Services Region associated with the automated backup.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The license model information for this DB cluster automated backup.
     * </p>
     * 
     * @return The license model information for this DB cluster automated backup.
     */
    public final String licenseModel() {
        return licenseModel;
    }

    /**
     * <p>
     * A list of status information for an automated backup:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>retained</code> - Automated backups for deleted clusters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A list of status information for an automated backup:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>retained</code> - Automated backups for deleted clusters.
     *         </p>
     *         </li>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * True if mapping of Amazon Web Services Identity and Access Management (IAM) accounts to database accounts is
     * enabled, and otherwise false.
     * </p>
     * 
     * @return True if mapping of Amazon Web Services Identity and Access Management (IAM) accounts to database accounts
     *         is enabled, and otherwise false.
     */
    public final Boolean iamDatabaseAuthenticationEnabled() {
        return iamDatabaseAuthenticationEnabled;
    }

    /**
     * <p>
     * The time when the DB cluster was created, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return The time when the DB cluster was created, in Universal Coordinated Time (UTC).
     */
    public final Instant clusterCreateTime() {
        return clusterCreateTime;
    }

    /**
     * <p>
     * Specifies whether the source DB cluster is encrypted.
     * </p>
     * 
     * @return Specifies whether the source DB cluster is encrypted.
     */
    public final Boolean storageEncrypted() {
        return storageEncrypted;
    }

    /**
     * <p>
     * For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated storage size
     * in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because Aurora DB cluster storage
     * size isn't fixed, but instead automatically adjusts as needed.
     * </p>
     * 
     * @return For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated
     *         storage size in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because
     *         Aurora DB cluster storage size isn't fixed, but instead automatically adjusts as needed.
     */
    public final Integer allocatedStorage() {
        return allocatedStorage;
    }

    /**
     * <p>
     * The version of the database engine for the automated backup.
     * </p>
     * 
     * @return The version of the database engine for the automated backup.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the source DB cluster.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the source DB cluster.
     */
    public final String dbClusterArn() {
        return dbClusterArn;
    }

    /**
     * <p>
     * The retention period for the automated backups.
     * </p>
     * 
     * @return The retention period for the automated backups.
     */
    public final Integer backupRetentionPeriod() {
        return backupRetentionPeriod;
    }

    /**
     * <p>
     * The engine mode of the database engine for the automated backup.
     * </p>
     * 
     * @return The engine mode of the database engine for the automated backup.
     */
    public final String engineMode() {
        return engineMode;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Availability Zones where instances in the DB cluster can be created. For information on Amazon Web Services
     * Regions and Availability Zones, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html"
     * >Regions and Availability Zones</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return The Availability Zones where instances in the DB cluster can be created. For information on Amazon Web
     *         Services Regions and Availability Zones, see <a href=
     *         "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html"
     *         >Regions and Availability Zones</a>.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * The port number that the automated backup used for connections.
     * </p>
     * <p>
     * Default: Inherits from the source DB cluster
     * </p>
     * <p>
     * Valid Values: <code>1150-65535</code>
     * </p>
     * 
     * @return The port number that the automated backup used for connections.</p>
     *         <p>
     *         Default: Inherits from the source DB cluster
     *         </p>
     *         <p>
     *         Valid Values: <code>1150-65535</code>
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The Amazon Web Services KMS key ID for an automated backup.
     * </p>
     * <p>
     * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
     * </p>
     * 
     * @return The Amazon Web Services KMS key ID for an automated backup.</p>
     *         <p>
     *         The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS
     *         key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The storage type associated with the DB cluster.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The storage type associated with the DB cluster.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final String storageType() {
        return storageType;
    }

    /**
     * <p>
     * The IOPS (I/O operations per second) value for the automated backup.
     * </p>
     * <p>
     * This setting is only for non-Aurora Multi-AZ DB clusters.
     * </p>
     * 
     * @return The IOPS (I/O operations per second) value for the automated backup.</p>
     *         <p>
     *         This setting is only for non-Aurora Multi-AZ DB clusters.
     */
    public final Integer iops() {
        return iops;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterAutomatedBackupsArn());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(restoreWindow());
        hashCode = 31 * hashCode + Objects.hashCode(masterUsername());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(licenseModel());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(iamDatabaseAuthenticationEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(clusterCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(storageEncrypted());
        hashCode = 31 * hashCode + Objects.hashCode(allocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(backupRetentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(engineMode());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBClusterAutomatedBackup)) {
            return false;
        }
        DBClusterAutomatedBackup other = (DBClusterAutomatedBackup) obj;
        return Objects.equals(engine(), other.engine()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(dbClusterAutomatedBackupsArn(), other.dbClusterAutomatedBackupsArn())
                && Objects.equals(dbClusterIdentifier(), other.dbClusterIdentifier())
                && Objects.equals(restoreWindow(), other.restoreWindow())
                && Objects.equals(masterUsername(), other.masterUsername())
                && Objects.equals(dbClusterResourceId(), other.dbClusterResourceId()) && Objects.equals(region(), other.region())
                && Objects.equals(licenseModel(), other.licenseModel()) && Objects.equals(status(), other.status())
                && Objects.equals(iamDatabaseAuthenticationEnabled(), other.iamDatabaseAuthenticationEnabled())
                && Objects.equals(clusterCreateTime(), other.clusterCreateTime())
                && Objects.equals(storageEncrypted(), other.storageEncrypted())
                && Objects.equals(allocatedStorage(), other.allocatedStorage())
                && Objects.equals(engineVersion(), other.engineVersion()) && Objects.equals(dbClusterArn(), other.dbClusterArn())
                && Objects.equals(backupRetentionPeriod(), other.backupRetentionPeriod())
                && Objects.equals(engineMode(), other.engineMode()) && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones()) && Objects.equals(port(), other.port())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(storageType(), other.storageType())
                && Objects.equals(iops(), other.iops());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DBClusterAutomatedBackup").add("Engine", engine()).add("VpcId", vpcId())
                .add("DBClusterAutomatedBackupsArn", dbClusterAutomatedBackupsArn())
                .add("DBClusterIdentifier", dbClusterIdentifier()).add("RestoreWindow", restoreWindow())
                .add("MasterUsername", masterUsername()).add("DbClusterResourceId", dbClusterResourceId())
                .add("Region", region()).add("LicenseModel", licenseModel()).add("Status", status())
                .add("IAMDatabaseAuthenticationEnabled", iamDatabaseAuthenticationEnabled())
                .add("ClusterCreateTime", clusterCreateTime()).add("StorageEncrypted", storageEncrypted())
                .add("AllocatedStorage", allocatedStorage()).add("EngineVersion", engineVersion())
                .add("DBClusterArn", dbClusterArn()).add("BackupRetentionPeriod", backupRetentionPeriod())
                .add("EngineMode", engineMode()).add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null)
                .add("Port", port()).add("KmsKeyId", kmsKeyId()).add("StorageType", storageType()).add("Iops", iops()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "DBClusterAutomatedBackupsArn":
            return Optional.ofNullable(clazz.cast(dbClusterAutomatedBackupsArn()));
        case "DBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterIdentifier()));
        case "RestoreWindow":
            return Optional.ofNullable(clazz.cast(restoreWindow()));
        case "MasterUsername":
            return Optional.ofNullable(clazz.cast(masterUsername()));
        case "DbClusterResourceId":
            return Optional.ofNullable(clazz.cast(dbClusterResourceId()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "LicenseModel":
            return Optional.ofNullable(clazz.cast(licenseModel()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "IAMDatabaseAuthenticationEnabled":
            return Optional.ofNullable(clazz.cast(iamDatabaseAuthenticationEnabled()));
        case "ClusterCreateTime":
            return Optional.ofNullable(clazz.cast(clusterCreateTime()));
        case "StorageEncrypted":
            return Optional.ofNullable(clazz.cast(storageEncrypted()));
        case "AllocatedStorage":
            return Optional.ofNullable(clazz.cast(allocatedStorage()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "DBClusterArn":
            return Optional.ofNullable(clazz.cast(dbClusterArn()));
        case "BackupRetentionPeriod":
            return Optional.ofNullable(clazz.cast(backupRetentionPeriod()));
        case "EngineMode":
            return Optional.ofNullable(clazz.cast(engineMode()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBClusterAutomatedBackup, T> g) {
        return obj -> g.apply((DBClusterAutomatedBackup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBClusterAutomatedBackup> {
        /**
         * <p>
         * The name of the database engine for this automated backup.
         * </p>
         * 
         * @param engine
         *        The name of the database engine for this automated backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The VPC ID associated with the DB cluster.
         * </p>
         * 
         * @param vpcId
         *        The VPC ID associated with the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the automated backups.
         * </p>
         * 
         * @param dbClusterAutomatedBackupsArn
         *        The Amazon Resource Name (ARN) for the automated backups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterAutomatedBackupsArn(String dbClusterAutomatedBackupsArn);

        /**
         * <p>
         * The identifier for the source DB cluster, which can't be changed and which is unique to an Amazon Web
         * Services Region.
         * </p>
         * 
         * @param dbClusterIdentifier
         *        The identifier for the source DB cluster, which can't be changed and which is unique to an Amazon Web
         *        Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterIdentifier(String dbClusterIdentifier);

        /**
         * Sets the value of the RestoreWindow property for this object.
         *
         * @param restoreWindow
         *        The new value for the RestoreWindow property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restoreWindow(RestoreWindow restoreWindow);

        /**
         * Sets the value of the RestoreWindow property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RestoreWindow.Builder} avoiding the need
         * to create one manually via {@link RestoreWindow#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RestoreWindow.Builder#build()} is called immediately and its
         * result is passed to {@link #restoreWindow(RestoreWindow)}.
         * 
         * @param restoreWindow
         *        a consumer that will call methods on {@link RestoreWindow.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #restoreWindow(RestoreWindow)
         */
        default Builder restoreWindow(Consumer<RestoreWindow.Builder> restoreWindow) {
            return restoreWindow(RestoreWindow.builder().applyMutation(restoreWindow).build());
        }

        /**
         * <p>
         * The master user name of the automated backup.
         * </p>
         * 
         * @param masterUsername
         *        The master user name of the automated backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUsername(String masterUsername);

        /**
         * <p>
         * The resource ID for the source DB cluster, which can't be changed and which is unique to an Amazon Web
         * Services Region.
         * </p>
         * 
         * @param dbClusterResourceId
         *        The resource ID for the source DB cluster, which can't be changed and which is unique to an Amazon Web
         *        Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterResourceId(String dbClusterResourceId);

        /**
         * <p>
         * The Amazon Web Services Region associated with the automated backup.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region associated with the automated backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The license model information for this DB cluster automated backup.
         * </p>
         * 
         * @param licenseModel
         *        The license model information for this DB cluster automated backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseModel(String licenseModel);

        /**
         * <p>
         * A list of status information for an automated backup:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>retained</code> - Automated backups for deleted clusters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        A list of status information for an automated backup:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>retained</code> - Automated backups for deleted clusters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * True if mapping of Amazon Web Services Identity and Access Management (IAM) accounts to database accounts is
         * enabled, and otherwise false.
         * </p>
         * 
         * @param iamDatabaseAuthenticationEnabled
         *        True if mapping of Amazon Web Services Identity and Access Management (IAM) accounts to database
         *        accounts is enabled, and otherwise false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled);

        /**
         * <p>
         * The time when the DB cluster was created, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param clusterCreateTime
         *        The time when the DB cluster was created, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterCreateTime(Instant clusterCreateTime);

        /**
         * <p>
         * Specifies whether the source DB cluster is encrypted.
         * </p>
         * 
         * @param storageEncrypted
         *        Specifies whether the source DB cluster is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageEncrypted(Boolean storageEncrypted);

        /**
         * <p>
         * For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated storage
         * size in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because Aurora DB
         * cluster storage size isn't fixed, but instead automatically adjusts as needed.
         * </p>
         * 
         * @param allocatedStorage
         *        For all database engines except Amazon Aurora, <code>AllocatedStorage</code> specifies the allocated
         *        storage size in gibibytes (GiB). For Aurora, <code>AllocatedStorage</code> always returns 1, because
         *        Aurora DB cluster storage size isn't fixed, but instead automatically adjusts as needed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocatedStorage(Integer allocatedStorage);

        /**
         * <p>
         * The version of the database engine for the automated backup.
         * </p>
         * 
         * @param engineVersion
         *        The version of the database engine for the automated backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the source DB cluster.
         * </p>
         * 
         * @param dbClusterArn
         *        The Amazon Resource Name (ARN) for the source DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterArn(String dbClusterArn);

        /**
         * <p>
         * The retention period for the automated backups.
         * </p>
         * 
         * @param backupRetentionPeriod
         *        The retention period for the automated backups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupRetentionPeriod(Integer backupRetentionPeriod);

        /**
         * <p>
         * The engine mode of the database engine for the automated backup.
         * </p>
         * 
         * @param engineMode
         *        The engine mode of the database engine for the automated backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineMode(String engineMode);

        /**
         * <p>
         * The Availability Zones where instances in the DB cluster can be created. For information on Amazon Web
         * Services Regions and Availability Zones, see <a href=
         * "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html"
         * >Regions and Availability Zones</a>.
         * </p>
         * 
         * @param availabilityZones
         *        The Availability Zones where instances in the DB cluster can be created. For information on Amazon Web
         *        Services Regions and Availability Zones, see <a href=
         *        "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html"
         *        >Regions and Availability Zones</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * The Availability Zones where instances in the DB cluster can be created. For information on Amazon Web
         * Services Regions and Availability Zones, see <a href=
         * "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html"
         * >Regions and Availability Zones</a>.
         * </p>
         * 
         * @param availabilityZones
         *        The Availability Zones where instances in the DB cluster can be created. For information on Amazon Web
         *        Services Regions and Availability Zones, see <a href=
         *        "https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html"
         *        >Regions and Availability Zones</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * The port number that the automated backup used for connections.
         * </p>
         * <p>
         * Default: Inherits from the source DB cluster
         * </p>
         * <p>
         * Valid Values: <code>1150-65535</code>
         * </p>
         * 
         * @param port
         *        The port number that the automated backup used for connections.</p>
         *        <p>
         *        Default: Inherits from the source DB cluster
         *        </p>
         *        <p>
         *        Valid Values: <code>1150-65535</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The Amazon Web Services KMS key ID for an automated backup.
         * </p>
         * <p>
         * The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
         * </p>
         * 
         * @param kmsKeyId
         *        The Amazon Web Services KMS key ID for an automated backup.</p>
         *        <p>
         *        The Amazon Web Services KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the
         *        KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The storage type associated with the DB cluster.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param storageType
         *        The storage type associated with the DB cluster.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The IOPS (I/O operations per second) value for the automated backup.
         * </p>
         * <p>
         * This setting is only for non-Aurora Multi-AZ DB clusters.
         * </p>
         * 
         * @param iops
         *        The IOPS (I/O operations per second) value for the automated backup.</p>
         *        <p>
         *        This setting is only for non-Aurora Multi-AZ DB clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);
    }

    static final class BuilderImpl implements Builder {
        private String engine;

        private String vpcId;

        private String dbClusterAutomatedBackupsArn;

        private String dbClusterIdentifier;

        private RestoreWindow restoreWindow;

        private String masterUsername;

        private String dbClusterResourceId;

        private String region;

        private String licenseModel;

        private String status;

        private Boolean iamDatabaseAuthenticationEnabled;

        private Instant clusterCreateTime;

        private Boolean storageEncrypted;

        private Integer allocatedStorage;

        private String engineVersion;

        private String dbClusterArn;

        private Integer backupRetentionPeriod;

        private String engineMode;

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private Integer port;

        private String kmsKeyId;

        private String storageType;

        private Integer iops;

        private BuilderImpl() {
        }

        private BuilderImpl(DBClusterAutomatedBackup model) {
            engine(model.engine);
            vpcId(model.vpcId);
            dbClusterAutomatedBackupsArn(model.dbClusterAutomatedBackupsArn);
            dbClusterIdentifier(model.dbClusterIdentifier);
            restoreWindow(model.restoreWindow);
            masterUsername(model.masterUsername);
            dbClusterResourceId(model.dbClusterResourceId);
            region(model.region);
            licenseModel(model.licenseModel);
            status(model.status);
            iamDatabaseAuthenticationEnabled(model.iamDatabaseAuthenticationEnabled);
            clusterCreateTime(model.clusterCreateTime);
            storageEncrypted(model.storageEncrypted);
            allocatedStorage(model.allocatedStorage);
            engineVersion(model.engineVersion);
            dbClusterArn(model.dbClusterArn);
            backupRetentionPeriod(model.backupRetentionPeriod);
            engineMode(model.engineMode);
            availabilityZones(model.availabilityZones);
            port(model.port);
            kmsKeyId(model.kmsKeyId);
            storageType(model.storageType);
            iops(model.iops);
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getDbClusterAutomatedBackupsArn() {
            return dbClusterAutomatedBackupsArn;
        }

        public final void setDbClusterAutomatedBackupsArn(String dbClusterAutomatedBackupsArn) {
            this.dbClusterAutomatedBackupsArn = dbClusterAutomatedBackupsArn;
        }

        @Override
        public final Builder dbClusterAutomatedBackupsArn(String dbClusterAutomatedBackupsArn) {
            this.dbClusterAutomatedBackupsArn = dbClusterAutomatedBackupsArn;
            return this;
        }

        public final String getDbClusterIdentifier() {
            return dbClusterIdentifier;
        }

        public final void setDbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
        }

        @Override
        public final Builder dbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
            return this;
        }

        public final RestoreWindow.Builder getRestoreWindow() {
            return restoreWindow != null ? restoreWindow.toBuilder() : null;
        }

        public final void setRestoreWindow(RestoreWindow.BuilderImpl restoreWindow) {
            this.restoreWindow = restoreWindow != null ? restoreWindow.build() : null;
        }

        @Override
        public final Builder restoreWindow(RestoreWindow restoreWindow) {
            this.restoreWindow = restoreWindow;
            return this;
        }

        public final String getMasterUsername() {
            return masterUsername;
        }

        public final void setMasterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
        }

        @Override
        public final Builder masterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
            return this;
        }

        public final String getDbClusterResourceId() {
            return dbClusterResourceId;
        }

        public final void setDbClusterResourceId(String dbClusterResourceId) {
            this.dbClusterResourceId = dbClusterResourceId;
        }

        @Override
        public final Builder dbClusterResourceId(String dbClusterResourceId) {
            this.dbClusterResourceId = dbClusterResourceId;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final String getLicenseModel() {
            return licenseModel;
        }

        public final void setLicenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
        }

        @Override
        public final Builder licenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Boolean getIamDatabaseAuthenticationEnabled() {
            return iamDatabaseAuthenticationEnabled;
        }

        public final void setIamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
        }

        @Override
        public final Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
            return this;
        }

        public final Instant getClusterCreateTime() {
            return clusterCreateTime;
        }

        public final void setClusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
        }

        @Override
        public final Builder clusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
            return this;
        }

        public final Boolean getStorageEncrypted() {
            return storageEncrypted;
        }

        public final void setStorageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
        }

        @Override
        public final Builder storageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
            return this;
        }

        public final Integer getAllocatedStorage() {
            return allocatedStorage;
        }

        public final void setAllocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
        }

        @Override
        public final Builder allocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getDbClusterArn() {
            return dbClusterArn;
        }

        public final void setDbClusterArn(String dbClusterArn) {
            this.dbClusterArn = dbClusterArn;
        }

        @Override
        public final Builder dbClusterArn(String dbClusterArn) {
            this.dbClusterArn = dbClusterArn;
            return this;
        }

        public final Integer getBackupRetentionPeriod() {
            return backupRetentionPeriod;
        }

        public final void setBackupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
        }

        @Override
        public final Builder backupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
            return this;
        }

        public final String getEngineMode() {
            return engineMode;
        }

        public final void setEngineMode(String engineMode) {
            this.engineMode = engineMode;
        }

        @Override
        public final Builder engineMode(String engineMode) {
            this.engineMode = engineMode;
            return this;
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final Integer getIops() {
            return iops;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        @Override
        public DBClusterAutomatedBackup build() {
            return new DBClusterAutomatedBackup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
