/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.proton.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.proton.ProtonClient;
import software.amazon.awssdk.services.proton.model.GetComponentRequest;
import software.amazon.awssdk.services.proton.model.GetComponentResponse;
import software.amazon.awssdk.services.proton.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.proton.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.proton.model.GetEnvironmentTemplateVersionRequest;
import software.amazon.awssdk.services.proton.model.GetEnvironmentTemplateVersionResponse;
import software.amazon.awssdk.services.proton.model.GetServiceInstanceRequest;
import software.amazon.awssdk.services.proton.model.GetServiceInstanceResponse;
import software.amazon.awssdk.services.proton.model.GetServiceRequest;
import software.amazon.awssdk.services.proton.model.GetServiceResponse;
import software.amazon.awssdk.services.proton.model.GetServiceTemplateVersionRequest;
import software.amazon.awssdk.services.proton.model.GetServiceTemplateVersionResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface ProtonWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param getComponentRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeleted(GetComponentRequest getComponentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetComponentRequest#builder()}
     *
     * @param getComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeleted(
            Consumer<GetComponentRequest.Builder> getComponentRequest) {
        return waitUntilComponentDeleted(GetComponentRequest.builder().applyMutation(getComponentRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param getComponentRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeleted(GetComponentRequest getComponentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeleted(
            Consumer<GetComponentRequest.Builder> getComponentRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilComponentDeleted(GetComponentRequest.builder().applyMutation(getComponentRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getComponentRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeployed(GetComponentRequest getComponentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetComponentRequest#builder()}
     *
     * @param getComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeployed(
            Consumer<GetComponentRequest.Builder> getComponentRequest) {
        return waitUntilComponentDeployed(GetComponentRequest.builder().applyMutation(getComponentRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getComponentRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeployed(GetComponentRequest getComponentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getComponent} API until the desired condition {@code ComponentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getComponentRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetComponentResponse> waitUntilComponentDeployed(
            Consumer<GetComponentRequest.Builder> getComponentRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilComponentDeployed(GetComponentRequest.builder().applyMutation(getComponentRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getEnvironment} API until the desired condition {@code EnvironmentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getEnvironmentRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentDeployed(GetEnvironmentRequest getEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getEnvironment} API until the desired condition {@code EnvironmentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetEnvironmentRequest#builder()}
     *
     * @param getEnvironmentRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentDeployed(
            Consumer<GetEnvironmentRequest.Builder> getEnvironmentRequest) {
        return waitUntilEnvironmentDeployed(GetEnvironmentRequest.builder().applyMutation(getEnvironmentRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getEnvironment} API until the desired condition {@code EnvironmentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getEnvironmentRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentDeployed(GetEnvironmentRequest getEnvironmentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getEnvironment} API until the desired condition {@code EnvironmentDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getEnvironmentRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentDeployed(
            Consumer<GetEnvironmentRequest.Builder> getEnvironmentRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilEnvironmentDeployed(GetEnvironmentRequest.builder().applyMutation(getEnvironmentRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getEnvironmentTemplateVersion} API until the desired condition
     * {@code EnvironmentTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state
     *
     * @param getEnvironmentTemplateVersionRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentTemplateVersionResponse> waitUntilEnvironmentTemplateVersionRegistered(
            GetEnvironmentTemplateVersionRequest getEnvironmentTemplateVersionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getEnvironmentTemplateVersion} API until the desired condition
     * {@code EnvironmentTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetEnvironmentTemplateVersionRequest#builder()}
     *
     * @param getEnvironmentTemplateVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentTemplateVersionResponse> waitUntilEnvironmentTemplateVersionRegistered(
            Consumer<GetEnvironmentTemplateVersionRequest.Builder> getEnvironmentTemplateVersionRequest) {
        return waitUntilEnvironmentTemplateVersionRegistered(GetEnvironmentTemplateVersionRequest.builder()
                .applyMutation(getEnvironmentTemplateVersionRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getEnvironmentTemplateVersion} API until the desired condition
     * {@code EnvironmentTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state
     *
     * @param getEnvironmentTemplateVersionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentTemplateVersionResponse> waitUntilEnvironmentTemplateVersionRegistered(
            GetEnvironmentTemplateVersionRequest getEnvironmentTemplateVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getEnvironmentTemplateVersion} API until the desired condition
     * {@code EnvironmentTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getEnvironmentTemplateVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetEnvironmentTemplateVersionResponse> waitUntilEnvironmentTemplateVersionRegistered(
            Consumer<GetEnvironmentTemplateVersionRequest.Builder> getEnvironmentTemplateVersionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilEnvironmentTemplateVersionRegistered(
                GetEnvironmentTemplateVersionRequest.builder().applyMutation(getEnvironmentTemplateVersionRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceCreated(GetServiceRequest getServiceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetServiceRequest#builder()}
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceCreated(Consumer<GetServiceRequest.Builder> getServiceRequest) {
        return waitUntilServiceCreated(GetServiceRequest.builder().applyMutation(getServiceRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceCreated(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceCreated} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceCreated(Consumer<GetServiceRequest.Builder> getServiceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilServiceCreated(GetServiceRequest.builder().applyMutation(getServiceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceDeleted(GetServiceRequest getServiceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetServiceRequest#builder()}
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceDeleted(Consumer<GetServiceRequest.Builder> getServiceRequest) {
        return waitUntilServiceDeleted(GetServiceRequest.builder().applyMutation(getServiceRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceDeleted(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceDeleted} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceDeleted(Consumer<GetServiceRequest.Builder> getServiceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilServiceDeleted(GetServiceRequest.builder().applyMutation(getServiceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getServiceInstance} API until the desired condition {@code ServiceInstanceDeployed} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getServiceInstanceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceInstanceResponse> waitUntilServiceInstanceDeployed(
            GetServiceInstanceRequest getServiceInstanceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getServiceInstance} API until the desired condition {@code ServiceInstanceDeployed} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetServiceInstanceRequest#builder()}
     *
     * @param getServiceInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceInstanceResponse> waitUntilServiceInstanceDeployed(
            Consumer<GetServiceInstanceRequest.Builder> getServiceInstanceRequest) {
        return waitUntilServiceInstanceDeployed(GetServiceInstanceRequest.builder().applyMutation(getServiceInstanceRequest)
                .build());
    }

    /**
     * Polls {@link ProtonClient#getServiceInstance} API until the desired condition {@code ServiceInstanceDeployed} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getServiceInstanceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceInstanceResponse> waitUntilServiceInstanceDeployed(
            GetServiceInstanceRequest getServiceInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getServiceInstance} API until the desired condition {@code ServiceInstanceDeployed} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getServiceInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceInstanceResponse> waitUntilServiceInstanceDeployed(
            Consumer<GetServiceInstanceRequest.Builder> getServiceInstanceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilServiceInstanceDeployed(GetServiceInstanceRequest.builder().applyMutation(getServiceInstanceRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServicePipelineDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServicePipelineDeployed(GetServiceRequest getServiceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServicePipelineDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetServiceRequest#builder()}
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServicePipelineDeployed(
            Consumer<GetServiceRequest.Builder> getServiceRequest) {
        return waitUntilServicePipelineDeployed(GetServiceRequest.builder().applyMutation(getServiceRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServicePipelineDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServicePipelineDeployed(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServicePipelineDeployed} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServicePipelineDeployed(
            Consumer<GetServiceRequest.Builder> getServiceRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilServicePipelineDeployed(GetServiceRequest.builder().applyMutation(getServiceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getServiceTemplateVersion} API until the desired condition
     * {@code ServiceTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state
     *
     * @param getServiceTemplateVersionRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceTemplateVersionResponse> waitUntilServiceTemplateVersionRegistered(
            GetServiceTemplateVersionRequest getServiceTemplateVersionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getServiceTemplateVersion} API until the desired condition
     * {@code ServiceTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetServiceTemplateVersionRequest#builder()}
     *
     * @param getServiceTemplateVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceTemplateVersionResponse> waitUntilServiceTemplateVersionRegistered(
            Consumer<GetServiceTemplateVersionRequest.Builder> getServiceTemplateVersionRequest) {
        return waitUntilServiceTemplateVersionRegistered(GetServiceTemplateVersionRequest.builder()
                .applyMutation(getServiceTemplateVersionRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getServiceTemplateVersion} API until the desired condition
     * {@code ServiceTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state
     *
     * @param getServiceTemplateVersionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceTemplateVersionResponse> waitUntilServiceTemplateVersionRegistered(
            GetServiceTemplateVersionRequest getServiceTemplateVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getServiceTemplateVersion} API until the desired condition
     * {@code ServiceTemplateVersionRegistered} is met, or until it is determined that the resource will never enter
     * into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getServiceTemplateVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceTemplateVersionResponse> waitUntilServiceTemplateVersionRegistered(
            Consumer<GetServiceTemplateVersionRequest.Builder> getServiceTemplateVersionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilServiceTemplateVersionRegistered(
                GetServiceTemplateVersionRequest.builder().applyMutation(getServiceTemplateVersionRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceUpdated} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceUpdated(GetServiceRequest getServiceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceUpdated} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetServiceRequest#builder()}
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceUpdated(Consumer<GetServiceRequest.Builder> getServiceRequest) {
        return waitUntilServiceUpdated(GetServiceRequest.builder().applyMutation(getServiceRequest).build());
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceUpdated} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getServiceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceUpdated(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ProtonClient#getService} API until the desired condition {@code ServiceUpdated} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getServiceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<GetServiceResponse> waitUntilServiceUpdated(Consumer<GetServiceRequest.Builder> getServiceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilServiceUpdated(GetServiceRequest.builder().applyMutation(getServiceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link ProtonWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultProtonWaiter.builder();
    }

    /**
     * Create an instance of {@link ProtonWaiter} with the default configuration.
     * <p>
     * <b>A default {@link ProtonClient} will be created to poll resources. It is recommended to share a single instance
     * of the waiter created via this method. If it is not desirable to share a waiter instance, invoke {@link #close()}
     * to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link ProtonWaiter}
     */
    static ProtonWaiter create() {
        return DefaultProtonWaiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link ProtonClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(ProtonClient client);

        /**
         * Builds an instance of {@link ProtonWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link ProtonWaiter}
         */
        ProtonWaiter build();
    }
}
