/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pricing;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pricing.model.AccessDeniedException;
import software.amazon.awssdk.services.pricing.model.DescribeServicesRequest;
import software.amazon.awssdk.services.pricing.model.DescribeServicesResponse;
import software.amazon.awssdk.services.pricing.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.pricing.model.GetAttributeValuesRequest;
import software.amazon.awssdk.services.pricing.model.GetAttributeValuesResponse;
import software.amazon.awssdk.services.pricing.model.GetPriceListFileUrlRequest;
import software.amazon.awssdk.services.pricing.model.GetPriceListFileUrlResponse;
import software.amazon.awssdk.services.pricing.model.GetProductsRequest;
import software.amazon.awssdk.services.pricing.model.GetProductsResponse;
import software.amazon.awssdk.services.pricing.model.InternalErrorException;
import software.amazon.awssdk.services.pricing.model.InvalidNextTokenException;
import software.amazon.awssdk.services.pricing.model.InvalidParameterException;
import software.amazon.awssdk.services.pricing.model.ListPriceListsRequest;
import software.amazon.awssdk.services.pricing.model.ListPriceListsResponse;
import software.amazon.awssdk.services.pricing.model.NotFoundException;
import software.amazon.awssdk.services.pricing.model.PricingException;
import software.amazon.awssdk.services.pricing.transform.DescribeServicesRequestMarshaller;
import software.amazon.awssdk.services.pricing.transform.GetAttributeValuesRequestMarshaller;
import software.amazon.awssdk.services.pricing.transform.GetPriceListFileUrlRequestMarshaller;
import software.amazon.awssdk.services.pricing.transform.GetProductsRequestMarshaller;
import software.amazon.awssdk.services.pricing.transform.ListPriceListsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PricingAsyncClient}.
 *
 * @see PricingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPricingAsyncClient implements PricingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPricingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PricingServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPricingAsyncClient(PricingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns the metadata for one service or a list of the metadata for all services. Use this without a service code
     * to get the service codes for all services. Use it with a service code, such as <code>AmazonEC2</code>, to get
     * information specific to that service, such as the attribute names available for that service. For example, some
     * of the attribute names available for EC2 are <code>volumeType</code>, <code>maxIopsVolume</code>,
     * <code>operation</code>, <code>locationType</code>, and <code>instanceCapacity10xlarge</code>.
     * </p>
     *
     * @param describeServicesRequest
     * @return A Java Future containing the result of the DescribeServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters had an invalid value.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>NotFoundException The requested resource can't be found.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ExpiredNextTokenException The pagination token expired. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PricingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PricingAsyncClient.DescribeServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServicesResponse> describeServices(DescribeServicesRequest describeServicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pricing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServicesRequest, DescribeServicesResponse>()
                            .withOperationName("DescribeServices")
                            .withMarshaller(new DescribeServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeServicesRequest));
            CompletableFuture<DescribeServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of attribute values. Attributes are similar to the details in a Price List API offer file. For a
     * list of available attributes, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/reading-an-offer.html#pps-defs">Offer File
     * Definitions</a> in the <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-what-is.html">Billing and Cost
     * Management User Guide</a>.
     * </p>
     *
     * @param getAttributeValuesRequest
     * @return A Java Future containing the result of the GetAttributeValues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters had an invalid value.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>NotFoundException The requested resource can't be found.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ExpiredNextTokenException The pagination token expired. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PricingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PricingAsyncClient.GetAttributeValues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetAttributeValues" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAttributeValuesResponse> getAttributeValues(GetAttributeValuesRequest getAttributeValuesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAttributeValuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pricing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAttributeValues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAttributeValuesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAttributeValuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAttributeValuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAttributeValuesRequest, GetAttributeValuesResponse>()
                            .withOperationName("GetAttributeValues")
                            .withMarshaller(new GetAttributeValuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAttributeValuesRequest));
            CompletableFuture<GetAttributeValuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <i> <b>This feature is in preview release and is subject to change. Your use of Amazon Web Services Price List
     * API is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">Amazon Web Services Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * <p>
     * This returns the URL that you can retrieve your Price List file from. This URL is based on the
     * <code>PriceListArn</code> and <code>FileFormat</code> that you retrieve from the <a
     * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_pricing_ListPriceLists.html">
     * <code>ListPriceLists</code> </a> response.
     * </p>
     *
     * @param getPriceListFileUrlRequest
     * @return A Java Future containing the result of the GetPriceListFileUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters had an invalid value.</li>
     *         <li>NotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException General authentication failure. The request wasn't signed correctly.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PricingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PricingAsyncClient.GetPriceListFileUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetPriceListFileUrl" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPriceListFileUrlResponse> getPriceListFileUrl(
            GetPriceListFileUrlRequest getPriceListFileUrlRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPriceListFileUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pricing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPriceListFileUrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPriceListFileUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPriceListFileUrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPriceListFileUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPriceListFileUrlRequest, GetPriceListFileUrlResponse>()
                            .withOperationName("GetPriceListFileUrl")
                            .withMarshaller(new GetPriceListFileUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPriceListFileUrlRequest));
            CompletableFuture<GetPriceListFileUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all products that match the filter criteria.
     * </p>
     *
     * @param getProductsRequest
     * @return A Java Future containing the result of the GetProducts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters had an invalid value.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>NotFoundException The requested resource can't be found.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ExpiredNextTokenException The pagination token expired. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PricingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PricingAsyncClient.GetProducts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/GetProducts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProductsResponse> getProducts(GetProductsRequest getProductsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProductsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pricing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProducts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProductsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProductsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProductsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProductsRequest, GetProductsResponse>()
                            .withOperationName("GetProducts").withMarshaller(new GetProductsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProductsRequest));
            CompletableFuture<GetProductsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <i> <b>This feature is in preview release and is subject to change. Your use of Amazon Web Services Price List
     * API is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">Amazon Web Services Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * <p>
     * This returns a list of Price List references that the requester if authorized to view, given a
     * <code>ServiceCode</code>, <code>CurrencyCode</code>, and an <code>EffectiveDate</code>. Use without a
     * <code>RegionCode</code> filter to list Price List references from all available Amazon Web Services Regions. Use
     * with a <code>RegionCode</code> filter to get the Price List reference that's specific to a specific Amazon Web
     * Services Region. You can use the <code>PriceListArn</code> from the response to get your preferred Price List
     * files through the <a
     * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_pricing_GetPriceListFileUrl.html">
     * <code>GetPriceListFileUrl</code> </a> API.
     * </p>
     *
     * @param listPriceListsRequest
     * @return A Java Future containing the result of the ListPriceLists operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters had an invalid value.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>NotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException General authentication failure. The request wasn't signed correctly.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ExpiredNextTokenException The pagination token expired. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PricingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PricingAsyncClient.ListPriceLists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pricing-2017-10-15/ListPriceLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPriceListsResponse> listPriceLists(ListPriceListsRequest listPriceListsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPriceListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pricing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPriceLists");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPriceListsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPriceListsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPriceListsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPriceListsRequest, ListPriceListsResponse>()
                            .withOperationName("ListPriceLists")
                            .withMarshaller(new ListPriceListsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPriceListsRequest));
            CompletableFuture<ListPriceListsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PricingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PricingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredNextTokenException")
                                .exceptionBuilderSupplier(ExpiredNextTokenException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
