/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpointsmsvoice;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pinpointsmsvoice.internal.PinpointSmsVoiceServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pinpointsmsvoice.model.AlreadyExistsException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.BadRequestException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.CreateConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.CreateConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.CreateConfigurationSetRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.CreateConfigurationSetResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.DeleteConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.DeleteConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.DeleteConfigurationSetRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.DeleteConfigurationSetResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.GetConfigurationSetEventDestinationsRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.GetConfigurationSetEventDestinationsResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.InternalServiceErrorException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.LimitExceededException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.ListConfigurationSetsRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.ListConfigurationSetsResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.NotFoundException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.PinpointSmsVoiceException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.SendVoiceMessageRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.SendVoiceMessageResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.model.TooManyRequestsException;
import software.amazon.awssdk.services.pinpointsmsvoice.model.UpdateConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.pinpointsmsvoice.model.UpdateConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.CreateConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.CreateConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.DeleteConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.DeleteConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.GetConfigurationSetEventDestinationsRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.ListConfigurationSetsRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.SendVoiceMessageRequestMarshaller;
import software.amazon.awssdk.services.pinpointsmsvoice.transform.UpdateConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PinpointSmsVoiceAsyncClient}.
 *
 * @see PinpointSmsVoiceAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPinpointSmsVoiceAsyncClient implements PinpointSmsVoiceAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPinpointSmsVoiceAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPinpointSmsVoiceAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Create a new configuration set. After you create the configuration set, you can add one or more event
     * destinations to it.
     *
     * @param createConfigurationSetRequest
     *        A request to create a new configuration set.
     * @return A Java Future containing the result of the CreateConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>LimitExceededException LimitExceededException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>AlreadyExistsException AlreadyExistsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.CreateConfigurationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/CreateConfigurationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationSetResponse> createConfigurationSet(
            CreateConfigurationSetRequest createConfigurationSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationSetRequest, CreateConfigurationSetResponse>()
                            .withOperationName("CreateConfigurationSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationSetRequest));
            CompletableFuture<CreateConfigurationSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new event destination in a configuration set.
     *
     * @param createConfigurationSetEventDestinationRequest
     *        Create a new event destination in a configuration set.
     * @return A Java Future containing the result of the CreateConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>LimitExceededException LimitExceededException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>NotFoundException NotFoundException</li>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>AlreadyExistsException AlreadyExistsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.CreateConfigurationSetEventDestination
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/CreateConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationSetEventDestinationResponse> createConfigurationSetEventDestination(
            CreateConfigurationSetEventDestinationRequest createConfigurationSetEventDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationSetEventDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createConfigurationSetEventDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationSetEventDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationSetEventDestinationRequest, CreateConfigurationSetEventDestinationResponse>()
                            .withOperationName("CreateConfigurationSetEventDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationSetEventDestinationRequest));
            CompletableFuture<CreateConfigurationSetEventDestinationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an existing configuration set.
     *
     * @param deleteConfigurationSetRequest
     * @return A Java Future containing the result of the DeleteConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException NotFoundException</li>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.DeleteConfigurationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/DeleteConfigurationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationSetResponse> deleteConfigurationSet(
            DeleteConfigurationSetRequest deleteConfigurationSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigurationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationSetRequest, DeleteConfigurationSetResponse>()
                            .withOperationName("DeleteConfigurationSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigurationSetRequest));
            CompletableFuture<DeleteConfigurationSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an event destination in a configuration set.
     *
     * @param deleteConfigurationSetEventDestinationRequest
     * @return A Java Future containing the result of the DeleteConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException NotFoundException</li>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.DeleteConfigurationSetEventDestination
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/DeleteConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationSetEventDestinationResponse> deleteConfigurationSetEventDestination(
            DeleteConfigurationSetEventDestinationRequest deleteConfigurationSetEventDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationSetEventDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteConfigurationSetEventDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationSetEventDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationSetEventDestinationRequest, DeleteConfigurationSetEventDestinationResponse>()
                            .withOperationName("DeleteConfigurationSetEventDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigurationSetEventDestinationRequest));
            CompletableFuture<DeleteConfigurationSetEventDestinationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Obtain information about an event destination, including the types of events it reports, the Amazon Resource Name
     * (ARN) of the destination, and the name of the event destination.
     *
     * @param getConfigurationSetEventDestinationsRequest
     * @return A Java Future containing the result of the GetConfigurationSetEventDestinations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException NotFoundException</li>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.GetConfigurationSetEventDestinations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/GetConfigurationSetEventDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationSetEventDestinationsResponse> getConfigurationSetEventDestinations(
            GetConfigurationSetEventDestinationsRequest getConfigurationSetEventDestinationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationSetEventDestinationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getConfigurationSetEventDestinationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationSetEventDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationSetEventDestinationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetConfigurationSetEventDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationSetEventDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationSetEventDestinationsRequest, GetConfigurationSetEventDestinationsResponse>()
                            .withOperationName("GetConfigurationSetEventDestinations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConfigurationSetEventDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConfigurationSetEventDestinationsRequest));
            CompletableFuture<GetConfigurationSetEventDestinationsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List all of the configuration sets associated with your Amazon Pinpoint account in the current region.
     *
     * @param listConfigurationSetsRequest
     * @return A Java Future containing the result of the ListConfigurationSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.ListConfigurationSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/ListConfigurationSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationSetsResponse> listConfigurationSets(
            ListConfigurationSetsRequest listConfigurationSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationSetsRequest, ListConfigurationSetsResponse>()
                            .withOperationName("ListConfigurationSets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConfigurationSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigurationSetsRequest));
            CompletableFuture<ListConfigurationSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new voice message and send it to a recipient's phone number.
     *
     * @param sendVoiceMessageRequest
     *        SendVoiceMessageRequest
     * @return A Java Future containing the result of the SendVoiceMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.SendVoiceMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/SendVoiceMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendVoiceMessageResponse> sendVoiceMessage(SendVoiceMessageRequest sendVoiceMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendVoiceMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendVoiceMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendVoiceMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendVoiceMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendVoiceMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendVoiceMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendVoiceMessageRequest, SendVoiceMessageResponse>()
                            .withOperationName("SendVoiceMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendVoiceMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendVoiceMessageRequest));
            CompletableFuture<SendVoiceMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Update an event destination in a configuration set. An event destination is a location that you publish
     * information about your voice calls to. For example, you can log an event to an Amazon CloudWatch destination when
     * a call fails.
     *
     * @param updateConfigurationSetEventDestinationRequest
     *        UpdateConfigurationSetEventDestinationRequest
     * @return A Java Future containing the result of the UpdateConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException NotFoundException</li>
     *         <li>TooManyRequestsException TooManyRequestsException</li>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServiceErrorException InternalServiceErrorException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointSmsVoiceException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PinpointSmsVoiceAsyncClient.UpdateConfigurationSetEventDestination
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-2018-09-05/UpdateConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationSetEventDestinationResponse> updateConfigurationSetEventDestination(
            UpdateConfigurationSetEventDestinationRequest updateConfigurationSetEventDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationSetEventDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateConfigurationSetEventDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint SMS Voice");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationSetEventDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationSetEventDestinationRequest, UpdateConfigurationSetEventDestinationResponse>()
                            .withOperationName("UpdateConfigurationSetEventDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigurationSetEventDestinationRequest));
            CompletableFuture<UpdateConfigurationSetEventDestinationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PinpointSmsVoiceServiceClientConfiguration serviceClientConfiguration() {
        return new PinpointSmsVoiceServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PinpointSmsVoiceException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(412).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        PinpointSmsVoiceServiceClientConfigurationBuilder serviceConfigBuilder = new PinpointSmsVoiceServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
