/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFilterRequest extends PersonalizeRequest implements
        ToCopyableBuilder<CreateFilterRequest.Builder, CreateFilterRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CreateFilterRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DATASET_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("datasetGroupArn").getter(getter(CreateFilterRequest::datasetGroupArn))
            .setter(setter(Builder::datasetGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("datasetGroupArn").build()).build();

    private static final SdkField<String> FILTER_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("filterExpression").getter(getter(CreateFilterRequest::filterExpression))
            .setter(setter(Builder::filterExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filterExpression").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(CreateFilterRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            DATASET_GROUP_ARN_FIELD, FILTER_EXPRESSION_FIELD, TAGS_FIELD));

    private final String name;

    private final String datasetGroupArn;

    private final String filterExpression;

    private final List<Tag> tags;

    private CreateFilterRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.datasetGroupArn = builder.datasetGroupArn;
        this.filterExpression = builder.filterExpression;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the filter to create.
     * </p>
     * 
     * @return The name of the filter to create.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the dataset group that the filter will belong to.
     * </p>
     * 
     * @return The ARN of the dataset group that the filter will belong to.
     */
    public final String datasetGroupArn() {
        return datasetGroupArn;
    }

    /**
     * <p>
     * The filter expression defines which items are included or excluded from recommendations. Filter expression must
     * follow specific format rules. For information about filter expression structure and syntax, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/filter-expressions.html">Filter expressions</a>.
     * </p>
     * 
     * @return The filter expression defines which items are included or excluded from recommendations. Filter
     *         expression must follow specific format rules. For information about filter expression structure and
     *         syntax, see <a href="https://docs.aws.amazon.com/personalize/latest/dg/filter-expressions.html">Filter
     *         expressions</a>.
     */
    public final String filterExpression() {
        return filterExpression;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a> to apply to
     * the filter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a> to
     *         apply to the filter.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(datasetGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(filterExpression());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFilterRequest)) {
            return false;
        }
        CreateFilterRequest other = (CreateFilterRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(datasetGroupArn(), other.datasetGroupArn())
                && Objects.equals(filterExpression(), other.filterExpression()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFilterRequest").add("Name", name()).add("DatasetGroupArn", datasetGroupArn())
                .add("FilterExpression", filterExpression() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "datasetGroupArn":
            return Optional.ofNullable(clazz.cast(datasetGroupArn()));
        case "filterExpression":
            return Optional.ofNullable(clazz.cast(filterExpression()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFilterRequest, T> g) {
        return obj -> g.apply((CreateFilterRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends PersonalizeRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateFilterRequest> {
        /**
         * <p>
         * The name of the filter to create.
         * </p>
         * 
         * @param name
         *        The name of the filter to create.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the dataset group that the filter will belong to.
         * </p>
         * 
         * @param datasetGroupArn
         *        The ARN of the dataset group that the filter will belong to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetGroupArn(String datasetGroupArn);

        /**
         * <p>
         * The filter expression defines which items are included or excluded from recommendations. Filter expression
         * must follow specific format rules. For information about filter expression structure and syntax, see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/filter-expressions.html">Filter expressions</a>.
         * </p>
         * 
         * @param filterExpression
         *        The filter expression defines which items are included or excluded from recommendations. Filter
         *        expression must follow specific format rules. For information about filter expression structure and
         *        syntax, see <a href="https://docs.aws.amazon.com/personalize/latest/dg/filter-expressions.html">Filter
         *        expressions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterExpression(String filterExpression);

        /**
         * <p>
         * A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a> to
         * apply to the filter.
         * </p>
         * 
         * @param tags
         *        A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a>
         *        to apply to the filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a> to
         * apply to the filter.
         * </p>
         * 
         * @param tags
         *        A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a>
         *        to apply to the filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of <a href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a> to
         * apply to the filter.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.personalize.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.personalize.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.personalize.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.personalize.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends PersonalizeRequest.BuilderImpl implements Builder {
        private String name;

        private String datasetGroupArn;

        private String filterExpression;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFilterRequest model) {
            super(model);
            name(model.name);
            datasetGroupArn(model.datasetGroupArn);
            filterExpression(model.filterExpression);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDatasetGroupArn() {
            return datasetGroupArn;
        }

        public final void setDatasetGroupArn(String datasetGroupArn) {
            this.datasetGroupArn = datasetGroupArn;
        }

        @Override
        public final Builder datasetGroupArn(String datasetGroupArn) {
            this.datasetGroupArn = datasetGroupArn;
            return this;
        }

        public final String getFilterExpression() {
            return filterExpression;
        }

        public final void setFilterExpression(String filterExpression) {
            this.filterExpression = filterExpression;
        }

        @Override
        public final Builder filterExpression(String filterExpression) {
            this.filterExpression = filterExpression;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateFilterRequest build() {
            return new CreateFilterRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
