/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.outposts.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about compute hardware assets.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComputeAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<ComputeAttributes.Builder, ComputeAttributes> {
    private static final SdkField<String> HOST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("HostId")
            .getter(getter(ComputeAttributes::hostId)).setter(setter(Builder::hostId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(ComputeAttributes::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<List<String>> INSTANCE_FAMILIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceFamilies")
            .getter(getter(ComputeAttributes::instanceFamilies))
            .setter(setter(Builder::instanceFamilies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFamilies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOST_ID_FIELD, STATE_FIELD,
            INSTANCE_FAMILIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String hostId;

    private final String state;

    private final List<String> instanceFamilies;

    private ComputeAttributes(BuilderImpl builder) {
        this.hostId = builder.hostId;
        this.state = builder.state;
        this.instanceFamilies = builder.instanceFamilies;
    }

    /**
     * <p>
     * The host ID of the Dedicated Host on the asset.
     * </p>
     * 
     * @return The host ID of the Dedicated Host on the asset.
     */
    public final String hostId() {
        return hostId;
    }

    /**
     * <p>
     * The state.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACTIVE - The asset is available and can provide capacity for new compute resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources. Existing
     * compute resources on the asset are not affected.
     * </p>
     * </li>
     * <li>
     * <p>
     * RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is reduced.
     * Amazon Web Services sends notifications for resources that must be stopped before the asset can be replaced.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ComputeAssetState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACTIVE - The asset is available and can provide capacity for new compute resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources.
     *         Existing compute resources on the asset are not affected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is
     *         reduced. Amazon Web Services sends notifications for resources that must be stopped before the asset can
     *         be replaced.
     *         </p>
     *         </li>
     * @see ComputeAssetState
     */
    public final ComputeAssetState state() {
        return ComputeAssetState.fromValue(state);
    }

    /**
     * <p>
     * The state.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ACTIVE - The asset is available and can provide capacity for new compute resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources. Existing
     * compute resources on the asset are not affected.
     * </p>
     * </li>
     * <li>
     * <p>
     * RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is reduced.
     * Amazon Web Services sends notifications for resources that must be stopped before the asset can be replaced.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ComputeAssetState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ACTIVE - The asset is available and can provide capacity for new compute resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources.
     *         Existing compute resources on the asset are not affected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is
     *         reduced. Amazon Web Services sends notifications for resources that must be stopped before the asset can
     *         be replaced.
     *         </p>
     *         </li>
     * @see ComputeAssetState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceFamilies property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceFamilies() {
        return instanceFamilies != null && !(instanceFamilies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the names of instance families that are currently associated with a given asset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceFamilies} method.
     * </p>
     * 
     * @return A list of the names of instance families that are currently associated with a given asset.
     */
    public final List<String> instanceFamilies() {
        return instanceFamilies;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hostId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceFamilies() ? instanceFamilies() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComputeAttributes)) {
            return false;
        }
        ComputeAttributes other = (ComputeAttributes) obj;
        return Objects.equals(hostId(), other.hostId()) && Objects.equals(stateAsString(), other.stateAsString())
                && hasInstanceFamilies() == other.hasInstanceFamilies()
                && Objects.equals(instanceFamilies(), other.instanceFamilies());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComputeAttributes").add("HostId", hostId()).add("State", stateAsString())
                .add("InstanceFamilies", hasInstanceFamilies() ? instanceFamilies() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HostId":
            return Optional.ofNullable(clazz.cast(hostId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "InstanceFamilies":
            return Optional.ofNullable(clazz.cast(instanceFamilies()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComputeAttributes, T> g) {
        return obj -> g.apply((ComputeAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComputeAttributes> {
        /**
         * <p>
         * The host ID of the Dedicated Host on the asset.
         * </p>
         * 
         * @param hostId
         *        The host ID of the Dedicated Host on the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostId(String hostId);

        /**
         * <p>
         * The state.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACTIVE - The asset is available and can provide capacity for new compute resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources. Existing
         * compute resources on the asset are not affected.
         * </p>
         * </li>
         * <li>
         * <p>
         * RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is reduced.
         * Amazon Web Services sends notifications for resources that must be stopped before the asset can be replaced.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACTIVE - The asset is available and can provide capacity for new compute resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources.
         *        Existing compute resources on the asset are not affected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is
         *        reduced. Amazon Web Services sends notifications for resources that must be stopped before the asset
         *        can be replaced.
         *        </p>
         *        </li>
         * @see ComputeAssetState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeAssetState
         */
        Builder state(String state);

        /**
         * <p>
         * The state.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ACTIVE - The asset is available and can provide capacity for new compute resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources. Existing
         * compute resources on the asset are not affected.
         * </p>
         * </li>
         * <li>
         * <p>
         * RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is reduced.
         * Amazon Web Services sends notifications for resources that must be stopped before the asset can be replaced.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ACTIVE - The asset is available and can provide capacity for new compute resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ISOLATED - The asset is undergoing maintenance and can't provide capacity for new compute resources.
         *        Existing compute resources on the asset are not affected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RETIRING - The underlying hardware for the asset is degraded. Capacity for new compute resources is
         *        reduced. Amazon Web Services sends notifications for resources that must be stopped before the asset
         *        can be replaced.
         *        </p>
         *        </li>
         * @see ComputeAssetState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeAssetState
         */
        Builder state(ComputeAssetState state);

        /**
         * <p>
         * A list of the names of instance families that are currently associated with a given asset.
         * </p>
         * 
         * @param instanceFamilies
         *        A list of the names of instance families that are currently associated with a given asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFamilies(Collection<String> instanceFamilies);

        /**
         * <p>
         * A list of the names of instance families that are currently associated with a given asset.
         * </p>
         * 
         * @param instanceFamilies
         *        A list of the names of instance families that are currently associated with a given asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFamilies(String... instanceFamilies);
    }

    static final class BuilderImpl implements Builder {
        private String hostId;

        private String state;

        private List<String> instanceFamilies = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComputeAttributes model) {
            hostId(model.hostId);
            state(model.state);
            instanceFamilies(model.instanceFamilies);
        }

        public final String getHostId() {
            return hostId;
        }

        public final void setHostId(String hostId) {
            this.hostId = hostId;
        }

        @Override
        public final Builder hostId(String hostId) {
            this.hostId = hostId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ComputeAssetState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Collection<String> getInstanceFamilies() {
            if (instanceFamilies instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceFamilies;
        }

        public final void setInstanceFamilies(Collection<String> instanceFamilies) {
            this.instanceFamilies = InstanceFamiliesCopier.copy(instanceFamilies);
        }

        @Override
        public final Builder instanceFamilies(Collection<String> instanceFamilies) {
            this.instanceFamilies = InstanceFamiliesCopier.copy(instanceFamilies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceFamilies(String... instanceFamilies) {
            instanceFamilies(Arrays.asList(instanceFamilies));
            return this;
        }

        @Override
        public ComputeAttributes build() {
            return new ComputeAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
