/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhub;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhub.internal.MigrationHubServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.migrationhub.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhub.model.AssociateCreatedArtifactRequest;
import software.amazon.awssdk.services.migrationhub.model.AssociateCreatedArtifactResponse;
import software.amazon.awssdk.services.migrationhub.model.AssociateDiscoveredResourceRequest;
import software.amazon.awssdk.services.migrationhub.model.AssociateDiscoveredResourceResponse;
import software.amazon.awssdk.services.migrationhub.model.CreateProgressUpdateStreamRequest;
import software.amazon.awssdk.services.migrationhub.model.CreateProgressUpdateStreamResponse;
import software.amazon.awssdk.services.migrationhub.model.DeleteProgressUpdateStreamRequest;
import software.amazon.awssdk.services.migrationhub.model.DeleteProgressUpdateStreamResponse;
import software.amazon.awssdk.services.migrationhub.model.DescribeApplicationStateRequest;
import software.amazon.awssdk.services.migrationhub.model.DescribeApplicationStateResponse;
import software.amazon.awssdk.services.migrationhub.model.DescribeMigrationTaskRequest;
import software.amazon.awssdk.services.migrationhub.model.DescribeMigrationTaskResponse;
import software.amazon.awssdk.services.migrationhub.model.DisassociateCreatedArtifactRequest;
import software.amazon.awssdk.services.migrationhub.model.DisassociateCreatedArtifactResponse;
import software.amazon.awssdk.services.migrationhub.model.DisassociateDiscoveredResourceRequest;
import software.amazon.awssdk.services.migrationhub.model.DisassociateDiscoveredResourceResponse;
import software.amazon.awssdk.services.migrationhub.model.DryRunOperationException;
import software.amazon.awssdk.services.migrationhub.model.HomeRegionNotSetException;
import software.amazon.awssdk.services.migrationhub.model.ImportMigrationTaskRequest;
import software.amazon.awssdk.services.migrationhub.model.ImportMigrationTaskResponse;
import software.amazon.awssdk.services.migrationhub.model.InternalServerErrorException;
import software.amazon.awssdk.services.migrationhub.model.InvalidInputException;
import software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesRequest;
import software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesResponse;
import software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsRequest;
import software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsResponse;
import software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesRequest;
import software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesResponse;
import software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksRequest;
import software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksResponse;
import software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsRequest;
import software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsResponse;
import software.amazon.awssdk.services.migrationhub.model.MigrationHubException;
import software.amazon.awssdk.services.migrationhub.model.NotifyApplicationStateRequest;
import software.amazon.awssdk.services.migrationhub.model.NotifyApplicationStateResponse;
import software.amazon.awssdk.services.migrationhub.model.NotifyMigrationTaskStateRequest;
import software.amazon.awssdk.services.migrationhub.model.NotifyMigrationTaskStateResponse;
import software.amazon.awssdk.services.migrationhub.model.PolicyErrorException;
import software.amazon.awssdk.services.migrationhub.model.PutResourceAttributesRequest;
import software.amazon.awssdk.services.migrationhub.model.PutResourceAttributesResponse;
import software.amazon.awssdk.services.migrationhub.model.ResourceNotFoundException;
import software.amazon.awssdk.services.migrationhub.model.ServiceUnavailableException;
import software.amazon.awssdk.services.migrationhub.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhub.model.UnauthorizedOperationException;
import software.amazon.awssdk.services.migrationhub.transform.AssociateCreatedArtifactRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.AssociateDiscoveredResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.CreateProgressUpdateStreamRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DeleteProgressUpdateStreamRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DescribeApplicationStateRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DescribeMigrationTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DisassociateCreatedArtifactRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DisassociateDiscoveredResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ImportMigrationTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListApplicationStatesRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListCreatedArtifactsRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListDiscoveredResourcesRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListMigrationTasksRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListProgressUpdateStreamsRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.NotifyApplicationStateRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.NotifyMigrationTaskStateRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.PutResourceAttributesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MigrationHubAsyncClient}.
 *
 * @see MigrationHubAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubAsyncClient implements MigrationHubAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMigrationHubAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a created artifact of an AWS cloud resource, the target receiving the migration, with the migration
     * task performed by a migration tool. This API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Migration tools can call the <code>AssociateCreatedArtifact</code> operation to indicate which AWS artifact is
     * associated with a migration task.
     * </p>
     * </li>
     * <li>
     * <p>
     * The created artifact name must be provided in ARN (Amazon Resource Name) format which will contain information
     * about type and region; for example: <code>arn:aws:ec2:us-east-1:488216288981:image/ami-6d0ba87b</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Examples of the AWS resource behind the created artifact are, AMI's, EC2 instance, or DMS endpoint, etc.
     * </p>
     * </li>
     * </ul>
     *
     * @param associateCreatedArtifactRequest
     * @return A Java Future containing the result of the AssociateCreatedArtifact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.AssociateCreatedArtifact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/AssociateCreatedArtifact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateCreatedArtifactResponse> associateCreatedArtifact(
            AssociateCreatedArtifactRequest associateCreatedArtifactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateCreatedArtifactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateCreatedArtifactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateCreatedArtifact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateCreatedArtifactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateCreatedArtifactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateCreatedArtifactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateCreatedArtifactRequest, AssociateCreatedArtifactResponse>()
                            .withOperationName("AssociateCreatedArtifact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateCreatedArtifactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateCreatedArtifactRequest));
            CompletableFuture<AssociateCreatedArtifactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a discovered resource ID from Application Discovery Service with a migration task.
     * </p>
     *
     * @param associateDiscoveredResourceRequest
     * @return A Java Future containing the result of the AssociateDiscoveredResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>PolicyErrorException Exception raised when there are problems accessing Application Discovery Service
     *         (Application Discovery Service); most likely due to a misconfigured policy or the
     *         <code>migrationhub-discovery</code> role is missing or not configured correctly.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.AssociateDiscoveredResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/AssociateDiscoveredResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDiscoveredResourceResponse> associateDiscoveredResource(
            AssociateDiscoveredResourceRequest associateDiscoveredResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateDiscoveredResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDiscoveredResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDiscoveredResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDiscoveredResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDiscoveredResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDiscoveredResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDiscoveredResourceRequest, AssociateDiscoveredResourceResponse>()
                            .withOperationName("AssociateDiscoveredResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateDiscoveredResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateDiscoveredResourceRequest));
            CompletableFuture<AssociateDiscoveredResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a progress update stream which is an AWS resource used for access control as well as a namespace for
     * migration task names that is implicitly linked to your AWS account. It must uniquely identify the migration tool
     * as it is used for all updates made by the tool; however, it does not need to be unique for each AWS account
     * because it is scoped to the AWS account.
     * </p>
     *
     * @param createProgressUpdateStreamRequest
     * @return A Java Future containing the result of the CreateProgressUpdateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.CreateProgressUpdateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/CreateProgressUpdateStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProgressUpdateStreamResponse> createProgressUpdateStream(
            CreateProgressUpdateStreamRequest createProgressUpdateStreamRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProgressUpdateStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProgressUpdateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProgressUpdateStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProgressUpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProgressUpdateStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProgressUpdateStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProgressUpdateStreamRequest, CreateProgressUpdateStreamResponse>()
                            .withOperationName("CreateProgressUpdateStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProgressUpdateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProgressUpdateStreamRequest));
            CompletableFuture<CreateProgressUpdateStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a progress update stream, including all of its tasks, which was previously created as an AWS resource
     * used for access control. This API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The only parameter needed for <code>DeleteProgressUpdateStream</code> is the stream name (same as a
     * <code>CreateProgressUpdateStream</code> call).
     * </p>
     * </li>
     * <li>
     * <p>
     * The call will return, and a background process will asynchronously delete the stream and all of its resources
     * (tasks, associated resources, resource attributes, created artifacts).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the stream takes time to be deleted, it might still show up on a <code>ListProgressUpdateStreams</code> call.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreateProgressUpdateStream</code>, <code>ImportMigrationTask</code>, <code>NotifyMigrationTaskState</code>,
     * and all Associate[*] APIs related to the tasks belonging to the stream will throw "InvalidInputException" if the
     * stream of the same name is in the process of being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * Once the stream and all of its resources are deleted, <code>CreateProgressUpdateStream</code> for a stream of the
     * same name will succeed, and that stream will be an entirely new logical resource (without any resources
     * associated with the old stream).
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteProgressUpdateStreamRequest
     * @return A Java Future containing the result of the DeleteProgressUpdateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.DeleteProgressUpdateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DeleteProgressUpdateStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProgressUpdateStreamResponse> deleteProgressUpdateStream(
            DeleteProgressUpdateStreamRequest deleteProgressUpdateStreamRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProgressUpdateStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProgressUpdateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProgressUpdateStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProgressUpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProgressUpdateStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProgressUpdateStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProgressUpdateStreamRequest, DeleteProgressUpdateStreamResponse>()
                            .withOperationName("DeleteProgressUpdateStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProgressUpdateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProgressUpdateStreamRequest));
            CompletableFuture<DeleteProgressUpdateStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the migration status of an application.
     * </p>
     *
     * @param describeApplicationStateRequest
     * @return A Java Future containing the result of the DescribeApplicationState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>PolicyErrorException Exception raised when there are problems accessing Application Discovery Service
     *         (Application Discovery Service); most likely due to a misconfigured policy or the
     *         <code>migrationhub-discovery</code> role is missing or not configured correctly.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.DescribeApplicationState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DescribeApplicationState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeApplicationStateResponse> describeApplicationState(
            DescribeApplicationStateRequest describeApplicationStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeApplicationStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApplicationStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplicationState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeApplicationStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeApplicationStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeApplicationStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationStateRequest, DescribeApplicationStateResponse>()
                            .withOperationName("DescribeApplicationState").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeApplicationStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeApplicationStateRequest));
            CompletableFuture<DescribeApplicationStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all attributes associated with a specific migration task.
     * </p>
     *
     * @param describeMigrationTaskRequest
     * @return A Java Future containing the result of the DescribeMigrationTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.DescribeMigrationTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DescribeMigrationTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMigrationTaskResponse> describeMigrationTask(
            DescribeMigrationTaskRequest describeMigrationTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMigrationTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMigrationTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMigrationTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMigrationTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMigrationTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMigrationTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMigrationTaskRequest, DescribeMigrationTaskResponse>()
                            .withOperationName("DescribeMigrationTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeMigrationTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMigrationTaskRequest));
            CompletableFuture<DescribeMigrationTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a created artifact of an AWS resource with a migration task performed by a migration tool that was
     * previously associated. This API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A migration user can call the <code>DisassociateCreatedArtifacts</code> operation to disassociate a created AWS
     * Artifact from a migration task.
     * </p>
     * </li>
     * <li>
     * <p>
     * The created artifact name must be provided in ARN (Amazon Resource Name) format which will contain information
     * about type and region; for example: <code>arn:aws:ec2:us-east-1:488216288981:image/ami-6d0ba87b</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Examples of the AWS resource behind the created artifact are, AMI's, EC2 instance, or RDS instance, etc.
     * </p>
     * </li>
     * </ul>
     *
     * @param disassociateCreatedArtifactRequest
     * @return A Java Future containing the result of the DisassociateCreatedArtifact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.DisassociateCreatedArtifact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DisassociateCreatedArtifact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateCreatedArtifactResponse> disassociateCreatedArtifact(
            DisassociateCreatedArtifactRequest disassociateCreatedArtifactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateCreatedArtifactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateCreatedArtifactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateCreatedArtifact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateCreatedArtifactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateCreatedArtifactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateCreatedArtifactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateCreatedArtifactRequest, DisassociateCreatedArtifactResponse>()
                            .withOperationName("DisassociateCreatedArtifact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateCreatedArtifactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateCreatedArtifactRequest));
            CompletableFuture<DisassociateCreatedArtifactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociate an Application Discovery Service discovered resource from a migration task.
     * </p>
     *
     * @param disassociateDiscoveredResourceRequest
     * @return A Java Future containing the result of the DisassociateDiscoveredResource operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.DisassociateDiscoveredResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DisassociateDiscoveredResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDiscoveredResourceResponse> disassociateDiscoveredResource(
            DisassociateDiscoveredResourceRequest disassociateDiscoveredResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateDiscoveredResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateDiscoveredResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDiscoveredResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDiscoveredResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDiscoveredResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDiscoveredResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDiscoveredResourceRequest, DisassociateDiscoveredResourceResponse>()
                            .withOperationName("DisassociateDiscoveredResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateDiscoveredResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateDiscoveredResourceRequest));
            CompletableFuture<DisassociateDiscoveredResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a new migration task which represents a server, database, etc., being migrated to AWS by a migration
     * tool.
     * </p>
     * <p>
     * This API is a prerequisite to calling the <code>NotifyMigrationTaskState</code> API as the migration tool must
     * first register the migration task with Migration Hub.
     * </p>
     *
     * @param importMigrationTaskRequest
     * @return A Java Future containing the result of the ImportMigrationTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.ImportMigrationTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ImportMigrationTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportMigrationTaskResponse> importMigrationTask(
            ImportMigrationTaskRequest importMigrationTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importMigrationTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importMigrationTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportMigrationTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportMigrationTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportMigrationTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportMigrationTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportMigrationTaskRequest, ImportMigrationTaskResponse>()
                            .withOperationName("ImportMigrationTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportMigrationTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(importMigrationTaskRequest));
            CompletableFuture<ImportMigrationTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the migration statuses for your applications. If you use the optional <code>ApplicationIds</code>
     * parameter, only the migration statuses for those applications will be returned.
     * </p>
     *
     * @param listApplicationStatesRequest
     * @return A Java Future containing the result of the ListApplicationStates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.ListApplicationStates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListApplicationStates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationStatesResponse> listApplicationStates(
            ListApplicationStatesRequest listApplicationStatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationStatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationStatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationStates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationStatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationStatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationStatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationStatesRequest, ListApplicationStatesResponse>()
                            .withOperationName("ListApplicationStates").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListApplicationStatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listApplicationStatesRequest));
            CompletableFuture<ListApplicationStatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the created artifacts attached to a given migration task in an update stream. This API has the following
     * traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Gets the list of the created artifacts while migration is taking place.
     * </p>
     * </li>
     * <li>
     * <p>
     * Shows the artifacts created by the migration tool that was associated by the
     * <code>AssociateCreatedArtifact</code> API.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists created artifacts in a paginated interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param listCreatedArtifactsRequest
     * @return A Java Future containing the result of the ListCreatedArtifacts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.ListCreatedArtifacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListCreatedArtifacts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCreatedArtifactsResponse> listCreatedArtifacts(
            ListCreatedArtifactsRequest listCreatedArtifactsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCreatedArtifactsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCreatedArtifactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCreatedArtifacts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCreatedArtifactsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCreatedArtifactsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCreatedArtifactsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCreatedArtifactsRequest, ListCreatedArtifactsResponse>()
                            .withOperationName("ListCreatedArtifacts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCreatedArtifactsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCreatedArtifactsRequest));
            CompletableFuture<ListCreatedArtifactsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists discovered resources associated with the given <code>MigrationTask</code>.
     * </p>
     *
     * @param listDiscoveredResourcesRequest
     * @return A Java Future containing the result of the ListDiscoveredResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.ListDiscoveredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListDiscoveredResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDiscoveredResourcesResponse> listDiscoveredResources(
            ListDiscoveredResourcesRequest listDiscoveredResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDiscoveredResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDiscoveredResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDiscoveredResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDiscoveredResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDiscoveredResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDiscoveredResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDiscoveredResourcesRequest, ListDiscoveredResourcesResponse>()
                            .withOperationName("ListDiscoveredResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDiscoveredResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDiscoveredResourcesRequest));
            CompletableFuture<ListDiscoveredResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all, or filtered by resource name, migration tasks associated with the user account making this call. This
     * API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Can show a summary list of the most recent migration tasks.
     * </p>
     * </li>
     * <li>
     * <p>
     * Can show a summary list of migration tasks associated with a given discovered resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists migration tasks in a paginated interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param listMigrationTasksRequest
     * @return A Java Future containing the result of the ListMigrationTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>PolicyErrorException Exception raised when there are problems accessing Application Discovery Service
     *         (Application Discovery Service); most likely due to a misconfigured policy or the
     *         <code>migrationhub-discovery</code> role is missing or not configured correctly.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.ListMigrationTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListMigrationTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMigrationTasksResponse> listMigrationTasks(ListMigrationTasksRequest listMigrationTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMigrationTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMigrationTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMigrationTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMigrationTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMigrationTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMigrationTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMigrationTasksRequest, ListMigrationTasksResponse>()
                            .withOperationName("ListMigrationTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMigrationTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMigrationTasksRequest));
            CompletableFuture<ListMigrationTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists progress update streams associated with the user account making this call.
     * </p>
     *
     * @param listProgressUpdateStreamsRequest
     * @return A Java Future containing the result of the ListProgressUpdateStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.ListProgressUpdateStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListProgressUpdateStreams"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProgressUpdateStreamsResponse> listProgressUpdateStreams(
            ListProgressUpdateStreamsRequest listProgressUpdateStreamsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProgressUpdateStreamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProgressUpdateStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProgressUpdateStreams");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProgressUpdateStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProgressUpdateStreamsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProgressUpdateStreamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProgressUpdateStreamsRequest, ListProgressUpdateStreamsResponse>()
                            .withOperationName("ListProgressUpdateStreams").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProgressUpdateStreamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProgressUpdateStreamsRequest));
            CompletableFuture<ListProgressUpdateStreamsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the migration state of an application. For a given application identified by the value passed to
     * <code>ApplicationId</code>, its status is set or updated by passing one of three values to <code>Status</code>:
     * <code>NOT_STARTED | IN_PROGRESS | COMPLETED</code>.
     * </p>
     *
     * @param notifyApplicationStateRequest
     * @return A Java Future containing the result of the NotifyApplicationState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>PolicyErrorException Exception raised when there are problems accessing Application Discovery Service
     *         (Application Discovery Service); most likely due to a misconfigured policy or the
     *         <code>migrationhub-discovery</code> role is missing or not configured correctly.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.NotifyApplicationState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/NotifyApplicationState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<NotifyApplicationStateResponse> notifyApplicationState(
            NotifyApplicationStateRequest notifyApplicationStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(notifyApplicationStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyApplicationStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyApplicationState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<NotifyApplicationStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, NotifyApplicationStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<NotifyApplicationStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<NotifyApplicationStateRequest, NotifyApplicationStateResponse>()
                            .withOperationName("NotifyApplicationState").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new NotifyApplicationStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(notifyApplicationStateRequest));
            CompletableFuture<NotifyApplicationStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Notifies Migration Hub of the current status, progress, or other detail regarding a migration task. This API has
     * the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Migration tools will call the <code>NotifyMigrationTaskState</code> API to share the latest progress and status.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MigrationTaskName</code> is used for addressing updates to the correct target.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ProgressUpdateStream</code> is used for access control and to provide a namespace for each migration tool.
     * </p>
     * </li>
     * </ul>
     *
     * @param notifyMigrationTaskStateRequest
     * @return A Java Future containing the result of the NotifyMigrationTaskState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.NotifyMigrationTaskState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/NotifyMigrationTaskState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<NotifyMigrationTaskStateResponse> notifyMigrationTaskState(
            NotifyMigrationTaskStateRequest notifyMigrationTaskStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(notifyMigrationTaskStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyMigrationTaskStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyMigrationTaskState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<NotifyMigrationTaskStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, NotifyMigrationTaskStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<NotifyMigrationTaskStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<NotifyMigrationTaskStateRequest, NotifyMigrationTaskStateResponse>()
                            .withOperationName("NotifyMigrationTaskState").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new NotifyMigrationTaskStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(notifyMigrationTaskStateRequest));
            CompletableFuture<NotifyMigrationTaskStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides identifying details of the resource being migrated so that it can be associated in the Application
     * Discovery Service repository. This association occurs asynchronously after <code>PutResourceAttributes</code>
     * returns.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Keep in mind that subsequent calls to PutResourceAttributes will override previously stored attributes. For
     * example, if it is first called with a MAC address, but later, it is desired to <i>add</i> an IP address, it will
     * then be required to call it with <i>both</i> the IP and MAC addresses to prevent overriding the MAC address.
     * </p>
     * </li>
     * <li>
     * <p>
     * Note the instructions regarding the special use case of the <a href=
     * "https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html#migrationhub-PutResourceAttributes-request-ResourceAttributeList"
     * > <code>ResourceAttributeList</code> </a> parameter when specifying any "VM" related value.
     * </p>
     * </li>
     * </ul>
     * </important> <note>
     * <p>
     * Because this is an asynchronous call, it will always return 200, whether an association occurs or not. To confirm
     * if an association was found based on the provided details, call <code>ListDiscoveredResources</code>.
     * </p>
     * </note>
     *
     * @param putResourceAttributesRequest
     * @return A Java Future containing the result of the PutResourceAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when there is an internal, configuration, or dependency
     *         error encountered.</li>
     *         <li>DryRunOperationException Exception raised to indicate a successfully authorized action when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>UnauthorizedOperationException Exception raised to indicate a request was not authorized when the
     *         <code>DryRun</code> flag is set to "true".</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>ResourceNotFoundException Exception raised when the request references a resource (Application
     *         Discovery Service configuration, update stream, migration task, etc.) that does not exist in Application
     *         Discovery Service (Application Discovery Service) or in Migration Hub's repository.</li>
     *         <li>HomeRegionNotSetException The home region is not set. Set the home region to continue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MigrationHubAsyncClient.PutResourceAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/PutResourceAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourceAttributesResponse> putResourceAttributes(
            PutResourceAttributesRequest putResourceAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourceAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourceAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourceAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourceAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourceAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourceAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourceAttributesRequest, PutResourceAttributesResponse>()
                            .withOperationName("PutResourceAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutResourceAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putResourceAttributesRequest));
            CompletableFuture<PutResourceAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MigrationHubServiceClientConfiguration serviceClientConfiguration() {
        return new MigrationHubServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedOperation")
                                .exceptionBuilderSupplier(UnauthorizedOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyErrorException")
                                .exceptionBuilderSupplier(PolicyErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("HomeRegionNotSetException")
                                .exceptionBuilderSupplier(HomeRegionNotSetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DryRunOperation")
                                .exceptionBuilderSupplier(DryRunOperationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        MigrationHubServiceClientConfigurationBuilder serviceConfigBuilder = new MigrationHubServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
