/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacecatalog;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.marketplacecatalog.internal.MarketplaceCatalogServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.marketplacecatalog.model.AccessDeniedException;
import software.amazon.awssdk.services.marketplacecatalog.model.CancelChangeSetRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.CancelChangeSetResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeChangeSetRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeChangeSetResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeEntityRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.DescribeEntityResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.InternalServiceException;
import software.amazon.awssdk.services.marketplacecatalog.model.ListChangeSetsRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.ListChangeSetsResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ListEntitiesRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.ListEntitiesResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.MarketplaceCatalogException;
import software.amazon.awssdk.services.marketplacecatalog.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ResourceInUseException;
import software.amazon.awssdk.services.marketplacecatalog.model.ResourceNotFoundException;
import software.amazon.awssdk.services.marketplacecatalog.model.ResourceNotSupportedException;
import software.amazon.awssdk.services.marketplacecatalog.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.marketplacecatalog.model.StartChangeSetRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.StartChangeSetResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.TagResourceRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.TagResourceResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ThrottlingException;
import software.amazon.awssdk.services.marketplacecatalog.model.UntagResourceRequest;
import software.amazon.awssdk.services.marketplacecatalog.model.UntagResourceResponse;
import software.amazon.awssdk.services.marketplacecatalog.model.ValidationException;
import software.amazon.awssdk.services.marketplacecatalog.transform.CancelChangeSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.DescribeChangeSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.DescribeEntityRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.ListChangeSetsRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.ListEntitiesRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.StartChangeSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.marketplacecatalog.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MarketplaceCatalogClient}.
 *
 * @see MarketplaceCatalogClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMarketplaceCatalogClient implements MarketplaceCatalogClient {
    private static final Logger log = Logger.loggerFor(DefaultMarketplaceCatalogClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MarketplaceCatalogServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMarketplaceCatalogClient(MarketplaceCatalogServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Used to cancel an open change request. Must be sent before the status of the request changes to
     * <code>APPLYING</code>, the final stage of completing your change request. You can describe a change during the
     * 60-day request history retention period for API calls.
     * </p>
     *
     * @param cancelChangeSetRequest
     * @return Result of the CancelChangeSet operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ResourceInUseException
     *         The resource is currently in use.
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.CancelChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/CancelChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelChangeSetResponse cancelChangeSet(CancelChangeSetRequest cancelChangeSetRequest)
            throws InternalServiceException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            ResourceInUseException, ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelChangeSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelChangeSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelChangeSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelChangeSet");

            return clientHandler.execute(new ClientExecutionParams<CancelChangeSetRequest, CancelChangeSetResponse>()
                    .withOperationName("CancelChangeSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(cancelChangeSetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelChangeSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a resource-based policy on an entity that is identified by its resource ARN.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws InternalServiceException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                    .withOperationName("DeleteResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteResourcePolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about a given change set.
     * </p>
     *
     * @param describeChangeSetRequest
     * @return Result of the DescribeChangeSet operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.DescribeChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/DescribeChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeChangeSetResponse describeChangeSet(DescribeChangeSetRequest describeChangeSetRequest)
            throws InternalServiceException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeChangeSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeChangeSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChangeSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChangeSet");

            return clientHandler.execute(new ClientExecutionParams<DescribeChangeSetRequest, DescribeChangeSetResponse>()
                    .withOperationName("DescribeChangeSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeChangeSetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeChangeSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the metadata and content of the entity.
     * </p>
     *
     * @param describeEntityRequest
     * @return Result of the DescribeEntity operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotSupportedException
     *         Currently, the specified resource is not supported.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.DescribeEntity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/DescribeEntity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEntityResponse describeEntity(DescribeEntityRequest describeEntityRequest) throws InternalServiceException,
            AccessDeniedException, ValidationException, ResourceNotSupportedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEntityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEntityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEntityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEntityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEntity");

            return clientHandler.execute(new ClientExecutionParams<DescribeEntityRequest, DescribeEntityResponse>()
                    .withOperationName("DescribeEntity").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeEntityRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEntityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a resource-based policy of an entity that is identified by its resource ARN.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws InternalServiceException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getResourcePolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of change sets owned by the account being used to make the call. You can filter this list by
     * providing any combination of <code>entityId</code>, <code>ChangeSetName</code>, and status. If you provide more
     * than one filter, the API operation applies a logical AND between the filters.
     * </p>
     * <p>
     * You can describe a change during the 60-day request history retention period for API calls.
     * </p>
     *
     * @param listChangeSetsRequest
     * @return Result of the ListChangeSets operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.ListChangeSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/ListChangeSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListChangeSetsResponse listChangeSets(ListChangeSetsRequest listChangeSetsRequest) throws InternalServiceException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChangeSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChangeSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChangeSetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChangeSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChangeSets");

            return clientHandler.execute(new ClientExecutionParams<ListChangeSetsRequest, ListChangeSetsResponse>()
                    .withOperationName("ListChangeSets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listChangeSetsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChangeSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides the list of entities of a given type.
     * </p>
     *
     * @param listEntitiesRequest
     * @return Result of the ListEntities operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.ListEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/ListEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEntitiesResponse listEntities(ListEntitiesRequest listEntitiesRequest) throws InternalServiceException,
            AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEntitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEntitiesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEntitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEntities");

            return clientHandler.execute(new ClientExecutionParams<ListEntitiesRequest, ListEntitiesResponse>()
                    .withOperationName("ListEntities").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listEntitiesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEntitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags that have been added to a resource (either an <a
     * href="https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#catalog-api-entities"
     * >entity</a> or <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets"
     * >change set</a>).
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404
     * @throws InternalServiceException
     *         There was an internal service exception.
     *         </p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServiceException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches a resource-based policy to an entity. Examples of an entity include: <code>AmiProduct</code> and
     * <code>ContainerProduct</code>.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/PutResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws InternalServiceException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                    .withOperationName("PutResourcePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putResourcePolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to request changes for your entities. Within a single <code>ChangeSet</code>, you can't start the same
     * change type against the same entity multiple times. Additionally, when a <code>ChangeSet</code> is running, all
     * the entities targeted by the different changes are locked until the change set has completed (either succeeded,
     * cancelled, or failed). If you try to start a change set containing a change against an entity that is already
     * locked, you will receive a <code>ResourceInUseException</code> error.
     * </p>
     * <p>
     * For example, you can't start the <code>ChangeSet</code> described in the <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/API_StartChangeSet.html#API_StartChangeSet_Examples"
     * >example</a> later in this topic because it contains two changes to run the same change type (
     * <code>AddRevisions</code>) against the same entity (<code>entity-id@1</code>).
     * </p>
     * <p>
     * For more information about working with change sets, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets">
     * Working with change sets</a>. For information about change types for single-AMI products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/ami-products.html#working-with-single-AMI-products"
     * >Working with single-AMI products</a>. Also, for more information about change types available for
     * container-based products, see <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/container-products.html#working-with-container-products"
     * >Working with container products</a>.
     * </p>
     *
     * @param startChangeSetRequest
     * @return Result of the StartChangeSet operation returned by the service.
     * @throws InternalServiceException
     *         There was an internal service exception.</p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     *         </p>
     *         <p>
     *         HTTP status code: 404
     * @throws ResourceInUseException
     *         The resource is currently in use.
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws ServiceQuotaExceededException
     *         The maximum number of open requests per account has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.StartChangeSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/StartChangeSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartChangeSetResponse startChangeSet(StartChangeSetRequest startChangeSetRequest) throws InternalServiceException,
            AccessDeniedException, ValidationException, ResourceNotFoundException, ResourceInUseException, ThrottlingException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartChangeSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartChangeSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startChangeSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startChangeSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartChangeSet");

            return clientHandler.execute(new ClientExecutionParams<StartChangeSetRequest, StartChangeSetResponse>()
                    .withOperationName("StartChangeSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startChangeSetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartChangeSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags a resource (either an <a
     * href="https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#catalog-api-entities"
     * >entity</a> or <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets"
     * >change set</a>).
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404
     * @throws InternalServiceException
     *         There was an internal service exception.
     *         </p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServiceException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag or list of tags from a resource (either an <a
     * href="https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#catalog-api-entities"
     * >entity</a> or <a href=
     * "https://docs.aws.amazon.com/marketplace-catalog/latest/api-reference/welcome.html#working-with-change-sets"
     * >change set</a>).
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.</p>
     *         <p>
     *         HTTP status code: 404
     * @throws InternalServiceException
     *         There was an internal service exception.
     *         </p>
     *         <p>
     *         HTTP status code: 500
     * @throws AccessDeniedException
     *         Access is denied.
     *         </p>
     *         <p>
     *         HTTP status code: 403
     * @throws ValidationException
     *         An error occurred during validation.
     *         </p>
     *         <p>
     *         HTTP status code: 422
     * @throws ThrottlingException
     *         Too many requests.
     *         </p>
     *         <p>
     *         HTTP status code: 429
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCatalogClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-catalog-2018-09-17/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServiceException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, MarketplaceCatalogException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Catalog");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        MarketplaceCatalogServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = MarketplaceCatalogServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MarketplaceCatalogException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(423).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotSupportedException")
                                .exceptionBuilderSupplier(ResourceNotSupportedException::builder).httpStatusCode(415).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public final MarketplaceCatalogServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
