/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutequipment;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lookoutequipment.model.AccessDeniedException;
import software.amazon.awssdk.services.lookoutequipment.model.ConflictException;
import software.amazon.awssdk.services.lookoutequipment.model.CreateDatasetRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateDatasetResponse;
import software.amazon.awssdk.services.lookoutequipment.model.CreateInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.CreateLabelGroupRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateLabelGroupResponse;
import software.amazon.awssdk.services.lookoutequipment.model.CreateLabelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateLabelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.CreateModelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateModelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteLabelGroupRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteLabelGroupResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteLabelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteLabelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteModelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteModelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDataIngestionJobRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDataIngestionJobResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeLabelGroupRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeLabelGroupResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeLabelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeLabelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeModelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeModelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.InternalServerException;
import software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceEventsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceEventsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListLabelGroupsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListLabelGroupsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListLabelsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListLabelsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListModelsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListModelsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListSensorStatisticsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListSensorStatisticsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.lookoutequipment.model.LookoutEquipmentException;
import software.amazon.awssdk.services.lookoutequipment.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lookoutequipment.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.lookoutequipment.model.StartDataIngestionJobRequest;
import software.amazon.awssdk.services.lookoutequipment.model.StartDataIngestionJobResponse;
import software.amazon.awssdk.services.lookoutequipment.model.StartInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.StartInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.StopInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.StopInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.TagResourceRequest;
import software.amazon.awssdk.services.lookoutequipment.model.TagResourceResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ThrottlingException;
import software.amazon.awssdk.services.lookoutequipment.model.UntagResourceRequest;
import software.amazon.awssdk.services.lookoutequipment.model.UntagResourceResponse;
import software.amazon.awssdk.services.lookoutequipment.model.UpdateInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.UpdateInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.UpdateLabelGroupRequest;
import software.amazon.awssdk.services.lookoutequipment.model.UpdateLabelGroupResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ValidationException;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateLabelGroupRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateLabelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteLabelGroupRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteLabelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeDataIngestionJobRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeLabelGroupRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeLabelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListDataIngestionJobsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListInferenceEventsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListInferenceExecutionsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListInferenceSchedulersRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListLabelGroupsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListLabelsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListModelsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListSensorStatisticsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.StartDataIngestionJobRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.StartInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.StopInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.UpdateInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.UpdateLabelGroupRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LookoutEquipmentClient}.
 *
 * @see LookoutEquipmentClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLookoutEquipmentClient implements LookoutEquipmentClient {
    private static final Logger log = Logger.loggerFor(DefaultLookoutEquipmentClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LookoutEquipmentServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLookoutEquipmentClient(LookoutEquipmentServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a container for a collection of data being ingested for analysis. The dataset contains the metadata
     * describing where the data is and what the data actually looks like. In other words, it contains the location of
     * the data source, the data schema, and other information. A dataset also contains any tags associated with the
     * ingested data.
     * </p>
     *
     * @param createDatasetRequest
     * @return Result of the CreateDataset operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDatasetResponse createDataset(CreateDatasetRequest createDatasetRequest) throws ValidationException,
            ConflictException, ThrottlingException, ServiceQuotaExceededException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");

            return clientHandler.execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                    .withOperationName("CreateDataset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a scheduled inference. Scheduling an inference is setting up a continuous real-time inference plan to
     * analyze new measurement data. When setting up the schedule, you provide an S3 bucket location for the input data,
     * assign it a delimiter between separate entries in the data, set an offset delay if desired, and set the frequency
     * of inferencing. You must also provide an S3 bucket location for the output data.
     * </p>
     *
     * @param createInferenceSchedulerRequest
     * @return Result of the CreateInferenceScheduler operation returned by the service.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.CreateInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateInferenceSchedulerResponse createInferenceScheduler(
            CreateInferenceSchedulerRequest createInferenceSchedulerRequest) throws ConflictException, ResourceNotFoundException,
            ValidationException, ServiceQuotaExceededException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateInferenceSchedulerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInferenceScheduler");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateInferenceSchedulerRequest, CreateInferenceSchedulerResponse>()
                            .withOperationName("CreateInferenceScheduler").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createInferenceSchedulerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateInferenceSchedulerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a label for an event.
     * </p>
     *
     * @param createLabelRequest
     * @return Result of the CreateLabel operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.CreateLabel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateLabel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateLabelResponse createLabel(CreateLabelRequest createLabelRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, ThrottlingException, ServiceQuotaExceededException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLabelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLabelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLabelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLabel");

            return clientHandler.execute(new ClientExecutionParams<CreateLabelRequest, CreateLabelResponse>()
                    .withOperationName("CreateLabel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLabelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLabelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a group of labels.
     * </p>
     *
     * @param createLabelGroupRequest
     * @return Result of the CreateLabelGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.CreateLabelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateLabelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateLabelGroupResponse createLabelGroup(CreateLabelGroupRequest createLabelGroupRequest) throws ValidationException,
            ConflictException, ThrottlingException, ServiceQuotaExceededException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLabelGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLabelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLabelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLabelGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateLabelGroupRequest, CreateLabelGroupResponse>()
                    .withOperationName("CreateLabelGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLabelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLabelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an ML model for data inference.
     * </p>
     * <p>
     * A machine-learning (ML) model is a mathematical model that finds patterns in your data. In Amazon Lookout for
     * Equipment, the model learns the patterns of normal behavior and detects abnormal behavior that could be potential
     * equipment failure (or maintenance events). The models are made by analyzing normal data and abnormalities in
     * machine behavior that have already occurred.
     * </p>
     * <p>
     * Your model is trained using a portion of the data from your dataset and uses that data to learn patterns of
     * normal behavior and abnormal patterns that lead to equipment failure. Another portion of the data is used to
     * evaluate the model's accuracy.
     * </p>
     *
     * @param createModelRequest
     * @return Result of the CreateModel operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.CreateModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateModelResponse createModel(CreateModelRequest createModelRequest) throws ValidationException, ConflictException,
            ThrottlingException, ServiceQuotaExceededException, InternalServerException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateModel");

            return clientHandler.execute(new ClientExecutionParams<CreateModelRequest, CreateModelResponse>()
                    .withOperationName("CreateModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a dataset and associated artifacts. The operation will check to see if any inference scheduler or data
     * ingestion job is currently using the dataset, and if there isn't, the dataset, its metadata, and any associated
     * data stored in S3 will be deleted. This does not affect any models that used this dataset for training and
     * evaluation, but does prevent it from being used in the future.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return Result of the DeleteDataset operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDatasetResponse deleteDataset(DeleteDatasetRequest deleteDatasetRequest) throws ResourceNotFoundException,
            ThrottlingException, InternalServerException, AccessDeniedException, ConflictException, ValidationException,
            AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");

            return clientHandler.execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                    .withOperationName("DeleteDataset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an inference scheduler that has been set up. Already processed output results are not affected.
     * </p>
     *
     * @param deleteInferenceSchedulerRequest
     * @return Result of the DeleteInferenceScheduler operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DeleteInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteInferenceSchedulerResponse deleteInferenceScheduler(
            DeleteInferenceSchedulerRequest deleteInferenceSchedulerRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteInferenceSchedulerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInferenceScheduler");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteInferenceSchedulerRequest, DeleteInferenceSchedulerResponse>()
                            .withOperationName("DeleteInferenceScheduler").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteInferenceSchedulerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteInferenceSchedulerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a label.
     * </p>
     *
     * @param deleteLabelRequest
     * @return Result of the DeleteLabel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DeleteLabel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteLabel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteLabelResponse deleteLabel(DeleteLabelRequest deleteLabelRequest) throws ResourceNotFoundException,
            ThrottlingException, InternalServerException, AccessDeniedException, ConflictException, ValidationException,
            AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLabelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLabelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLabelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLabel");

            return clientHandler.execute(new ClientExecutionParams<DeleteLabelRequest, DeleteLabelResponse>()
                    .withOperationName("DeleteLabel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLabelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLabelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a group of labels.
     * </p>
     *
     * @param deleteLabelGroupRequest
     * @return Result of the DeleteLabelGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DeleteLabelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteLabelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLabelGroupResponse deleteLabelGroup(DeleteLabelGroupRequest deleteLabelGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, InternalServerException, AccessDeniedException,
            ConflictException, ValidationException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLabelGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLabelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLabelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLabelGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteLabelGroupRequest, DeleteLabelGroupResponse>()
                    .withOperationName("DeleteLabelGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLabelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLabelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an ML model currently available for Amazon Lookout for Equipment. This will prevent it from being used
     * with an inference scheduler, even one that is already set up.
     * </p>
     *
     * @param deleteModelRequest
     * @return Result of the DeleteModel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DeleteModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteModelResponse deleteModel(DeleteModelRequest deleteModelRequest) throws ResourceNotFoundException,
            ThrottlingException, InternalServerException, ConflictException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteModelRequest, DeleteModelResponse>()
                    .withOperationName("DeleteModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information on a specific data ingestion job such as creation time, dataset ARN, and status.
     * </p>
     *
     * @param describeDataIngestionJobRequest
     * @return Result of the DescribeDataIngestionJob operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DescribeDataIngestionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeDataIngestionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataIngestionJobResponse describeDataIngestionJob(
            DescribeDataIngestionJobRequest describeDataIngestionJobRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataIngestionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataIngestionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataIngestionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataIngestionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDataIngestionJobRequest, DescribeDataIngestionJobResponse>()
                            .withOperationName("DescribeDataIngestionJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeDataIngestionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDataIngestionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a JSON description of the data in each time series dataset, including names, column names, and data
     * types.
     * </p>
     *
     * @param describeDatasetRequest
     * @return Result of the DescribeDataset operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDatasetResponse describeDataset(DescribeDatasetRequest describeDatasetRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");

            return clientHandler.execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                    .withOperationName("DescribeDataset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Specifies information about the inference scheduler being used, including name, model, status, and associated
     * metadata
     * </p>
     *
     * @param describeInferenceSchedulerRequest
     * @return Result of the DescribeInferenceScheduler operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DescribeInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeInferenceSchedulerResponse describeInferenceScheduler(
            DescribeInferenceSchedulerRequest describeInferenceSchedulerRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeInferenceSchedulerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInferenceScheduler");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeInferenceSchedulerRequest, DescribeInferenceSchedulerResponse>()
                            .withOperationName("DescribeInferenceScheduler").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeInferenceSchedulerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeInferenceSchedulerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the name of the label.
     * </p>
     *
     * @param describeLabelRequest
     * @return Result of the DescribeLabel operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DescribeLabel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeLabel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLabelResponse describeLabel(DescribeLabelRequest describeLabelRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLabelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeLabelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLabelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLabel");

            return clientHandler.execute(new ClientExecutionParams<DescribeLabelRequest, DescribeLabelResponse>()
                    .withOperationName("DescribeLabel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeLabelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLabelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the label group.
     * </p>
     *
     * @param describeLabelGroupRequest
     * @return Result of the DescribeLabelGroup operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DescribeLabelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeLabelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLabelGroupResponse describeLabelGroup(DescribeLabelGroupRequest describeLabelGroupRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLabelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLabelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLabelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLabelGroup");

            return clientHandler.execute(new ClientExecutionParams<DescribeLabelGroupRequest, DescribeLabelGroupResponse>()
                    .withOperationName("DescribeLabelGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeLabelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLabelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a JSON containing the overall information about a specific ML model, including model name and ARN,
     * dataset, training and evaluation information, status, and so on.
     * </p>
     *
     * @param describeModelRequest
     * @return Result of the DescribeModel operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.DescribeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeModelResponse describeModel(DescribeModelRequest describeModelRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeModel");

            return clientHandler.execute(new ClientExecutionParams<DescribeModelRequest, DescribeModelResponse>()
                    .withOperationName("DescribeModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of all data ingestion jobs, including dataset name and ARN, S3 location of the input data,
     * status, and so on.
     * </p>
     *
     * @param listDataIngestionJobsRequest
     * @return Result of the ListDataIngestionJobs operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListDataIngestionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListDataIngestionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataIngestionJobsResponse listDataIngestionJobs(ListDataIngestionJobsRequest listDataIngestionJobsRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataIngestionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataIngestionJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataIngestionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataIngestionJobs");

            return clientHandler.execute(new ClientExecutionParams<ListDataIngestionJobsRequest, ListDataIngestionJobsResponse>()
                    .withOperationName("ListDataIngestionJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDataIngestionJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDataIngestionJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all datasets currently available in your account, filtering on the dataset name.
     * </p>
     *
     * @param listDatasetsRequest
     * @return Result of the ListDatasets operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListDatasets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDatasetsResponse listDatasets(ListDatasetsRequest listDatasetsRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatasetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");

            return clientHandler.execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                    .withOperationName("ListDatasets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDatasetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all inference events that have been found for the specified inference scheduler.
     * </p>
     *
     * @param listInferenceEventsRequest
     * @return Result of the ListInferenceEvents operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListInferenceEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInferenceEventsResponse listInferenceEvents(ListInferenceEventsRequest listInferenceEventsRequest)
            throws ValidationException, ThrottlingException, ResourceNotFoundException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInferenceEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInferenceEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInferenceEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInferenceEvents");

            return clientHandler.execute(new ClientExecutionParams<ListInferenceEventsRequest, ListInferenceEventsResponse>()
                    .withOperationName("ListInferenceEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listInferenceEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInferenceEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all inference executions that have been performed by the specified inference scheduler.
     * </p>
     *
     * @param listInferenceExecutionsRequest
     * @return Result of the ListInferenceExecutions operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListInferenceExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInferenceExecutionsResponse listInferenceExecutions(ListInferenceExecutionsRequest listInferenceExecutionsRequest)
            throws ValidationException, ThrottlingException, ResourceNotFoundException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInferenceExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInferenceExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInferenceExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInferenceExecutions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListInferenceExecutionsRequest, ListInferenceExecutionsResponse>()
                            .withOperationName("ListInferenceExecutions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listInferenceExecutionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListInferenceExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all inference schedulers currently available for your account.
     * </p>
     *
     * @param listInferenceSchedulersRequest
     * @return Result of the ListInferenceSchedulers operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListInferenceSchedulers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceSchedulers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInferenceSchedulersResponse listInferenceSchedulers(ListInferenceSchedulersRequest listInferenceSchedulersRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInferenceSchedulersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInferenceSchedulersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInferenceSchedulersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInferenceSchedulers");

            return clientHandler
                    .execute(new ClientExecutionParams<ListInferenceSchedulersRequest, ListInferenceSchedulersResponse>()
                            .withOperationName("ListInferenceSchedulers").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listInferenceSchedulersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListInferenceSchedulersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the label groups.
     * </p>
     *
     * @param listLabelGroupsRequest
     * @return Result of the ListLabelGroups operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListLabelGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListLabelGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLabelGroupsResponse listLabelGroups(ListLabelGroupsRequest listLabelGroupsRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLabelGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLabelGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLabelGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLabelGroups");

            return clientHandler.execute(new ClientExecutionParams<ListLabelGroupsRequest, ListLabelGroupsResponse>()
                    .withOperationName("ListLabelGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLabelGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLabelGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of labels.
     * </p>
     *
     * @param listLabelsRequest
     * @return Result of the ListLabels operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListLabels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListLabels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListLabelsResponse listLabels(ListLabelsRequest listLabelsRequest) throws ValidationException, ThrottlingException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLabelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLabelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLabelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLabels");

            return clientHandler
                    .execute(new ClientExecutionParams<ListLabelsRequest, ListLabelsResponse>().withOperationName("ListLabels")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listLabelsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListLabelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates a list of all models in the account, including model name and ARN, dataset, and status.
     * </p>
     *
     * @param listModelsRequest
     * @return Result of the ListModels operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListModelsResponse listModels(ListModelsRequest listModelsRequest) throws ValidationException, ThrottlingException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModels");

            return clientHandler
                    .execute(new ClientExecutionParams<ListModelsRequest, ListModelsResponse>().withOperationName("ListModels")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listModelsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists statistics about the data collected for each of the sensors that have been successfully ingested in the
     * particular dataset. Can also be used to retreive Sensor Statistics for a previous ingestion job.
     * </p>
     *
     * @param listSensorStatisticsRequest
     * @return Result of the ListSensorStatistics operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListSensorStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListSensorStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSensorStatisticsResponse listSensorStatistics(ListSensorStatisticsRequest listSensorStatisticsRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSensorStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSensorStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSensorStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSensorStatistics");

            return clientHandler.execute(new ClientExecutionParams<ListSensorStatisticsRequest, ListSensorStatisticsResponse>()
                    .withOperationName("ListSensorStatistics").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSensorStatisticsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSensorStatisticsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the tags for a specified resource, including key and value.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a data ingestion job. Amazon Lookout for Equipment returns the job status.
     * </p>
     *
     * @param startDataIngestionJobRequest
     * @return Result of the StartDataIngestionJob operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.StartDataIngestionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/StartDataIngestionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDataIngestionJobResponse startDataIngestionJob(StartDataIngestionJobRequest startDataIngestionJobRequest)
            throws ValidationException, ResourceNotFoundException, ConflictException, ThrottlingException,
            ServiceQuotaExceededException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDataIngestionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDataIngestionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDataIngestionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDataIngestionJob");

            return clientHandler.execute(new ClientExecutionParams<StartDataIngestionJobRequest, StartDataIngestionJobResponse>()
                    .withOperationName("StartDataIngestionJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startDataIngestionJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDataIngestionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an inference scheduler.
     * </p>
     *
     * @param startInferenceSchedulerRequest
     * @return Result of the StartInferenceScheduler operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.StartInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/StartInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartInferenceSchedulerResponse startInferenceScheduler(StartInferenceSchedulerRequest startInferenceSchedulerRequest)
            throws ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartInferenceSchedulerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInferenceScheduler");

            return clientHandler
                    .execute(new ClientExecutionParams<StartInferenceSchedulerRequest, StartInferenceSchedulerResponse>()
                            .withOperationName("StartInferenceScheduler").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startInferenceSchedulerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartInferenceSchedulerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an inference scheduler.
     * </p>
     *
     * @param stopInferenceSchedulerRequest
     * @return Result of the StopInferenceScheduler operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.StopInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/StopInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopInferenceSchedulerResponse stopInferenceScheduler(StopInferenceSchedulerRequest stopInferenceSchedulerRequest)
            throws ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopInferenceSchedulerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopInferenceScheduler");

            return clientHandler
                    .execute(new ClientExecutionParams<StopInferenceSchedulerRequest, StopInferenceSchedulerResponse>()
                            .withOperationName("StopInferenceScheduler").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(stopInferenceSchedulerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopInferenceSchedulerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a given tag to a resource in your account. A tag is a key-value pair which can be added to an Amazon
     * Lookout for Equipment resource as metadata. Tags can be used for organizing your resources as well as helping you
     * to search and filter by tag. Multiple tags can be added to a resource, either when you create it, or later. Up to
     * 50 tags can be associated with each resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ServiceQuotaExceededException
     *         Resource limitations have been exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a specific tag from a given resource. The tag is specified by its key.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an inference scheduler.
     * </p>
     *
     * @param updateInferenceSchedulerRequest
     * @return Result of the UpdateInferenceScheduler operation returned by the service.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.UpdateInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/UpdateInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateInferenceSchedulerResponse updateInferenceScheduler(
            UpdateInferenceSchedulerRequest updateInferenceSchedulerRequest) throws ConflictException, ResourceNotFoundException,
            ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateInferenceSchedulerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInferenceScheduler");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateInferenceSchedulerRequest, UpdateInferenceSchedulerResponse>()
                            .withOperationName("UpdateInferenceScheduler").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateInferenceSchedulerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateInferenceSchedulerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the label group.
     * </p>
     *
     * @param updateLabelGroupRequest
     * @return Result of the UpdateLabelGroup operation returned by the service.
     * @throws ConflictException
     *         The request could not be completed due to a conflict with the current state of the target resource.
     * @throws ResourceNotFoundException
     *         The resource requested could not be found. Verify the resource ID and retry your request.
     * @throws ValidationException
     *         The input fails to satisfy constraints specified by Amazon Lookout for Equipment or a related AWS service
     *         that's being utilized.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         The request could not be completed because you do not have access to the resource.
     * @throws InternalServerException
     *         Processing of the request has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutEquipmentException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutEquipmentClient.UpdateLabelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/UpdateLabelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLabelGroupResponse updateLabelGroup(UpdateLabelGroupRequest updateLabelGroupRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, LookoutEquipmentException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLabelGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLabelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLabelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLabelGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateLabelGroupRequest, UpdateLabelGroupResponse>()
                    .withOperationName("UpdateLabelGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLabelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLabelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LookoutEquipmentException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final LookoutEquipmentServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
