/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lexruntimev2.internal.LexRuntimeV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.lexruntimev2.model.AccessDeniedException;
import software.amazon.awssdk.services.lexruntimev2.model.BadGatewayException;
import software.amazon.awssdk.services.lexruntimev2.model.ConflictException;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.DeleteSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.DependencyFailedException;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.GetSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.InternalServerException;
import software.amazon.awssdk.services.lexruntimev2.model.LexRuntimeV2Exception;
import software.amazon.awssdk.services.lexruntimev2.model.LexRuntimeV2Request;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionRequest;
import software.amazon.awssdk.services.lexruntimev2.model.PutSessionResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeTextResponse;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceRequest;
import software.amazon.awssdk.services.lexruntimev2.model.RecognizeUtteranceResponse;
import software.amazon.awssdk.services.lexruntimev2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lexruntimev2.model.ThrottlingException;
import software.amazon.awssdk.services.lexruntimev2.model.ValidationException;
import software.amazon.awssdk.services.lexruntimev2.transform.DeleteSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.PutSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.RecognizeTextRequestMarshaller;
import software.amazon.awssdk.services.lexruntimev2.transform.RecognizeUtteranceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LexRuntimeV2Client}.
 *
 * @see LexRuntimeV2Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLexRuntimeV2Client implements LexRuntimeV2Client {
    private static final Logger log = Logger.loggerFor(DefaultLexRuntimeV2Client.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLexRuntimeV2Client(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     * <p>
     * You can use this operation to restart a conversation with a bot. When you remove a session, the entire history of
     * the session is removed so that you can start again.
     * </p>
     * <p>
     * You don't need to delete a session. Sessions have a time limit and will expire. Set the session time limit when
     * you create the bot. The default is 5 minutes, but you can specify anything between 1 minute and 24 hours.
     * </p>
     * <p>
     * If you specify a bot or alias ID that doesn't exist, you receive a <code>BadRequestException.</code>
     * </p>
     * <p>
     * If the locale doesn't exist in the bot, or if the locale hasn't been enables for the alias, you receive a
     * <code>BadRequestException</code>.
     * </p>
     *
     * @param deleteSessionRequest
     * @return Result of the DeleteSession operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/DeleteSession" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteSessionResponse deleteSession(DeleteSessionRequest deleteSessionRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSession");

            return clientHandler.execute(new ClientExecutionParams<DeleteSessionRequest, DeleteSessionResponse>()
                    .withOperationName("DeleteSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user.
     * </p>
     * <p>
     * For example, you can use this operation to retrieve session information for a user that has left a long-running
     * session in use.
     * </p>
     * <p>
     * If the bot, alias, or session identifier doesn't exist, Amazon Lex V2 returns a <code>BadRequestException</code>.
     * If the locale doesn't exist or is not enabled for the alias, you receive a <code>BadRequestException</code>.
     * </p>
     *
     * @param getSessionRequest
     * @return Result of the GetSession operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSessionResponse getSession(GetSessionRequest getSessionRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, LexRuntimeV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");

            return clientHandler.execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>()
                    .withOperationName("GetSession").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSessionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex V2 bot. Use this operation to enable
     * your application to set the state of the bot.
     * </p>
     *
     * @param putSessionRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled PutSessionResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        If the requested content type was audio, the audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT putSession(PutSessionRequest putSessionRequest,
            ResponseTransformer<PutSessionResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<PutSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSession");

            return clientHandler.execute(
                    new ClientExecutionParams<PutSessionRequest, PutSessionResponse>().withOperationName("PutSession")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(putSessionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutSessionRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. Client applications use this API to send requests to Amazon Lex V2 at runtime.
     * Amazon Lex V2 then interprets the user input using the machine learning model that it build for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex V2 returns the next message to convey to the user and an optional response card to
     * display.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeTextRequest
     * @return Result of the RecognizeText operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeText" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RecognizeTextResponse recognizeText(RecognizeTextRequest recognizeTextRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RecognizeTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RecognizeTextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(recognizeTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, recognizeTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecognizeText");

            return clientHandler.execute(new ClientExecutionParams<RecognizeTextRequest, RecognizeTextResponse>()
                    .withOperationName("RecognizeText").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(recognizeTextRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RecognizeTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends user input to Amazon Lex V2. You can send text or speech. Clients use this API to send text and audio
     * requests to Amazon Lex V2 at runtime. Amazon Lex V2 interprets the user input using the machine learning model
     * built for the bot.
     * </p>
     * <p>
     * The following request fields must be compressed with gzip and then base64 encoded before you send them to Amazon
     * Lex V2.
     * </p>
     * <ul>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following response fields are compressed using gzip and then base64 encoded by Amazon Lex V2. Before you can
     * use these fields, you must decode and decompress them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * inputTranscript
     * </p>
     * </li>
     * <li>
     * <p>
     * interpretations
     * </p>
     * </li>
     * <li>
     * <p>
     * messages
     * </p>
     * </li>
     * <li>
     * <p>
     * requestAttributes
     * </p>
     * </li>
     * <li>
     * <p>
     * sessionState
     * </p>
     * </li>
     * </ul>
     * <p>
     * The example contains a Java application that compresses and encodes a Java object to send to Amazon Lex V2, and a
     * second that decodes and decompresses a response from Amazon Lex V2.
     * </p>
     * <p>
     * If the optional post-fulfillment response is specified, the messages are returned as follows. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/API_PostFulfillmentStatusSpecification.html"
     * >PostFulfillmentStatusSpecification</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Success message</b> - Returned if the Lambda function completes successfully and the intent state is fulfilled
     * or ready fulfillment if the message is present.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Failed message</b> - The failed message is returned if the Lambda function throws an exception or if the
     * Lambda function returns a failed intent state without a message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Timeout message</b> - If you don't configure a timeout message and a timeout, and the Lambda function doesn't
     * return within 30 seconds, the timeout message is returned. If you configure a timeout, the timeout message is
     * returned when the period times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/lexv2/latest/dg/streaming-progress.html#progress-complete.html">Completion
     * message</a>.
     * </p>
     *
     * @param recognizeUtteranceRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>requestContentType</code>
     *        parameter.
     *        </p>
     *        '
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        RecognizeUtteranceResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The prompt or statement to send to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex V2 did not understand the user intent, it sends the
     *        <code>clarificationPrompt</code> configured for the bot. If the intent requires confirmation before taking
     *        the fulfillment action, it sends the <code>confirmationPrompt</code>. Another example: Suppose that the
     *        Lambda function successfully fulfilled the intent, and sent a message to convey to the user. Then Amazon
     *        Lex V2 sends that message in the response.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws ConflictException
     * @throws DependencyFailedException
     * @throws BadGatewayException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeV2Client.RecognizeUtterance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex.v2-2020-08-07/RecognizeUtterance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT recognizeUtterance(RecognizeUtteranceRequest recognizeUtteranceRequest, RequestBody requestBody,
            ResponseTransformer<RecognizeUtteranceResponse, ReturnT> responseTransformer) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, ThrottlingException, InternalServerException, ConflictException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeV2Exception {
        recognizeUtteranceRequest = applySignerOverride(recognizeUtteranceRequest, Aws4UnsignedPayloadSigner.create());
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<RecognizeUtteranceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RecognizeUtteranceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(recognizeUtteranceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, recognizeUtteranceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RecognizeUtterance");

            return clientHandler.execute(
                    new ClientExecutionParams<RecognizeUtteranceRequest, RecognizeUtteranceResponse>()
                            .withOperationName("RecognizeUtterance")
                            .withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(recognizeUtteranceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withRequestBody(requestBody)
                            .withMarshaller(
                                    StreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new RecognizeUtteranceRequestMarshaller(protocolFactory))
                                            .requestBody(requestBody).transferEncoding(true).useHttp2(true).build()),
                    responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends LexRuntimeV2Request> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        LexRuntimeV2ServiceClientConfigurationBuilder serviceConfigBuilder = new LexRuntimeV2ServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LexRuntimeV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .contentType("application/json")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build());
    }

    @Override
    public final LexRuntimeV2ServiceClientConfiguration serviceClientConfiguration() {
        return new LexRuntimeV2ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
