/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalytics.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This documentation is for version 1 of the Amazon Kinesis Data Analytics API, which only supports SQL applications.
 * Version 2 of the API supports SQL and Java applications. For more information about version 2, see <a
 * href="/kinesisanalytics/latest/apiv2/Welcome.html">Amazon Kinesis Data Analytics API V2 Documentation</a>.
 * </p>
 * </note>
 * <p>
 * Provides a description of the application, including the application Amazon Resource Name (ARN), status, latest
 * version, and input and output configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ApplicationDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<ApplicationDetail.Builder, ApplicationDetail> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationName").getter(getter(ApplicationDetail::applicationName))
            .setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> APPLICATION_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationDescription").getter(getter(ApplicationDetail::applicationDescription))
            .setter(setter(Builder::applicationDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationDescription").build())
            .build();

    private static final SdkField<String> APPLICATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationARN").getter(getter(ApplicationDetail::applicationARN))
            .setter(setter(Builder::applicationARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationARN").build()).build();

    private static final SdkField<String> APPLICATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationStatus").getter(getter(ApplicationDetail::applicationStatusAsString))
            .setter(setter(Builder::applicationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationStatus").build()).build();

    private static final SdkField<Instant> CREATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTimestamp").getter(getter(ApplicationDetail::createTimestamp))
            .setter(setter(Builder::createTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTimestamp").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateTimestamp").getter(getter(ApplicationDetail::lastUpdateTimestamp))
            .setter(setter(Builder::lastUpdateTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTimestamp").build())
            .build();

    private static final SdkField<List<InputDescription>> INPUT_DESCRIPTIONS_FIELD = SdkField
            .<List<InputDescription>> builder(MarshallingType.LIST)
            .memberName("InputDescriptions")
            .getter(getter(ApplicationDetail::inputDescriptions))
            .setter(setter(Builder::inputDescriptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDescriptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InputDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(InputDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<OutputDescription>> OUTPUT_DESCRIPTIONS_FIELD = SdkField
            .<List<OutputDescription>> builder(MarshallingType.LIST)
            .memberName("OutputDescriptions")
            .getter(getter(ApplicationDetail::outputDescriptions))
            .setter(setter(Builder::outputDescriptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputDescriptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OutputDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(OutputDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ReferenceDataSourceDescription>> REFERENCE_DATA_SOURCE_DESCRIPTIONS_FIELD = SdkField
            .<List<ReferenceDataSourceDescription>> builder(MarshallingType.LIST)
            .memberName("ReferenceDataSourceDescriptions")
            .getter(getter(ApplicationDetail::referenceDataSourceDescriptions))
            .setter(setter(Builder::referenceDataSourceDescriptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferenceDataSourceDescriptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReferenceDataSourceDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReferenceDataSourceDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<CloudWatchLoggingOptionDescription>> CLOUD_WATCH_LOGGING_OPTION_DESCRIPTIONS_FIELD = SdkField
            .<List<CloudWatchLoggingOptionDescription>> builder(MarshallingType.LIST)
            .memberName("CloudWatchLoggingOptionDescriptions")
            .getter(getter(ApplicationDetail::cloudWatchLoggingOptionDescriptions))
            .setter(setter(Builder::cloudWatchLoggingOptionDescriptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CloudWatchLoggingOptionDescriptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CloudWatchLoggingOptionDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(CloudWatchLoggingOptionDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APPLICATION_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationCode").getter(getter(ApplicationDetail::applicationCode))
            .setter(setter(Builder::applicationCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationCode").build()).build();

    private static final SdkField<Long> APPLICATION_VERSION_ID_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ApplicationVersionId").getter(getter(ApplicationDetail::applicationVersionId))
            .setter(setter(Builder::applicationVersionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationVersionId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            APPLICATION_DESCRIPTION_FIELD, APPLICATION_ARN_FIELD, APPLICATION_STATUS_FIELD, CREATE_TIMESTAMP_FIELD,
            LAST_UPDATE_TIMESTAMP_FIELD, INPUT_DESCRIPTIONS_FIELD, OUTPUT_DESCRIPTIONS_FIELD,
            REFERENCE_DATA_SOURCE_DESCRIPTIONS_FIELD, CLOUD_WATCH_LOGGING_OPTION_DESCRIPTIONS_FIELD, APPLICATION_CODE_FIELD,
            APPLICATION_VERSION_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String applicationName;

    private final String applicationDescription;

    private final String applicationARN;

    private final String applicationStatus;

    private final Instant createTimestamp;

    private final Instant lastUpdateTimestamp;

    private final List<InputDescription> inputDescriptions;

    private final List<OutputDescription> outputDescriptions;

    private final List<ReferenceDataSourceDescription> referenceDataSourceDescriptions;

    private final List<CloudWatchLoggingOptionDescription> cloudWatchLoggingOptionDescriptions;

    private final String applicationCode;

    private final Long applicationVersionId;

    private ApplicationDetail(BuilderImpl builder) {
        this.applicationName = builder.applicationName;
        this.applicationDescription = builder.applicationDescription;
        this.applicationARN = builder.applicationARN;
        this.applicationStatus = builder.applicationStatus;
        this.createTimestamp = builder.createTimestamp;
        this.lastUpdateTimestamp = builder.lastUpdateTimestamp;
        this.inputDescriptions = builder.inputDescriptions;
        this.outputDescriptions = builder.outputDescriptions;
        this.referenceDataSourceDescriptions = builder.referenceDataSourceDescriptions;
        this.cloudWatchLoggingOptionDescriptions = builder.cloudWatchLoggingOptionDescriptions;
        this.applicationCode = builder.applicationCode;
        this.applicationVersionId = builder.applicationVersionId;
    }

    /**
     * <p>
     * Name of the application.
     * </p>
     * 
     * @return Name of the application.
     */
    public final String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * Description of the application.
     * </p>
     * 
     * @return Description of the application.
     */
    public final String applicationDescription() {
        return applicationDescription;
    }

    /**
     * <p>
     * ARN of the application.
     * </p>
     * 
     * @return ARN of the application.
     */
    public final String applicationARN() {
        return applicationARN;
    }

    /**
     * <p>
     * Status of the application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #applicationStatus}
     * will return {@link ApplicationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #applicationStatusAsString}.
     * </p>
     * 
     * @return Status of the application.
     * @see ApplicationStatus
     */
    public final ApplicationStatus applicationStatus() {
        return ApplicationStatus.fromValue(applicationStatus);
    }

    /**
     * <p>
     * Status of the application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #applicationStatus}
     * will return {@link ApplicationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #applicationStatusAsString}.
     * </p>
     * 
     * @return Status of the application.
     * @see ApplicationStatus
     */
    public final String applicationStatusAsString() {
        return applicationStatus;
    }

    /**
     * <p>
     * Time stamp when the application version was created.
     * </p>
     * 
     * @return Time stamp when the application version was created.
     */
    public final Instant createTimestamp() {
        return createTimestamp;
    }

    /**
     * <p>
     * Time stamp when the application was last updated.
     * </p>
     * 
     * @return Time stamp when the application was last updated.
     */
    public final Instant lastUpdateTimestamp() {
        return lastUpdateTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the InputDescriptions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInputDescriptions() {
        return inputDescriptions != null && !(inputDescriptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the application input configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring Application
     * Input</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputDescriptions} method.
     * </p>
     * 
     * @return Describes the application input configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
     *         Application Input</a>.
     */
    public final List<InputDescription> inputDescriptions() {
        return inputDescriptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputDescriptions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputDescriptions() {
        return outputDescriptions != null && !(outputDescriptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the application output configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring Application
     * Output</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputDescriptions} method.
     * </p>
     * 
     * @return Describes the application output configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring
     *         Application Output</a>.
     */
    public final List<OutputDescription> outputDescriptions() {
        return outputDescriptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReferenceDataSourceDescriptions
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasReferenceDataSourceDescriptions() {
        return referenceDataSourceDescriptions != null && !(referenceDataSourceDescriptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes reference data sources configured for the application. For more information, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring Application
     * Input</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReferenceDataSourceDescriptions}
     * method.
     * </p>
     * 
     * @return Describes reference data sources configured for the application. For more information, see <a
     *         href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
     *         Application Input</a>.
     */
    public final List<ReferenceDataSourceDescription> referenceDataSourceDescriptions() {
        return referenceDataSourceDescriptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the CloudWatchLoggingOptionDescriptions
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasCloudWatchLoggingOptionDescriptions() {
        return cloudWatchLoggingOptionDescriptions != null
                && !(cloudWatchLoggingOptionDescriptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the CloudWatch log streams that are configured to receive application messages. For more information
     * about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
     * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
     * CloudWatch Logs</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCloudWatchLoggingOptionDescriptions}
     * method.
     * </p>
     * 
     * @return Describes the CloudWatch log streams that are configured to receive application messages. For more
     *         information about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
     *         href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
     *         CloudWatch Logs</a>.
     */
    public final List<CloudWatchLoggingOptionDescription> cloudWatchLoggingOptionDescriptions() {
        return cloudWatchLoggingOptionDescriptions;
    }

    /**
     * <p>
     * Returns the application code that you provided to perform data analysis on any of the in-application streams in
     * your application.
     * </p>
     * 
     * @return Returns the application code that you provided to perform data analysis on any of the in-application
     *         streams in your application.
     */
    public final String applicationCode() {
        return applicationCode;
    }

    /**
     * <p>
     * Provides the current application version.
     * </p>
     * 
     * @return Provides the current application version.
     */
    public final Long applicationVersionId() {
        return applicationVersionId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(applicationDescription());
        hashCode = 31 * hashCode + Objects.hashCode(applicationARN());
        hashCode = 31 * hashCode + Objects.hashCode(applicationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasInputDescriptions() ? inputDescriptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputDescriptions() ? outputDescriptions() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasReferenceDataSourceDescriptions() ? referenceDataSourceDescriptions() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasCloudWatchLoggingOptionDescriptions() ? cloudWatchLoggingOptionDescriptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(applicationCode());
        hashCode = 31 * hashCode + Objects.hashCode(applicationVersionId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ApplicationDetail)) {
            return false;
        }
        ApplicationDetail other = (ApplicationDetail) obj;
        return Objects.equals(applicationName(), other.applicationName())
                && Objects.equals(applicationDescription(), other.applicationDescription())
                && Objects.equals(applicationARN(), other.applicationARN())
                && Objects.equals(applicationStatusAsString(), other.applicationStatusAsString())
                && Objects.equals(createTimestamp(), other.createTimestamp())
                && Objects.equals(lastUpdateTimestamp(), other.lastUpdateTimestamp())
                && hasInputDescriptions() == other.hasInputDescriptions()
                && Objects.equals(inputDescriptions(), other.inputDescriptions())
                && hasOutputDescriptions() == other.hasOutputDescriptions()
                && Objects.equals(outputDescriptions(), other.outputDescriptions())
                && hasReferenceDataSourceDescriptions() == other.hasReferenceDataSourceDescriptions()
                && Objects.equals(referenceDataSourceDescriptions(), other.referenceDataSourceDescriptions())
                && hasCloudWatchLoggingOptionDescriptions() == other.hasCloudWatchLoggingOptionDescriptions()
                && Objects.equals(cloudWatchLoggingOptionDescriptions(), other.cloudWatchLoggingOptionDescriptions())
                && Objects.equals(applicationCode(), other.applicationCode())
                && Objects.equals(applicationVersionId(), other.applicationVersionId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("ApplicationDetail")
                .add("ApplicationName", applicationName())
                .add("ApplicationDescription", applicationDescription())
                .add("ApplicationARN", applicationARN())
                .add("ApplicationStatus", applicationStatusAsString())
                .add("CreateTimestamp", createTimestamp())
                .add("LastUpdateTimestamp", lastUpdateTimestamp())
                .add("InputDescriptions", hasInputDescriptions() ? inputDescriptions() : null)
                .add("OutputDescriptions", hasOutputDescriptions() ? outputDescriptions() : null)
                .add("ReferenceDataSourceDescriptions",
                        hasReferenceDataSourceDescriptions() ? referenceDataSourceDescriptions() : null)
                .add("CloudWatchLoggingOptionDescriptions",
                        hasCloudWatchLoggingOptionDescriptions() ? cloudWatchLoggingOptionDescriptions() : null)
                .add("ApplicationCode", applicationCode()).add("ApplicationVersionId", applicationVersionId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "ApplicationDescription":
            return Optional.ofNullable(clazz.cast(applicationDescription()));
        case "ApplicationARN":
            return Optional.ofNullable(clazz.cast(applicationARN()));
        case "ApplicationStatus":
            return Optional.ofNullable(clazz.cast(applicationStatusAsString()));
        case "CreateTimestamp":
            return Optional.ofNullable(clazz.cast(createTimestamp()));
        case "LastUpdateTimestamp":
            return Optional.ofNullable(clazz.cast(lastUpdateTimestamp()));
        case "InputDescriptions":
            return Optional.ofNullable(clazz.cast(inputDescriptions()));
        case "OutputDescriptions":
            return Optional.ofNullable(clazz.cast(outputDescriptions()));
        case "ReferenceDataSourceDescriptions":
            return Optional.ofNullable(clazz.cast(referenceDataSourceDescriptions()));
        case "CloudWatchLoggingOptionDescriptions":
            return Optional.ofNullable(clazz.cast(cloudWatchLoggingOptionDescriptions()));
        case "ApplicationCode":
            return Optional.ofNullable(clazz.cast(applicationCode()));
        case "ApplicationVersionId":
            return Optional.ofNullable(clazz.cast(applicationVersionId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ApplicationDetail, T> g) {
        return obj -> g.apply((ApplicationDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ApplicationDetail> {
        /**
         * <p>
         * Name of the application.
         * </p>
         * 
         * @param applicationName
         *        Name of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * Description of the application.
         * </p>
         * 
         * @param applicationDescription
         *        Description of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationDescription(String applicationDescription);

        /**
         * <p>
         * ARN of the application.
         * </p>
         * 
         * @param applicationARN
         *        ARN of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationARN(String applicationARN);

        /**
         * <p>
         * Status of the application.
         * </p>
         * 
         * @param applicationStatus
         *        Status of the application.
         * @see ApplicationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApplicationStatus
         */
        Builder applicationStatus(String applicationStatus);

        /**
         * <p>
         * Status of the application.
         * </p>
         * 
         * @param applicationStatus
         *        Status of the application.
         * @see ApplicationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApplicationStatus
         */
        Builder applicationStatus(ApplicationStatus applicationStatus);

        /**
         * <p>
         * Time stamp when the application version was created.
         * </p>
         * 
         * @param createTimestamp
         *        Time stamp when the application version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTimestamp(Instant createTimestamp);

        /**
         * <p>
         * Time stamp when the application was last updated.
         * </p>
         * 
         * @param lastUpdateTimestamp
         *        Time stamp when the application was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimestamp(Instant lastUpdateTimestamp);

        /**
         * <p>
         * Describes the application input configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         * Application Input</a>.
         * </p>
         * 
         * @param inputDescriptions
         *        Describes the application input configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         *        Application Input</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDescriptions(Collection<InputDescription> inputDescriptions);

        /**
         * <p>
         * Describes the application input configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         * Application Input</a>.
         * </p>
         * 
         * @param inputDescriptions
         *        Describes the application input configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         *        Application Input</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDescriptions(InputDescription... inputDescriptions);

        /**
         * <p>
         * Describes the application input configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         * Application Input</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.InputDescription.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.InputDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.InputDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #inputDescriptions(List<InputDescription>)}.
         * 
         * @param inputDescriptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kinesisanalytics.model.InputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDescriptions(java.util.Collection<InputDescription>)
         */
        Builder inputDescriptions(Consumer<InputDescription.Builder>... inputDescriptions);

        /**
         * <p>
         * Describes the application output configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring
         * Application Output</a>.
         * </p>
         * 
         * @param outputDescriptions
         *        Describes the application output configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring
         *        Application Output</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputDescriptions(Collection<OutputDescription> outputDescriptions);

        /**
         * <p>
         * Describes the application output configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring
         * Application Output</a>.
         * </p>
         * 
         * @param outputDescriptions
         *        Describes the application output configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring
         *        Application Output</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputDescriptions(OutputDescription... outputDescriptions);

        /**
         * <p>
         * Describes the application output configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-output.html">Configuring
         * Application Output</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.OutputDescription.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.OutputDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.OutputDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #outputDescriptions(List<OutputDescription>)}.
         * 
         * @param outputDescriptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kinesisanalytics.model.OutputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputDescriptions(java.util.Collection<OutputDescription>)
         */
        Builder outputDescriptions(Consumer<OutputDescription.Builder>... outputDescriptions);

        /**
         * <p>
         * Describes reference data sources configured for the application. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         * Application Input</a>.
         * </p>
         * 
         * @param referenceDataSourceDescriptions
         *        Describes reference data sources configured for the application. For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         *        Application Input</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceDataSourceDescriptions(Collection<ReferenceDataSourceDescription> referenceDataSourceDescriptions);

        /**
         * <p>
         * Describes reference data sources configured for the application. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         * Application Input</a>.
         * </p>
         * 
         * @param referenceDataSourceDescriptions
         *        Describes reference data sources configured for the application. For more information, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         *        Application Input</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceDataSourceDescriptions(ReferenceDataSourceDescription... referenceDataSourceDescriptions);

        /**
         * <p>
         * Describes reference data sources configured for the application. For more information, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/how-it-works-input.html">Configuring
         * Application Input</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.ReferenceDataSourceDescription.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.ReferenceDataSourceDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.ReferenceDataSourceDescription.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #referenceDataSourceDescriptions(List<ReferenceDataSourceDescription>)}.
         * 
         * @param referenceDataSourceDescriptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kinesisanalytics.model.ReferenceDataSourceDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #referenceDataSourceDescriptions(java.util.Collection<ReferenceDataSourceDescription>)
         */
        Builder referenceDataSourceDescriptions(
                Consumer<ReferenceDataSourceDescription.Builder>... referenceDataSourceDescriptions);

        /**
         * <p>
         * Describes the CloudWatch log streams that are configured to receive application messages. For more
         * information about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
         * CloudWatch Logs</a>.
         * </p>
         * 
         * @param cloudWatchLoggingOptionDescriptions
         *        Describes the CloudWatch log streams that are configured to receive application messages. For more
         *        information about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with
         *        Amazon CloudWatch Logs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLoggingOptionDescriptions(
                Collection<CloudWatchLoggingOptionDescription> cloudWatchLoggingOptionDescriptions);

        /**
         * <p>
         * Describes the CloudWatch log streams that are configured to receive application messages. For more
         * information about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
         * CloudWatch Logs</a>.
         * </p>
         * 
         * @param cloudWatchLoggingOptionDescriptions
         *        Describes the CloudWatch log streams that are configured to receive application messages. For more
         *        information about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
         *        href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with
         *        Amazon CloudWatch Logs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLoggingOptionDescriptions(CloudWatchLoggingOptionDescription... cloudWatchLoggingOptionDescriptions);

        /**
         * <p>
         * Describes the CloudWatch log streams that are configured to receive application messages. For more
         * information about using CloudWatch log streams with Amazon Kinesis Analytics applications, see <a
         * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/cloudwatch-logs.html">Working with Amazon
         * CloudWatch Logs</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.CloudWatchLoggingOptionDescription.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.CloudWatchLoggingOptionDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kinesisanalytics.model.CloudWatchLoggingOptionDescription.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #cloudWatchLoggingOptionDescriptions(List<CloudWatchLoggingOptionDescription>)}.
         * 
         * @param cloudWatchLoggingOptionDescriptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kinesisanalytics.model.CloudWatchLoggingOptionDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudWatchLoggingOptionDescriptions(java.util.Collection<CloudWatchLoggingOptionDescription>)
         */
        Builder cloudWatchLoggingOptionDescriptions(
                Consumer<CloudWatchLoggingOptionDescription.Builder>... cloudWatchLoggingOptionDescriptions);

        /**
         * <p>
         * Returns the application code that you provided to perform data analysis on any of the in-application streams
         * in your application.
         * </p>
         * 
         * @param applicationCode
         *        Returns the application code that you provided to perform data analysis on any of the in-application
         *        streams in your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationCode(String applicationCode);

        /**
         * <p>
         * Provides the current application version.
         * </p>
         * 
         * @param applicationVersionId
         *        Provides the current application version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationVersionId(Long applicationVersionId);
    }

    static final class BuilderImpl implements Builder {
        private String applicationName;

        private String applicationDescription;

        private String applicationARN;

        private String applicationStatus;

        private Instant createTimestamp;

        private Instant lastUpdateTimestamp;

        private List<InputDescription> inputDescriptions = DefaultSdkAutoConstructList.getInstance();

        private List<OutputDescription> outputDescriptions = DefaultSdkAutoConstructList.getInstance();

        private List<ReferenceDataSourceDescription> referenceDataSourceDescriptions = DefaultSdkAutoConstructList.getInstance();

        private List<CloudWatchLoggingOptionDescription> cloudWatchLoggingOptionDescriptions = DefaultSdkAutoConstructList
                .getInstance();

        private String applicationCode;

        private Long applicationVersionId;

        private BuilderImpl() {
        }

        private BuilderImpl(ApplicationDetail model) {
            applicationName(model.applicationName);
            applicationDescription(model.applicationDescription);
            applicationARN(model.applicationARN);
            applicationStatus(model.applicationStatus);
            createTimestamp(model.createTimestamp);
            lastUpdateTimestamp(model.lastUpdateTimestamp);
            inputDescriptions(model.inputDescriptions);
            outputDescriptions(model.outputDescriptions);
            referenceDataSourceDescriptions(model.referenceDataSourceDescriptions);
            cloudWatchLoggingOptionDescriptions(model.cloudWatchLoggingOptionDescriptions);
            applicationCode(model.applicationCode);
            applicationVersionId(model.applicationVersionId);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final String getApplicationDescription() {
            return applicationDescription;
        }

        public final void setApplicationDescription(String applicationDescription) {
            this.applicationDescription = applicationDescription;
        }

        @Override
        public final Builder applicationDescription(String applicationDescription) {
            this.applicationDescription = applicationDescription;
            return this;
        }

        public final String getApplicationARN() {
            return applicationARN;
        }

        public final void setApplicationARN(String applicationARN) {
            this.applicationARN = applicationARN;
        }

        @Override
        public final Builder applicationARN(String applicationARN) {
            this.applicationARN = applicationARN;
            return this;
        }

        public final String getApplicationStatus() {
            return applicationStatus;
        }

        public final void setApplicationStatus(String applicationStatus) {
            this.applicationStatus = applicationStatus;
        }

        @Override
        public final Builder applicationStatus(String applicationStatus) {
            this.applicationStatus = applicationStatus;
            return this;
        }

        @Override
        public final Builder applicationStatus(ApplicationStatus applicationStatus) {
            this.applicationStatus(applicationStatus == null ? null : applicationStatus.toString());
            return this;
        }

        public final Instant getCreateTimestamp() {
            return createTimestamp;
        }

        public final void setCreateTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
        }

        @Override
        public final Builder createTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
            return this;
        }

        public final Instant getLastUpdateTimestamp() {
            return lastUpdateTimestamp;
        }

        public final void setLastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
        }

        @Override
        public final Builder lastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
            return this;
        }

        public final List<InputDescription.Builder> getInputDescriptions() {
            List<InputDescription.Builder> result = InputDescriptionsCopier.copyToBuilder(this.inputDescriptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInputDescriptions(Collection<InputDescription.BuilderImpl> inputDescriptions) {
            this.inputDescriptions = InputDescriptionsCopier.copyFromBuilder(inputDescriptions);
        }

        @Override
        public final Builder inputDescriptions(Collection<InputDescription> inputDescriptions) {
            this.inputDescriptions = InputDescriptionsCopier.copy(inputDescriptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputDescriptions(InputDescription... inputDescriptions) {
            inputDescriptions(Arrays.asList(inputDescriptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputDescriptions(Consumer<InputDescription.Builder>... inputDescriptions) {
            inputDescriptions(Stream.of(inputDescriptions).map(c -> InputDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<OutputDescription.Builder> getOutputDescriptions() {
            List<OutputDescription.Builder> result = OutputDescriptionsCopier.copyToBuilder(this.outputDescriptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputDescriptions(Collection<OutputDescription.BuilderImpl> outputDescriptions) {
            this.outputDescriptions = OutputDescriptionsCopier.copyFromBuilder(outputDescriptions);
        }

        @Override
        public final Builder outputDescriptions(Collection<OutputDescription> outputDescriptions) {
            this.outputDescriptions = OutputDescriptionsCopier.copy(outputDescriptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputDescriptions(OutputDescription... outputDescriptions) {
            outputDescriptions(Arrays.asList(outputDescriptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputDescriptions(Consumer<OutputDescription.Builder>... outputDescriptions) {
            outputDescriptions(Stream.of(outputDescriptions).map(c -> OutputDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ReferenceDataSourceDescription.Builder> getReferenceDataSourceDescriptions() {
            List<ReferenceDataSourceDescription.Builder> result = ReferenceDataSourceDescriptionsCopier
                    .copyToBuilder(this.referenceDataSourceDescriptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setReferenceDataSourceDescriptions(
                Collection<ReferenceDataSourceDescription.BuilderImpl> referenceDataSourceDescriptions) {
            this.referenceDataSourceDescriptions = ReferenceDataSourceDescriptionsCopier
                    .copyFromBuilder(referenceDataSourceDescriptions);
        }

        @Override
        public final Builder referenceDataSourceDescriptions(
                Collection<ReferenceDataSourceDescription> referenceDataSourceDescriptions) {
            this.referenceDataSourceDescriptions = ReferenceDataSourceDescriptionsCopier.copy(referenceDataSourceDescriptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceDataSourceDescriptions(ReferenceDataSourceDescription... referenceDataSourceDescriptions) {
            referenceDataSourceDescriptions(Arrays.asList(referenceDataSourceDescriptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceDataSourceDescriptions(
                Consumer<ReferenceDataSourceDescription.Builder>... referenceDataSourceDescriptions) {
            referenceDataSourceDescriptions(Stream.of(referenceDataSourceDescriptions)
                    .map(c -> ReferenceDataSourceDescription.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<CloudWatchLoggingOptionDescription.Builder> getCloudWatchLoggingOptionDescriptions() {
            List<CloudWatchLoggingOptionDescription.Builder> result = CloudWatchLoggingOptionDescriptionsCopier
                    .copyToBuilder(this.cloudWatchLoggingOptionDescriptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCloudWatchLoggingOptionDescriptions(
                Collection<CloudWatchLoggingOptionDescription.BuilderImpl> cloudWatchLoggingOptionDescriptions) {
            this.cloudWatchLoggingOptionDescriptions = CloudWatchLoggingOptionDescriptionsCopier
                    .copyFromBuilder(cloudWatchLoggingOptionDescriptions);
        }

        @Override
        public final Builder cloudWatchLoggingOptionDescriptions(
                Collection<CloudWatchLoggingOptionDescription> cloudWatchLoggingOptionDescriptions) {
            this.cloudWatchLoggingOptionDescriptions = CloudWatchLoggingOptionDescriptionsCopier
                    .copy(cloudWatchLoggingOptionDescriptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cloudWatchLoggingOptionDescriptions(
                CloudWatchLoggingOptionDescription... cloudWatchLoggingOptionDescriptions) {
            cloudWatchLoggingOptionDescriptions(Arrays.asList(cloudWatchLoggingOptionDescriptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cloudWatchLoggingOptionDescriptions(
                Consumer<CloudWatchLoggingOptionDescription.Builder>... cloudWatchLoggingOptionDescriptions) {
            cloudWatchLoggingOptionDescriptions(Stream.of(cloudWatchLoggingOptionDescriptions)
                    .map(c -> CloudWatchLoggingOptionDescription.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getApplicationCode() {
            return applicationCode;
        }

        public final void setApplicationCode(String applicationCode) {
            this.applicationCode = applicationCode;
        }

        @Override
        public final Builder applicationCode(String applicationCode) {
            this.applicationCode = applicationCode;
            return this;
        }

        public final Long getApplicationVersionId() {
            return applicationVersionId;
        }

        public final void setApplicationVersionId(Long applicationVersionId) {
            this.applicationVersionId = applicationVersionId;
        }

        @Override
        public final Builder applicationVersionId(Long applicationVersionId) {
            this.applicationVersionId = applicationVersionId;
            return this;
        }

        @Override
        public ApplicationDetail build() {
            return new ApplicationDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
