/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotroborunner;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotroborunner.internal.IotRoboRunnerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotroborunner.model.AccessDeniedException;
import software.amazon.awssdk.services.iotroborunner.model.ConflictException;
import software.amazon.awssdk.services.iotroborunner.model.CreateDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.CreateSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.InternalServerException;
import software.amazon.awssdk.services.iotroborunner.model.IotRoboRunnerException;
import software.amazon.awssdk.services.iotroborunner.model.ListDestinationsRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListDestinationsResponse;
import software.amazon.awssdk.services.iotroborunner.model.ListSitesRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListSitesResponse;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsResponse;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkersRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkersResponse;
import software.amazon.awssdk.services.iotroborunner.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotroborunner.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.iotroborunner.model.ThrottlingException;
import software.amazon.awssdk.services.iotroborunner.model.UpdateDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.UpdateSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.ValidationException;
import software.amazon.awssdk.services.iotroborunner.transform.CreateDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.CreateSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.CreateWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.CreateWorkerRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteWorkerRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetWorkerRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListDestinationsRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListSitesRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListWorkerFleetsRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListWorkersRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateWorkerRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IotRoboRunnerClient}.
 *
 * @see IotRoboRunnerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotRoboRunnerClient implements IotRoboRunnerClient {
    private static final Logger log = Logger.loggerFor(DefaultIotRoboRunnerClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final IotRoboRunnerServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIotRoboRunnerClient(IotRoboRunnerServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Grants permission to create a destination
     * </p>
     *
     * @param createDestinationRequest
     * @return Result of the CreateDestination operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws ServiceQuotaExceededException
     *         Exception thrown if the user's AWS account has reached a service limit and the operation cannot proceed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.CreateDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDestinationResponse createDestination(CreateDestinationRequest createDestinationRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDestinationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDestination");

            return clientHandler.execute(new ClientExecutionParams<CreateDestinationRequest, CreateDestinationResponse>()
                    .withOperationName("CreateDestination").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createDestinationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to create a site
     * </p>
     *
     * @param createSiteRequest
     * @return Result of the CreateSite operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws ServiceQuotaExceededException
     *         Exception thrown if the user's AWS account has reached a service limit and the operation cannot proceed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.CreateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSiteResponse createSite(CreateSiteRequest createSiteRequest) throws ConflictException, AccessDeniedException,
            ValidationException, ThrottlingException, InternalServerException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSite");

            return clientHandler.execute(new ClientExecutionParams<CreateSiteRequest, CreateSiteResponse>()
                    .withOperationName("CreateSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createSiteRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to create a worker
     * </p>
     *
     * @param createWorkerRequest
     * @return Result of the CreateWorker operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws ServiceQuotaExceededException
     *         Exception thrown if the user's AWS account has reached a service limit and the operation cannot proceed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.CreateWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateWorker" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateWorkerResponse createWorker(CreateWorkerRequest createWorkerRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWorkerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorker");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkerRequest, CreateWorkerResponse>()
                    .withOperationName("CreateWorker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createWorkerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to create a worker fleet
     * </p>
     *
     * @param createWorkerFleetRequest
     * @return Result of the CreateWorkerFleet operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws ServiceQuotaExceededException
     *         Exception thrown if the user's AWS account has reached a service limit and the operation cannot proceed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.CreateWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateWorkerFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateWorkerFleetResponse createWorkerFleet(CreateWorkerFleetRequest createWorkerFleetRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWorkerFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkerFleetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkerFleet");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkerFleetRequest, CreateWorkerFleetResponse>()
                    .withOperationName("CreateWorkerFleet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createWorkerFleetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkerFleetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to delete a destination
     * </p>
     *
     * @param deleteDestinationRequest
     * @return Result of the DeleteDestination operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.DeleteDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDestinationResponse deleteDestination(DeleteDestinationRequest deleteDestinationRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDestinationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDestination");

            return clientHandler.execute(new ClientExecutionParams<DeleteDestinationRequest, DeleteDestinationResponse>()
                    .withOperationName("DeleteDestination").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteDestinationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to delete a site
     * </p>
     *
     * @param deleteSiteRequest
     * @return Result of the DeleteSite operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.DeleteSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSiteResponse deleteSite(DeleteSiteRequest deleteSiteRequest) throws ConflictException, AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSite");

            return clientHandler.execute(new ClientExecutionParams<DeleteSiteRequest, DeleteSiteResponse>()
                    .withOperationName("DeleteSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteSiteRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to delete a worker
     * </p>
     *
     * @param deleteWorkerRequest
     * @return Result of the DeleteWorker operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.DeleteWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteWorker" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteWorkerResponse deleteWorker(DeleteWorkerRequest deleteWorkerRequest) throws ConflictException,
            AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWorkerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorker");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkerRequest, DeleteWorkerResponse>()
                    .withOperationName("DeleteWorker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteWorkerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to delete a worker fleet
     * </p>
     *
     * @param deleteWorkerFleetRequest
     * @return Result of the DeleteWorkerFleet operation returned by the service.
     * @throws ConflictException
     *         Exception thrown if a resource in a create request already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.DeleteWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteWorkerFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteWorkerFleetResponse deleteWorkerFleet(DeleteWorkerFleetRequest deleteWorkerFleetRequest)
            throws ConflictException, AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWorkerFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkerFleetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkerFleet");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkerFleetRequest, DeleteWorkerFleetResponse>()
                    .withOperationName("DeleteWorkerFleet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteWorkerFleetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkerFleetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to get a destination
     * </p>
     *
     * @param getDestinationRequest
     * @return Result of the GetDestination operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.GetDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDestinationResponse getDestination(GetDestinationRequest getDestinationRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDestinationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDestinationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDestination");

            return clientHandler.execute(new ClientExecutionParams<GetDestinationRequest, GetDestinationResponse>()
                    .withOperationName("GetDestination").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getDestinationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to get a site
     * </p>
     *
     * @param getSiteRequest
     * @return Result of the GetSite operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.GetSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSiteResponse getSite(GetSiteRequest getSiteRequest) throws AccessDeniedException, ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSite");

            return clientHandler.execute(new ClientExecutionParams<GetSiteRequest, GetSiteResponse>()
                    .withOperationName("GetSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSiteRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to get a worker
     * </p>
     *
     * @param getWorkerRequest
     * @return Result of the GetWorker operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.GetWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetWorker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetWorkerResponse getWorker(GetWorkerRequest getWorkerRequest) throws AccessDeniedException, ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetWorkerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorker");

            return clientHandler.execute(new ClientExecutionParams<GetWorkerRequest, GetWorkerResponse>()
                    .withOperationName("GetWorker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getWorkerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetWorkerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to get a worker fleet
     * </p>
     *
     * @param getWorkerFleetRequest
     * @return Result of the GetWorkerFleet operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.GetWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetWorkerFleet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetWorkerFleetResponse getWorkerFleet(GetWorkerFleetRequest getWorkerFleetRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetWorkerFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkerFleetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkerFleet");

            return clientHandler.execute(new ClientExecutionParams<GetWorkerFleetRequest, GetWorkerFleetResponse>()
                    .withOperationName("GetWorkerFleet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getWorkerFleetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetWorkerFleetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to list destinations
     * </p>
     *
     * @param listDestinationsRequest
     * @return Result of the ListDestinations operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.ListDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDestinationsResponse listDestinations(ListDestinationsRequest listDestinationsRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDestinationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDestinationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDestinations");

            return clientHandler.execute(new ClientExecutionParams<ListDestinationsRequest, ListDestinationsResponse>()
                    .withOperationName("ListDestinations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listDestinationsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDestinationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to list sites
     * </p>
     *
     * @param listSitesRequest
     * @return Result of the ListSites operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSitesResponse listSites(ListSitesRequest listSitesRequest) throws AccessDeniedException, ValidationException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSitesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSitesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSites");

            return clientHandler.execute(new ClientExecutionParams<ListSitesRequest, ListSitesResponse>()
                    .withOperationName("ListSites").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listSitesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSitesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to list worker fleets
     * </p>
     *
     * @param listWorkerFleetsRequest
     * @return Result of the ListWorkerFleets operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.ListWorkerFleets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListWorkerFleets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkerFleetsResponse listWorkerFleets(ListWorkerFleetsRequest listWorkerFleetsRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkerFleetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkerFleetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkerFleetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkerFleetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkerFleets");

            return clientHandler.execute(new ClientExecutionParams<ListWorkerFleetsRequest, ListWorkerFleetsResponse>()
                    .withOperationName("ListWorkerFleets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listWorkerFleetsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkerFleetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to list workers
     * </p>
     *
     * @param listWorkersRequest
     * @return Result of the ListWorkers operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.ListWorkers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListWorkers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListWorkersResponse listWorkers(ListWorkersRequest listWorkersRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkers");

            return clientHandler.execute(new ClientExecutionParams<ListWorkersRequest, ListWorkersResponse>()
                    .withOperationName("ListWorkers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listWorkersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to update a destination
     * </p>
     *
     * @param updateDestinationRequest
     * @return Result of the UpdateDestination operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.UpdateDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDestinationResponse updateDestination(UpdateDestinationRequest updateDestinationRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDestinationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDestination");

            return clientHandler.execute(new ClientExecutionParams<UpdateDestinationRequest, UpdateDestinationResponse>()
                    .withOperationName("UpdateDestination").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateDestinationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to update a site
     * </p>
     *
     * @param updateSiteRequest
     * @return Result of the UpdateSite operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.UpdateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSiteResponse updateSite(UpdateSiteRequest updateSiteRequest) throws AccessDeniedException, ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSiteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSite");

            return clientHandler.execute(new ClientExecutionParams<UpdateSiteRequest, UpdateSiteResponse>()
                    .withOperationName("UpdateSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateSiteRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to update a worker
     * </p>
     *
     * @param updateWorkerRequest
     * @return Result of the UpdateWorker operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.UpdateWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateWorker" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateWorkerResponse updateWorker(UpdateWorkerRequest updateWorkerRequest) throws AccessDeniedException,
            ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWorkerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWorkerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorker");

            return clientHandler.execute(new ClientExecutionParams<UpdateWorkerRequest, UpdateWorkerResponse>()
                    .withOperationName("UpdateWorker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateWorkerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWorkerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Grants permission to update a worker fleet
     * </p>
     *
     * @param updateWorkerFleetRequest
     * @return Result of the UpdateWorkerFleet operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         Exception thrown if an invalid parameter is provided to an API.
     * @throws ResourceNotFoundException
     *         Exception thrown if a resource referenced in the request doesn't exist.
     * @throws ThrottlingException
     *         Exception thrown if the api has been called too quickly be the client.
     * @throws InternalServerException
     *         Exception thrown if something goes wrong within the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotRoboRunnerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotRoboRunnerClient.UpdateWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateWorkerFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateWorkerFleetResponse updateWorkerFleet(UpdateWorkerFleetRequest updateWorkerFleetRequest)
            throws AccessDeniedException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, IotRoboRunnerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWorkerFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWorkerFleetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkerFleet");

            return clientHandler.execute(new ClientExecutionParams<UpdateWorkerFleetRequest, UpdateWorkerFleetResponse>()
                    .withOperationName("UpdateWorkerFleet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateWorkerFleetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWorkerFleetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        IotRoboRunnerServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = IotRoboRunnerServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotRoboRunnerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final IotRoboRunnerServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
