/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdeviceadvisor;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotdeviceadvisor.internal.IotDeviceAdvisorServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ConflictException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.CreateSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.CreateSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.DeleteSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.DeleteSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetEndpointRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetEndpointResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunReportRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunReportResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.InternalServerException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.IotDeviceAdvisorException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StartSuiteRunRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StartSuiteRunResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StopSuiteRunRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StopSuiteRunResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.TagResourceRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.TagResourceResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UpdateSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UpdateSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ValidationException;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.CreateSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.DeleteSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetEndpointRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetSuiteRunReportRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetSuiteRunRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.ListSuiteDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.ListSuiteRunsRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.StartSuiteRunRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.StopSuiteRunRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.UpdateSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotDeviceAdvisorAsyncClient}.
 *
 * @see IotDeviceAdvisorAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotDeviceAdvisorAsyncClient implements IotDeviceAdvisorAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotDeviceAdvisorAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final IotDeviceAdvisorServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIotDeviceAdvisorAsyncClient(IotDeviceAdvisorServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >CreateSuiteDefinition</a> action.
     * </p>
     *
     * @param createSuiteDefinitionRequest
     * @return A Java Future containing the result of the CreateSuiteDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.CreateSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/CreateSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSuiteDefinitionResponse> createSuiteDefinition(
            CreateSuiteDefinitionRequest createSuiteDefinitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSuiteDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSuiteDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSuiteDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSuiteDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSuiteDefinitionRequest, CreateSuiteDefinitionResponse>()
                            .withOperationName("CreateSuiteDefinition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSuiteDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSuiteDefinitionRequest));
            CompletableFuture<CreateSuiteDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteSuiteDefinition</a> action.
     * </p>
     *
     * @param deleteSuiteDefinitionRequest
     * @return A Java Future containing the result of the DeleteSuiteDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.DeleteSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/DeleteSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSuiteDefinitionResponse> deleteSuiteDefinition(
            DeleteSuiteDefinitionRequest deleteSuiteDefinitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSuiteDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSuiteDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSuiteDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSuiteDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSuiteDefinitionRequest, DeleteSuiteDefinitionResponse>()
                            .withOperationName("DeleteSuiteDefinition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSuiteDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSuiteDefinitionRequest));
            CompletableFuture<DeleteSuiteDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about an Device Advisor endpoint.
     * </p>
     *
     * @param getEndpointRequest
     * @return A Java Future containing the result of the GetEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.GetEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEndpointResponse> getEndpoint(GetEndpointRequest getEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEndpointRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEndpointResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEndpointRequest, GetEndpointResponse>()
                            .withOperationName("GetEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEndpointRequest));
            CompletableFuture<GetEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetSuiteDefinition</a> action.
     * </p>
     *
     * @param getSuiteDefinitionRequest
     * @return A Java Future containing the result of the GetSuiteDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.GetSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSuiteDefinitionResponse> getSuiteDefinition(GetSuiteDefinitionRequest getSuiteDefinitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSuiteDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSuiteDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSuiteDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSuiteDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSuiteDefinitionRequest, GetSuiteDefinitionResponse>()
                            .withOperationName("GetSuiteDefinition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSuiteDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSuiteDefinitionRequest));
            CompletableFuture<GetSuiteDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a Device Advisor test suite run.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetSuiteRun</a> action.
     * </p>
     *
     * @param getSuiteRunRequest
     * @return A Java Future containing the result of the GetSuiteRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.GetSuiteRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetSuiteRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSuiteRunResponse> getSuiteRun(GetSuiteRunRequest getSuiteRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSuiteRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSuiteRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSuiteRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSuiteRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSuiteRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSuiteRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSuiteRunRequest, GetSuiteRunResponse>()
                            .withOperationName("GetSuiteRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSuiteRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSuiteRunRequest));
            CompletableFuture<GetSuiteRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a report download link for a successful Device Advisor qualifying test suite run.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetSuiteRunReport</a> action.
     * </p>
     *
     * @param getSuiteRunReportRequest
     * @return A Java Future containing the result of the GetSuiteRunReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.GetSuiteRunReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetSuiteRunReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSuiteRunReportResponse> getSuiteRunReport(GetSuiteRunReportRequest getSuiteRunReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSuiteRunReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSuiteRunReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSuiteRunReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSuiteRunReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSuiteRunReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSuiteRunReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSuiteRunReportRequest, GetSuiteRunReportResponse>()
                            .withOperationName("GetSuiteRunReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSuiteRunReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSuiteRunReportRequest));
            CompletableFuture<GetSuiteRunReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Device Advisor test suites you have created.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListSuiteDefinitions</a> action.
     * </p>
     *
     * @param listSuiteDefinitionsRequest
     * @return A Java Future containing the result of the ListSuiteDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.ListSuiteDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListSuiteDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSuiteDefinitionsResponse> listSuiteDefinitions(
            ListSuiteDefinitionsRequest listSuiteDefinitionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSuiteDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSuiteDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSuiteDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSuiteDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSuiteDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSuiteDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSuiteDefinitionsRequest, ListSuiteDefinitionsResponse>()
                            .withOperationName("ListSuiteDefinitions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSuiteDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSuiteDefinitionsRequest));
            CompletableFuture<ListSuiteDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists runs of the specified Device Advisor test suite. You can list all runs of the test suite, or the runs of a
     * specific version of the test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListSuiteRuns</a> action.
     * </p>
     *
     * @param listSuiteRunsRequest
     * @return A Java Future containing the result of the ListSuiteRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.ListSuiteRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListSuiteRuns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSuiteRunsResponse> listSuiteRuns(ListSuiteRunsRequest listSuiteRunsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSuiteRunsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSuiteRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSuiteRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSuiteRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSuiteRunsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSuiteRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSuiteRunsRequest, ListSuiteRunsResponse>()
                            .withOperationName("ListSuiteRuns").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSuiteRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSuiteRunsRequest));
            CompletableFuture<ListSuiteRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags attached to an IoT Device Advisor resource.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListTagsForResource</a> action.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a Device Advisor test suite run.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >StartSuiteRun</a> action.
     * </p>
     *
     * @param startSuiteRunRequest
     * @return A Java Future containing the result of the StartSuiteRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ConflictException Sends a Conflict Exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.StartSuiteRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/StartSuiteRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSuiteRunResponse> startSuiteRun(StartSuiteRunRequest startSuiteRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSuiteRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSuiteRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSuiteRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSuiteRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartSuiteRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSuiteRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSuiteRunRequest, StartSuiteRunResponse>()
                            .withOperationName("StartSuiteRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSuiteRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSuiteRunRequest));
            CompletableFuture<StartSuiteRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a Device Advisor test suite run that is currently running.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >StopSuiteRun</a> action.
     * </p>
     *
     * @param stopSuiteRunRequest
     * @return A Java Future containing the result of the StopSuiteRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.StopSuiteRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/StopSuiteRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopSuiteRunResponse> stopSuiteRun(StopSuiteRunRequest stopSuiteRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopSuiteRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopSuiteRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopSuiteRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopSuiteRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopSuiteRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopSuiteRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopSuiteRunRequest, StopSuiteRunResponse>()
                            .withOperationName("StopSuiteRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopSuiteRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopSuiteRunRequest));
            CompletableFuture<StopSuiteRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to and modifies existing tags of an IoT Device Advisor resource.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >TagResource</a> action.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an IoT Device Advisor resource.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UntagResource</a> action.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>ResourceNotFoundException Sends a Resource Not Found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateSuiteDefinition</a> action.
     * </p>
     *
     * @param updateSuiteDefinitionRequest
     * @return A Java Future containing the result of the UpdateSuiteDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Sends a validation exception.</li>
     *         <li>InternalServerException Sends an Internal Failure exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDeviceAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotDeviceAdvisorAsyncClient.UpdateSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/UpdateSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSuiteDefinitionResponse> updateSuiteDefinition(
            UpdateSuiteDefinitionRequest updateSuiteDefinitionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSuiteDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSuiteDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSuiteDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSuiteDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSuiteDefinitionRequest, UpdateSuiteDefinitionResponse>()
                            .withOperationName("UpdateSuiteDefinition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSuiteDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSuiteDefinitionRequest));
            CompletableFuture<UpdateSuiteDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IotDeviceAdvisorServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotDeviceAdvisorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        IotDeviceAdvisorServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = IotDeviceAdvisorServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
