/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.inspector2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about the resource involved in a finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Resource implements SdkPojo, Serializable, ToCopyableBuilder<Resource.Builder, Resource> {
    private static final SdkField<ResourceDetails> DETAILS_FIELD = SdkField.<ResourceDetails> builder(MarshallingType.SDK_POJO)
            .memberName("details").getter(getter(Resource::details)).setter(setter(Builder::details))
            .constructor(ResourceDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("details").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Resource::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> PARTITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("partition").getter(getter(Resource::partition)).setter(setter(Builder::partition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partition").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(Resource::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Resource::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Resource::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DETAILS_FIELD, ID_FIELD,
            PARTITION_FIELD, REGION_FIELD, TAGS_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final ResourceDetails details;

    private final String id;

    private final String partition;

    private final String region;

    private final Map<String, String> tags;

    private final String type;

    private Resource(BuilderImpl builder) {
        this.details = builder.details;
        this.id = builder.id;
        this.partition = builder.partition;
        this.region = builder.region;
        this.tags = builder.tags;
        this.type = builder.type;
    }

    /**
     * <p>
     * An object that contains details about the resource involved in a finding.
     * </p>
     * 
     * @return An object that contains details about the resource involved in a finding.
     */
    public final ResourceDetails details() {
        return details;
    }

    /**
     * <p>
     * The ID of the resource.
     * </p>
     * 
     * @return The ID of the resource.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The partition of the resource.
     * </p>
     * 
     * @return The partition of the resource.
     */
    public final String partition() {
        return partition;
    }

    /**
     * <p>
     * The Amazon Web Services Region the impacted resource is located in.
     * </p>
     * 
     * @return The Amazon Web Services Region the impacted resource is located in.
     */
    public final String region() {
        return region;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags attached to the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags attached to the resource.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The type of resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of resource.
     * @see ResourceType
     */
    public final ResourceType type() {
        return ResourceType.fromValue(type);
    }

    /**
     * <p>
     * The type of resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of resource.
     * @see ResourceType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(details());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(partition());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Resource)) {
            return false;
        }
        Resource other = (Resource) obj;
        return Objects.equals(details(), other.details()) && Objects.equals(id(), other.id())
                && Objects.equals(partition(), other.partition()) && Objects.equals(region(), other.region())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Resource").add("Details", details()).add("Id", id()).add("Partition", partition())
                .add("Region", region()).add("Tags", hasTags() ? tags() : null).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "details":
            return Optional.ofNullable(clazz.cast(details()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "partition":
            return Optional.ofNullable(clazz.cast(partition()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Resource, T> g) {
        return obj -> g.apply((Resource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Resource> {
        /**
         * <p>
         * An object that contains details about the resource involved in a finding.
         * </p>
         * 
         * @param details
         *        An object that contains details about the resource involved in a finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(ResourceDetails details);

        /**
         * <p>
         * An object that contains details about the resource involved in a finding.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceDetails.Builder} avoiding the
         * need to create one manually via {@link ResourceDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #details(ResourceDetails)}.
         * 
         * @param details
         *        a consumer that will call methods on {@link ResourceDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #details(ResourceDetails)
         */
        default Builder details(Consumer<ResourceDetails.Builder> details) {
            return details(ResourceDetails.builder().applyMutation(details).build());
        }

        /**
         * <p>
         * The ID of the resource.
         * </p>
         * 
         * @param id
         *        The ID of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The partition of the resource.
         * </p>
         * 
         * @param partition
         *        The partition of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partition(String partition);

        /**
         * <p>
         * The Amazon Web Services Region the impacted resource is located in.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region the impacted resource is located in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The tags attached to the resource.
         * </p>
         * 
         * @param tags
         *        The tags attached to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The type of resource.
         * </p>
         * 
         * @param type
         *        The type of resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of resource.
         * </p>
         * 
         * @param type
         *        The type of resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder type(ResourceType type);
    }

    static final class BuilderImpl implements Builder {
        private ResourceDetails details;

        private String id;

        private String partition;

        private String region;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(Resource model) {
            details(model.details);
            id(model.id);
            partition(model.partition);
            region(model.region);
            tags(model.tags);
            type(model.type);
        }

        public final ResourceDetails.Builder getDetails() {
            return details != null ? details.toBuilder() : null;
        }

        public final void setDetails(ResourceDetails.BuilderImpl details) {
            this.details = details != null ? details.build() : null;
        }

        @Override
        public final Builder details(ResourceDetails details) {
            this.details = details;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getPartition() {
            return partition;
        }

        public final void setPartition(String partition) {
            this.partition = partition;
        }

        @Override
        public final Builder partition(String partition) {
            this.partition = partition;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ResourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public Resource build() {
            return new Resource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
