/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the latest version of a component.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComponentLatestVersion implements SdkPojo, Serializable,
        ToCopyableBuilder<ComponentLatestVersion.Builder, ComponentLatestVersion> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ComponentLatestVersion::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> COMPONENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("componentVersion").getter(getter(ComponentLatestVersion::componentVersion))
            .setter(setter(Builder::componentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("componentVersion").build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTimestamp").getter(getter(ComponentLatestVersion::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTimestamp").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(ComponentLatestVersion::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> PUBLISHER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("publisher").getter(getter(ComponentLatestVersion::publisher)).setter(setter(Builder::publisher))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publisher").build()).build();

    private static final SdkField<List<ComponentPlatform>> PLATFORMS_FIELD = SdkField
            .<List<ComponentPlatform>> builder(MarshallingType.LIST)
            .memberName("platforms")
            .getter(getter(ComponentLatestVersion::platforms))
            .setter(setter(Builder::platforms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platforms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ComponentPlatform> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentPlatform::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            COMPONENT_VERSION_FIELD, CREATION_TIMESTAMP_FIELD, DESCRIPTION_FIELD, PUBLISHER_FIELD, PLATFORMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String componentVersion;

    private final Instant creationTimestamp;

    private final String description;

    private final String publisher;

    private final List<ComponentPlatform> platforms;

    private ComponentLatestVersion(BuilderImpl builder) {
        this.arn = builder.arn;
        this.componentVersion = builder.componentVersion;
        this.creationTimestamp = builder.creationTimestamp;
        this.description = builder.description;
        this.publisher = builder.publisher;
        this.platforms = builder.platforms;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a> of the component
     * version.
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a> of the
     *         component version.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The version of the component.
     * </p>
     * 
     * @return The version of the component.
     */
    public final String componentVersion() {
        return componentVersion;
    }

    /**
     * <p>
     * The time at which the component was created, expressed in ISO 8601 format.
     * </p>
     * 
     * @return The time at which the component was created, expressed in ISO 8601 format.
     */
    public final Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * <p>
     * The description of the component version.
     * </p>
     * 
     * @return The description of the component version.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The publisher of the component version.
     * </p>
     * 
     * @return The publisher of the component version.
     */
    public final String publisher() {
        return publisher;
    }

    /**
     * For responses, this returns true if the service returned a value for the Platforms property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPlatforms() {
        return platforms != null && !(platforms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The platforms that the component version supports.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatforms} method.
     * </p>
     * 
     * @return The platforms that the component version supports.
     */
    public final List<ComponentPlatform> platforms() {
        return platforms;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(componentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(publisher());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatforms() ? platforms() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComponentLatestVersion)) {
            return false;
        }
        ComponentLatestVersion other = (ComponentLatestVersion) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(componentVersion(), other.componentVersion())
                && Objects.equals(creationTimestamp(), other.creationTimestamp())
                && Objects.equals(description(), other.description()) && Objects.equals(publisher(), other.publisher())
                && hasPlatforms() == other.hasPlatforms() && Objects.equals(platforms(), other.platforms());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComponentLatestVersion").add("Arn", arn()).add("ComponentVersion", componentVersion())
                .add("CreationTimestamp", creationTimestamp()).add("Description", description()).add("Publisher", publisher())
                .add("Platforms", hasPlatforms() ? platforms() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "componentVersion":
            return Optional.ofNullable(clazz.cast(componentVersion()));
        case "creationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "publisher":
            return Optional.ofNullable(clazz.cast(publisher()));
        case "platforms":
            return Optional.ofNullable(clazz.cast(platforms()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComponentLatestVersion, T> g) {
        return obj -> g.apply((ComponentLatestVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComponentLatestVersion> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a> of the
         * component version.
         * </p>
         * 
         * @param arn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a> of
         *        the component version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The version of the component.
         * </p>
         * 
         * @param componentVersion
         *        The version of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder componentVersion(String componentVersion);

        /**
         * <p>
         * The time at which the component was created, expressed in ISO 8601 format.
         * </p>
         * 
         * @param creationTimestamp
         *        The time at which the component was created, expressed in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * The description of the component version.
         * </p>
         * 
         * @param description
         *        The description of the component version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The publisher of the component version.
         * </p>
         * 
         * @param publisher
         *        The publisher of the component version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publisher(String publisher);

        /**
         * <p>
         * The platforms that the component version supports.
         * </p>
         * 
         * @param platforms
         *        The platforms that the component version supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platforms(Collection<ComponentPlatform> platforms);

        /**
         * <p>
         * The platforms that the component version supports.
         * </p>
         * 
         * @param platforms
         *        The platforms that the component version supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platforms(ComponentPlatform... platforms);

        /**
         * <p>
         * The platforms that the component version supports.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.greengrassv2.model.ComponentPlatform.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.greengrassv2.model.ComponentPlatform#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.greengrassv2.model.ComponentPlatform.Builder#build()} is called
         * immediately and its result is passed to {@link #platforms(List<ComponentPlatform>)}.
         * 
         * @param platforms
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.greengrassv2.model.ComponentPlatform.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #platforms(java.util.Collection<ComponentPlatform>)
         */
        Builder platforms(Consumer<ComponentPlatform.Builder>... platforms);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String componentVersion;

        private Instant creationTimestamp;

        private String description;

        private String publisher;

        private List<ComponentPlatform> platforms = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComponentLatestVersion model) {
            arn(model.arn);
            componentVersion(model.componentVersion);
            creationTimestamp(model.creationTimestamp);
            description(model.description);
            publisher(model.publisher);
            platforms(model.platforms);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getComponentVersion() {
            return componentVersion;
        }

        public final void setComponentVersion(String componentVersion) {
            this.componentVersion = componentVersion;
        }

        @Override
        public final Builder componentVersion(String componentVersion) {
            this.componentVersion = componentVersion;
            return this;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getPublisher() {
            return publisher;
        }

        public final void setPublisher(String publisher) {
            this.publisher = publisher;
        }

        @Override
        public final Builder publisher(String publisher) {
            this.publisher = publisher;
            return this;
        }

        public final List<ComponentPlatform.Builder> getPlatforms() {
            List<ComponentPlatform.Builder> result = ComponentPlatformListCopier.copyToBuilder(this.platforms);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlatforms(Collection<ComponentPlatform.BuilderImpl> platforms) {
            this.platforms = ComponentPlatformListCopier.copyFromBuilder(platforms);
        }

        @Override
        public final Builder platforms(Collection<ComponentPlatform> platforms) {
            this.platforms = ComponentPlatformListCopier.copy(platforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platforms(ComponentPlatform... platforms) {
            platforms(Arrays.asList(platforms));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platforms(Consumer<ComponentPlatform.Builder>... platforms) {
            platforms(Stream.of(platforms).map(c -> ComponentPlatform.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ComponentLatestVersion build() {
            return new ComponentLatestVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
