/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.firehose.model.ConcurrentModificationException;
import software.amazon.awssdk.services.firehose.model.CreateDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.CreateDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.DeleteDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.DeleteDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.DescribeDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.DescribeDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.FirehoseException;
import software.amazon.awssdk.services.firehose.model.InvalidArgumentException;
import software.amazon.awssdk.services.firehose.model.InvalidKmsResourceException;
import software.amazon.awssdk.services.firehose.model.LimitExceededException;
import software.amazon.awssdk.services.firehose.model.ListDeliveryStreamsRequest;
import software.amazon.awssdk.services.firehose.model.ListDeliveryStreamsResponse;
import software.amazon.awssdk.services.firehose.model.ListTagsForDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.ListTagsForDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.PutRecordBatchRequest;
import software.amazon.awssdk.services.firehose.model.PutRecordBatchResponse;
import software.amazon.awssdk.services.firehose.model.PutRecordRequest;
import software.amazon.awssdk.services.firehose.model.PutRecordResponse;
import software.amazon.awssdk.services.firehose.model.ResourceInUseException;
import software.amazon.awssdk.services.firehose.model.ResourceNotFoundException;
import software.amazon.awssdk.services.firehose.model.ServiceUnavailableException;
import software.amazon.awssdk.services.firehose.model.StartDeliveryStreamEncryptionRequest;
import software.amazon.awssdk.services.firehose.model.StartDeliveryStreamEncryptionResponse;
import software.amazon.awssdk.services.firehose.model.StopDeliveryStreamEncryptionRequest;
import software.amazon.awssdk.services.firehose.model.StopDeliveryStreamEncryptionResponse;
import software.amazon.awssdk.services.firehose.model.TagDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.TagDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.UntagDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.UntagDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.UpdateDestinationRequest;
import software.amazon.awssdk.services.firehose.model.UpdateDestinationResponse;
import software.amazon.awssdk.services.firehose.transform.CreateDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.DeleteDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.DescribeDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.ListDeliveryStreamsRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.ListTagsForDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.PutRecordBatchRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.PutRecordRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.StartDeliveryStreamEncryptionRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.StopDeliveryStreamEncryptionRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.TagDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.UntagDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.UpdateDestinationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FirehoseAsyncClient}.
 *
 * @see FirehoseAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFirehoseAsyncClient implements FirehoseAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFirehoseAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final FirehoseServiceClientConfiguration serviceClientConfiguration;

    protected DefaultFirehoseAsyncClient(FirehoseServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Kinesis Data Firehose delivery stream.
     * </p>
     * <p>
     * By default, you can create up to 50 delivery streams per Amazon Web Services Region.
     * </p>
     * <p>
     * This is an asynchronous operation that immediately returns. The initial status of the delivery stream is
     * <code>CREATING</code>. After the delivery stream is created, its status is <code>ACTIVE</code> and it now accepts
     * data. If the delivery stream creation fails, the status transitions to <code>CREATING_FAILED</code>. Attempts to
     * send data to a delivery stream that is not in the <code>ACTIVE</code> state cause an exception. To check the
     * state of a delivery stream, use <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * If the status of a delivery stream is <code>CREATING_FAILED</code>, this status doesn't change, and you can't
     * invoke <code>CreateDeliveryStream</code> again on it. However, you can invoke the <a>DeleteDeliveryStream</a>
     * operation to delete it.
     * </p>
     * <p>
     * A Kinesis Data Firehose delivery stream can be configured to receive records directly from providers using
     * <a>PutRecord</a> or <a>PutRecordBatch</a>, or it can be configured to use an existing Kinesis stream as its
     * source. To specify a Kinesis data stream as input, set the <code>DeliveryStreamType</code> parameter to
     * <code>KinesisStreamAsSource</code>, and provide the Kinesis stream Amazon Resource Name (ARN) and role ARN in the
     * <code>KinesisStreamSourceConfiguration</code> parameter.
     * </p>
     * <p>
     * To create a delivery stream with server-side encryption (SSE) enabled, include
     * <a>DeliveryStreamEncryptionConfigurationInput</a> in your request. This is optional. You can also invoke
     * <a>StartDeliveryStreamEncryption</a> to turn on SSE for an existing delivery stream that doesn't have SSE
     * enabled.
     * </p>
     * <p>
     * A delivery stream is configured with a single destination: Amazon S3, Amazon ES, Amazon Redshift, or Splunk. You
     * must specify only one of the following destination configuration parameters:
     * <code>ExtendedS3DestinationConfiguration</code>, <code>S3DestinationConfiguration</code>,
     * <code>ElasticsearchDestinationConfiguration</code>, <code>RedshiftDestinationConfiguration</code>, or
     * <code>SplunkDestinationConfiguration</code>.
     * </p>
     * <p>
     * When you specify <code>S3DestinationConfiguration</code>, you can also provide the following optional values:
     * BufferingHints, <code>EncryptionConfiguration</code>, and <code>CompressionFormat</code>. By default, if no
     * <code>BufferingHints</code> value is provided, Kinesis Data Firehose buffers data up to 5 MB or for 5 minutes,
     * whichever condition is satisfied first. <code>BufferingHints</code> is a hint, so there are some cases where the
     * service cannot adhere to these conditions strictly. For example, record boundaries might be such that the size is
     * a little over or under the configured buffering size. By default, no encryption is performed. We strongly
     * recommend that you enable encryption to ensure secure data storage in Amazon S3.
     * </p>
     * <p>
     * A few notes about Amazon Redshift as a destination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon Redshift destination requires an S3 bucket as intermediate location. Kinesis Data Firehose first
     * delivers data to Amazon S3 and then uses <code>COPY</code> syntax to load data into an Amazon Redshift table.
     * This is specified in the <code>RedshiftDestinationConfiguration.S3Configuration</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * The compression formats <code>SNAPPY</code> or <code>ZIP</code> cannot be specified in
     * <code>RedshiftDestinationConfiguration.S3Configuration</code> because the Amazon Redshift <code>COPY</code>
     * operation that reads from the S3 bucket doesn't support these compression formats.
     * </p>
     * </li>
     * <li>
     * <p>
     * We strongly recommend that you use the user name and password you provide exclusively with Kinesis Data Firehose,
     * and that the permissions for the account are restricted for Amazon Redshift <code>INSERT</code> permissions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Kinesis Data Firehose assumes the IAM role that is configured as part of the destination. The role should allow
     * the Kinesis Data Firehose principal to assume the role, and the role should have permissions that allow the
     * service to deliver the data. For more information, see <a
     * href="https://docs.aws.amazon.com/firehose/latest/dev/controlling-access.html#using-iam-s3">Grant Kinesis Data
     * Firehose Access to an Amazon S3 Destination</a> in the <i>Amazon Kinesis Data Firehose Developer Guide</i>.
     * </p>
     *
     * @param createDeliveryStreamRequest
     * @return A Java Future containing the result of the CreateDeliveryStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>LimitExceededException You have already reached the limit for a requested resource.</li>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>InvalidKmsResourceException Kinesis Data Firehose throws this exception when an attempt to put
     *         records or to start or stop delivery stream encryption fails. This happens when the KMS service throws
     *         one of the following exception types: <code>AccessDeniedException</code>,
     *         <code>InvalidStateException</code>, <code>DisabledException</code>, or <code>NotFoundException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.CreateDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/CreateDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeliveryStreamResponse> createDeliveryStream(
            CreateDeliveryStreamRequest createDeliveryStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeliveryStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeliveryStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeliveryStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeliveryStreamRequest, CreateDeliveryStreamResponse>()
                            .withOperationName("CreateDeliveryStream")
                            .withMarshaller(new CreateDeliveryStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDeliveryStreamRequest));
            CompletableFuture<CreateDeliveryStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a delivery stream and its data.
     * </p>
     * <p>
     * To check the state of a delivery stream, use <a>DescribeDeliveryStream</a>. You can delete a delivery stream only
     * if it is in one of the following states: <code>ACTIVE</code>, <code>DELETING</code>, <code>CREATING_FAILED</code>
     * , or <code>DELETING_FAILED</code>. You can't delete a delivery stream that is in the <code>CREATING</code> state.
     * While the deletion request is in process, the delivery stream is in the <code>DELETING</code> state.
     * </p>
     * <p>
     * While the delivery stream is in the <code>DELETING</code> state, the service might continue to accept records,
     * but it doesn't make any guarantees with respect to delivering the data. Therefore, as a best practice, first stop
     * any applications that are sending records before you delete a delivery stream.
     * </p>
     *
     * @param deleteDeliveryStreamRequest
     * @return A Java Future containing the result of the DeleteDeliveryStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.DeleteDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/DeleteDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDeliveryStreamResponse> deleteDeliveryStream(
            DeleteDeliveryStreamRequest deleteDeliveryStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeliveryStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDeliveryStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDeliveryStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDeliveryStreamRequest, DeleteDeliveryStreamResponse>()
                            .withOperationName("DeleteDeliveryStream")
                            .withMarshaller(new DeleteDeliveryStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDeliveryStreamRequest));
            CompletableFuture<DeleteDeliveryStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified delivery stream and its status. For example, after your delivery stream is created, call
     * <code>DescribeDeliveryStream</code> to see whether the delivery stream is <code>ACTIVE</code> and therefore ready
     * for data to be sent to it.
     * </p>
     * <p>
     * If the status of a delivery stream is <code>CREATING_FAILED</code>, this status doesn't change, and you can't
     * invoke <a>CreateDeliveryStream</a> again on it. However, you can invoke the <a>DeleteDeliveryStream</a> operation
     * to delete it. If the status is <code>DELETING_FAILED</code>, you can force deletion by invoking
     * <a>DeleteDeliveryStream</a> again but with <a>DeleteDeliveryStreamInput&#36AllowForceDelete</a> set to true.
     * </p>
     *
     * @param describeDeliveryStreamRequest
     * @return A Java Future containing the result of the DescribeDeliveryStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.DescribeDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/DescribeDeliveryStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeliveryStreamResponse> describeDeliveryStream(
            DescribeDeliveryStreamRequest describeDeliveryStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDeliveryStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeliveryStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeliveryStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeliveryStreamRequest, DescribeDeliveryStreamResponse>()
                            .withOperationName("DescribeDeliveryStream")
                            .withMarshaller(new DescribeDeliveryStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDeliveryStreamRequest));
            CompletableFuture<DescribeDeliveryStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your delivery streams in alphabetical order of their names.
     * </p>
     * <p>
     * The number of delivery streams might be too large to return using a single call to
     * <code>ListDeliveryStreams</code>. You can limit the number of delivery streams returned, using the
     * <code>Limit</code> parameter. To determine whether there are more delivery streams to list, check the value of
     * <code>HasMoreDeliveryStreams</code> in the output. If there are more delivery streams to list, you can request
     * them by calling this operation again and setting the <code>ExclusiveStartDeliveryStreamName</code> parameter to
     * the name of the last delivery stream returned in the last call.
     * </p>
     *
     * @param listDeliveryStreamsRequest
     * @return A Java Future containing the result of the ListDeliveryStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.ListDeliveryStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/ListDeliveryStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeliveryStreamsResponse> listDeliveryStreams(
            ListDeliveryStreamsRequest listDeliveryStreamsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeliveryStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeliveryStreams");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeliveryStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeliveryStreamsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeliveryStreamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeliveryStreamsRequest, ListDeliveryStreamsResponse>()
                            .withOperationName("ListDeliveryStreams")
                            .withMarshaller(new ListDeliveryStreamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDeliveryStreamsRequest));
            CompletableFuture<ListDeliveryStreamsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified delivery stream. This operation has a limit of five transactions per second per
     * account.
     * </p>
     *
     * @param listTagsForDeliveryStreamRequest
     * @return A Java Future containing the result of the ListTagsForDeliveryStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>LimitExceededException You have already reached the limit for a requested resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.ListTagsForDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/ListTagsForDeliveryStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForDeliveryStreamResponse> listTagsForDeliveryStream(
            ListTagsForDeliveryStreamRequest listTagsForDeliveryStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForDeliveryStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForDeliveryStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForDeliveryStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForDeliveryStreamRequest, ListTagsForDeliveryStreamResponse>()
                            .withOperationName("ListTagsForDeliveryStream")
                            .withMarshaller(new ListTagsForDeliveryStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForDeliveryStreamRequest));
            CompletableFuture<ListTagsForDeliveryStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Writes a single data record into an Amazon Kinesis Data Firehose delivery stream. To write multiple data records
     * into a delivery stream, use <a>PutRecordBatch</a>. Applications using these operations are referred to as
     * producers.
     * </p>
     * <p>
     * By default, each delivery stream can take in up to 2,000 transactions per second, 5,000 records per second, or 5
     * MB per second. If you use <a>PutRecord</a> and <a>PutRecordBatch</a>, the limits are an aggregate across these
     * two operations for each delivery stream. For more information about limits and how to request an increase, see <a
     * href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Kinesis Data Firehose Limits</a>.
     * </p>
     * <p>
     * You must specify the name of the delivery stream and the data record when using <a>PutRecord</a>. The data record
     * consists of a data blob that can be up to 1,000 KiB in size, and any kind of data. For example, it can be a
     * segment from a log file, geographic location data, website clickstream data, and so on.
     * </p>
     * <p>
     * Kinesis Data Firehose buffers records before delivering them to the destination. To disambiguate the data blobs
     * at the destination, a common solution is to use delimiters in the data, such as a newline (<code>\n</code>) or
     * some other character unique within the data. This allows the consumer application to parse individual data items
     * when reading the data from the destination.
     * </p>
     * <p>
     * The <code>PutRecord</code> operation returns a <code>RecordId</code>, which is a unique string assigned to each
     * record. Producer applications can use this ID for purposes such as auditability and investigation.
     * </p>
     * <p>
     * If the <code>PutRecord</code> operation throws a <code>ServiceUnavailableException</code>, back off and retry. If
     * the exception persists, it is possible that the throughput limits have been exceeded for the delivery stream.
     * </p>
     * <p>
     * Data records sent to Kinesis Data Firehose are stored for 24 hours from the time they are added to a delivery
     * stream as it tries to send the records to the destination. If the destination is unreachable for more than 24
     * hours, the data is no longer available.
     * </p>
     * <important>
     * <p>
     * Don't concatenate two or more base64 strings to form the data fields of your records. Instead, concatenate the
     * raw data, then perform base64 encoding.
     * </p>
     * </important>
     *
     * @param putRecordRequest
     * @return A Java Future containing the result of the PutRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>InvalidKmsResourceException Kinesis Data Firehose throws this exception when an attempt to put
     *         records or to start or stop delivery stream encryption fails. This happens when the KMS service throws
     *         one of the following exception types: <code>AccessDeniedException</code>,
     *         <code>InvalidStateException</code>, <code>DisabledException</code>, or <code>NotFoundException</code>.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation. If you
     *         continue to see the exception, throughput limits for the delivery stream may have been exceeded. For more
     *         information about limits and how to request an increase, see <a
     *         href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Kinesis Data Firehose
     *         Limits</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.PutRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/PutRecord" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecordResponse> putRecord(PutRecordRequest putRecordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRecordRequest, PutRecordResponse>().withOperationName("PutRecord")
                            .withMarshaller(new PutRecordRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRecordRequest));
            CompletableFuture<PutRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per
     * producer than when writing single records. To write single data records into a delivery stream, use
     * <a>PutRecord</a>. Applications using these operations are referred to as producers.
     * </p>
     * <p>
     * For information about service quota, see <a
     * href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Kinesis Data Firehose Quota</a>.
     * </p>
     * <p>
     * Each <a>PutRecordBatch</a> request supports up to 500 records. Each record in the request can be as large as
     * 1,000 KB (before base64 encoding), up to a limit of 4 MB for the entire request. These limits cannot be changed.
     * </p>
     * <p>
     * You must specify the name of the delivery stream and the data record when using <a>PutRecord</a>. The data record
     * consists of a data blob that can be up to 1,000 KB in size, and any kind of data. For example, it could be a
     * segment from a log file, geographic location data, website clickstream data, and so on.
     * </p>
     * <p>
     * Kinesis Data Firehose buffers records before delivering them to the destination. To disambiguate the data blobs
     * at the destination, a common solution is to use delimiters in the data, such as a newline (<code>\n</code>) or
     * some other character unique within the data. This allows the consumer application to parse individual data items
     * when reading the data from the destination.
     * </p>
     * <p>
     * The <a>PutRecordBatch</a> response includes a count of failed records, <code>FailedPutCount</code>, and an array
     * of responses, <code>RequestResponses</code>. Even if the <a>PutRecordBatch</a> call succeeds, the value of
     * <code>FailedPutCount</code> may be greater than 0, indicating that there are records for which the operation
     * didn't succeed. Each entry in the <code>RequestResponses</code> array provides additional information about the
     * processed record. It directly correlates with a record in the request array using the same ordering, from the top
     * to the bottom. The response array always includes the same number of records as the request array.
     * <code>RequestResponses</code> includes both successfully and unsuccessfully processed records. Kinesis Data
     * Firehose tries to process all records in each <a>PutRecordBatch</a> request. A single record failure does not
     * stop the processing of subsequent records.
     * </p>
     * <p>
     * A successfully processed record includes a <code>RecordId</code> value, which is unique for the record. An
     * unsuccessfully processed record includes <code>ErrorCode</code> and <code>ErrorMessage</code> values.
     * <code>ErrorCode</code> reflects the type of error, and is one of the following values:
     * <code>ServiceUnavailableException</code> or <code>InternalFailure</code>. <code>ErrorMessage</code> provides more
     * detailed information about the error.
     * </p>
     * <p>
     * If there is an internal server error or a timeout, the write might have completed or it might have failed. If
     * <code>FailedPutCount</code> is greater than 0, retry the request, resending only those records that might have
     * failed processing. This minimizes the possible duplicate records and also reduces the total bytes sent (and
     * corresponding charges). We recommend that you handle any duplicates at the destination.
     * </p>
     * <p>
     * If <a>PutRecordBatch</a> throws <code>ServiceUnavailableException</code>, back off and retry. If the exception
     * persists, it is possible that the throughput limits have been exceeded for the delivery stream.
     * </p>
     * <p>
     * Data records sent to Kinesis Data Firehose are stored for 24 hours from the time they are added to a delivery
     * stream as it attempts to send the records to the destination. If the destination is unreachable for more than 24
     * hours, the data is no longer available.
     * </p>
     * <important>
     * <p>
     * Don't concatenate two or more base64 strings to form the data fields of your records. Instead, concatenate the
     * raw data, then perform base64 encoding.
     * </p>
     * </important>
     *
     * @param putRecordBatchRequest
     * @return A Java Future containing the result of the PutRecordBatch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>InvalidKmsResourceException Kinesis Data Firehose throws this exception when an attempt to put
     *         records or to start or stop delivery stream encryption fails. This happens when the KMS service throws
     *         one of the following exception types: <code>AccessDeniedException</code>,
     *         <code>InvalidStateException</code>, <code>DisabledException</code>, or <code>NotFoundException</code>.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation. If you
     *         continue to see the exception, throughput limits for the delivery stream may have been exceeded. For more
     *         information about limits and how to request an increase, see <a
     *         href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Kinesis Data Firehose
     *         Limits</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.PutRecordBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/PutRecordBatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecordBatchResponse> putRecordBatch(PutRecordBatchRequest putRecordBatchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecordBatch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRecordBatchResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRecordBatchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRecordBatchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRecordBatchRequest, PutRecordBatchResponse>()
                            .withOperationName("PutRecordBatch")
                            .withMarshaller(new PutRecordBatchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putRecordBatchRequest));
            CompletableFuture<PutRecordBatchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables server-side encryption (SSE) for the delivery stream.
     * </p>
     * <p>
     * This operation is asynchronous. It returns immediately. When you invoke it, Kinesis Data Firehose first sets the
     * encryption status of the stream to <code>ENABLING</code>, and then to <code>ENABLED</code>. The encryption status
     * of a delivery stream is the <code>Status</code> property in <a>DeliveryStreamEncryptionConfiguration</a>. If the
     * operation fails, the encryption status changes to <code>ENABLING_FAILED</code>. You can continue to read and
     * write data to your delivery stream while the encryption status is <code>ENABLING</code>, but the data is not
     * encrypted. It can take up to 5 seconds after the encryption status changes to <code>ENABLED</code> before all
     * records written to the delivery stream are encrypted. To find out whether a record or a batch of records was
     * encrypted, check the response elements <a>PutRecordOutput&#36Encrypted</a> and
     * <a>PutRecordBatchOutput&#36Encrypted</a>, respectively.
     * </p>
     * <p>
     * To check the encryption status of a delivery stream, use <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * Even if encryption is currently enabled for a delivery stream, you can still invoke this operation on it to
     * change the ARN of the CMK or both its type and ARN. If you invoke this method to change the CMK, and the old CMK
     * is of type <code>CUSTOMER_MANAGED_CMK</code>, Kinesis Data Firehose schedules the grant it had on the old CMK for
     * retirement. If the new CMK is of type <code>CUSTOMER_MANAGED_CMK</code>, Kinesis Data Firehose creates a grant
     * that enables it to use the new CMK to encrypt and decrypt data and to manage the grant.
     * </p>
     * <p>
     * If a delivery stream already has encryption enabled and then you invoke this operation to change the ARN of the
     * CMK or both its type and ARN and you get <code>ENABLING_FAILED</code>, this only means that the attempt to change
     * the CMK failed. In this case, encryption remains enabled with the old CMK.
     * </p>
     * <p>
     * If the encryption status of your delivery stream is <code>ENABLING_FAILED</code>, you can invoke this operation
     * again with a valid CMK. The CMK must be enabled and the key policy mustn't explicitly deny the permission for
     * Kinesis Data Firehose to invoke KMS encrypt and decrypt operations.
     * </p>
     * <p>
     * You can enable SSE for a delivery stream only if it's a delivery stream that uses <code>DirectPut</code> as its
     * source.
     * </p>
     * <p>
     * The <code>StartDeliveryStreamEncryption</code> and <code>StopDeliveryStreamEncryption</code> operations have a
     * combined limit of 25 calls per delivery stream per 24 hours. For example, you reach the limit if you call
     * <code>StartDeliveryStreamEncryption</code> 13 times and <code>StopDeliveryStreamEncryption</code> 12 times for
     * the same delivery stream in a 24-hour period.
     * </p>
     *
     * @param startDeliveryStreamEncryptionRequest
     * @return A Java Future containing the result of the StartDeliveryStreamEncryption operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>LimitExceededException You have already reached the limit for a requested resource.</li>
     *         <li>InvalidKmsResourceException Kinesis Data Firehose throws this exception when an attempt to put
     *         records or to start or stop delivery stream encryption fails. This happens when the KMS service throws
     *         one of the following exception types: <code>AccessDeniedException</code>,
     *         <code>InvalidStateException</code>, <code>DisabledException</code>, or <code>NotFoundException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.StartDeliveryStreamEncryption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/StartDeliveryStreamEncryption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeliveryStreamEncryptionResponse> startDeliveryStreamEncryption(
            StartDeliveryStreamEncryptionRequest startDeliveryStreamEncryptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startDeliveryStreamEncryptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeliveryStreamEncryption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeliveryStreamEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeliveryStreamEncryptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDeliveryStreamEncryptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeliveryStreamEncryptionRequest, StartDeliveryStreamEncryptionResponse>()
                            .withOperationName("StartDeliveryStreamEncryption")
                            .withMarshaller(new StartDeliveryStreamEncryptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startDeliveryStreamEncryptionRequest));
            CompletableFuture<StartDeliveryStreamEncryptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables server-side encryption (SSE) for the delivery stream.
     * </p>
     * <p>
     * This operation is asynchronous. It returns immediately. When you invoke it, Kinesis Data Firehose first sets the
     * encryption status of the stream to <code>DISABLING</code>, and then to <code>DISABLED</code>. You can continue to
     * read and write data to your stream while its status is <code>DISABLING</code>. It can take up to 5 seconds after
     * the encryption status changes to <code>DISABLED</code> before all records written to the delivery stream are no
     * longer subject to encryption. To find out whether a record or a batch of records was encrypted, check the
     * response elements <a>PutRecordOutput&#36Encrypted</a> and <a>PutRecordBatchOutput&#36Encrypted</a>, respectively.
     * </p>
     * <p>
     * To check the encryption state of a delivery stream, use <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * If SSE is enabled using a customer managed CMK and then you invoke <code>StopDeliveryStreamEncryption</code>,
     * Kinesis Data Firehose schedules the related KMS grant for retirement and then retires it after it ensures that it
     * is finished delivering records to the destination.
     * </p>
     * <p>
     * The <code>StartDeliveryStreamEncryption</code> and <code>StopDeliveryStreamEncryption</code> operations have a
     * combined limit of 25 calls per delivery stream per 24 hours. For example, you reach the limit if you call
     * <code>StartDeliveryStreamEncryption</code> 13 times and <code>StopDeliveryStreamEncryption</code> 12 times for
     * the same delivery stream in a 24-hour period.
     * </p>
     *
     * @param stopDeliveryStreamEncryptionRequest
     * @return A Java Future containing the result of the StopDeliveryStreamEncryption operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>LimitExceededException You have already reached the limit for a requested resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.StopDeliveryStreamEncryption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/StopDeliveryStreamEncryption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopDeliveryStreamEncryptionResponse> stopDeliveryStreamEncryption(
            StopDeliveryStreamEncryptionRequest stopDeliveryStreamEncryptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDeliveryStreamEncryptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDeliveryStreamEncryption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopDeliveryStreamEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopDeliveryStreamEncryptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopDeliveryStreamEncryptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopDeliveryStreamEncryptionRequest, StopDeliveryStreamEncryptionResponse>()
                            .withOperationName("StopDeliveryStreamEncryption")
                            .withMarshaller(new StopDeliveryStreamEncryptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopDeliveryStreamEncryptionRequest));
            CompletableFuture<StopDeliveryStreamEncryptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tags for the specified delivery stream. A tag is a key-value pair that you can define and assign
     * to Amazon Web Services resources. If you specify a tag that already exists, the tag value is replaced with the
     * value that you specify in the request. Tags are metadata. For example, you can add friendly names and
     * descriptions or other types of information that can help you distinguish the delivery stream. For more
     * information about tags, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * Each delivery stream can have up to 50 tags.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second per account.
     * </p>
     *
     * @param tagDeliveryStreamRequest
     * @return A Java Future containing the result of the TagDeliveryStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>LimitExceededException You have already reached the limit for a requested resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.TagDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/TagDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagDeliveryStreamResponse> tagDeliveryStream(TagDeliveryStreamRequest tagDeliveryStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagDeliveryStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TagDeliveryStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagDeliveryStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagDeliveryStreamRequest, TagDeliveryStreamResponse>()
                            .withOperationName("TagDeliveryStream")
                            .withMarshaller(new TagDeliveryStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagDeliveryStreamRequest));
            CompletableFuture<TagDeliveryStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified delivery stream. Removed tags are deleted, and you can't recover them after this
     * operation successfully completes.
     * </p>
     * <p>
     * If you specify a tag that doesn't exist, the operation ignores it.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second per account.
     * </p>
     *
     * @param untagDeliveryStreamRequest
     * @return A Java Future containing the result of the UntagDeliveryStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>LimitExceededException You have already reached the limit for a requested resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.UntagDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/UntagDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagDeliveryStreamResponse> untagDeliveryStream(
            UntagDeliveryStreamRequest untagDeliveryStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagDeliveryStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UntagDeliveryStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagDeliveryStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagDeliveryStreamRequest, UntagDeliveryStreamResponse>()
                            .withOperationName("UntagDeliveryStream")
                            .withMarshaller(new UntagDeliveryStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagDeliveryStreamRequest));
            CompletableFuture<UntagDeliveryStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified destination of the specified delivery stream.
     * </p>
     * <p>
     * Use this operation to change the destination type (for example, to replace the Amazon S3 destination with Amazon
     * Redshift) or change the parameters associated with a destination (for example, to change the bucket name of the
     * Amazon S3 destination). The update might not occur immediately. The target delivery stream remains active while
     * the configurations are updated, so data writes to the delivery stream can continue during this process. The
     * updated configurations are usually effective within a few minutes.
     * </p>
     * <p>
     * Switching between Amazon ES and other services is not supported. For an Amazon ES destination, you can only
     * update to another Amazon ES destination.
     * </p>
     * <p>
     * If the destination type is the same, Kinesis Data Firehose merges the configuration parameters specified with the
     * destination configuration that already exists on the delivery stream. If any of the parameters are not specified
     * in the call, the existing values are retained. For example, in the Amazon S3 destination, if
     * <a>EncryptionConfiguration</a> is not specified, then the existing <code>EncryptionConfiguration</code> is
     * maintained on the destination.
     * </p>
     * <p>
     * If the destination type is not the same, for example, changing the destination from Amazon S3 to Amazon Redshift,
     * Kinesis Data Firehose does not merge any parameters. In this case, all parameters must be specified.
     * </p>
     * <p>
     * Kinesis Data Firehose uses <code>CurrentDeliveryStreamVersionId</code> to avoid race conditions and conflicting
     * merges. This is a required field, and the service updates the configuration only if the existing configuration
     * has a version ID that matches. After the update is applied successfully, the version ID is updated, and can be
     * retrieved using <a>DescribeDeliveryStream</a>. Use the new version ID to set
     * <code>CurrentDeliveryStreamVersionId</code> in the next call.
     * </p>
     *
     * @param updateDestinationRequest
     * @return A Java Future containing the result of the UpdateDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException The specified input parameter has a value that is not valid.</li>
     *         <li>ResourceInUseException The resource is already in use and not available for this operation.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ConcurrentModificationException Another modification has already happened. Fetch
     *         <code>VersionId</code> again and use it to update the destination.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FirehoseException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FirehoseAsyncClient.UpdateDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/UpdateDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDestinationResponse> updateDestination(UpdateDestinationRequest updateDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDestinationRequest, UpdateDestinationResponse>()
                            .withOperationName("UpdateDestination")
                            .withMarshaller(new UpdateDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDestinationRequest));
            CompletableFuture<UpdateDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final FirehoseServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FirehoseException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKMSResourceException")
                                .exceptionBuilderSupplier(InvalidKmsResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
