/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspace;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.finspace.internal.FinspaceServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.finspace.model.AccessDeniedException;
import software.amazon.awssdk.services.finspace.model.ConflictException;
import software.amazon.awssdk.services.finspace.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxChangesetRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxChangesetResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxClusterRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxClusterResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxUserRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxUserResponse;
import software.amazon.awssdk.services.finspace.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxClusterRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxClusterResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxUserRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxUserResponse;
import software.amazon.awssdk.services.finspace.model.FinspaceException;
import software.amazon.awssdk.services.finspace.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.GetKxChangesetRequest;
import software.amazon.awssdk.services.finspace.model.GetKxChangesetResponse;
import software.amazon.awssdk.services.finspace.model.GetKxClusterRequest;
import software.amazon.awssdk.services.finspace.model.GetKxClusterResponse;
import software.amazon.awssdk.services.finspace.model.GetKxConnectionStringRequest;
import software.amazon.awssdk.services.finspace.model.GetKxConnectionStringResponse;
import software.amazon.awssdk.services.finspace.model.GetKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.GetKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.GetKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.GetKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.GetKxUserRequest;
import software.amazon.awssdk.services.finspace.model.GetKxUserResponse;
import software.amazon.awssdk.services.finspace.model.InternalServerException;
import software.amazon.awssdk.services.finspace.model.InvalidRequestException;
import software.amazon.awssdk.services.finspace.model.LimitExceededException;
import software.amazon.awssdk.services.finspace.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.finspace.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.finspace.model.ListKxChangesetsRequest;
import software.amazon.awssdk.services.finspace.model.ListKxChangesetsResponse;
import software.amazon.awssdk.services.finspace.model.ListKxClusterNodesRequest;
import software.amazon.awssdk.services.finspace.model.ListKxClusterNodesResponse;
import software.amazon.awssdk.services.finspace.model.ListKxClustersRequest;
import software.amazon.awssdk.services.finspace.model.ListKxClustersResponse;
import software.amazon.awssdk.services.finspace.model.ListKxDatabasesRequest;
import software.amazon.awssdk.services.finspace.model.ListKxDatabasesResponse;
import software.amazon.awssdk.services.finspace.model.ListKxEnvironmentsRequest;
import software.amazon.awssdk.services.finspace.model.ListKxEnvironmentsResponse;
import software.amazon.awssdk.services.finspace.model.ListKxUsersRequest;
import software.amazon.awssdk.services.finspace.model.ListKxUsersResponse;
import software.amazon.awssdk.services.finspace.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.finspace.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.finspace.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.finspace.model.ResourceNotFoundException;
import software.amazon.awssdk.services.finspace.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.finspace.model.TagResourceRequest;
import software.amazon.awssdk.services.finspace.model.TagResourceResponse;
import software.amazon.awssdk.services.finspace.model.ThrottlingException;
import software.amazon.awssdk.services.finspace.model.UntagResourceRequest;
import software.amazon.awssdk.services.finspace.model.UntagResourceResponse;
import software.amazon.awssdk.services.finspace.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterCodeConfigurationRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterCodeConfigurationResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterDatabasesRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterDatabasesResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentNetworkRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentNetworkResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxUserRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxUserResponse;
import software.amazon.awssdk.services.finspace.model.ValidationException;
import software.amazon.awssdk.services.finspace.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxChangesetRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxClusterRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxUserRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxClusterRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxUserRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxChangesetRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxClusterRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxConnectionStringRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxUserRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxChangesetsRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxClusterNodesRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxClustersRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxDatabasesRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxUsersRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxClusterCodeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxClusterDatabasesRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxEnvironmentNetworkRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxUserRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FinspaceClient}.
 *
 * @see FinspaceClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFinspaceClient implements FinspaceClient {
    private static final Logger log = Logger.loggerFor(DefaultFinspaceClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final FinspaceServiceClientConfiguration serviceClientConfiguration;

    protected DefaultFinspaceClient(FinspaceServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a new FinSpace environment.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use Service Quotas to request a service quota increase.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ThrottlingException,
            ServiceQuotaExceededException, LimitExceededException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");

            return clientHandler.execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                    .withOperationName("CreateEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a changeset for a kdb database. A changeset allows you to add and delete existing files by using an
     * ordered list of change requests.
     * </p>
     *
     * @param createKxChangesetRequest
     * @return Result of the CreateKxChangeset operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.CreateKxChangeset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxChangeset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateKxChangesetResponse createKxChangeset(CreateKxChangesetRequest createKxChangesetRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, ConflictException, LimitExceededException, AwsServiceException, SdkClientException,
            FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKxChangesetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKxChangesetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxChangesetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxChangesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxChangeset");

            return clientHandler.execute(new ClientExecutionParams<CreateKxChangesetRequest, CreateKxChangesetResponse>()
                    .withOperationName("CreateKxChangeset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createKxChangesetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKxChangesetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new kdb cluster.
     * </p>
     *
     * @param createKxClusterRequest
     * @return Result of the CreateKxCluster operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.CreateKxCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateKxClusterResponse createKxCluster(CreateKxClusterRequest createKxClusterRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, LimitExceededException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKxClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKxClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateKxClusterRequest, CreateKxClusterResponse>()
                    .withOperationName("CreateKxCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createKxClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKxClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new kdb database in the environment.
     * </p>
     *
     * @param createKxDatabaseRequest
     * @return Result of the CreateKxDatabase operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceAlreadyExistsException
     *         The specified resource group already exists.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.CreateKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateKxDatabaseResponse createKxDatabase(CreateKxDatabaseRequest createKxDatabaseRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, ConflictException,
            ResourceAlreadyExistsException, ResourceNotFoundException, LimitExceededException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKxDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxDatabaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxDatabase");

            return clientHandler.execute(new ClientExecutionParams<CreateKxDatabaseRequest, CreateKxDatabaseResponse>()
                    .withOperationName("CreateKxDatabase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createKxDatabaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKxDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a managed kdb environment for the account.
     * </p>
     *
     * @param createKxEnvironmentRequest
     * @return Result of the CreateKxEnvironment operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use Service Quotas to request a service quota increase.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.CreateKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateKxEnvironmentResponse createKxEnvironment(CreateKxEnvironmentRequest createKxEnvironmentRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ThrottlingException,
            ServiceQuotaExceededException, LimitExceededException, ConflictException, AwsServiceException, SdkClientException,
            FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateKxEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxEnvironment");

            return clientHandler.execute(new ClientExecutionParams<CreateKxEnvironmentRequest, CreateKxEnvironmentResponse>()
                    .withOperationName("CreateKxEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createKxEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKxEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a user in FinSpace kdb environment with an associated IAM role.
     * </p>
     *
     * @param createKxUserRequest
     * @return Result of the CreateKxUser operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceAlreadyExistsException
     *         The specified resource group already exists.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.CreateKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateKxUserResponse createKxUser(CreateKxUserRequest createKxUserRequest) throws ResourceNotFoundException,
            InternalServerException, ThrottlingException, AccessDeniedException, ValidationException,
            ResourceAlreadyExistsException, LimitExceededException, ConflictException, AwsServiceException, SdkClientException,
            FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKxUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxUser");

            return clientHandler.execute(new ClientExecutionParams<CreateKxUserRequest, CreateKxUserResponse>()
                    .withOperationName("CreateKxUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createKxUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKxUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an FinSpace environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");

            return clientHandler.execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                    .withOperationName("DeleteEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a kdb cluster.
     * </p>
     *
     * @param deleteKxClusterRequest
     * @return Result of the DeleteKxCluster operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.DeleteKxCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteKxClusterResponse deleteKxCluster(DeleteKxClusterRequest deleteKxClusterRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, LimitExceededException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKxClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteKxClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxCluster");

            return clientHandler.execute(new ClientExecutionParams<DeleteKxClusterRequest, DeleteKxClusterResponse>()
                    .withOperationName("DeleteKxCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteKxClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKxClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified database and all of its associated data. This action is irreversible. You must copy any
     * data out of the database before deleting it if the data is to be retained.
     * </p>
     *
     * @param deleteKxDatabaseRequest
     * @return Result of the DeleteKxDatabase operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.DeleteKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteKxDatabaseResponse deleteKxDatabase(DeleteKxDatabaseRequest deleteKxDatabaseRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteKxDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxDatabaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxDatabase");

            return clientHandler.execute(new ClientExecutionParams<DeleteKxDatabaseRequest, DeleteKxDatabaseResponse>()
                    .withOperationName("DeleteKxDatabase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteKxDatabaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKxDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the kdb environment. This action is irreversible. Deleting a kdb environment will remove all the
     * associated data and any services running in it.
     * </p>
     *
     * @param deleteKxEnvironmentRequest
     * @return Result of the DeleteKxEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.DeleteKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteKxEnvironmentResponse deleteKxEnvironment(DeleteKxEnvironmentRequest deleteKxEnvironmentRequest)
            throws ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteKxEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxEnvironment");

            return clientHandler.execute(new ClientExecutionParams<DeleteKxEnvironmentRequest, DeleteKxEnvironmentResponse>()
                    .withOperationName("DeleteKxEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteKxEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKxEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a user in the specified kdb environment.
     * </p>
     *
     * @param deleteKxUserRequest
     * @return Result of the DeleteKxUser operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.DeleteKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteKxUserResponse deleteKxUser(DeleteKxUserRequest deleteKxUserRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteKxUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxUser");

            return clientHandler.execute(new ClientExecutionParams<DeleteKxUserRequest, DeleteKxUserResponse>()
                    .withOperationName("DeleteKxUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteKxUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKxUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the FinSpace environment object.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnvironmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");

            return clientHandler.execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                    .withOperationName("GetEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a kdb changeset.
     * </p>
     *
     * @param getKxChangesetRequest
     * @return Result of the GetKxChangeset operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetKxChangeset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxChangeset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKxChangesetResponse getKxChangeset(GetKxChangesetRequest getKxChangesetRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKxChangesetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKxChangesetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxChangesetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxChangesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxChangeset");

            return clientHandler.execute(new ClientExecutionParams<GetKxChangesetRequest, GetKxChangesetResponse>()
                    .withOperationName("GetKxChangeset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getKxChangesetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKxChangesetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a kdb cluster.
     * </p>
     *
     * @param getKxClusterRequest
     * @return Result of the GetKxCluster operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetKxCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKxClusterResponse getKxCluster(GetKxClusterRequest getKxClusterRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, LimitExceededException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKxClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKxClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxCluster");

            return clientHandler.execute(new ClientExecutionParams<GetKxClusterRequest, GetKxClusterResponse>()
                    .withOperationName("GetKxCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getKxClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKxClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a connection string for a user to connect to a kdb cluster. You must call this API using the same role
     * that you have defined while creating a user.
     * </p>
     *
     * @param getKxConnectionStringRequest
     * @return Result of the GetKxConnectionString operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetKxConnectionString
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxConnectionString"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetKxConnectionStringResponse getKxConnectionString(GetKxConnectionStringRequest getKxConnectionStringRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKxConnectionStringResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetKxConnectionStringResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxConnectionStringRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxConnectionStringRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxConnectionString");

            return clientHandler.execute(new ClientExecutionParams<GetKxConnectionStringRequest, GetKxConnectionStringResponse>()
                    .withOperationName("GetKxConnectionString").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getKxConnectionStringRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKxConnectionStringRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns database information for the specified environment ID.
     * </p>
     *
     * @param getKxDatabaseRequest
     * @return Result of the GetKxDatabase operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKxDatabaseResponse getKxDatabase(GetKxDatabaseRequest getKxDatabaseRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKxDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxDatabaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxDatabase");

            return clientHandler.execute(new ClientExecutionParams<GetKxDatabaseRequest, GetKxDatabaseResponse>()
                    .withOperationName("GetKxDatabase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getKxDatabaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKxDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves all the information for the specified kdb environment.
     * </p>
     *
     * @param getKxEnvironmentRequest
     * @return Result of the GetKxEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKxEnvironmentResponse getKxEnvironment(GetKxEnvironmentRequest getKxEnvironmentRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKxEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxEnvironment");

            return clientHandler.execute(new ClientExecutionParams<GetKxEnvironmentRequest, GetKxEnvironmentResponse>()
                    .withOperationName("GetKxEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getKxEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKxEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified kdb user.
     * </p>
     *
     * @param getKxUserRequest
     * @return Result of the GetKxUser operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.GetKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKxUserResponse getKxUser(GetKxUserRequest getKxUserRequest) throws InternalServerException, ThrottlingException,
            AccessDeniedException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKxUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxUser");

            return clientHandler.execute(new ClientExecutionParams<GetKxUserRequest, GetKxUserResponse>()
                    .withOperationName("GetKxUser").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getKxUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKxUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A list of all of your FinSpace environments.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");

            return clientHandler.execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                    .withOperationName("ListEnvironments").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEnvironmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the changesets for a database.
     * </p>
     *
     * @param listKxChangesetsRequest
     * @return Result of the ListKxChangesets operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListKxChangesets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxChangesets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKxChangesetsResponse listKxChangesets(ListKxChangesetsRequest listKxChangesetsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKxChangesetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKxChangesetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxChangesetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxChangesetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxChangesets");

            return clientHandler.execute(new ClientExecutionParams<ListKxChangesetsRequest, ListKxChangesetsResponse>()
                    .withOperationName("ListKxChangesets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listKxChangesetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKxChangesetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the nodes in a kdb cluster.
     * </p>
     *
     * @param listKxClusterNodesRequest
     * @return Result of the ListKxClusterNodes operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListKxClusterNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxClusterNodes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListKxClusterNodesResponse listKxClusterNodes(ListKxClusterNodesRequest listKxClusterNodesRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, LimitExceededException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKxClusterNodesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListKxClusterNodesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxClusterNodesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxClusterNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxClusterNodes");

            return clientHandler.execute(new ClientExecutionParams<ListKxClusterNodesRequest, ListKxClusterNodesResponse>()
                    .withOperationName("ListKxClusterNodes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listKxClusterNodesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKxClusterNodesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of clusters.
     * </p>
     *
     * @param listKxClustersRequest
     * @return Result of the ListKxClusters operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListKxClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKxClustersResponse listKxClusters(ListKxClustersRequest listKxClustersRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, LimitExceededException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKxClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKxClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxClusters");

            return clientHandler.execute(new ClientExecutionParams<ListKxClustersRequest, ListKxClustersResponse>()
                    .withOperationName("ListKxClusters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listKxClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKxClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the databases in the kdb environment.
     * </p>
     *
     * @param listKxDatabasesRequest
     * @return Result of the ListKxDatabases operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListKxDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKxDatabasesResponse listKxDatabases(ListKxDatabasesRequest listKxDatabasesRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKxDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKxDatabasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxDatabasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxDatabases");

            return clientHandler.execute(new ClientExecutionParams<ListKxDatabasesRequest, ListKxDatabasesResponse>()
                    .withOperationName("ListKxDatabases").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listKxDatabasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKxDatabasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of kdb environments created in an account.
     * </p>
     *
     * @param listKxEnvironmentsRequest
     * @return Result of the ListKxEnvironments operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListKxEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListKxEnvironmentsResponse listKxEnvironments(ListKxEnvironmentsRequest listKxEnvironmentsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKxEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListKxEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxEnvironments");

            return clientHandler.execute(new ClientExecutionParams<ListKxEnvironmentsRequest, ListKxEnvironmentsResponse>()
                    .withOperationName("ListKxEnvironments").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listKxEnvironmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKxEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the users in a kdb environment.
     * </p>
     *
     * @param listKxUsersRequest
     * @return Result of the ListKxUsers operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListKxUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKxUsersResponse listKxUsers(ListKxUsersRequest listKxUsersRequest) throws ResourceNotFoundException,
            InternalServerException, ThrottlingException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKxUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKxUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxUsers");

            return clientHandler.execute(new ClientExecutionParams<ListKxUsersRequest, ListKxUsersResponse>()
                    .withOperationName("ListKxUsers").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listKxUsersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKxUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A list of all tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws InvalidRequestException
     *         The request is invalid. Something is wrong with the input to the request.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds metadata tags to a FinSpace resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws InvalidRequestException
     *         The request is invalid. Something is wrong with the input to the request.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes metadata tags from a FinSpace resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws InvalidRequestException
     *         The request is invalid. Something is wrong with the input to the request.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update your FinSpace environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");

            return clientHandler.execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                    .withOperationName("UpdateEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to update code configuration on a running cluster. By using this API you can update the code, the
     * initialization script path, and the command line arguments for a specific cluster. The configuration that you
     * want to update will override any existing configurations on the cluster.
     * </p>
     *
     * @param updateKxClusterCodeConfigurationRequest
     * @return Result of the UpdateKxClusterCodeConfiguration operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateKxClusterCodeConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxClusterCodeConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateKxClusterCodeConfigurationResponse updateKxClusterCodeConfiguration(
            UpdateKxClusterCodeConfigurationRequest updateKxClusterCodeConfigurationRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, LimitExceededException, ValidationException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKxClusterCodeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKxClusterCodeConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxClusterCodeConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateKxClusterCodeConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxClusterCodeConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateKxClusterCodeConfigurationRequest, UpdateKxClusterCodeConfigurationResponse>()
                            .withOperationName("UpdateKxClusterCodeConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateKxClusterCodeConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateKxClusterCodeConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the databases mounted on a kdb cluster, which includes the <code>changesetId</code> and all the dbPaths
     * to be cached. This API does not allow you to change a database name or add a database if you created a cluster
     * without one.
     * </p>
     * <p>
     * Using this API you can point a cluster to a different changeset and modify a list of partitions being cached.
     * </p>
     *
     * @param updateKxClusterDatabasesRequest
     * @return Result of the UpdateKxClusterDatabases operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateKxClusterDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxClusterDatabases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateKxClusterDatabasesResponse updateKxClusterDatabases(
            UpdateKxClusterDatabasesRequest updateKxClusterDatabasesRequest) throws InternalServerException, ThrottlingException,
            AccessDeniedException, LimitExceededException, ValidationException, ConflictException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKxClusterDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKxClusterDatabasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxClusterDatabasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxClusterDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxClusterDatabases");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateKxClusterDatabasesRequest, UpdateKxClusterDatabasesResponse>()
                            .withOperationName("UpdateKxClusterDatabases").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateKxClusterDatabasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateKxClusterDatabasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates information for the given kdb database.
     * </p>
     *
     * @param updateKxDatabaseRequest
     * @return Result of the UpdateKxDatabase operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateKxDatabaseResponse updateKxDatabase(UpdateKxDatabaseRequest updateKxDatabaseRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateKxDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxDatabaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxDatabase");

            return clientHandler.execute(new ClientExecutionParams<UpdateKxDatabaseRequest, UpdateKxDatabaseResponse>()
                    .withOperationName("UpdateKxDatabase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateKxDatabaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateKxDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates information for the given kdb environment.
     * </p>
     *
     * @param updateKxEnvironmentRequest
     * @return Result of the UpdateKxEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateKxEnvironmentResponse updateKxEnvironment(UpdateKxEnvironmentRequest updateKxEnvironmentRequest)
            throws ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKxEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxEnvironment");

            return clientHandler.execute(new ClientExecutionParams<UpdateKxEnvironmentRequest, UpdateKxEnvironmentResponse>()
                    .withOperationName("UpdateKxEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateKxEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateKxEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates environment network to connect to your internal network by using a transit gateway. This API supports
     * request to create a transit gateway attachment from FinSpace VPC to your transit gateway ID and create a custom
     * Route-53 outbound resolvers.
     * </p>
     * <p>
     * Once you send a request to update a network, you cannot change it again. Network update might require termination
     * of any clusters that are running in the existing network.
     * </p>
     *
     * @param updateKxEnvironmentNetworkRequest
     * @return Result of the UpdateKxEnvironmentNetwork operation returned by the service.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateKxEnvironmentNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxEnvironmentNetwork"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateKxEnvironmentNetworkResponse updateKxEnvironmentNetwork(
            UpdateKxEnvironmentNetworkRequest updateKxEnvironmentNetworkRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, ConflictException,
            AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKxEnvironmentNetworkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKxEnvironmentNetworkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxEnvironmentNetworkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxEnvironmentNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxEnvironmentNetwork");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateKxEnvironmentNetworkRequest, UpdateKxEnvironmentNetworkResponse>()
                            .withOperationName("UpdateKxEnvironmentNetwork").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateKxEnvironmentNetworkRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateKxEnvironmentNetworkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the user details. You can only update the IAM role associated with a user.
     * </p>
     *
     * @param updateKxUserRequest
     * @return Result of the UpdateKxUser operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         One or more resources can't be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws LimitExceededException
     *         A service limit or quota is exceeded.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FinspaceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FinspaceClient.UpdateKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateKxUserResponse updateKxUser(UpdateKxUserRequest updateKxUserRequest) throws InternalServerException,
            ThrottlingException, AccessDeniedException, ResourceNotFoundException, ValidationException, LimitExceededException,
            ConflictException, AwsServiceException, SdkClientException, FinspaceException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateKxUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxUser");

            return clientHandler.execute(new ClientExecutionParams<UpdateKxUserRequest, UpdateKxUserResponse>()
                    .withOperationName("UpdateKxUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateKxUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateKxUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        FinspaceServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = FinspaceServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FinspaceException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build());
    }

    @Override
    public final FinspaceServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
